"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const assert_2 = require("assert");
const batch = require("../lib");
describe('Batch Compute Evironment', () => {
    let expectedManagedDefaultComputeProps;
    let defaultServiceRole;
    let stack;
    let vpc;
    beforeEach(() => {
        // GIVEN
        stack = new cdk.Stack();
        vpc = new ec2.Vpc(stack, 'test-vpc');
        defaultServiceRole = {
            ServiceRole: {
                'Fn::GetAtt': [
                    'testcomputeenvResourceServiceInstanceRole105069A5',
                    'Arn'
                ],
            },
        };
        expectedManagedDefaultComputeProps = (overrides) => {
            return {
                ComputeResources: {
                    AllocationStrategy: batch.AllocationStrategy.BEST_FIT,
                    InstanceRole: {
                        'Fn::GetAtt': [
                            'testcomputeenvInstanceProfileCBD87EAB',
                            'Arn'
                        ]
                    },
                    InstanceTypes: [
                        'optimal'
                    ],
                    MaxvCpus: 256,
                    MinvCpus: 0,
                    Subnets: [
                        {
                            Ref: 'testvpcPrivateSubnet1Subnet865FB50A'
                        },
                        {
                            Ref: 'testvpcPrivateSubnet2Subnet23D3396F'
                        }
                    ],
                    Type: batch.ComputeResourceType.ON_DEMAND,
                    ...overrides,
                }
            };
        };
    });
    describe('when validating props', () => {
        test('should deny setting compute resources when using type unmanaged', () => {
            // THEN
            assert_2.throws(() => {
                // WHEN
                new batch.ComputeEnvironment(stack, 'test-compute-env', {
                    managed: false,
                    computeResources: {
                        vpc
                    },
                });
            });
        });
        test('should deny if creating a managed environment with no provided compute resource props', () => {
            // THEN
            assert_2.throws(() => {
                // WHEN
                new batch.ComputeEnvironment(stack, 'test-compute-env', {
                    managed: true,
                });
            });
        });
    });
    describe('using spot resources', () => {
        test('should provide a spot fleet role if one is not given and allocationStrategy is BEST_FIT', () => {
            // WHEN
            new batch.ComputeEnvironment(stack, 'test-compute-env', {
                managed: true,
                computeResources: {
                    type: batch.ComputeResourceType.SPOT,
                    allocationStrategy: batch.AllocationStrategy.BEST_FIT,
                    vpc
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Batch::ComputeEnvironment', {
                Type: 'MANAGED',
                ...expectedManagedDefaultComputeProps({
                    Type: batch.ComputeResourceType.SPOT,
                    SpotIamFleetRole: {
                        'Fn::Join': [
                            '',
                            [
                                'arn',
                                {
                                    Ref: 'AWS::Partition'
                                },
                                'iam::',
                                {
                                    Ref: 'AWS::AccountId'
                                },
                                ':role/aws-service-role/spotfleet.amazonaws.com/AWSServiceRoleForEC2SpotFleet',
                            ],
                        ],
                    },
                }),
            }, assert_1.ResourcePart.Properties));
        });
        describe('with a bid percentage', () => {
            test('should deny my bid if set below 0', () => {
                // THEN
                assert_2.throws(() => {
                    // WHEN
                    new batch.ComputeEnvironment(stack, 'test-compute-env', {
                        managed: true,
                        computeResources: {
                            vpc,
                            type: batch.ComputeResourceType.SPOT,
                            bidPercentage: -1,
                        },
                    });
                });
            });
            test('should deny my bid if above 100', () => {
                // THEN
                assert_2.throws(() => {
                    // WHEN
                    new batch.ComputeEnvironment(stack, 'test-compute-env', {
                        managed: true,
                        computeResources: {
                            vpc,
                            type: batch.ComputeResourceType.SPOT,
                            bidPercentage: 101,
                        },
                    });
                });
            });
        });
    });
    describe('with properties specified', () => {
        test('renders the correct cloudformation properties', () => {
            // WHEN
            const props = {
                computeEnvironmentName: 'my-test-compute-env',
                computeResources: {
                    allocationStrategy: batch.AllocationStrategy.BEST_FIT,
                    vpc,
                    computeResourcesTags: new cdk.Tag('foo', 'bar'),
                    desiredvCpus: 1,
                    ec2KeyPair: 'my-key-pair',
                    image: new ecs.EcsOptimizedAmi({
                        generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
                        hardwareType: ecs.AmiHardwareType.STANDARD,
                    }),
                    instanceRole: new iam.CfnInstanceProfile(stack, 'Instance-Profile', {
                        roles: [new iam.Role(stack, 'Ecs-Instance-Role', {
                                assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
                                managedPolicies: [
                                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2ContainerServiceforEC2Role')
                                ]
                            }).roleName]
                    }).attrArn,
                    instanceTypes: [
                        ec2.InstanceType.of(ec2.InstanceClass.T2, ec2.InstanceSize.MICRO),
                    ],
                    maxvCpus: 4,
                    minvCpus: 1,
                    securityGroups: [
                        new ec2.SecurityGroup(stack, 'test-sg', {
                            vpc,
                            allowAllOutbound: true,
                        }),
                    ],
                    type: batch.ComputeResourceType.ON_DEMAND,
                    vpcSubnets: {
                        subnetType: ec2.SubnetType.PRIVATE,
                    },
                },
                enabled: false,
                managed: true,
            };
            new batch.ComputeEnvironment(stack, 'test-compute-env', props);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Batch::ComputeEnvironment', {
                ComputeEnvironmentName: 'my-test-compute-env',
                Type: 'MANAGED',
                State: 'DISABLED',
                ServiceRole: {
                    'Fn::GetAtt': [
                        'testcomputeenvResourceServiceInstanceRole105069A5',
                        'Arn'
                    ],
                },
                ComputeResources: {
                    AllocationStrategy: batch.AllocationStrategy.BEST_FIT,
                    DesiredvCpus: props.computeResources.desiredvCpus,
                    Ec2KeyPair: props.computeResources.ec2KeyPair,
                    ImageId: {
                        Ref: 'SsmParameterValueawsserviceecsoptimizedamiamazonlinux2recommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter'
                    },
                    InstanceRole: {
                        'Fn::GetAtt': [
                            props.computeResources.instanceRole ? 'InstanceProfile' : '',
                            'Arn'
                        ]
                    },
                    InstanceTypes: [
                        props.computeResources.instanceTypes ? props.computeResources.instanceTypes[0].toString() : '',
                    ],
                    MaxvCpus: props.computeResources.maxvCpus,
                    MinvCpus: props.computeResources.minvCpus,
                    SecurityGroupIds: [
                        {
                            'Fn::GetAtt': [
                                'testsg872EB48A',
                                'GroupId'
                            ]
                        }
                    ],
                    Subnets: [
                        {
                            Ref: `${vpc.node.uniqueId}PrivateSubnet1Subnet865FB50A`
                        },
                        {
                            Ref: `${vpc.node.uniqueId}PrivateSubnet2Subnet23D3396F`
                        }
                    ],
                    Tags: {
                        key: 'foo',
                        props: {},
                        defaultPriority: 100,
                        value: 'bar'
                    },
                    Type: 'EC2'
                },
            }, assert_1.ResourcePart.Properties));
        });
        describe('with no allocation strategy specified', () => {
            test('should default to a best_fit strategy', () => {
                // WHEN
                new batch.ComputeEnvironment(stack, 'test-compute-env', {
                    managed: true,
                    computeResources: {
                        vpc,
                    },
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Batch::ComputeEnvironment', {
                    Type: 'MANAGED',
                    ServiceRole: {
                        'Fn::GetAtt': [
                            'testcomputeenvResourceServiceInstanceRole105069A5',
                            'Arn'
                        ],
                    },
                }, assert_1.ResourcePart.Properties));
            });
        });
        describe('with a min vcpu value', () => {
            test('should deny less than 0', () => {
                // THEN
                assert_2.throws(() => {
                    // WHEN
                    new batch.ComputeEnvironment(stack, 'test-compute-env', {
                        computeResources: {
                            vpc,
                            minvCpus: -1,
                        },
                    });
                });
            });
            test('cannot be greater than the max vcpu value', () => {
                // THEN
                assert_2.throws(() => {
                    // WHEN
                    new batch.ComputeEnvironment(stack, 'test-compute-env', {
                        computeResources: {
                            vpc,
                            minvCpus: 2,
                            maxvCpus: 1,
                        },
                    });
                });
            });
        });
        describe('with no min vcpu value provided', () => {
            test('should default to 0', () => {
                // WHEN
                new batch.ComputeEnvironment(stack, 'test-compute-env', {
                    managed: true,
                    computeResources: {
                        vpc,
                    },
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Batch::ComputeEnvironment', {
                    ...defaultServiceRole,
                    ...expectedManagedDefaultComputeProps({
                        MinvCpus: 0,
                    }),
                }, assert_1.ResourcePart.Properties));
            });
        });
        describe('with no max vcpu value provided', () => {
            test('should default to 256', () => {
                // WHEN
                new batch.ComputeEnvironment(stack, 'test-compute-env', {
                    managed: true,
                    computeResources: {
                        vpc,
                    },
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Batch::ComputeEnvironment', {
                    ...expectedManagedDefaultComputeProps({
                        MaxvCpus: 256,
                    }),
                }, assert_1.ResourcePart.Properties));
            });
        });
        describe('with no instance role specified', () => {
            test('should generate a role for me', () => {
                // WHEN
                new batch.ComputeEnvironment(stack, 'test-compute-env', {
                    managed: true,
                    computeResources: {
                        vpc,
                    },
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResource('AWS::Batch::ComputeEnvironment'));
                assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role'));
            });
        });
        describe('with no instance type defined', () => {
            test('should default to optimal matching', () => {
                // WHEN
                new batch.ComputeEnvironment(stack, 'test-compute-env', {
                    managed: true,
                    computeResources: {
                        vpc,
                    },
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Batch::ComputeEnvironment', {
                    ...expectedManagedDefaultComputeProps({
                        InstanceTypes: ['optimal'],
                    }),
                }, assert_1.ResourcePart.Properties));
            });
        });
        describe('with no type specified', () => {
            test('should default to EC2', () => {
                // WHEN
                new batch.ComputeEnvironment(stack, 'test-compute-env', {
                    managed: true,
                    computeResources: {
                        vpc,
                    },
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Batch::ComputeEnvironment', {
                    ...expectedManagedDefaultComputeProps({
                        Type: batch.ComputeResourceType.ON_DEMAND,
                    }),
                }, assert_1.ResourcePart.Properties));
            });
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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