"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AutomationStepSimulation = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const __1 = require("..");
const cancellation_exception_1 = require("../domain/cancellation-exception");
const data_type_1 = require("../domain/data-type");
const non_retriable_exception_1 = require("../domain/non-retriable-exception");
const response_code_1 = require("../domain/response-code");
const approve_hook_1 = require("../interface/approve-hook");
const aws_invoker_1 = require("../interface/aws-invoker");
const observer_1 = require("../interface/observer");
const pause_hook_1 = require("../interface/pause-hook");
const api_run_command_hook_1 = require("../interface/run-command-hook/api-run-command-hook");
const sleep_hook_1 = require("../interface/sleep-hook");
const webhook_1 = require("../interface/webhook");
const approve_simulation_1 = require("./automation/approve-simulation");
const assert_aws_resource_simulation_1 = require("./automation/assert-aws-resource-simulation");
const aws_api_simulation_1 = require("./automation/aws-api-simulation");
const branch_simulation_1 = require("./automation/branch-simulation");
const change_instance_state_simulation_1 = require("./automation/change-instance-state-simulation");
const copy_image_simulation_1 = require("./automation/copy-image-simulation");
const create_image_simulation_1 = require("./automation/create-image-simulation");
const create_stack_simulation_1 = require("./automation/create-stack-simulation");
const create_tags_simulation_1 = require("./automation/create-tags-simulation");
const delete_image_simulation_1 = require("./automation/delete-image-simulation");
const delete_stack_simulation_1 = require("./automation/delete-stack-simulation");
const execute_automation_simulation_1 = require("./automation/execute-automation-simulation");
const execute_script_simulation_1 = require("./automation/execute-script-simulation");
const execute_state_machine_simulation_1 = require("./automation/execute-state-machine-simulation");
const invoke_lambda_function_simulation_1 = require("./automation/invoke-lambda-function-simulation");
const invoke_webhook_simulation_1 = require("./automation/invoke-webhook-simulation");
const pause_simulation_1 = require("./automation/pause-simulation");
const run_command_simulation_1 = require("./automation/run-command-simulation");
const run_instance_simulation_1 = require("./automation/run-instance-simulation");
const sleep_simulation_1 = require("./automation/sleep-simulation");
const wait_for_resource_simulation_1 = require("./automation/wait-for-resource-simulation");
// eslint-disable-next-line
const jp = require('jsonpath');
class AutomationStepSimulation {
    constructor(step, props) {
        const awsInvoker = props.awsInvoker ?? new aws_invoker_1.ReflectiveAwsInvoker();
        const sleepHook = props.sleepHook ?? new sleep_hook_1.SleepImpl();
        this.props = {
            sleepHook,
            awsInvoker,
            pauseHook: props.pauseHook ?? new pause_hook_1.PauseImpl(),
            inputObserver: props.inputObserver ?? new observer_1.NoopObserver(),
            outputObserver: props.outputObserver ?? new observer_1.NoopObserver(),
            approveHook: props.approveHook ?? new approve_hook_1.ApproveImpl(),
            parameterResolver: props.parameterResolver ?? { resolve: (x) => x },
            webhook: props.webhook ?? new webhook_1.WebhookImpl(),
            runCommandHook: props.runCommandHook ?? new api_run_command_hook_1.ApiRunCommandHook(awsInvoker, sleepHook),
            executeAutomationHook: props.executeAutomationHook ?? new __1.ApiExecuteAutomationHook(awsInvoker, sleepHook),
        };
        this.step = step;
    }
    /**
     * Invokes the current step on the input and will return a SimulationResult.
     * @param inputs must contain all of the inputs declared by the current step.
     * @returns a SimulationResult with the step outputs in the case of success or stacktrace in the case of failure.
     */
    invoke(inputs) {
        console.log('Executing Step: ' + this.step.name);
        return this.invokeWithFallback(inputs);
    }
    /**
     * If fallback/retries are specified for this step, the retry or skip logic is handled in this function.
     */
    invokeWithFallback(allInputs) {
        try {
            const filteredInputs = this.filterInputs(allInputs);
            this.step.inputObserver.accept(filteredInputs);
            const response = this.executeWithRetries(filteredInputs);
            this.step.outputObserver.accept(response);
            Object.assign(allInputs, response);
            const nextStep = this.mySimulation().nextStep(allInputs);
            if (nextStep && !this.step.isEnd) {
                const nextStepRes = new AutomationStepSimulation(nextStep, this.props).invoke(allInputs);
                if (nextStepRes.responseCode != response_code_1.ResponseCode.SUCCESS) {
                    return {
                        responseCode: nextStepRes.responseCode,
                        stackTrace: nextStepRes.stackTrace,
                        executedSteps: this.prependSelf(nextStepRes.executedSteps),
                    };
                }
                else {
                    return {
                        responseCode: response_code_1.ResponseCode.SUCCESS,
                        outputs: { ...response, ...nextStepRes.outputs },
                        executedSteps: this.prependSelf(nextStepRes.executedSteps),
                    };
                }
            }
            else {
                return {
                    responseCode: response_code_1.ResponseCode.SUCCESS,
                    outputs: response,
                    executedSteps: [this.step.name],
                };
            }
            // JSII does not differentiate between exceptions. For the API we therefore return a ResponseCode
        }
        catch (error) {
            if (error instanceof non_retriable_exception_1.NonRetriableException) {
                throw error;
            }
            else if (error instanceof cancellation_exception_1.CancellationException) {
                if (this.step.onCancel.stepToInvoke(this.step)) {
                    const onCancelStepName = this.step.onCancel.stepToInvoke(this.step);
                    console.log('Step cancelled: ' + this.step.name + '. Executing onCancel step ' + onCancelStepName);
                    const onCancelStep = this.step.allStepsInExecution?.filter(s => s.name == this.step.onCancel.stepToInvoke(this.step));
                    if (onCancelStep == undefined || onCancelStep.length != 1) {
                        throw new Error('Could not find cancellation step ' + onCancelStepName);
                    }
                    return new AutomationStepSimulation(onCancelStep[0], this.props).invoke(allInputs);
                }
                else {
                    return {
                        responseCode: response_code_1.ResponseCode.CANCELED,
                        stackTrace: error.message + '\n' + error.stack,
                        executedSteps: [this.step.name],
                    };
                }
            }
            else {
                if (this.step.onFailure.stepToInvoke(this.step)) {
                    const onFailureStepName = this.step.onFailure.stepToInvoke(this.step);
                    console.log('Step failed: ' + this.step.name + '. Executing onFailure step ' + onFailureStepName);
                    const onFailureStep = this.step.allStepsInExecution?.filter(s => s.name == this.step.onFailure.stepToInvoke(this.step));
                    if (onFailureStep == undefined || onFailureStep.length != 1) {
                        throw new Error('Could not find onFailure step ' + onFailureStepName);
                    }
                    return new AutomationStepSimulation(onFailureStep[0], this.props).invoke(allInputs);
                }
                else {
                    return {
                        responseCode: response_code_1.ResponseCode.FAILED,
                        outputs: {},
                        stackTrace: error.message + '\n' + error.stack,
                        executedSteps: [this.step.name],
                    };
                }
            }
        }
    }
    /**
     * This function receives all of the available inputs and returns the subset of inputs that were requested.
     * If any input requested in the current step is not available in the received inputs, this will throw.
     * @param inputs all available inputs for the currently executing step
     * @returns the subset of inputs which are requested by the current step
     */
    filterInputs(inputs) {
        const foundAll = this.step.listInputs().every(required => Object.keys(inputs).includes(required));
        if (!foundAll) {
            throw new non_retriable_exception_1.NonRetriableException(`Not all inputs required were provided. Required: ${this.step.listInputs()}. Provided: ${JSON.stringify(inputs)}`);
        }
        const filteredEntries = Object.entries(inputs).filter(entry => {
            const [key] = entry;
            return key.startsWith('global:') || key.startsWith('automation:') || this.step.listInputs().includes(key);
        });
        return Object.fromEntries(filteredEntries);
    }
    /**
     * Adds this step name to the list of executed steps.
     * Since the steps are invoked as a chain, the current step is prepended as invocation stack is popped.
     */
    prependSelf(subsequentSteps) {
        return [this.step.name, ...subsequentSteps];
    }
    executeWithRetries(inputs) {
        let exception;
        for (let i = 0; i < this.step.maxAttempts; i++) {
            try {
                console.log('Invoking step ' + this.step.name + ': attempt ' + i);
                return this.tryExecute(inputs);
            }
            catch (error) {
                if (error instanceof non_retriable_exception_1.NonRetriableException) {
                    throw error;
                }
                else {
                    exception = error;
                }
            }
        }
        console.log('Exception occurred in this step: ' + this.step.name);
        throw exception;
    }
    tryExecute(inputs) {
        const start = Date.now();
        console.log(`Execute step ${this.step.name} start: ${start}`);
        const response = this.mySimulation().executeStep(inputs);
        console.log(`Execute step ${this.step.name} end: ${Date.now()}`);
        const relevantResponse = this.getSelectedResponse(response);
        this.checkExecutionTime(start);
        return relevantResponse;
    }
    mySimulation() {
        switch (this.step.action) {
            case 'aws:approve':
                return new approve_simulation_1.ApproveSimulation(this.step, this.props);
            case 'aws:assertAwsResourceProperty':
                return new assert_aws_resource_simulation_1.AssertAwsResourceSimulation(this.step, this.props);
            case 'aws:executeAwsApi':
                return new aws_api_simulation_1.AwsApiSimulation(this.step, this.props);
            case 'aws:branch':
                return new branch_simulation_1.BranchSimulation(this.step);
            case 'aws:changeInstanceState':
                return new change_instance_state_simulation_1.ChangeInstanceStateSimulation(this.step, this.props);
            case 'aws:copyImage':
                return new copy_image_simulation_1.CopyImageSimulation(this.step, this.props);
            case 'aws:createImage':
                return new create_image_simulation_1.CreateImageSimulation(this.step, this.props);
            case 'aws:createStack':
                return new create_stack_simulation_1.CreateStackSimulation(this.step, this.props);
            case 'aws:createTags':
                return new create_tags_simulation_1.CreateTagsSimulation(this.step, this.props);
            case 'aws:deleteImage':
                return new delete_image_simulation_1.DeleteImageSimulation(this.step, this.props);
            case 'aws:deleteStack':
                return new delete_stack_simulation_1.DeleteStackSimulation(this.step, this.props);
            case 'aws:executeAutomation':
                return new execute_automation_simulation_1.ExecuteAutomationSimulation(this.step, this.props);
            case 'aws:executeScript':
                return new execute_script_simulation_1.ExecuteScriptSimulation(this.step);
            case 'aws:executeStateMachine':
                return new execute_state_machine_simulation_1.ExecuteStateMachineSimulation(this.step, this.props);
            case 'aws:invokeLambdaFunction':
                return new invoke_lambda_function_simulation_1.InvokeLambdaFunctionSimulation(this.step, this.props);
            case 'aws:invokeWebhook':
                return new invoke_webhook_simulation_1.InvokeWebhookSimulation(this.step, this.props);
            case 'aws:pause':
                return new pause_simulation_1.PauseSimulation(this.step, this.props);
            case 'aws:runCommand':
                return new run_command_simulation_1.RunCommandSimulation(this.step, this.props);
            case 'aws:runInstances':
                return new run_instance_simulation_1.RunInstanceSimulation(this.step, this.props);
            case 'aws:sleep':
                return new sleep_simulation_1.SleepSimulation(this.step, this.props);
            case 'aws:waitForAwsResourceProperty':
                return new wait_for_resource_simulation_1.WaitForResourceSimulation(this.step, this.props);
            default:
                throw new Error('No simulation available for action: ' + this.step.name);
        }
    }
    /**
     * A timeout may be set as a property of a step.
     * This function checks the timeout to ensure that the execution time for the step was not exceeded.
     * The execution will not be killed when timeout is reached.
     * Rather an error will be thrown after the fact if time is exceeded.
     */
    checkExecutionTime(start) {
        const executionTime = Date.now() - start;
        if (executionTime > this.step.timeoutSeconds * 1000) {
            throw new Error('Execution time exceeded timeout: timeout set to ' +
                this.step.timeoutSeconds + ' seconds but took ' + executionTime + ' ms');
        }
    }
    /**
     * Finds each declared step output using the json selector specified.
     * @returns the mapping of output name to the value selected from the step execution response.
     */
    getSelectedResponse(response) {
        const relevantResponse = {};
        this.step.listOutputs().forEach(declaredOutput => {
            if (response == undefined) {
                throw new Error(`Output ${declaredOutput.name} specified selector ${declaredOutput.selector} but response was undefined for step ${this.step.name}`);
            }
            // I cannot explain the hack below. But it needs to be reformed into an object.
            const hackedResponse = JSON.parse(JSON.stringify(response));
            const selectedResponse = jp.query(hackedResponse, declaredOutput.selector)[0];
            if (selectedResponse === undefined) {
                throw new Error(`Output ${declaredOutput.name} specified selector ${declaredOutput.selector} but not found in response ${JSON.stringify(response)}`);
            }
            relevantResponse[this.step.name + '.' + declaredOutput.name] = selectedResponse;
            if (!new data_type_1.DataType(declaredOutput.outputType).validateType(selectedResponse)) {
                throw new Error(`DataType of output ${declaredOutput.name}:${data_type_1.DataTypeEnum[declaredOutput.outputType]} does not match response type ${JSON.stringify(selectedResponse)}`);
            }
        });
        return relevantResponse;
    }
}
exports.AutomationStepSimulation = AutomationStepSimulation;
_a = JSII_RTTI_SYMBOL_1;
AutomationStepSimulation[_a] = { fqn: "@cdklabs/cdk-ssm-documents.AutomationStepSimulation", version: "0.0.26" };
//# sourceMappingURL=data:application/json;base64,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