import { Construct } from 'constructs';
import { Output } from '../../domain/output';
import { IGenericVariable } from '../../interface/variables/variable';
import { AutomationStep, AutomationStepProps } from '../automation-step';
/**
 * The code to run for the execution.
 * See "script" parameter here:
 * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-executeScript.html
 * Attachments are not yet supported.
 */
export declare abstract class ScriptCode {
    /**
     * Inline code to be executed. String will be used to produce function in yaml/json.
     * Simulation will execute the function in this string using the language specified.
     */
    static inline(code: string): InlineScriptCode;
    /**
     * Full path to the code to execute. File is parsed to produce yaml/json.
     * Simulation will execute this file using the language specified.
     * (Attachments not yet supported)
     */
    static fromFile(fullPath: string): FileScriptCode;
    /**
     * @returns code as a string
     */
    abstract codeAsString(): string;
    /**
     * If there is a file for this code, return it.
     * Otherwise, create a file with the specified suffix.
     * @param suffix of the file to create (such as ".py")
     */
    abstract createOrGetFile(suffix: string): string;
}
export declare class InlineScriptCode extends ScriptCode {
    private static toFile;
    readonly inlineCode: string;
    constructor(inlineCode: string);
    codeAsString(): string;
    createOrGetFile(suffix: string): string;
}
export declare class FileScriptCode extends ScriptCode {
    readonly fullPath: string;
    constructor(fullPath: string);
    codeAsString(): string;
    createOrGetFile(_suffix: string): string;
}
/**
 * Specifies the script language as described in the "Runtime" argument here:
 * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-executeScript.html
 */
export declare abstract class ScriptLanguage {
    /**
     * Create a new ScriptLanguage for python execution.
     * @param version is the pythonVersion to use when writing the document (for simulation will not matter).
     * @param handlerName is the function name in code as entry point for script handler.
     */
    static python(version: PythonVersion, handlerName: string): ScriptLanguage;
    /**
     * Creates a ScriptLanguage based on the provided runtime.
     * Prefer one of the other static constructors if possible.
     * @param runtime is the runtime name (such as "python3.6").
     * @param handlerName to be provided for python executions.
     */
    static fromRuntime(runtime: string, handlerName?: string): ScriptLanguage;
    private static pythonVersion;
    /**
     * Builds the ssm inputs.
     */
    abstract ssmInputs(): {
        [name: string]: string;
    };
    /**
     * The associated runtime of this ScriptLanguage.
     */
    abstract runtime(): string;
    /**
     * The suffix to apply to file names of this type of execution.
     */
    abstract fileSuffix(): string;
    /**
     * Simulate an execution of this ScriptLanguage.
     * Provide the inputs after replaced with the actual values (not variables).
     */
    abstract simulate(code: ScriptCode, inputs: {
        [name: string]: string;
    }): {
        [name: string]: string;
    };
}
/**
 * Python runtime to use when writing SSM Document.
 * Simulation will use local python version.
 */
export declare enum PythonVersion {
    VERSION_3_6 = 0,
    VERSION_3_7 = 1,
    VERSION_3_8 = 2
}
/**
 * Properties for ExecuteScriptStep
 */
export interface ExecuteScriptStepProps extends AutomationStepProps {
    /**
     * (Required) Language used to execute the script.
     */
    readonly language: ScriptLanguage;
    /**
     * Inline code to be executed. String will be used to produce function in yaml/json.
     * Simulation will execute the function in this code using the language specified.
     */
    readonly code: ScriptCode;
    /**
     * (Optional) Outputs that the function is expected to return.
     * Be sure to prefix the selector for these outputs with "$.Payload." for executeScript step outputs.
     * @default []
     */
    readonly outputs?: Output[];
    /**
     * InputPayload that will be passed to the first parameter of the handler.
     * This can be used to pass input data to the script.
     * The key of this dict is the variable name that will be available to the code.
     * The value is the Variable object.
     */
    readonly inputPayload: {
        [name: string]: IGenericVariable;
    };
}
/**
 * AutomationStep implementation for aws:executeScript
 * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-executeScript.html
 */
export declare class ExecuteScriptStep extends AutomationStep {
    readonly language: ScriptLanguage;
    readonly code: ScriptCode;
    readonly outputs: Output[];
    readonly inputs: {
        [name: string]: IGenericVariable;
    };
    readonly action: string;
    constructor(stage: Construct, id: string, props: ExecuteScriptStepProps);
    listOutputs(): Output[];
    listInputs(): string[];
    toSsmEntry(): {
        [name: string]: any;
    };
}
