import { Construct } from 'constructs';
import { IAutomationComponent } from '../construct/document-component';
import { AutomationDocumentBuilder } from '../document/document-builder';
import { OnCancel, OnFailure } from '../interface/on-failure';
import { StepRef } from '../interface/step-ref';
import { Step, StepProps } from './step';
export interface AutomationStepProps extends StepProps {
    /**
     * Whether to stop document execution after this step.
     * @default false
     */
    readonly isEnd?: boolean;
    /**
     * (Optional) Fallback action to take in the event that this step fails.
     * @default undefined
     */
    readonly onFailure?: OnFailure;
    /**
     * (Optional) Fallback action to take in the event that this step is cancelled.
     * @default undefined
     */
    readonly onCancel?: OnCancel;
    /**
     * (Optional) max attempts to run this step if there are failures.
     * @default Step.DEFAULT_MAX_ATTEMPTS
     */
    readonly maxAttempts?: number;
    /**
     * (Optional) timeout seconds to run this step.
     * In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.
     * @default Step.DEFAULT_TIMEOUT
     */
    readonly timeoutSeconds?: number;
    /**
     * (Optional) explicit step to go to after this step completes.
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp
     * @default will implicitly choose the next step in the sequence that the steps are added to the document.
     */
    readonly explicitNextStep?: StepRef;
}
/**
 * Parent class for AutomationSteps. These steps are using in an AutomationDocument.
 * You can instantiate steps using the AutomationBuilder for convenience.
 * You can use these steps to simulate their execution (mimics the SSM run) AND to build their yaml/json declaration.
 * Control flow of the subsequent step is determined by the currently executing step.
 * The flow of the execution therefore follows a chain-of-responsibility pattern.
 * The inputs received into a step AND the outputs of previous steps are merged to form inputs of subsequent steps.
 */
export declare abstract class AutomationStep extends Step implements IAutomationComponent {
    static readonly DEFAULT_TIMEOUT = 3600;
    static readonly DEFAULT_MAX_ATTEMPTS = 1;
    readonly maxAttempts: number;
    readonly timeoutSeconds: number;
    readonly isEnd: boolean;
    readonly onFailure: OnFailure;
    readonly onCancel: OnCancel;
    readonly explicitNextStep?: StepRef;
    nextStep?: AutomationStep;
    allStepsInExecution?: AutomationStep[];
    constructor(scope: Construct, id: string, props: AutomationStepProps);
    addToDocument(doc: AutomationDocumentBuilder): void;
    protected prepareSsmEntry(inputs: {
        [name: string]: any;
    }): {
        [name: string]: any;
    };
    /**
     * Formats the provided output into an object that can be used to generate the SSM yaml/json representation.
     */
    private formatOutput;
    variables(): {
        [name: string]: any;
    };
}
