###################################################
##                                               ##
## This file is part of the KinBot code v2.0     ##
##                                               ##
## The contents are covered by the terms of the  ##
## BSD 3-clause license included in the LICENSE  ##
## file, found at the root.                      ##
##                                               ##
## Copyright 2018 National Technology &          ##
## Engineering Solutions of Sandia, LLC (NTESS). ##
## Under the terms of Contract DE-NA0003525 with ##
## NTESS, the U.S. Government retains certain    ##
## rights to this software.                      ##
##                                               ##
## Authors:                                      ##
##   Judit Zador                                 ##
##   Ruben Van de Vijver                         ##
##                                               ##
###################################################
"""
This class tests the resonance structure generation algorithm

The data is a dictionary of which the keys are the smiles
and the values are the expected number of resonance isomers
"""
import unittest
import numpy as np

from kinbot.stationary_pt import StationaryPoint
import kinbot.frequencies as frequencies


class TestFrequencies(unittest.TestCase):
    def setUp(self):
        pass

    def testFrequencies(self):
        """
        Test the frequency calculator based on the hessian
        by projecting out the translations, external rotations
        and internal rotations
        """
        # smiles of the speciesdata
        smi = 'CCCC'
        # geometry of the species
        geom = [[ 0.91879892, -0.04331223,  0.12849174],
                [ 0.40326992,  1.40563299,  0.15982756],
                [ 0.89480303,  2.27635373, -1.01612317],
                [ 0.37978671,  1.82281002, -2.39283836],
                [ 0.56484506, -0.58104398, -0.75857814],
                [ 0.58013979, -0.60154787,  1.00923604],
                [ 2.01620768, -0.06868878,  0.11831706],
                [-0.69710354,  1.40285362,  0.16867873],
                [ 0.71679336,  1.87619548,  1.10252902],
                [ 0.58079101,  3.31524962, -0.84143554],
                [ 1.99516746,  2.28446562, -1.02104071],
                [-0.71762425,  1.80824531, -2.41526122],
                [ 0.72083567,  2.50053168, -3.18429864],
                [ 0.73195359,  0.81628395, -2.64600957]]
        geom = np.array(geom)

        # hessian matrix
        hess = [[ 0.557479281000,  0.036110582900, -0.001644922990, -0.094996623900,  0.033793056500,  0.001227901360,  0.004435637720, -0.010301447300, -0.000714141524, -0.000402278745,  0.000207775781,  0.000217081305, -0.079155177000, -0.041443873100, -0.069861527200, -0.075582477100, -0.041043274800,  0.066112977000, -0.304832510000,  0.000618737475,  0.001591972160, -0.011587172100,  0.035782319500,  0.001443997110,  0.003566576500, -0.009402669700, -0.000876604471, -0.000457712074, -0.002463678640,  0.002595980330,  0.000811779351, -0.001430576500,  0.000032881273,  0.000129362928, -0.000033252298,  0.000111660638,  0.000157802640, -0.000276891050,  0.000074700177,  0.000433511158, -0.000116808738, -0.000311955226],
                [ 0.036110582900,  0.481460445000, -0.001497124110,  0.032529383300, -0.168787802000, -0.006867396700,  0.006092748400, -0.023937567300,  0.003730351660, -0.000262084184,  0.001441647380,  0.000316178114, -0.041989591900, -0.107986603000, -0.100880979000, -0.041503733500, -0.111783765000,  0.104729927000,  0.001080107490, -0.048474562500, -0.000241263827, -0.000265123753, -0.003449223290, -0.000371820596,  0.007618533730, -0.018717902300,  0.000837312337, -0.000555734072, -0.000078154904,  0.001999374050,  0.000646690960,  0.001260483370, -0.001360417570,  0.000088413780, -0.000085577196,  0.000164034911, -0.000076331932, -0.000238612455, -0.000622223468,  0.000486138870, -0.000622805294,  0.000064046666],
                [-0.001644922990, -0.001497124110,  0.575851102000,  0.001964210840, -0.005551021620, -0.083609101600, -0.009286228620,  0.027728004900,  0.001373639180, -0.000137246872,  0.000910607647,  0.001051929890, -0.069899107500, -0.100194460000, -0.221142958000,  0.066308541300,  0.103907569000, -0.219256026000,  0.001583703380, -0.000080359730, -0.050839140900, -0.000325149456,  0.000352021138,  0.000321544325,  0.010807393200, -0.028918001700, -0.000253208170,  0.001475456680,  0.004079204780, -0.004016767400, -0.000771004699, -0.000846365220,  0.001019809380, -0.000285363620,  0.000091087635, -0.000075059137, -0.000051358279, -0.000192607794,  0.000123367411,  0.000261076583,  0.000211445408, -0.000549130393],
                [-0.094996623900,  0.032529383300,  0.001964210840,  0.570648686000,  0.013726467600,  0.029803019800, -0.093894628800, -0.017540296600,  0.025240052300,  0.004430547620,  0.007089239030, -0.008470974800,  0.003985664600, -0.011160808500, -0.000344880024,  0.003352509560, -0.011319168200, -0.000141509338, -0.012312369900,  0.035615059900,  0.001341132350, -0.296633947000, -0.002814851200, -0.001857303680, -0.074252517800, -0.032055118900, -0.063305094900,  0.003016262880,  0.007165737430, -0.008147262180, -0.013171250400, -0.021050691300,  0.028389638100,  0.000515556167,  0.000833930927, -0.001531483210, -0.000733957654, -0.000834470892, -0.003543541800,  0.000046069520, -0.000184412716,  0.000603996490],
                [ 0.033793056500, -0.168787802000, -0.005551021620,  0.013726467600,  0.471256701000,  0.045550224600, -0.019055559200, -0.110456357000,  0.036788918500,  0.003690246720,  0.008042833690, -0.005946132620,  0.007450531200, -0.016991805700,  0.000566738577,  0.007662075620, -0.017473385800, -0.003434017560, -0.000361925575, -0.003081819410,  0.000169600942, -0.002141608270, -0.048419727000, -0.001543381080, -0.032001313400, -0.092648760200, -0.091748171900, -0.012965805500, -0.019490702300,  0.027544257800,  0.000385145313, -0.001607085100,  0.002202751810,  0.000097042391,  0.000342026851, -0.000085156640, -0.000935724242, -0.000536578498, -0.003946142400,  0.000657370855, -0.000147538049, -0.000568468443],
                [ 0.001227901360, -0.006867396700, -0.083609101600,  0.029803019800,  0.045550224600,  0.530185827000,  0.024067639100,  0.038254723800, -0.133557501000,  0.009656895150,  0.018083768400, -0.030339996500,  0.011176997800, -0.027940521700,  0.000197610447, -0.010718813500,  0.027683119400,  0.001610715360, -0.000005380242,  0.000353950668, -0.000058645732, -0.001385963250, -0.001490240310, -0.050945021400, -0.064025490200, -0.091980721200, -0.231554605000, -0.000068702298, -0.003001675120,  0.001414867140,  0.000375944632,  0.002432713030, -0.002619663450,  0.000967738003,  0.000578480710,  0.002131790000, -0.001665661510, -0.002059701750, -0.003910908180,  0.000593875240,  0.000403276118,  0.001054632250],
                [ 0.004435637720,  0.006092748400, -0.009286228620, -0.093894628800, -0.019055559200,  0.024067639100,  0.570900497000, -0.032595232800, -0.004232877500, -0.095277421800, -0.011406581600, -0.030607826800,  0.000043159915, -0.000514838503,  0.000373962854, -0.000738247952,  0.003637838290, -0.000237140037,  0.000536641625,  0.001192182950, -0.001206594120, -0.013027642100, -0.020980119300,  0.028551488200,  0.003004879320,  0.005695987090, -0.009218334250, -0.074383197800,  0.070019040800,  0.012170299900, -0.296727831000,  0.002527681320,  0.001997681770, -0.012251640800, -0.011694206300, -0.033682439000,  0.003373175160,  0.003467893310,  0.010833941000,  0.004006619800,  0.003613165550,  0.010476427500],
                [-0.010301447300, -0.023937567300,  0.027728004900, -0.017540296600, -0.110456357000,  0.038254723800, -0.032595232800,  0.550390617000,  0.021569897200, -0.011111359900, -0.094250354100, -0.029239287500, -0.000711297419,  0.000972151769,  0.000156692325,  0.001860692500, -0.005293631350, -0.000606360776, -0.000954457648,  0.002153894160,  0.000019371190, -0.000397838529, -0.001228430010,  0.002336752460,  0.003859667870,  0.006484974130, -0.010940182100,  0.070740800200, -0.270987312000, -0.037645349500,  0.001896728730, -0.051464188100, -0.000592835086,  0.000107061175, -0.000144181197, -0.000504021609,  0.007980792090,  0.006761052700,  0.020233362800, -0.012833812400, -0.009000668610, -0.030770768100],
                [-0.000714141524,  0.003730351660,  0.001373639180,  0.025240052300,  0.036788918500, -0.133557501000, -0.004232877500,  0.021569897200,  0.451154731000, -0.032090962200, -0.028197579400, -0.158290659000, -0.000492084585, -0.000980152816,  0.000034862639,  0.000407801351, -0.002484344690,  0.000853888771,  0.000202037882, -0.000630194064,  0.000364010457, -0.000276661410,  0.002092844910, -0.002923760270,  0.012375619200,  0.019611685600, -0.024571878400,  0.011914527900, -0.037319661200, -0.053342229400,  0.001505776140, -0.000642112942, -0.047826423400,  0.000374606934, -0.000789448823, -0.002922212160, -0.010438695000, -0.010818867900, -0.022572722700, -0.003774999570, -0.001931336020, -0.007773745060],
                [-0.000402278745, -0.000262084184, -0.000137246872,  0.004430547620,  0.003690246720,  0.009656895150, -0.095277421800, -0.011111359900, -0.032090962200,  0.558209213000, -0.009125619780, -0.034709539800,  0.000415340383,  0.000358171193,  0.000000905828,  0.000159128422,  0.000009821173,  0.000284856041,  0.000131990037, -0.000097828362,  0.000069311231,  0.000786266725,  0.000375904422,  0.001427105050, -0.000456009865, -0.001763774920,  0.003115461590,  0.003579063430,  0.003594264790,  0.008763686330, -0.011547639900, -0.011824677300, -0.033753097500, -0.305000167000, -0.001072859680, -0.000565352953, -0.076013943900, -0.051443542800,  0.058887756100, -0.079014088600,  0.078673338600,  0.019050222000],
                [ 0.000207775781,  0.001441647380,  0.000910607647,  0.007089239030,  0.008042833690,  0.018083768400, -0.011406581600, -0.094250354100, -0.028197579400, -0.009125619780,  0.566853551000, -0.028314924300, -0.000405813331, -0.000316988707,  0.000087265018,  0.000073046030, -0.000135428465, -0.000227947707,  0.000240096907, -0.000003210597,  0.000052913364,  0.000562715807,  0.000386702658, -0.000586865840, -0.001249462100, -0.001984311450,  0.004661147410, -0.012569365100, -0.009659129560, -0.031620324000,  0.000388942723, -0.000027569785, -0.000685633791, -0.001195716590, -0.050831902500,  0.000607484566, -0.051498752000, -0.151306241000,  0.115957274000,  0.078889494300, -0.268209599000, -0.050727185300],
                [ 0.000217081305,  0.000316178114,  0.001051929890, -0.008470974800, -0.005946132620, -0.030339996500, -0.030607826800, -0.029239287500, -0.158290659000, -0.034709539800, -0.028314924300,  0.489796505000, -0.000391829054,  0.000142067560, -0.000688267160,  0.000058546643, -0.000657571583,  0.000021899914, -0.000169145634,  0.000141233266, -0.000156069524, -0.000870684066, -0.001083763300,  0.001824089140,  0.000966571121,  0.002585026930, -0.002110201450, -0.004138394740, -0.001918643950, -0.009243037780,  0.000150249840, -0.001426379210, -0.003164852780, -0.000834451580,  0.000361231775, -0.048518990600,  0.059364188600,  0.116743826000, -0.179560342000,  0.019436209100, -0.051702861200, -0.060622006300],
                [-0.079155177000, -0.041989591900, -0.069899107500,  0.003985664600,  0.007450531200,  0.011176997800,  0.000043159915, -0.000711297419, -0.000492084585,  0.000415340383, -0.000405813331, -0.000391829054,  0.082170568700,  0.043883239500,  0.078044112600,  0.003495229520,  0.005401929110,  0.009954809490, -0.011350612800, -0.016458449400, -0.029453532500,  0.000665737135, -0.000625010895,  0.000511858440,  0.000059291070,  0.003007766240,  0.000159750828, -0.000060170797, -0.000108422372,  0.000205866966,  0.000026998545, -0.000052468999, -0.000060908234, -0.000223489041,  0.000140719611, -0.000038210872, -0.000144869281,  0.000150427844,  0.000065183444,  0.000072328995,  0.000316440754,  0.000217093184],
                [-0.041443873100, -0.107986603000, -0.100194460000, -0.011160808500, -0.016991805700, -0.027940521700, -0.000514838503,  0.000972151769, -0.000980152816,  0.000358171193, -0.000316988707,  0.000142067560,  0.043883239500,  0.119840850000,  0.111814261000,  0.005906654280,  0.008066891770,  0.014503839500,  0.000290968064, -0.000150240230,  0.000002063185,  0.000111922593,  0.001514337600,  0.000276003926,  0.002702457800, -0.005775080730,  0.000445401249,  0.000111932829,  0.000347388012,  0.000177052371, -0.000006786294, -0.000145911482,  0.000108411307,  0.000058268378, -0.000061770157,  0.000096427007, -0.000010851744,  0.000535119260,  0.000099909733, -0.000286456542,  0.000151661591,  0.001449697830],
                [-0.069861527200, -0.100880979000, -0.221142958000, -0.000344880024,  0.000566738577,  0.000197610447,  0.000373962854,  0.000156692325,  0.000034862639,  0.000000905828,  0.000087265018, -0.000688267160,  0.078044112600,  0.111814261000,  0.242817300000, -0.010493946400, -0.013936543100, -0.022850703000,  0.001401288830,  0.000811652521,  0.000387534637,  0.000611035395, -0.000326880526,  0.000202559149,  0.000448151560,  0.000430071969,  0.001316161050, -0.000015506148, -0.000226147940,  0.000124908040,  0.000053325603,  0.000066745555, -0.000005338059,  0.000029804814, -0.000043582456, -0.000002800832,  0.000036585620,  0.000161681863,  0.000229922299, -0.000283313413,  0.001319024070, -0.000620791338],
                [-0.075582477100, -0.041503733500,  0.066308541300,  0.003352509560,  0.007662075620, -0.010718813500, -0.000738247952,  0.001860692500,  0.000407801351,  0.000159128422,  0.000073046030,  0.000058546643,  0.003495229520,  0.005906654280, -0.010493946400,  0.078647598400,  0.043929663400, -0.073931943100, -0.010722735000, -0.017236570000,  0.029041201600,  0.000394715293, -0.000537710690, -0.000362268343,  0.000241854679, -0.000788101770,  0.000060932686,  0.000467445735,  0.000365561370, -0.000302900477,  0.000423918964,  0.000300000013, -0.000134663171,  0.000006452519, -0.000023320384,  0.000023532240, -0.000000863259,  0.000025535986,  0.000019870589, -0.000144529814, -0.000033792868,  0.000024108483],
                [-0.041043274800, -0.111783765000,  0.103907569000, -0.011319168200, -0.017473385800,  0.027683119400,  0.003637838290, -0.005293631350, -0.002484344690,  0.000009821173, -0.000135428465, -0.000657571583,  0.005401929110,  0.008066891770, -0.013936543100,  0.043929663400,  0.124958213000, -0.115530393000,  0.000325281332, -0.000125007354, -0.000129258723,  0.000384143980,  0.001259101570,  0.000001306272, -0.000870123215,  0.001042362100, -0.000426380057, -0.000426090514, -0.001300320690,  0.001591298870,  0.000048406439,  0.000287275840, -0.000388354266,  0.000055737630,  0.000018935809,  0.000059096657, -0.000026671850,  0.000148561268,  0.000149502306, -0.000107492816,  0.000330197117,  0.000160952609],
                [ 0.066112977000,  0.104729927000, -0.219256026000, -0.000141509338, -0.003434017560,  0.001610715360, -0.000237140037, -0.000606360776,  0.000853888771,  0.000284856041, -0.000227947707,  0.000021899914,  0.009954809490,  0.014503839500, -0.022850703000, -0.073931943100, -0.115530393000,  0.238426713000, -0.001172378760, -0.000473740575, -0.000085975425, -0.000598006922,  0.000347859264, -0.000022101354, -0.000104883229,  0.000359811438,  0.000893666466,  0.000047613768, -0.000039210588, -0.000025405264,  0.000008736166, -0.000126420940, -0.000048007771, -0.000081103354,  0.000034200394,  0.000014479678, -0.000018292858,  0.000237094498,  0.000030248075, -0.000123734870,  0.000225358158,  0.000436607680],
                [-0.304832510000,  0.001080107490,  0.001583703380, -0.012312369900, -0.000361925575, -0.000005380242,  0.000536641625, -0.000954457648,  0.000202037882,  0.000131990037,  0.000240096907, -0.000169145634, -0.011350612800,  0.000290968064,  0.001401288830, -0.010722735000,  0.000325281332, -0.001172378760,  0.335968234000, -0.003796974450, -0.002196966940,  0.001014683130,  0.002301249850,  0.000503015956,  0.000603465592,  0.000253289215,  0.000532380180,  0.000412280046, -0.000005518885, -0.000002154967,  0.000632261852,  0.000727256027, -0.000699875116,  0.000142228293, -0.000106601886,  0.000148869356,  0.000006913742,  0.000060554518, -0.000076997589, -0.000230470931, -0.000053324956, -0.000048396325],
                [ 0.000618737475, -0.048474562500, -0.000080359730,  0.035615059900, -0.003081819410,  0.000353950668,  0.001192182950,  0.002153894160, -0.000630194064, -0.000097828362, -0.000003210597,  0.000141233266, -0.016458449400, -0.000150240230,  0.000811652521, -0.017236570000, -0.000125007354, -0.000473740575, -0.003796974450,  0.054324381900,  0.000328044615,  0.002236943710, -0.006269973860, -0.000090077542, -0.000556919926,  0.001404142770, -0.000144097996, -0.000333527294,  0.000376518698, -0.000398408893, -0.001046966920, -0.000163879470,  0.000271078925, -0.000111873714, -0.000047084158, -0.000192239409, -0.000015838536,  0.000041274254,  0.000051844795, -0.000007975487,  0.000015565803,  0.000051313421],
                [ 0.001591972160, -0.000241263827, -0.050839140900,  0.001341132350,  0.000169600942, -0.000058645732, -0.001206594120,  0.000019371190,  0.000364010457,  0.000069311231,  0.000052913364, -0.000156069524, -0.029453532500,  0.000002063185,  0.000387534637,  0.029041201600, -0.000129258723, -0.000085975425, -0.002196966940,  0.000328044615,  0.049536269700,  0.000009025212, -0.000188391390,  0.000859989334,  0.000485598736,  0.000167091798, -0.000050893837,  0.000128473971, -0.000065137481, -0.000104352963,  0.000159767814,  0.000051678617,  0.000154467411,  0.000145393609, -0.000107573025,  0.000044588785,  0.000030151342,  0.000025660583, -0.000005990485, -0.000144934523, -0.000084799847, -0.000045791452],
                [-0.011587172100, -0.000265123753, -0.000325149456, -0.296633947000, -0.002141608270, -0.001385963250, -0.013027642100, -0.000397838529, -0.000276661410,  0.000786266725,  0.000562715807, -0.000870684066,  0.000665737135,  0.000111922593,  0.000611035395,  0.000394715293,  0.000384143980, -0.000598006922,  0.001014683130,  0.002236943710,  0.000009025212,  0.327078658000,  0.001908306410, -0.000098643877, -0.011211103000, -0.000096625978,  0.000161794067,  0.000445774654, -0.000675251910, -0.000001994717,  0.000919513287, -0.001690768780,  0.001598510130,  0.000698874397,  0.000144489938,  0.001199897110,  0.000425109666, -0.000023343949, -0.000045138440,  0.000030532015, -0.000057961271,  0.000021980223],
                [ 0.035782319500, -0.003449223290,  0.000352021138, -0.002814851200, -0.048419727000, -0.001490240310, -0.020980119300, -0.001228430010,  0.002092844910,  0.000375904422,  0.000386702658, -0.001083763300, -0.000625010895,  0.001514337600, -0.000326880526, -0.000537710690,  0.001259101570,  0.000347859264,  0.002301249850, -0.006269973860, -0.000188391390,  0.001908306410,  0.057676638400, -0.002012297070, -0.015037876400, -0.000231322031, -0.000293743748,  0.000076710051,  0.000337089674, -0.000732923797, -0.001033320970, -0.001677935690,  0.003357378420,  0.000462630311,  0.000228833858,  0.000196170553,  0.000041609570, -0.000164615301, -0.000252036680,  0.000080159392,  0.000038523433,  0.000034002544],
                [ 0.001443997110, -0.000371820596,  0.000321544325, -0.001857303680, -0.001543381080, -0.050945021400,  0.028551488200,  0.002336752460, -0.002923760270,  0.001427105050, -0.000586865840,  0.001824089140,  0.000511858440,  0.000276003926,  0.000202559149, -0.000362268343,  0.000001306272, -0.000022101354,  0.000503015956, -0.000090077542,  0.000859989334, -0.000098643877, -0.002012297070,  0.053996240600, -0.030319186800, -0.000675798642, -0.000572841147, -0.000653076701, -0.000463440047,  0.000816895682,  0.002084942010,  0.003132456320, -0.003659014380, -0.000911429020, -0.000039664076, -0.000139936846, -0.000326542275,  0.000011716933,  0.000402783060,  0.000006043879,  0.000025108980, -0.000161425910],
                [ 0.003566576500,  0.007618533730,  0.010807393200, -0.074252517800, -0.032001313400, -0.064025490200,  0.003004879320,  0.003859667870,  0.012375619200, -0.000456009865, -0.001249462100,  0.000966571121,  0.000059291070,  0.002702457800,  0.000448151560,  0.000241854679, -0.000870123215, -0.000104883229,  0.000603465592, -0.000556919926,  0.000485598736, -0.011211103000, -0.015037876400, -0.030319186800,  0.076908890300,  0.034393823600,  0.069678815000,  0.000280541835,  0.000499146782, -0.000714739428,  0.000437960626,  0.000608390526, -0.000262541603,  0.000410586032, -0.000025447089,  0.000355884856,  0.000467119865,  0.000078313296,  0.000422103376, -0.000061535147, -0.000019191454, -0.000113295827],
                [-0.009402669700, -0.018717902300, -0.028918001700, -0.032055118900, -0.092648760200, -0.091980721200,  0.005695987090,  0.006484974130,  0.019611685600, -0.001763774920, -0.001984311450,  0.002585026930,  0.003007766240, -0.005775080730,  0.000430071969, -0.000788101770,  0.001042362100,  0.000359811438,  0.000253289215,  0.001404142770,  0.000167091798, -0.000096625978, -0.000231322031, -0.000675798642,  0.034393823600,  0.108705444000,  0.098282383300,  0.000536535704,  0.001045616250, -0.000233377753,  0.000197842819,  0.000436602581, -0.000746827584,  0.000002929675, -0.000191975080, -0.000225290437,  0.000183020208,  0.000300248641,  0.001103529870, -0.000164903323,  0.000129961184,  0.000240416270],
                [-0.000876604471,  0.000837312337, -0.000253208170, -0.063305094900, -0.091748171900, -0.231554605000, -0.009218334250, -0.010940182100, -0.024571878400,  0.003115461590,  0.004661147410, -0.002110201450,  0.000159750828,  0.000445401249,  0.001316161050,  0.000060932686, -0.000426380057,  0.000893666466,  0.000532380180, -0.000144097996, -0.000050893837,  0.000161794067, -0.000293743748, -0.000572841147,  0.069678815000,  0.098282383300,  0.256012547000, -0.000695797842,  0.000405605652,  0.000999258380,  0.000659004856, -0.000503505707,  0.000708239491,  0.000002552083,  0.000110082904,  0.000462709809, -0.000439580294, -0.000527497416, -0.001625311350,  0.000164720405, -0.000158353984,  0.000346357198],
                [-0.000457712074, -0.000555734072,  0.001475456680,  0.003016262880, -0.012965805500, -0.000068702298, -0.074383197800,  0.070740800200,  0.011914527900,  0.003579063430, -0.012569365100, -0.004138394740, -0.000060170797,  0.000111932829, -0.000015506148,  0.000467445735, -0.000426090514,  0.000047613768,  0.000412280046, -0.000333527294,  0.000128473971,  0.000445774654,  0.000076710051, -0.000653076701,  0.000280541835,  0.000536535704, -0.000695797842,  0.077003315500, -0.076817039100, -0.012542197900, -0.011205602800,  0.033354294600,  0.005531828340,  0.000601743047, -0.000294036321,  0.000670043314,  0.000236669508,  0.000359711857,  0.000799593785,  0.000063586822, -0.001218387310, -0.002453862220],
                [-0.002463678640, -0.000078154904,  0.004079204780,  0.007165737430, -0.019490702300, -0.003001675120,  0.070019040800, -0.270987312000, -0.037319661200,  0.003594264790, -0.009659129560, -0.001918643950, -0.000108422372,  0.000347388012, -0.000226147940,  0.000365561370, -0.001300320690, -0.000039210588, -0.000005518885,  0.000376518698, -0.000065137481, -0.000675251910,  0.000337089674, -0.000463440047,  0.000499146782,  0.001045616250,  0.000405605652, -0.076817039100,  0.298274531000,  0.040406725300, -0.000136212824, -0.000795615621, -0.000114897060, -0.000586260703,  0.000093614097,  0.000252036708,  0.000174692057,  0.000878659823, -0.000383102689, -0.001026058750,  0.000957816966, -0.001611656380],
                [ 0.002595980330,  0.001999374050, -0.004016767400, -0.008147262180,  0.027544257800,  0.001414867140,  0.012170299900, -0.037645349500, -0.053342229400,  0.008763686330, -0.031620324000, -0.009243037780,  0.000205866966,  0.000177052371,  0.000124908040, -0.000302900477,  0.001591298870, -0.000025405264, -0.000002154967, -0.000398408893, -0.000104352963, -0.000001994717, -0.000732923797,  0.000816895682, -0.000714739428, -0.000233377753,  0.000999258380, -0.012542197900,  0.040406725300,  0.066461670300,  0.000111476779, -0.000463310086, -0.000005613339, -0.000083374773,  0.000585667105,  0.001289783720,  0.000768879885,  0.000408824903,  0.001044358070, -0.002821565820, -0.001619506420, -0.005414335160],
                [ 0.000811779351,  0.000646690960, -0.000771004699, -0.013171250400,  0.000385145313,  0.000375944632, -0.296727831000,  0.001896728730,  0.001505776140, -0.011547639900,  0.000388942723,  0.000150249840,  0.000026998545, -0.000006786294,  0.000053325603,  0.000423918964,  0.000048406439,  0.000008736166,  0.000632261852, -0.001046966920,  0.000159767814,  0.000919513287, -0.001033320970,  0.002084942010,  0.000437960626,  0.000197842819,  0.000659004856, -0.011205602800, -0.000136212824,  0.000111476779,  0.327319130000, -0.000522519027, -0.001747096940,  0.001019228930, -0.000665600374, -0.002131478270,  0.000410888095,  0.000459443519, -0.000527224047,  0.000650645014, -0.000611794096,  0.000067580103],
                [-0.001430576500,  0.001260483370, -0.000846365220, -0.021050691300, -0.001607085100,  0.002432713030,  0.002527681320, -0.051464188100, -0.000642112942, -0.011824677300, -0.000027569785, -0.001426379210, -0.000052468999, -0.000145911482,  0.000066745555,  0.000300000013,  0.000287275840, -0.000126420940,  0.000727256027, -0.000163879470,  0.000051678617, -0.001690768780, -0.001677935690,  0.003132456320,  0.000608390526,  0.000436602581, -0.000503505707,  0.033354294600, -0.000795615621, -0.000463310086, -0.000522519027,  0.053211548800, -0.000605423301, -0.001149055730,  0.000174592071, -0.002056008930,  0.000508508927,  0.000184012806,  0.000337657158, -0.000305373851,  0.000327669725,  0.000648275660],
                [ 0.000032881273, -0.001360417570,  0.001019809380,  0.028389638100,  0.002202751810, -0.002619663450,  0.001997681770, -0.000592835086, -0.047826423400, -0.033753097500, -0.000685633791, -0.003164852780, -0.000060908234,  0.000108411307, -0.000005338059, -0.000134663171, -0.000388354266, -0.000048007771, -0.000699875116,  0.000271078925,  0.000154467411,  0.001598510130,  0.003357378420, -0.003659014380, -0.000262541603, -0.000746827584,  0.000708239491,  0.005531828340, -0.000114897060, -0.000005613339, -0.001747096940, -0.000605423301,  0.058534733700, -0.002049213140, -0.002165722420, -0.005580464400,  0.000418020512,  0.000671917929,  0.001075125330,  0.000738835582,  0.000048572694,  0.001417002250],
                [ 0.000129362928,  0.000088413780, -0.000285363620,  0.000515556167,  0.000097042391,  0.000967738003, -0.012251640800,  0.000107061175,  0.000374606934, -0.305000167000, -0.001195716590, -0.000834451580, -0.000223489041,  0.000058268378,  0.000029804814,  0.000006452519,  0.000055737630, -0.000081103354,  0.000142228293, -0.000111873714,  0.000145393609,  0.000698874397,  0.000462630311, -0.000911429020,  0.000410586032,  0.000002929675,  0.000002552083,  0.000601743047, -0.000586260703, -0.000083374773,  0.001019228930, -0.001149055730, -0.002049213140,  0.335981374000,  0.002561958750,  0.003297591000, -0.010801126300,  0.001048090670, -0.000668728837, -0.011228983200, -0.001439226010,  0.000095977887],
                [-0.000033252298, -0.000085577196,  0.000091087635,  0.000833930927,  0.000342026851,  0.000578480710, -0.011694206300, -0.000144181197, -0.000789448823, -0.001072859680, -0.050831902500,  0.000361231775,  0.000140719611, -0.000061770157, -0.000043582456, -0.000023320384,  0.000018935809,  0.000034200394, -0.000106601886, -0.000047084158, -0.000107573025,  0.000144489938,  0.000228833858, -0.000039664076, -0.000025447089, -0.000191975080,  0.000110082904, -0.000294036321,  0.000093614097,  0.000585667105, -0.000665600374,  0.000174592071, -0.002165722420,  0.002561958750,  0.050149008600,  0.001623164620, -0.022680777200, -0.000201125886, -0.000105607558,  0.032915002300,  0.000556604861, -0.000132316779],
                [ 0.000111660638,  0.000164034911, -0.000075059137, -0.001531483210, -0.000085156640,  0.002131790000, -0.033682439000, -0.000504021609, -0.002922212160, -0.000565352953,  0.000607484566, -0.048518990600, -0.000038210872,  0.000096427007, -0.000002800832,  0.000023532240,  0.000059096657,  0.000014479678,  0.000148869356, -0.000192239409,  0.000044588785,  0.001199897110,  0.000196170553, -0.000139936846,  0.000355884856, -0.000225290437,  0.000462709809,  0.000670043314,  0.000252036708,  0.001289783720, -0.002131478270, -0.002056008930, -0.005580464400,  0.003297591000,  0.001623164620,  0.053631786900,  0.024986835900, -0.000480266327,  0.000069007009,  0.007154649910,  0.000544568335, -0.000404681882],
                [ 0.000157802640, -0.000076331932, -0.000051358279, -0.000733957654, -0.000935724242, -0.001665661510,  0.003373175160,  0.007980792090, -0.010438695000, -0.076013943900, -0.051498752000,  0.059364188600, -0.000144869281, -0.000010851744,  0.000036585620, -0.000000863259, -0.000026671850, -0.000018292858,  0.000006913742, -0.000015838536,  0.000030151342,  0.000425109666,  0.000041609570, -0.000326542275,  0.000467119865,  0.000183020208, -0.000439580294,  0.000236669508,  0.000174692057,  0.000768879885,  0.000410888095,  0.000508508927,  0.000418020512, -0.010801126300, -0.022680777200,  0.024986835900,  0.079112534200,  0.058103695900, -0.063986801200,  0.003504547540,  0.008252628830, -0.008677730390],
                [-0.000276891050, -0.000238612455, -0.000192607794, -0.000834470892, -0.000536578498, -0.002059701750,  0.003467893310,  0.006761052700, -0.010818867900, -0.051443542800, -0.151306241000,  0.116743826000,  0.000150427844,  0.000535119260,  0.000161681863,  0.000025535986,  0.000148561268,  0.000237094498,  0.000060554518,  0.000041274254,  0.000025660583, -0.000023343949, -0.000164615301,  0.000011716933,  0.000078313296,  0.000300248641, -0.000527497416,  0.000359711857,  0.000878659823,  0.000408824903,  0.000459443519,  0.000184012806,  0.000671917929,  0.001048090670, -0.000201125886, -0.000480266327,  0.058103695900,  0.163615200000, -0.127320828000, -0.011175418200, -0.020016956200,  0.023139046100],
                [ 0.000074700177, -0.000622223468,  0.000123367411, -0.003543541800, -0.003946142400, -0.003910908180,  0.010833941000,  0.020233362800, -0.022572722700,  0.058887756100,  0.115957274000, -0.179560342000,  0.000065183444,  0.000099909733,  0.000229922299,  0.000019870589,  0.000149502306,  0.000030248075, -0.000076997589,  0.000051844795, -0.000005990485, -0.000045138440, -0.000252036680,  0.000402783060,  0.000422103376,  0.001103529870, -0.001625311350,  0.000799593785, -0.000383102689,  0.001044358070, -0.000527224047,  0.000337657158,  0.001075125330, -0.000668728837, -0.000105607558,  0.000069007009, -0.063986801200, -0.127320828000,  0.199467764000, -0.002254716580, -0.005303140270,  0.005232700240],
                [ 0.000433511158,  0.000486138870,  0.000261076583,  0.000046069520,  0.000657370855,  0.000593875240,  0.004006619800, -0.012833812400, -0.003774999570, -0.079014088600,  0.078889494300,  0.019436209100,  0.000072328995, -0.000286456542, -0.000283313413, -0.000144529814, -0.000107492816, -0.000123734870, -0.000230470931, -0.000007975487, -0.000144934523,  0.000030532015,  0.000080159392,  0.000006043879, -0.000061535147, -0.000164903323,  0.000164720405,  0.000063586822, -0.001026058750, -0.002821565820,  0.000650645014, -0.000305373851,  0.000738835582, -0.011228983200,  0.032915002300,  0.007154649910,  0.003504547540, -0.011175418200, -0.002254716580,  0.081871766900, -0.087120674300, -0.018952145900],
                [-0.000116808738, -0.000622805294,  0.000211445408, -0.000184412716, -0.000147538049,  0.000403276118,  0.003613165550, -0.009000668610, -0.001931336020,  0.078673338600, -0.268209599000, -0.051702861200,  0.000316440754,  0.000151661591,  0.001319024070, -0.000033792868,  0.000330197117,  0.000225358158, -0.000053324956,  0.000015565803, -0.000084799847, -0.000057961271,  0.000038523433,  0.000025108980, -0.000019191454,  0.000129961184, -0.000158353984, -0.001218387310,  0.000957816966, -0.001619506420, -0.000611794096,  0.000327669725,  0.000048572694, -0.001439226010,  0.000556604861,  0.000544568335,  0.008252628830, -0.020016956200, -0.005303140270, -0.087120674300,  0.295489566000,  0.058022644000],
                [-0.000311955226,  0.000064046666, -0.000549130393,  0.000603996490, -0.000568468443,  0.001054632250,  0.010476427500, -0.030770768100, -0.007773745060,  0.019050222000, -0.050727185300, -0.060622006300,  0.000217093184,  0.001449697830, -0.000620791338,  0.000024108483,  0.000160952609,  0.000436607680, -0.000048396325,  0.000051313421, -0.000045791452,  0.000021980223,  0.000034002544, -0.000161425910, -0.000113295827,  0.000240416270,  0.000346357198, -0.002453862220, -0.001611656380, -0.005414335160,  0.000067580103,  0.000648275660,  0.001417002250,  0.000095977887, -0.000132316779, -0.000404681882, -0.008677730390,  0.023139046100,  0.005232700240, -0.018952145900,  0.058022644000,  0.067104607900]]
        hess = np.array(hess)

        st_pt = StationaryPoint(smi, 0, 1, smiles=smi)
        st_pt.characterize()
        natom = st_pt.natom
        freq, reduced_freqs = frequencies.get_frequencies(st_pt, hess, geom)

        # check the number of frequencies when neglecting internal rotors
        warn = 'Wrong number of frequencies for '
        warn += '{}, expected {}, calculated {}'.format(smi, 3 * natom - 6, len(freq))
        self.assertEqual(len(freq), 3 * natom - 6, warn)

        # check the number of freauencies when including internal rotors
        warn = 'Wrong number of frequencies for '
        warn += '{}, expected {}, calculated {}'.format(smi, 3 * natom - 6 - 3, len(reduced_freqs))
        self.assertEqual(len(reduced_freqs), 3 * natom - 6 - 3, warn)

        # check if the frequencies are as expected
        exps = [114.00308362581993, 217.27775429467988, 258.3569949257438, 324.95225959788496, 444.19934314923285, 775.6200599866132, 809.834778642176, 851.31690919327, 978.7594097713534, 1001.6985870040352, 1016.7427610310635, 1110.2892683008433, 1182.5149554411457, 1226.4874456219302, 1318.400582167711, 1338.054870101307, 1396.7770994771845, 1417.5721206176297, 1452.5015242293932, 1460.9939823348343, 1533.7500760993864, 1534.471579819868, 1544.1270899676583, 1549.977599644375, 1552.1602547884133, 1556.1540474667054, 3018.3546194275327, 3020.597688051164, 3036.615370205335, 3039.4844797216215, 3056.030309062557, 3060.954764774267, 3108.0034857725796, 3110.0837061265997, 3114.029978361547, 3121.6966999165047]
        for i, exp in enumerate(exps):
            warn = 'Frequency values have wrong number: '
            warn += 'expected {}, calculated {}'.format(exp, freq[i])
            self.assertAlmostEqual(exp, freq[i], places=2, msg=warn)

        # check if the reduced frequencies are as expected
        exps = [323.2231139328416, 443.06424130022316, 670.1999625377555, 765.4302138965138, 805.4817609384712, 923.6453995768678, 998.4208652995756, 1010.1550075636384, 1026.2590199716992, 1035.9563299669474, 1110.728140649036, 1237.3899533374413, 1297.0701335112522, 1319.639259548847, 1338.4248295874343, 1402.476769356911, 1417.8484347981505, 1476.7358681560938, 1487.8976160436305, 1525.2008164401302, 1544.0032443689747, 1547.12404969892, 1549.9807205743023, 2426.2794467384574, 2507.323795982035, 2864.813636413937, 3024.7798090881265, 3034.7484086579275, 3037.1066584298032, 3052.278595042064, 3107.809971415559, 3113.170533468414, 3113.8054919074493]
        for i, exp in enumerate(exps):
            warn = 'Frequency values have wrong number: '
            warn += 'expected {}, calculated {}'.format(exp, reduced_freqs[i])
            self.assertAlmostEqual(exp, reduced_freqs[i], places=2, msg=warn)


if __name__ == "__main__":
    unittest.main()
