from setuptools import setup, Extension, find_packages
from distutils.command.build_ext import build_ext
from Cython.Build import cythonize
from Cython.Compiler import Options
import numpy


# Preserve `__doc__` on functions and classes
# http://docs.cython.org/en/latest/src/userguide/source_files_and_compilation.html#compiler-options
Options.docstrings = True

COMPILE_OPTIONS = {
    "msvc": ["/Ox", "/EHsc"],
    "mingw32": ["-O2", "-Wno-strict-prototypes", "-Wno-unused-function"],
    "other": ["-O2", "-Wno-strict-prototypes", "-Wno-unused-function"],
}
LINK_OPTIONS = {"msvc": ["-std=c++11"], "mingw32": ["-std=c++11"], "other": []}
COMPILER_DIRECTIVES = {
    "language_level": -3,
    "embedsignature": True,
    "annotation_typing": False,
}


# By subclassing build_extensions we have the actual compiler that will be used which is really known only after finalize_options
# http://stackoverflow.com/questions/724664/python-distutils-how-to-get-a-compiler-that-is-going-to-be-used
class build_ext_options:
    def build_options(self):
        for e in self.extensions:
            e.extra_compile_args += COMPILE_OPTIONS.get(
                self.compiler.compiler_type, COMPILE_OPTIONS["other"]
            )
        for e in self.extensions:
            e.extra_link_args += LINK_OPTIONS.get(
                self.compiler.compiler_type, LINK_OPTIONS["other"]
            )


class build_ext_subclass(build_ext, build_ext_options):
    def build_extensions(self):
        build_ext_options.build_options(self)
        build_ext.build_extensions(self)


def setup_package():
    ext_modules = [
        Extension(
            "spacy_experimental.biaffine_parser.arc_predicter",
            ["spacy_experimental/biaffine_parser/arc_predicter.pyx"],
            language="c++",
            include_dirs=[numpy.get_include()],
            extra_compile_args=["-std=c++11"],
        ),
        Extension(
            "spacy_experimental.biaffine_parser.arc_labeler",
            ["spacy_experimental/biaffine_parser/arc_labeler.pyx"],
            language="c++",
            include_dirs=[numpy.get_include()],
            extra_compile_args=["-std=c++11"],
        ),
        Extension(
            "spacy_experimental.biaffine_parser.mst",
            ["spacy_experimental/biaffine_parser/mst.pyx"],
            language="c++",
            include_dirs=[numpy.get_include()],
            extra_compile_args=["-std=c++11"],
        ),
        Extension(
            "spacy_experimental.char_tokenizer.char_tagger_tokenizer",
            ["spacy_experimental/char_tokenizer/char_tagger_tokenizer.pyx"],
            language="c++",
            include_dirs=[numpy.get_include()],
            extra_compile_args=["-std=c++11"],
        ),
        Extension(
            "spacy_experimental.char_tokenizer.char_ner_tokenizer",
            ["spacy_experimental/char_tokenizer/char_ner_tokenizer.pyx"],
            language="c++",
            include_dirs=[numpy.get_include()],
            extra_compile_args=["-std=c++11"],
        ),
    ]

    ext_modules = cythonize(ext_modules, compiler_directives=COMPILER_DIRECTIVES)

    setup(
        name="spacy-experimental",
        packages=find_packages(),
        ext_modules=ext_modules,
        cmdclass={"build_ext": build_ext_subclass},
        package_data={"": ["*.pyx", "*.pxd", "*.pxi"]},
    )


if __name__ == "__main__":
    setup_package()
