"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const cdk = require("aws-cdk-lib");
const util_1 = require("./util");
/**
 * Bundling
 */
class Bundling {
    constructor(props) {
        var _a, _b, _c, _d;
        this.props = props;
        Bundling.runsLocally = (_b = (_a = Bundling.runsLocally) !== null && _a !== void 0 ? _a : util_1.getGoBuildVersion()) !== null && _b !== void 0 ? _b : false;
        const projectRoot = path.dirname(props.moduleDir);
        this.relativeEntryPath = `./${path.relative(projectRoot, path.resolve(props.entry))}`;
        const cgoEnabled = props.cgoEnabled ? '1' : '0';
        const environment = {
            CGO_ENABLED: cgoEnabled,
            GO111MODULE: 'on',
            GOARCH: props.architecture.dockerPlatform.split('/')[1],
            GOOS: 'linux',
            ...props.environment,
        };
        // Docker bundling
        const shouldBuildImage = props.forcedDockerBundling || !Bundling.runsLocally;
        this.image = shouldBuildImage
            ? (_c = props.dockerImage) !== null && _c !== void 0 ? _c : cdk.DockerImage.fromBuild(path.join(__dirname, '../lib'), {
                buildArgs: {
                    ...(_d = props.buildArgs) !== null && _d !== void 0 ? _d : {},
                    IMAGE: aws_lambda_1.Runtime.GO_1_X.bundlingImage.image,
                },
                platform: props.architecture.dockerPlatform,
            }) : cdk.DockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand(cdk.AssetStaging.BUNDLING_INPUT_DIR, cdk.AssetStaging.BUNDLING_OUTPUT_DIR);
        this.command = ['bash', '-c', bundlingCommand];
        this.environment = environment;
        // Local bundling
        if (!props.forcedDockerBundling) { // only if Docker is not forced
            const osPlatform = os.platform();
            const createLocalCommand = (outputDir) => this.createBundlingCommand(projectRoot, outputDir, osPlatform);
            this.local = {
                tryBundle(outputDir) {
                    if (Bundling.runsLocally == false) {
                        process.stderr.write('go build cannot run locally. Switching to Docker bundling.\n');
                        return false;
                    }
                    const localCommand = createLocalCommand(outputDir);
                    util_1.exec(osPlatform === 'win32' ? 'cmd' : 'bash', [
                        osPlatform === 'win32' ? '/c' : '-c',
                        localCommand,
                    ], {
                        env: { ...process.env, ...environment !== null && environment !== void 0 ? environment : {} },
                        stdio: [
                            'ignore',
                            process.stderr,
                            'inherit',
                        ],
                        cwd: path.dirname(props.moduleDir),
                        windowsVerbatimArguments: osPlatform === 'win32',
                    });
                    return true;
                },
            };
        }
    }
    static bundle(options) {
        var _a;
        const bundling = new Bundling(options);
        return aws_lambda_1.Code.fromAsset(path.dirname(options.moduleDir), {
            assetHashType: (_a = options.assetHashType) !== null && _a !== void 0 ? _a : cdk.AssetHashType.OUTPUT,
            assetHash: options.assetHash,
            bundling: {
                image: bundling.image,
                command: bundling.command,
                environment: bundling.environment,
                local: bundling.local,
            },
        });
    }
    static clearRunsLocallyCache() {
        this.runsLocally = undefined;
    }
    createBundlingCommand(inputDir, outputDir, osPlatform = 'linux') {
        var _a, _b, _c, _d;
        const pathJoin = osPathJoin(osPlatform);
        const hasVendor = util_1.findUp('vendor', path.dirname(this.props.entry));
        const goBuildCommand = [
            'go', 'build',
            hasVendor ? '-mod=vendor' : '',
            '-o', `${pathJoin(outputDir, 'bootstrap')}`,
            `${this.props.goBuildFlags ? this.props.goBuildFlags.join(' ') : ''}`,
            `${this.relativeEntryPath.replace(/\\/g, '/')}`,
        ].filter(c => !!c).join(' ');
        return chain([
            ...(_b = (_a = this.props.commandHooks) === null || _a === void 0 ? void 0 : _a.beforeBundling(inputDir, outputDir)) !== null && _b !== void 0 ? _b : [],
            goBuildCommand,
            ...(_d = (_c = this.props.commandHooks) === null || _c === void 0 ? void 0 : _c.afterBundling(inputDir, outputDir)) !== null && _d !== void 0 ? _d : [],
        ]);
    }
}
exports.Bundling = Bundling;
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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