"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AutomatedWaf = exports.LogLevel = exports.Waf2ScopeOption = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const apigateway = require("@aws-cdk/aws-apigateway");
const athena = require("@aws-cdk/aws-athena");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const events = require("@aws-cdk/aws-events");
const targets = require("@aws-cdk/aws-events-targets");
const glue = require("@aws-cdk/aws-glue");
const iam = require("@aws-cdk/aws-iam");
const firehose = require("@aws-cdk/aws-kinesisfirehose");
const lambda = require("@aws-cdk/aws-lambda");
const s3 = require("@aws-cdk/aws-s3");
const s3n = require("@aws-cdk/aws-s3-notifications");
const wafv2 = require("@aws-cdk/aws-wafv2");
const cdk = require("@aws-cdk/core");
const core_1 = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
/**
 * @experimental
 */
var Waf2ScopeOption;
(function (Waf2ScopeOption) {
    Waf2ScopeOption["CLOUDFRONT"] = "CLOUDFRONT";
    Waf2ScopeOption["REGIONAL"] = "REGIONAL";
})(Waf2ScopeOption = exports.Waf2ScopeOption || (exports.Waf2ScopeOption = {}));
/**
 * @experimental
 */
var LogLevel;
(function (LogLevel) {
    LogLevel["DEBUG"] = "DEBUG";
    LogLevel["INFO"] = "INFO";
    LogLevel["WARNING"] = "WARNING";
    LogLevel["ERROR"] = "ERROR";
    LogLevel["CRITICAL"] = "CRITICAL";
})(LogLevel = exports.LogLevel || (exports.LogLevel = {}));
const Elastic_Load_Balancing_account_ID = {
    'us-east-1': '127311923021',
    'us-east-2': '033677994240',
    'us-west-1': '027434742980',
    'us-west-2': '797873946194',
    'af-south-1': '098369216593',
    'ca-central-1': '985666609251',
    'eu-central-1': '054676820928',
    'eu-west-1': '156460612806',
    'eu-west-2': '652711504416',
    'eu-south-1': '635631232127',
    'eu-west-3': '009996457667',
    'eu-north-1': '897822967062',
    'ap-east-1': '754344448648',
    'ap-northeast-1': '582318560864',
    'ap-northeast-2': '600734575887',
    'ap-northeast-3': '383597477331',
    'ap-southeast-1': '114774131450',
    'ap-southeast-2': '783225319266',
    'ap-south-1': '718504428378',
    'me-south-1': '076674570225',
    'sa-east-1': '507241528517',
    'us-gov-west-1': '048591011584',
    'us-gov-east-1': '190560391635',
    'cn-north-1': '638102146993',
    'cn-northwest-1': '037604701340',
};
/**
 * @experimental
 */
class AutomatedWaf extends cdk.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h;
        super(scope, id);
        const stack = cdk.Stack.of(this);
        if (props.resourceNamingPrefix &&
            !this.validateResourceNamingPrefix(props.resourceNamingPrefix)) {
            throw new Error('resourceNamingPrefix property only allow alphanumeric and "_" symbol because glue database naming is needed');
        }
        //default value
        const errorThreshold = (_b = props.errorThreshold) !== null && _b !== void 0 ? _b : 200;
        const countMode = (_c = props.countMode) !== null && _c !== void 0 ? _c : false;
        const requestThreshold = (_d = props.requestThreshold) !== null && _d !== void 0 ? _d : 1000;
        const blockPeriod = (_e = props.blockPeriod) !== null && _e !== void 0 ? _e : 240;
        const enableShieldAdvancedLambda = (_f = props.enableShieldAdvancedLambda) !== null && _f !== void 0 ? _f : false;
        const logLevel = (_g = props.logLevel) !== null && _g !== void 0 ? _g : 'INFO';
        const cloudWatchDashboardName = `WAFMonitoringDashboard${props.resourceNamingPrefix}${cdk.Aws.REGION}`;
        const reputationListName = cdk.Fn.ref('AWS::StackName') + 'IPReputationListsRule';
        const allowListName = cdk.Fn.ref('AWS::StackName') + 'WhitelistRule';
        const blacklistRuleName = cdk.Fn.ref('AWS::StackName') + 'BlacklistRule';
        const httpFloodRegularRuleName = cdk.Fn.ref('AWS::StackName') + 'HttpFloodRegularRule';
        const scannersProbesRuleName = cdk.Fn.ref('AWS::StackName') + 'ScannersProbesRule';
        const badBotRuleName = cdk.Fn.ref('AWS::StackName') + 'BadBotRule';
        const sqlInjectionRuleName = cdk.Fn.ref('AWS::StackName') + 'SqlInjectionRule';
        const xssRuleName = cdk.Fn.ref('AWS::StackName') + 'XssRule';
        const wafLogBucketAppendProps = props.wafLogBucketName
            ? { bucketName: props.wafLogBucketName }
            : {};
        //WafLogBucket
        const wafLogBucket = new s3.Bucket(this, 'WafLogBucket', {
            ...wafLogBucketAppendProps,
            publicReadAccess: false,
            encryption: s3.BucketEncryption.KMS_MANAGED,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
        });
        const accessLogBucketAppendProps = props.appAccessLogBucketName
            ? { bucketName: props.appAccessLogBucketName }
            : {};
        const accessLogBucket = new s3.Bucket(this, 'AccessLogBucket', {
            ...accessLogBucketAppendProps,
            publicReadAccess: false,
            encryption: s3.BucketEncryption.S3_MANAGED,
            accessControl: props.waf2Scope == Waf2ScopeOption.CLOUDFRONT ? s3.BucketAccessControl.LOG_DELIVERY_WRITE : s3.BucketAccessControl.PRIVATE,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
        });
        if (props.waf2Scope == Waf2ScopeOption.REGIONAL) {
            accessLogBucket.addToResourcePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['s3:PutObject'],
                resources: [
                    `${accessLogBucket.bucketArn}/AWSLogs/${cdk.Aws.ACCOUNT_ID}/*`,
                ],
                principals: [
                    new iam.AccountPrincipal(Elastic_Load_Balancing_account_ID[stack.region]),
                ],
            }));
            accessLogBucket.addToResourcePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['s3:PutObject'],
                resources: [
                    `${accessLogBucket.bucketArn}/AWSLogs/${cdk.Aws.ACCOUNT_ID}/*`,
                ],
                principals: [new iam.ServicePrincipal('delivery.logs.amazonaws.com')],
                conditions: {
                    StringEquals: {
                        's3:x-amz-acl': 'bucket-owner-full-control',
                    },
                },
            }));
            accessLogBucket.addToResourcePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['s3:GetBucketAcl'],
                resources: [accessLogBucket.bucketArn],
                principals: [new iam.ServicePrincipal('delivery.logs.amazonaws.com')],
            }));
        }
        // Setup Whitelist IP Set
        const whitelistIpSetV4 = new wafv2.CfnIPSet(this, 'WhitelistSetIPV4', {
            addresses: [],
            ipAddressVersion: 'IPV4',
            scope: props.waf2Scope,
            name: cdk.Fn.ref('AWS::StackName') + 'WhitelistSetIPV4',
            description: 'Allow whitelist for IPV4 addresses',
        });
        const whitelistIpSetV6 = new wafv2.CfnIPSet(this, 'WhitelistSetIPV6', {
            addresses: [],
            ipAddressVersion: 'IPV6',
            scope: props.waf2Scope,
            name: cdk.Fn.ref('AWS::StackName') + 'WhitelistSetIPV6',
            description: 'Allow whitelist for IPV6 addresses',
        });
        // Setup Blacklist IP Set
        const blacklistIpSetV4 = new wafv2.CfnIPSet(this, 'BlacklistSetIPV4', {
            addresses: [],
            ipAddressVersion: 'IPV4',
            scope: props.waf2Scope,
            name: cdk.Fn.ref('AWS::StackName') + 'BlacklistSetIPV4',
            description: 'Allow blacklist for IPV4 addresses',
        });
        const blacklistIpSetV6 = new wafv2.CfnIPSet(this, 'BlacklistSetIPV6', {
            addresses: [],
            ipAddressVersion: 'IPV6',
            scope: props.waf2Scope,
            name: cdk.Fn.ref('AWS::StackName') + 'BlacklistSetIPV6',
            description: 'Allow blacklist for IPV6 addresses',
        });
        // Setup HTTP Flood IP Set
        const httpFloodIpSetV4 = new wafv2.CfnIPSet(this, 'HTTPFloodSetIPV4', {
            addresses: [],
            ipAddressVersion: 'IPV4',
            scope: props.waf2Scope,
            name: cdk.Fn.ref('AWS::StackName') + 'HTTPFloodSetIPV4',
            description: 'Block HTTP Flood IPV4 addresses',
        });
        const httpFloodIpSetV6 = new wafv2.CfnIPSet(this, 'HTTPFloodSetIPV6', {
            addresses: [],
            ipAddressVersion: 'IPV6',
            scope: props.waf2Scope,
            name: cdk.Fn.ref('AWS::StackName') + 'HTTPFloodSetIPV6',
            description: 'Block HTTP Flood IPV6 addresses',
        });
        // Block Scanners/Probes IP Set
        const scannersProbesIpSetV4 = new wafv2.CfnIPSet(this, 'ScannersProbesSetIPV4', {
            addresses: [],
            ipAddressVersion: 'IPV4',
            scope: props.waf2Scope,
            name: cdk.Fn.ref('AWS::StackName') + 'ScannersProbesSetIPV4',
            description: 'Block Scanners/Probes IPV4 addresses',
        });
        const scannersProbesIpSetV6 = new wafv2.CfnIPSet(this, 'ScannersProbesSetIPV6', {
            addresses: [],
            ipAddressVersion: 'IPV6',
            scope: props.waf2Scope,
            name: cdk.Fn.ref('AWS::StackName') + 'ScannersProbesSetIPV6',
            description: 'Block Scanners/Probes IPV6 addresses',
        });
        // Block Reputation List IP Set
        const reputationListsIpSetV4 = new wafv2.CfnIPSet(this, 'IPReputationListsSetIPV4', {
            addresses: [],
            ipAddressVersion: 'IPV4',
            scope: props.waf2Scope,
            name: cdk.Fn.ref('AWS::StackName') + 'IPReputationListsSetIPV4',
            description: 'Block Reputation List IPV4 addresses',
        });
        const reputationListsIpSetV6 = new wafv2.CfnIPSet(this, 'IPReputationListsSetIPV6', {
            addresses: [],
            ipAddressVersion: 'IPV6',
            scope: props.waf2Scope,
            name: cdk.Fn.ref('AWS::StackName') + 'IPReputationListsSetIPV6',
            description: 'Block Reputation List IPV6 addresses',
        });
        // Block Bad Bot IP Set
        const badBotIpSetV4 = new wafv2.CfnIPSet(this, 'IPBadBotSetIPV4', {
            addresses: [],
            ipAddressVersion: 'IPV4',
            scope: props.waf2Scope,
            name: cdk.Fn.ref('AWS::StackName') + 'IPBadBotSetIPV4',
            description: 'Block Bad Bot IPV4 addresses',
        });
        const badBotIpSetV6 = new wafv2.CfnIPSet(this, 'IPBadBotSetIPV6', {
            addresses: [],
            ipAddressVersion: 'IPV6',
            scope: props.waf2Scope,
            name: cdk.Fn.ref('AWS::StackName') + 'IPBadBotSetIPV6',
            description: 'Block Bad Bot IPV6 addresses',
        });
        const badBotWafRules = [];
        if (props.waf2Scope == Waf2ScopeOption.CLOUDFRONT) {
            badBotWafRules.push({
                name: badBotRuleName,
                priority: 7,
                action: !countMode
                    ? {
                        block: {},
                    }
                    : { count: {} },
                visibilityConfig: {
                    sampledRequestsEnabled: true,
                    cloudWatchMetricsEnabled: true,
                    metricName: badBotRuleName,
                },
                statement: {
                    orStatement: {
                        statements: [
                            {
                                ipSetReferenceStatement: {
                                    arn: cdk.Fn.getAtt(badBotIpSetV4.logicalId, 'Arn').toString(),
                                },
                            },
                            {
                                ipSetReferenceStatement: {
                                    arn: cdk.Fn.getAtt(badBotIpSetV6.logicalId, 'Arn').toString(),
                                },
                            },
                        ],
                    },
                },
            });
        }
        // WAF Web ACL
        const wafweb = new wafv2.CfnWebACL(this, 'wafweb', {
            name: props.resourceNamingPrefix
                ? `${props.resourceNamingPrefix}-WAF`
                : 'CloudFront-Web-WAF',
            description: 'Custom WAFWebACL',
            defaultAction: {
                allow: {},
            },
            scope: props.waf2Scope,
            visibilityConfig: {
                cloudWatchMetricsEnabled: true,
                sampledRequestsEnabled: true,
                metricName: props.resourceNamingPrefix
                    ? `${props.resourceNamingPrefix}-WAF`
                    : 'CloudFront-Web-WAF',
            },
            rules: [
                {
                    name: 'AWS-AWSManagedRulesCommonRuleSet',
                    priority: 0,
                    overrideAction: !countMode
                        ? {
                            none: {},
                        }
                        : { count: {} },
                    visibilityConfig: {
                        cloudWatchMetricsEnabled: true,
                        sampledRequestsEnabled: true,
                        metricName: props.resourceNamingPrefix
                            ? `${props.resourceNamingPrefix}-waf-ipset-metrics`
                            : 'cloudfront-waf-ipset-metrics',
                    },
                    statement: {
                        managedRuleGroupStatement: {
                            vendorName: 'AWS',
                            name: 'AWSManagedRulesCommonRuleSet',
                        },
                    },
                },
                {
                    name: allowListName,
                    priority: 1,
                    action: !countMode
                        ? {
                            allow: {},
                        }
                        : { count: {} },
                    visibilityConfig: {
                        sampledRequestsEnabled: true,
                        cloudWatchMetricsEnabled: true,
                        metricName: allowListName,
                    },
                    statement: {
                        orStatement: {
                            statements: [
                                {
                                    ipSetReferenceStatement: {
                                        arn: cdk.Fn.getAtt(whitelistIpSetV4.logicalId, 'Arn').toString(),
                                    },
                                },
                                {
                                    ipSetReferenceStatement: {
                                        arn: cdk.Fn.getAtt(whitelistIpSetV6.logicalId, 'Arn').toString(),
                                    },
                                },
                            ],
                        },
                    },
                },
                {
                    name: blacklistRuleName,
                    priority: 2,
                    action: !countMode
                        ? {
                            block: {},
                        }
                        : { count: {} },
                    visibilityConfig: {
                        sampledRequestsEnabled: true,
                        cloudWatchMetricsEnabled: true,
                        metricName: blacklistRuleName,
                    },
                    statement: {
                        orStatement: {
                            statements: [
                                {
                                    ipSetReferenceStatement: {
                                        arn: cdk.Fn.getAtt(blacklistIpSetV4.logicalId, 'Arn').toString(),
                                    },
                                },
                                {
                                    ipSetReferenceStatement: {
                                        arn: cdk.Fn.getAtt(blacklistIpSetV6.logicalId, 'Arn').toString(),
                                    },
                                },
                            ],
                        },
                    },
                },
                {
                    name: httpFloodRegularRuleName,
                    priority: 3,
                    action: !countMode
                        ? {
                            block: {},
                        }
                        : { count: {} },
                    visibilityConfig: {
                        sampledRequestsEnabled: true,
                        cloudWatchMetricsEnabled: true,
                        metricName: httpFloodRegularRuleName,
                    },
                    statement: {
                        orStatement: {
                            statements: [
                                {
                                    ipSetReferenceStatement: {
                                        arn: cdk.Fn.getAtt(httpFloodIpSetV4.logicalId, 'Arn').toString(),
                                    },
                                },
                                {
                                    ipSetReferenceStatement: {
                                        arn: cdk.Fn.getAtt(httpFloodIpSetV6.logicalId, 'Arn').toString(),
                                    },
                                },
                            ],
                        },
                    },
                },
                {
                    name: scannersProbesRuleName,
                    priority: 5,
                    action: !countMode
                        ? {
                            block: {},
                        }
                        : { count: {} },
                    visibilityConfig: {
                        sampledRequestsEnabled: true,
                        cloudWatchMetricsEnabled: true,
                        metricName: scannersProbesRuleName,
                    },
                    statement: {
                        orStatement: {
                            statements: [
                                {
                                    ipSetReferenceStatement: {
                                        arn: cdk.Fn.getAtt(scannersProbesIpSetV4.logicalId, 'Arn').toString(),
                                    },
                                },
                                {
                                    ipSetReferenceStatement: {
                                        arn: cdk.Fn.getAtt(scannersProbesIpSetV6.logicalId, 'Arn').toString(),
                                    },
                                },
                            ],
                        },
                    },
                },
                {
                    name: reputationListName,
                    priority: 6,
                    action: !countMode
                        ? {
                            block: {},
                        }
                        : { count: {} },
                    visibilityConfig: {
                        sampledRequestsEnabled: true,
                        cloudWatchMetricsEnabled: true,
                        metricName: reputationListName,
                    },
                    statement: {
                        orStatement: {
                            statements: [
                                {
                                    ipSetReferenceStatement: {
                                        arn: cdk.Fn.getAtt(reputationListsIpSetV4.logicalId, 'Arn').toString(),
                                    },
                                },
                                {
                                    ipSetReferenceStatement: {
                                        arn: cdk.Fn.getAtt(reputationListsIpSetV6.logicalId, 'Arn').toString(),
                                    },
                                },
                            ],
                        },
                    },
                },
                {
                    name: sqlInjectionRuleName,
                    priority: 20,
                    action: !countMode
                        ? {
                            block: {},
                        }
                        : { count: {} },
                    visibilityConfig: {
                        sampledRequestsEnabled: true,
                        cloudWatchMetricsEnabled: true,
                        metricName: sqlInjectionRuleName,
                    },
                    statement: {
                        orStatement: {
                            statements: [
                                {
                                    sqliMatchStatement: {
                                        fieldToMatch: {
                                            queryString: {},
                                        },
                                        textTransformations: [
                                            {
                                                priority: 1,
                                                type: 'URL_DECODE',
                                            },
                                            {
                                                priority: 2,
                                                type: 'HTML_ENTITY_DECODE',
                                            },
                                        ],
                                    },
                                },
                                {
                                    sqliMatchStatement: {
                                        fieldToMatch: {
                                            body: {},
                                        },
                                        textTransformations: [
                                            {
                                                priority: 1,
                                                type: 'URL_DECODE',
                                            },
                                            {
                                                priority: 2,
                                                type: 'HTML_ENTITY_DECODE',
                                            },
                                        ],
                                    },
                                },
                                {
                                    sqliMatchStatement: {
                                        fieldToMatch: {
                                            uriPath: {},
                                        },
                                        textTransformations: [
                                            {
                                                priority: 1,
                                                type: 'URL_DECODE',
                                            },
                                            {
                                                priority: 2,
                                                type: 'HTML_ENTITY_DECODE',
                                            },
                                        ],
                                    },
                                },
                                {
                                    sqliMatchStatement: {
                                        fieldToMatch: {
                                            singleHeader: { Name: 'Authorization' },
                                        },
                                        textTransformations: [
                                            {
                                                priority: 1,
                                                type: 'URL_DECODE',
                                            },
                                            {
                                                priority: 2,
                                                type: 'HTML_ENTITY_DECODE',
                                            },
                                        ],
                                    },
                                },
                                {
                                    sqliMatchStatement: {
                                        fieldToMatch: {
                                            singleHeader: { Name: 'Cookie' },
                                        },
                                        textTransformations: [
                                            {
                                                priority: 1,
                                                type: 'URL_DECODE',
                                            },
                                            {
                                                priority: 2,
                                                type: 'HTML_ENTITY_DECODE',
                                            },
                                        ],
                                    },
                                },
                            ],
                        },
                    },
                },
                {
                    name: xssRuleName,
                    priority: 30,
                    action: !countMode
                        ? {
                            block: {},
                        }
                        : { count: {} },
                    visibilityConfig: {
                        sampledRequestsEnabled: true,
                        cloudWatchMetricsEnabled: true,
                        metricName: xssRuleName,
                    },
                    statement: {
                        orStatement: {
                            statements: [
                                {
                                    xssMatchStatement: {
                                        fieldToMatch: {
                                            queryString: {},
                                        },
                                        textTransformations: [
                                            {
                                                priority: 1,
                                                type: 'URL_DECODE',
                                            },
                                            {
                                                priority: 2,
                                                type: 'HTML_ENTITY_DECODE',
                                            },
                                        ],
                                    },
                                },
                                {
                                    xssMatchStatement: {
                                        fieldToMatch: {
                                            body: {},
                                        },
                                        textTransformations: [
                                            {
                                                priority: 1,
                                                type: 'URL_DECODE',
                                            },
                                            {
                                                priority: 2,
                                                type: 'HTML_ENTITY_DECODE',
                                            },
                                        ],
                                    },
                                },
                                {
                                    xssMatchStatement: {
                                        fieldToMatch: {
                                            uriPath: {},
                                        },
                                        textTransformations: [
                                            {
                                                priority: 1,
                                                type: 'URL_DECODE',
                                            },
                                            {
                                                priority: 2,
                                                type: 'HTML_ENTITY_DECODE',
                                            },
                                        ],
                                    },
                                },
                                {
                                    xssMatchStatement: {
                                        fieldToMatch: {
                                            singleHeader: {
                                                name: 'Cookie',
                                            },
                                        },
                                        textTransformations: [
                                            {
                                                priority: 1,
                                                type: 'URL_DECODE',
                                            },
                                            {
                                                priority: 2,
                                                type: 'HTML_ENTITY_DECODE',
                                            },
                                        ],
                                    },
                                },
                            ],
                        },
                    },
                },
                ...badBotWafRules,
            ],
        });
        if (props.associatedResourceArn) {
            new wafv2.CfnWebACLAssociation(this, 'wafAssociation', {
                webAclArn: wafweb.attrArn,
                resourceArn: props.associatedResourceArn,
            });
        }
        //Lambda
        new lambda.DockerImageFunction(this, 'Helper', {
            description: "This lambda function verifies the main project's dependencies, requirements and implement auxiliary functions.",
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../source/helper'), {
                cmd: ['helper.lambda_handler'],
            }),
            memorySize: 512,
            timeout: cdk.Duration.seconds(300),
            environment: {
                LOG_LEVEL: logLevel,
                SCOPE: props.waf2Scope,
            },
        });
        const logParserLambda = new lambda.DockerImageFunction(this, 'LogParser', {
            description: 'This function parses access logs to identify suspicious behavior, such as an abnormal amount of errors. It then blocks those IP addresses for a customer-defined period of time.',
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../source/log_parser'), {
                cmd: ['log-parser.lambda_handler'],
            }),
            memorySize: 512,
            timeout: cdk.Duration.seconds(300),
            environment: {
                APP_ACCESS_LOG_BUCKET: accessLogBucket.bucketName,
                WAF_ACCESS_LOG_BUCKET: wafLogBucket.bucketName,
                SEND_ANONYMOUS_USAGE_DATA: 'No',
                UUID: 'CreateUniqueID.UUID',
                LIMIT_IP_ADDRESS_RANGES_PER_IP_MATCH_CONDITION: '10000',
                MAX_AGE_TO_UPDATE: '30',
                REGION: 'AWS::Region',
                SCOPE: props.waf2Scope,
                LOG_TYPE: props.waf2Scope == Waf2ScopeOption.CLOUDFRONT ? 'cloudfront' : 'alb',
                METRIC_NAME_PREFIX: cdk.Fn.ref('AWS::StackName'),
                LOG_LEVEL: logLevel,
                STACK_NAME: cdk.Fn.ref('AWS::StackName'),
                IP_SET_ID_HTTP_FLOODV4: httpFloodIpSetV4.attrArn,
                IP_SET_ID_HTTP_FLOODV6: httpFloodIpSetV6.attrArn,
                IP_SET_NAME_HTTP_FLOODV4: httpFloodIpSetV4.name,
                IP_SET_NAME_HTTP_FLOODV6: httpFloodIpSetV6.name,
                IP_SET_ID_SCANNERS_PROBESV4: scannersProbesIpSetV4.attrArn,
                IP_SET_ID_SCANNERS_PROBESV6: scannersProbesIpSetV6.attrArn,
                IP_SET_NAME_SCANNERS_PROBESV4: scannersProbesIpSetV4.name,
                IP_SET_NAME_SCANNERS_PROBESV6: scannersProbesIpSetV6.name,
                WAF_BLOCK_PERIOD: String(blockPeriod),
                ERROR_THRESHOLD: String(errorThreshold),
                REQUEST_THRESHOLD: String(requestThreshold),
                SOLUTION_ID: 'SO8128',
                METRICS_URL: 'https://metrics.awssolutionsbuilder.com/generic',
            },
        });
        const moveLogToPartitionLambda = new lambda.DockerImageFunction(this, 'MoveS3LogsForPartition', {
            description: 'This function is triggered by S3 event to move log files(upon their arrival in s3) from their original location to a partitioned folder structure created per timestamps in file names, hence allowing the usage of partitioning within AWS Athena.',
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../source/log_parser'), {
                cmd: ['partition_s3_logs.lambda_handler'],
            }),
            memorySize: 512,
            timeout: cdk.Duration.seconds(300),
            environment: {
                ENDPOINT: props.waf2Scope == Waf2ScopeOption.CLOUDFRONT
                    ? 'CloudFront'
                    : 'Alb',
                KEEP_ORIGINAL_DATA: 'No',
                LOG_LEVEL: logLevel,
            },
        });
        new lambda.DockerImageFunction(this, 'CustomTimer', {
            description: 'This lambda function counts X seconds and can be used to slow down component creation in CloudFormation',
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../source/timer'), {
                cmd: ['timer.lambda_handler'],
            }),
            memorySize: 128,
            timeout: cdk.Duration.seconds(300),
            environment: {
                LOG_LEVEL: logLevel,
                SECONDS: '2',
            },
        });
        //IP reputation list Lambda
        const reputationListRole = new iam.Role(this, 'LambdaRoleReputationListsParser', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        });
        reputationListRole.attachInlinePolicy(new iam.Policy(this, 'CloudWatchLogs', {
            policyName: 'CloudWatchLogs',
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    resources: ['*'],
                    actions: [
                        'logs:CreateLogGroup',
                        'logs:CreateLogStream',
                        'logs:PutLogEvents',
                    ],
                }),
            ],
        }));
        reputationListRole.attachInlinePolicy(new iam.Policy(this, 'WAFGetAndUpdateIPSet', {
            policyName: 'WAFGetAndUpdateIPSet',
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    resources: ['*'],
                    actions: ['wafv2:GetIPSet', 'wafv2:UpdateIPSet'],
                }),
            ],
        }));
        reputationListRole.attachInlinePolicy(new iam.Policy(this, 'CloudFormationAccess', {
            policyName: 'CloudFormationAccess',
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    resources: ['*'],
                    actions: ['cloudformation:DescribeStacks'],
                }),
            ],
        }));
        reputationListRole.attachInlinePolicy(new iam.Policy(this, 'CloudWatchAccess', {
            policyName: 'CloudWatchAccess',
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    resources: ['*'],
                    actions: ['cloudwatch:GetMetricStatistics'],
                }),
            ],
        }));
        const reputationListsParserLambda = new lambda.DockerImageFunction(this, 'ReputationListsParser', {
            description: 'This lambda function checks third-party IP reputation lists hourly for new IP ranges to block. These lists include the Spamhaus Dont Route Or Peer (DROP) and Extended Drop (EDROP) lists, the Proofpoint Emerging Threats IP list, and the Tor exit node list.',
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../source/reputation_lists_parser'), {
                cmd: ['reputation-lists.lambda_handler'],
            }),
            memorySize: 512,
            timeout: cdk.Duration.seconds(300),
            role: reputationListRole,
            environment: {
                IP_SET_ID_REPUTATIONV4: reputationListsIpSetV4.attrArn,
                IP_SET_ID_REPUTATIONV6: reputationListsIpSetV6.attrArn,
                IP_SET_NAME_REPUTATIONV4: reputationListsIpSetV4.name,
                IP_SET_NAME_REPUTATIONV6: reputationListsIpSetV6.name,
                SCOPE: props.waf2Scope,
                LOG_LEVEL: logLevel,
                URL_LIST: '[{"url":"https://www.spamhaus.org/drop/drop.txt"},{"url":"https://www.spamhaus.org/drop/edrop.txt"},{"url":"https://check.torproject.org/exit-addresses", "prefix":"ExitAddress"},{"url":"https://rules.emergingthreats.net/fwrules/emerging-Block-IPs.txt"}]',
                SOLUTION_ID: 'SO8128',
                METRICS_URL: 'https://metrics.awssolutionsbuilder.com/generic',
                STACK_NAME: cdk.Fn.ref('AWS::StackName'),
                LOG_TYPE: props.waf2Scope == Waf2ScopeOption.CLOUDFRONT
                    ? 'cloudfront'
                    : 'alb',
                SEND_ANONYMOUS_USAGE_DATA: 'No',
                IPREPUTATIONLIST_METRICNAME: reputationListName,
            },
        });
        //Add CloudWatch event to Lambda
        const reputationListsParserRule = new events.Rule(this, 'ReputationListsParserEventsRule', {
            description: 'Security Automation - WAF Reputation Lists',
            schedule: events.Schedule.expression('rate(1 hour)'),
        });
        const reputationListsParserRuleInput = {
            URL_LIST: [
                {
                    url: 'https://www.spamhaus.org/drop/drop.txt',
                },
                {
                    url: 'https://www.spamhaus.org/drop/edrop.txt',
                },
                {
                    url: 'https://check.torproject.org/exit-addresses',
                    prefix: 'ExitAddress',
                },
                {
                    url: 'https://rules.emergingthreats.net/fwrules/emerging-Block-IPs.txt',
                },
            ],
            IP_SET_ID_REPUTATIONV4: reputationListsIpSetV4.attrArn,
            IP_SET_ID_REPUTATIONV6: reputationListsIpSetV6.attrArn,
            IP_SET_NAME_REPUTATIONV4: reputationListsIpSetV4.name,
            IP_SET_NAME_REPUTATIONV6: reputationListsIpSetV6.name,
            SCOPE: props.waf2Scope,
        };
        reputationListsParserRule.addTarget(new targets.LambdaFunction(reputationListsParserLambda, {
            event: events.RuleTargetInput.fromObject(reputationListsParserRuleInput),
        }));
        if (enableShieldAdvancedLambda) {
            //AWS Shield Advanced Lambda
            const shieldRole = new iam.Role(this, 'ShieldAdvanceRole', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            shieldRole.attachInlinePolicy(new iam.Policy(this, 'ShieldAdvanceAccess', {
                policyName: 'CloudFrontShieldAdvanceAccess',
                statements: [
                    new iam.PolicyStatement({
                        effect: iam.Effect.ALLOW,
                        resources: ['*'],
                        actions: ['shield:Create*'],
                    }),
                ],
            }));
            shieldRole.attachInlinePolicy(new iam.Policy(this, 'CloudFrontDistributionAccess', {
                policyName: 'CloudFrontDistributionAccess',
                statements: [
                    new iam.PolicyStatement({
                        effect: iam.Effect.ALLOW,
                        resources: ['*'],
                        actions: ['cloudfront:GetDistribution'],
                    }),
                ],
            }));
            shieldRole.attachInlinePolicy(new iam.Policy(this, 'ShieldLogAccess', {
                policyName: 'ShieldLogAccess',
                statements: [
                    new iam.PolicyStatement({
                        effect: iam.Effect.ALLOW,
                        resources: ['*'],
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:PutLogEvents',
                        ],
                    }),
                ],
            }));
            new lambda.DockerImageFunction(this, 'ShieldAdvancedLambda', {
                description: 'This lambda function create an AWS Shield resource protection and protection group for the cloudfront resource.',
                code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../source/shield_protection'), {
                    cmd: ['shield-protection.lambda_handler'],
                }),
                memorySize: 512,
                timeout: cdk.Duration.seconds(300),
                role: shieldRole,
                environment: {
                    SCOPE: props.waf2Scope,
                    LOG_LEVEL: logLevel,
                    SOLUTION_ID: 'SO8128',
                },
            });
        }
        if (props.waf2Scope == Waf2ScopeOption.CLOUDFRONT) {
            //Badbot protection Lambda
            const badBotRole = new iam.Role(this, 'BadBotRole', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            badBotRole.attachInlinePolicy(new iam.Policy(this, 'BadBotLogsAccess', {
                policyName: 'LogsAccess',
                statements: [
                    new iam.PolicyStatement({
                        effect: iam.Effect.ALLOW,
                        resources: ['*'],
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:PutLogEvents',
                        ],
                    }),
                ],
            }));
            badBotRole.attachInlinePolicy(new iam.Policy(this, 'BadBotCloudFormationAccess', {
                policyName: 'CloudFormationAccess',
                statements: [
                    new iam.PolicyStatement({
                        effect: iam.Effect.ALLOW,
                        resources: ['*'],
                        actions: ['cloudformation:DescribeStacks'],
                    }),
                ],
            }));
            badBotRole.attachInlinePolicy(new iam.Policy(this, 'BadBotCloudWatchAccess', {
                policyName: 'CloudWatchAccess',
                statements: [
                    new iam.PolicyStatement({
                        effect: iam.Effect.ALLOW,
                        resources: ['*'],
                        actions: ['cloudwatch:GetMetricStatistics'],
                    }),
                ],
            }));
            badBotRole.attachInlinePolicy(new iam.Policy(this, 'BadBotWAFGetAndUpdateIPSet', {
                policyName: 'WAFGetAndUpdateIPSet',
                statements: [
                    new iam.PolicyStatement({
                        effect: iam.Effect.ALLOW,
                        resources: ['*'],
                        actions: ['wafv2:GetIPSet', 'wafv2:UpdateIPSet'],
                    }),
                ],
            }));
            const badBotParserLambda = new lambda.DockerImageFunction(this, 'BadBotParser', {
                description: 'This lambda function will intercepts and inspects trap endpoint requests to extract its IP address, and then add it to an AWS WAF block list.',
                role: badBotRole,
                code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../source/access_handler'), {
                    cmd: ['access-handler.lambda_handler'],
                }),
                memorySize: 512,
                timeout: cdk.Duration.seconds(300),
                environment: {
                    SCOPE: props.waf2Scope,
                    IP_SET_ID_BAD_BOTV4: badBotIpSetV4.attrArn,
                    IP_SET_ID_BAD_BOTV6: badBotIpSetV6.attrArn,
                    IP_SET_NAME_BAD_BOTV4: badBotIpSetV4.name,
                    IP_SET_NAME_BAD_BOTV6: badBotIpSetV6.name,
                    SEND_ANONYMOUS_USAGE_DATA: 'No',
                    UUID: 'CreateUniqueID.UUID',
                    REGION: cdk.Fn.ref('AWS::Region'),
                    LOG_TYPE: 'cloudfront',
                    METRIC_NAME_PREFIX: cdk.Fn.ref('AWS::StackName'),
                    LOG_LEVEL: logLevel,
                    SOLUTION_ID: 'SO8128',
                    METRICS_URL: 'https://metrics.awssolutionsbuilder.com/generic',
                    STACK_NAME: cdk.Fn.ref('AWS::StackName'),
                },
            });
            //API Gateway for badbot detection
            const badBotApi = new apigateway.RestApi(this, 'ApiGatewayBadBot', {
                restApiName: 'Security Automation - WAF Bad Bot API',
                description: 'API created by AWS WAF Security Automation CloudFormation template. This endpoint will be used to capture bad bots.',
            });
            const integration = new apigateway.LambdaIntegration(badBotParserLambda);
            badBotApi.root.addMethod('ANY', integration, {
                requestParameters: {
                    'method.request.header.X-Forwarded-For': false,
                },
            });
            const badBotProxy = badBotApi.root.addResource('{proxy+}');
            badBotProxy.addMethod('ANY', integration, {
                requestParameters: {
                    'method.request.header.X-Forwarded-For': false,
                },
            });
        }
        const customResourceLambda = new lambda.DockerImageFunction(this, 'CustomResource', {
            description: 'This lambda function configures the Web ACL rules based on the features enabled in the CloudFormation template.',
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../source/custom_resource'), {
                cmd: ['custom-resource.lambda_handler'],
            }),
            memorySize: 512,
            timeout: cdk.Duration.seconds(300),
            environment: {
                LOG_LEVEL: logLevel,
                SCOPE: props.waf2Scope,
                SOLUTION_ID: 'SO8128',
                METRICS_URL: 'https://metrics.awssolutionsbuilder.com/generic',
            },
        });
        //Kinesis Data Firehose
        const firehoseRole = new iam.Role(this, 'FirehoseRole', {
            assumedBy: new iam.ServicePrincipal('firehose.amazonaws.com'),
        });
        firehoseRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'kinesis:DescribeStream',
                'kinesis:GetShardIterator',
                'kinesis:GetRecords',
            ],
            resources: ['*'],
        }));
        firehoseRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                's3:AbortMultipartUpload',
                's3:GetBucketLocation',
                's3:GetObject',
                's3:ListBucket',
                's3:ListBucketMultipartUploads',
                's3:PutObject',
            ],
            resources: [wafLogBucket.bucketArn, wafLogBucket.arnForObjects('*')],
        }));
        const firehoseStream = new firehose.CfnDeliveryStream(this, 'FirehoseWAFLogsDeliveryStream', {
            deliveryStreamName: 'aws-waf-logs-' + stack.stackName,
            deliveryStreamType: 'DirectPut',
            extendedS3DestinationConfiguration: {
                bucketArn: wafLogBucket.bucketArn,
                bufferingHints: {
                    intervalInSeconds: 300,
                    sizeInMBs: 5,
                },
                compressionFormat: 'GZIP',
                prefix: 'AWSLogs/year=!{timestamp:yyyy}/month=!{timestamp:MM}/day=!{timestamp:dd}/hour=!{timestamp:HH}/',
                errorOutputPrefix: 'AWSErrorLogs/result=!{firehose:error-output-type}/year=!{timestamp:yyyy}/month=!{timestamp:MM}/day=!{timestamp:dd}/hour=!{timestamp:HH}/',
                roleArn: firehoseRole.roleArn,
            },
        });
        //Glue DB & table
        const glueDatabasePostfix = props.resourceNamingPrefix
            ? `_${props.resourceNamingPrefix.toLowerCase()}`
            : '';
        const glueAccessLogsDatabase = new glue.Database(this, 'GlueAccessLogsDatabase', {
            databaseName: `glue_accesslogs_database${glueDatabasePostfix}`,
        });
        const glueWafAccessLogsTable = new glue.Table(this, 'GlueWafAccessLogsTable', {
            database: glueAccessLogsDatabase,
            tableName: 'waf_access_logs',
            description: stack.stackName + ' - WAF Logs',
            bucket: wafLogBucket,
            s3Prefix: 'AWSLogs/',
            columns: [
                {
                    name: 'timestamp',
                    type: glue.Schema.BIG_INT,
                },
                {
                    name: 'formatversion',
                    type: glue.Schema.INTEGER,
                },
                {
                    name: 'webaclid',
                    type: glue.Schema.STRING,
                },
                {
                    name: 'terminatingruleid',
                    type: glue.Schema.STRING,
                },
                {
                    name: 'terminatingruletype',
                    type: glue.Schema.STRING,
                },
                {
                    name: 'action',
                    type: glue.Schema.STRING,
                },
                {
                    name: 'httpsourcename',
                    type: glue.Schema.STRING,
                },
                {
                    name: 'httpsourceid',
                    type: glue.Schema.STRING,
                },
                {
                    name: 'rulegrouplist',
                    type: glue.Schema.array(glue.Schema.STRING),
                },
                {
                    name: 'ratebasedrulelist',
                    type: glue.Schema.array(glue.Schema.STRING),
                },
                {
                    name: 'nonterminatingmatchingrules',
                    type: glue.Schema.array(glue.Schema.STRING),
                },
                {
                    name: 'httprequest',
                    type: glue.Schema.struct([
                        {
                            name: 'clientip',
                            type: glue.Schema.STRING,
                        },
                        {
                            name: 'country',
                            type: glue.Schema.STRING,
                        },
                        {
                            name: 'headers',
                            type: glue.Schema.array(glue.Schema.struct([
                                {
                                    name: 'name',
                                    type: glue.Schema.STRING,
                                },
                                {
                                    name: 'value',
                                    type: glue.Schema.STRING,
                                },
                            ])),
                        },
                        {
                            name: 'uri',
                            type: glue.Schema.STRING,
                        },
                        {
                            name: 'args',
                            type: glue.Schema.STRING,
                        },
                        {
                            name: 'httpversion',
                            type: glue.Schema.STRING,
                        },
                        {
                            name: 'httpmethod',
                            type: glue.Schema.STRING,
                        },
                        {
                            name: 'requestid',
                            type: glue.Schema.STRING,
                        },
                    ]),
                },
            ],
            partitionKeys: [
                {
                    name: 'year',
                    type: glue.Schema.INTEGER,
                },
                {
                    name: 'month',
                    type: glue.Schema.INTEGER,
                },
                {
                    name: 'day',
                    type: glue.Schema.INTEGER,
                },
                {
                    name: 'hour',
                    type: glue.Schema.INTEGER,
                },
            ],
            dataFormat: {
                inputFormat: new glue.InputFormat('org.apache.hadoop.mapred.TextInputFormat'),
                outputFormat: new glue.OutputFormat('org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat'),
                serializationLibrary: new glue.SerializationLibrary('org.openx.data.jsonserde.JsonSerDe'),
            },
        });
        // const df = new glue.DataFormat({
        //   inputFormat: new glue.InputFormat('org.apache.hadoop.mapred.TextInputFormat'),
        //   outputFormat: new glue.OutputFormat('org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat'),
        //   serializationLibrary: new glue.SerializationLibrary('org.openx.data.jsonserde.JsonSerDe'),
        // });
        if (props.waf2Scope == Waf2ScopeOption.CLOUDFRONT) {
            new glue.CfnTable(this, 'glueAppAccessLogsTable', {
                databaseName: glueAccessLogsDatabase.databaseName,
                catalogId: cdk.Aws.ACCOUNT_ID,
                tableInput: {
                    name: 'app_access_logs',
                    description: stack.stackName + ' - APP Access Logs',
                    parameters: {
                        'skip.header.line.count': '2',
                        'EXTERNAL': 'TRUE',
                    },
                    storageDescriptor: {
                        columns: [
                            {
                                name: 'date',
                                type: glue.Schema.DATE.inputString,
                            },
                            {
                                name: 'time',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'location',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'bytes',
                                type: glue.Schema.BIG_INT.inputString,
                            },
                            {
                                name: 'requestip',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'method',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'host',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'uri',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'status',
                                type: glue.Schema.INTEGER.inputString,
                            },
                            {
                                name: 'referrer',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'useragent',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'querystring',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'cookie',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'resulttype',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'requestid',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'hostheader',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'requestprotocol',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'requestbytes',
                                type: glue.Schema.BIG_INT.inputString,
                            },
                            {
                                name: 'timetaken',
                                type: glue.Schema.FLOAT.inputString,
                            },
                            {
                                name: 'xforwardedfor',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'sslprotocol',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'sslcipher',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'responseresulttype',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'httpversion',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'filestatus',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'encryptedfields',
                                type: glue.Schema.INTEGER.inputString,
                            },
                        ],
                        compressed: false,
                        inputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                        location: 's3://' + accessLogBucket.bucketName + '/AWSLogs-Partitioned/',
                        outputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                        serdeInfo: {
                            serializationLibrary: 'org.apache.hadoop.hive.serde2.lazy.LazySimpleSerDe',
                            parameters: {
                                'field.delim': '	',
                                'serialization.format': '	',
                            },
                        },
                        storedAsSubDirectories: false,
                    },
                    tableType: 'EXTERNAL_TABLE',
                    partitionKeys: [
                        {
                            name: 'year',
                            type: glue.Schema.INTEGER.inputString,
                        },
                        {
                            name: 'month',
                            type: glue.Schema.INTEGER.inputString,
                        },
                        {
                            name: 'day',
                            type: glue.Schema.INTEGER.inputString,
                        },
                        {
                            name: 'hour',
                            type: glue.Schema.INTEGER.inputString,
                        },
                    ],
                },
            });
        }
        else {
            // glue access log format for Alb
            new glue.CfnTable(this, 'glueAppAccessLogsTable', {
                databaseName: glueAccessLogsDatabase.databaseName,
                catalogId: cdk.Aws.ACCOUNT_ID,
                tableInput: {
                    name: 'app_access_logs',
                    description: stack.stackName + ' - APP Access Logs',
                    parameters: {
                        EXTERNAL: 'TRUE',
                    },
                    storageDescriptor: {
                        columns: [
                            {
                                name: 'type',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'time',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'elb',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'client_ip',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'client_port',
                                type: glue.Schema.INTEGER.inputString,
                            },
                            {
                                name: 'target_ip',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'target_port',
                                type: glue.Schema.INTEGER.inputString,
                            },
                            {
                                name: 'request_processing_time',
                                type: glue.Schema.DOUBLE.inputString,
                            },
                            {
                                name: 'target_processing_time',
                                type: glue.Schema.DOUBLE.inputString,
                            },
                            {
                                name: 'response_processing_time',
                                type: glue.Schema.DOUBLE.inputString,
                            },
                            {
                                name: 'elb_status_code',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'target_status_code',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'received_bytes',
                                type: glue.Schema.BIG_INT.inputString,
                            },
                            {
                                name: 'sent_bytes',
                                type: glue.Schema.BIG_INT.inputString,
                            },
                            {
                                name: 'request_verb',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'request_url',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'request_proto',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'user_agent',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'ssl_cipher',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'ssl_protocol',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'target_group_arn',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'trace_id',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'domain_name',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'chosen_cert_arn',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'matched_rule_priority',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'request_creation_time',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'actions_executed',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'redirect_url',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'error_reason',
                                type: glue.Schema.STRING.inputString,
                            },
                        ],
                        compressed: false,
                        inputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                        location: 's3://' + accessLogBucket.bucketName + '/AWSLogs-Partitioned/',
                        outputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                        serdeInfo: {
                            serializationLibrary: 'org.apache.hadoop.hive.serde2.RegexSerDe',
                            parameters: {
                                'serialization.format': '1',
                                'input.regex': '([^ ]*) ([^ ]*) ([^ ]*) ([^ ]*):([0-9]*) ([^ ]*)[:-]([0-9]*) ([-.0-9]*) ([-.0-9]*) ([-.0-9]*) (|[-0-9]*) (-|[-0-9]*) ([-0-9]*) ([-0-9]*) "([^ ]*) ([^ ]*) (- |[^ ]*)" "([^"]*)" ([A-Z0-9-]+) ([A-Za-z0-9.-]*) ([^ ]*) "([^"]*)" "([^"]*)" "([^"]*)" ([-.0-9]*) ([^ ]*) "([^"]*)" "([^ ]*)" "([^ ].*)"',
                            },
                        },
                        storedAsSubDirectories: false,
                    },
                    tableType: 'EXTERNAL_TABLE',
                    partitionKeys: [
                        {
                            name: 'year',
                            type: glue.Schema.INTEGER.inputString,
                        },
                        {
                            name: 'month',
                            type: glue.Schema.INTEGER.inputString,
                        },
                        {
                            name: 'day',
                            type: glue.Schema.INTEGER.inputString,
                        },
                        {
                            name: 'hour',
                            type: glue.Schema.INTEGER.inputString,
                        },
                    ],
                },
            });
        }
        //Athena
        const athenaWorkGroupPostfix = (_h = props.resourceNamingPrefix) !== null && _h !== void 0 ? _h : '';
        const addPartitionAthenaQueryWorkGroup = new athena.CfnWorkGroup(this, 'WAFAddPartitionAthenaQueryWorkGroup', {
            name: `WAFAddPartitionAthenaQueryWorkGroup${athenaWorkGroupPostfix}`,
            description: 'Athena WorkGroup for adding Athena partition queries used by AWS WAF Security Automations Solution',
            state: 'ENABLED',
            recursiveDeleteOption: true,
            workGroupConfiguration: {
                publishCloudWatchMetricsEnabled: true,
                enforceWorkGroupConfiguration: false,
            },
        });
        const wAFLogAthenaQueryWorkGroup = new athena.CfnWorkGroup(this, 'WAFLogAthenaQueryWorkGroup', {
            name: `WAFLogAthenaQueryWorkGroup${athenaWorkGroupPostfix}`,
            description: 'Athena WorkGroup for WAF log queries used by AWS WAF Security Automations Solution',
            state: 'ENABLED',
            recursiveDeleteOption: true,
            workGroupConfiguration: {
                publishCloudWatchMetricsEnabled: true,
            },
        });
        const wAFAppAccessLogAthenaQueryWorkGroup = new athena.CfnWorkGroup(this, 'WAFAppAccessLogAthenaQueryWorkGroup', {
            name: `WAFAppAccessLogAthenaQueryWorkGroup${athenaWorkGroupPostfix}`,
            description: 'Athena WorkGroup for CloudFront or ALB application access log queries used by AWS WAF Security Automations Solution',
            state: 'ENABLED',
            recursiveDeleteOption: true,
            workGroupConfiguration: {
                publishCloudWatchMetricsEnabled: true,
            },
        });
        //Cloudwatch Dashboard
        new cloudwatch.CfnDashboard(this, 'MonitoringDashboard', {
            dashboardName: cloudWatchDashboardName,
            dashboardBody: JSON.stringify({
                widgets: [
                    {
                        type: 'metric',
                        x: 0,
                        y: 0,
                        width: 15,
                        height: 10,
                        properties: {
                            view: 'timeSeries',
                            stacked: false,
                            metrics: [
                                [
                                    'WAF',
                                    'BlockedRequests',
                                    'WebACL',
                                    'WAFWebACLMetricName',
                                    'Rule',
                                    'ALL',
                                ],
                                [
                                    'WAF',
                                    'AllowedRequests',
                                    'WebACL',
                                    'WAFWebACLMetricName',
                                    'Rule',
                                    'ALL',
                                ],
                            ],
                            region: 'us-east-1',
                            period: 300,
                        },
                    },
                ],
            }),
        });
        //Add CloudWatch event to Lambda LogParser
        const logParserWafLogRuleInput = {
            resourceType: 'LambdaAthenaWAFLogParser',
            glueAccessLogsDatabase: glueAccessLogsDatabase.databaseName,
            accessLogBucket: wafLogBucket.bucketName,
            glueWafAccessLogsTable: glueWafAccessLogsTable.tableName,
            athenaWorkGroup: wAFLogAthenaQueryWorkGroup.name,
        };
        new events.Rule(this, 'lambdaAthenaWAFLogParserRule', {
            description: 'Security Automation - WAF Logs Athena parser',
            schedule: events.Schedule.expression('rate(5 minutes)'),
            targets: [
                new targets.LambdaFunction(logParserLambda, {
                    event: events.RuleTargetInput.fromObject(logParserWafLogRuleInput),
                }),
            ],
        });
        const lambdaAthenaAppLogParserRuleInput = {
            resourceType: 'LambdaAthenaAppLogParser',
            glueAccessLogsDatabase: glueAccessLogsDatabase.databaseName,
            accessLogBucket: accessLogBucket.bucketName,
            glueAppAccessLogsTable: 'app_access_logs',
            athenaWorkGroup: wAFAppAccessLogAthenaQueryWorkGroup.name,
        };
        new events.Rule(this, 'lambdaAthenaAppLogParserRule', {
            description: 'Security Automation - App Logs Athena parser',
            schedule: events.Schedule.expression('rate(5 minutes)'),
            targets: [
                new targets.LambdaFunction(logParserLambda, {
                    event: events.RuleTargetInput.fromObject(lambdaAthenaAppLogParserRuleInput),
                }),
            ],
        });
        const addAthenaPartitionsLambda = new lambda.DockerImageFunction(this, 'AddAthenaPartitionsFunction', {
            description: 'This function adds a new hourly partition to athena table. It runs every hour, triggered by a CloudWatch event.',
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../source/log_parser'), {
                cmd: ['add_athena_partitions.lambda_handler'],
            }),
            memorySize: 512,
            timeout: cdk.Duration.minutes(5),
            maxEventAge: cdk.Duration.hours(6),
            retryAttempts: 2,
            environment: {
                LOG_LEVEL: logLevel,
            },
        });
        const addAthenaPartitionsRuleInput = {
            resourceType: 'LambdaAddAthenaPartitionsEventsRule',
            glueAccessLogsDatabase: glueAccessLogsDatabase.databaseName,
            accessLogBucket: accessLogBucket.bucketName,
            glueAppAccessLogsTable: 'app_access_logs',
            glueWafAccessLogsTable: glueWafAccessLogsTable.tableName,
            wafLogBucket: wafLogBucket.bucketName,
            athenaWorkGroup: addPartitionAthenaQueryWorkGroup.name,
        };
        new events.Rule(this, 'lambdaAddAthenaPartition', {
            description: 'Security Automations - Add partitions to Athena table',
            schedule: events.Schedule.expression('rate(1 hour)'),
            targets: [
                new targets.LambdaFunction(addAthenaPartitionsLambda, {
                    event: events.RuleTargetInput.fromObject(addAthenaPartitionsRuleInput),
                }),
            ],
        });
        addAthenaPartitionsLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                's3:AbortMultipartUpload',
                's3:GetBucketLocation',
                's3:GetObject',
                's3:ListBucket',
                's3:ListBucketMultipartUploads',
                's3:PutObject',
                'glue:GetTable',
                'glue:GetDatabase',
                'glue:UpdateDatabase',
                'glue:CreateDatabase',
                'glue:BatchCreatePartition',
                'logs:CreateLogGroup',
                'logs:CreateLogStream',
                'logs:PutLogEvents',
            ],
            resources: ['*'],
        }));
        addAthenaPartitionsLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['athena:StartQueryExecution'],
            resources: ['*'],
        }));
        logParserLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['glue:GetTable', 'glue:GetPartitions'],
            resources: [
                glueAccessLogsDatabase.catalogArn,
                glueAccessLogsDatabase.databaseArn,
                `arn:${cdk.Aws.PARTITION}:glue:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:table/${glueAccessLogsDatabase.databaseName}/waf_access_logs`,
                `arn:${cdk.Aws.PARTITION}:glue:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:table/${glueAccessLogsDatabase.databaseName}/app_access_logs`,
            ],
        }));
        logParserLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['wafv2:GetIPSet', 'wafv2:UpdateIPSet'],
            resources: [
                httpFloodIpSetV4.attrArn,
                httpFloodIpSetV6.attrArn,
                scannersProbesIpSetV4.attrArn,
                scannersProbesIpSetV6.attrArn,
            ],
        }));
        logParserLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['athena:GetNamedQuery', 'athena:StartQueryExecution'],
            resources: [
                `arn:${cdk.Aws.PARTITION}:athena:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:workgroup/WAF*`,
            ],
        }));
        logParserLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['cloudwatch:GetMetricStatistics'],
            resources: ['*'],
        }));
        logParserLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'logs:CreateLogGroup',
                'logs:CreateLogStream',
                'logs:PutLogEvents',
            ],
            resources: [
                `arn:${cdk.Aws.PARTITION}:athena:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*LogParser*`,
            ],
        }));
        logParserLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                's3:GetBucketLocation',
                's3:GetObject',
                's3:ListBucket',
                's3:ListBucketMultipartUploads',
                's3:ListMultipartUploadParts',
                's3:AbortMultipartUpload',
                's3:CreateBucket',
                's3:PutObject',
            ],
            resources: [
                accessLogBucket.bucketArn + '/athena_results/*',
                accessLogBucket.bucketArn + '/*',
                accessLogBucket.bucketArn,
                wafLogBucket.bucketArn + '/athena_results/*',
                wafLogBucket.bucketArn + '/*',
                wafLogBucket.bucketArn,
            ],
        }));
        // S3
        logParserLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                's3:GetBucketLocation',
                's3:GetObject',
                's3:ListBucket',
                's3:ListBucketMultipartUploads',
                's3:ListMultipartUploadParts',
                's3:AbortMultipartUpload',
                's3:CreateBucket',
                's3:PutObject',
            ],
            resources: [
                wafLogBucket.bucketArn + '/athena_results/*',
                accessLogBucket.bucketArn + '/athena_results/*',
                wafLogBucket.bucketArn,
                accessLogBucket.bucketArn,
            ],
        }));
        logParserLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject'],
            resources: [
                wafLogBucket.bucketArn + '/*',
                accessLogBucket.bucketArn + '/*',
            ],
        }));
        logParserLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:PutObject'],
            resources: [
                wafLogBucket.bucketArn + '/' + stack.stackName + '-waf_log_out.json',
                wafLogBucket.bucketArn + '/' + stack.stackName + '-waf_log_conf.json',
                accessLogBucket.bucketArn +
                    '/' +
                    stack.stackName +
                    '-waf_log_out.json',
                accessLogBucket.bucketArn +
                    '/' +
                    stack.stackName +
                    '-waf_log_conf.json',
            ],
        }));
        moveLogToPartitionLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'logs:CreateLogGroup',
                'logs:CreateLogStream',
                'logs:PutLogEvents',
            ],
            resources: [
                `arn:${cdk.Aws.PARTITION}:athena:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*MoveS3LogsForPartition*`,
            ],
        }));
        moveLogToPartitionLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:DeleteObject', 's3:PutObject'],
            resources: [accessLogBucket.bucketArn + '/*'],
        }));
        customResourceLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['cloudformation:DescribeStacks'],
            resources: [
                `arn:${cdk.Aws.PARTITION}:cloudformation:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:stack/${cdk.Stack.name}/*`,
            ],
        }));
        customResourceLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['lambda:InvokeFunction'],
            resources: [
                `arn:${cdk.Aws.PARTITION}:lambda:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:function:${stack.stackName}*`,
            ],
        }));
        customResourceLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'wafv2:GetIPSet',
                'wafv2:DeleteIPSet',
                'wafv2:GetWebACL',
                'wafv2:UpdateWebACL',
                'wafv2:DeleteLoggingConfiguration',
                'wafv2:PutLoggingConfiguration',
            ],
            resources: [wafweb.attrArn],
        }));
        customResourceLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['wafv2:GetIPSet', 'wafv2:DeleteIPSet'],
            resources: [
                `arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:regional/ipset/${stack.stackName}*`,
                `arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:global/ipset/${stack.stackName}*`,
            ],
        }));
        customResourceLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'logs:CreateLogGroup',
                'logs:CreateLogStream',
                'logs:PutLogEvents',
            ],
            resources: [
                `arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*CustomResource*`,
            ],
        }));
        customResourceLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                's3:GetBucketLocation',
                's3:ListBucket',
                's3:CreateBucket',
                's3:GetBucketNotification',
                's3:PutBucketNotification',
                's3:PutEncryptionConfiguration',
                's3:PutBucketPublicAccessBlock',
                's3:GetObject*',
                's3:PutObject*',
                's3:DeleteObject*',
            ],
            resources: [
                accessLogBucket.bucketArn,
                `${accessLogBucket.bucketArn}/*`,
            ],
        }));
        customResourceLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                's3:CreateBucket',
                's3:GetBucketNotification',
                's3:PutBucketNotification',
                's3:GetObject*',
                's3:PutObject*',
                's3:DeleteObject*',
            ],
            resources: [wafLogBucket.bucketArn, `${wafLogBucket.bucketArn}/*`],
        }));
        customResourceLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['iam:CreateServiceLinkedRole'],
            resources: [
                'arn:aws:iam::*:role/aws-service-role/wafv2.amazonaws.com/AWSServiceRoleForWAFV2Logging',
            ],
        }));
        wafLogBucket.addEventNotification(s3.EventType.OBJECT_CREATED, new s3n.LambdaDestination(logParserLambda), {
            prefix: 'athena_results/',
            suffix: 'csv',
        });
        accessLogBucket.addEventNotification(s3.EventType.OBJECT_CREATED, new s3n.LambdaDestination(moveLogToPartitionLambda), {
            prefix: 'AWSLogs/',
            suffix: 'gz',
        });
        accessLogBucket.addEventNotification(s3.EventType.OBJECT_CREATED, new s3n.LambdaDestination(logParserLambda), {
            prefix: 'athena_results/',
            suffix: 'csv',
        });
        const customResourceProvider = new cr.Provider(this, 'customResourceProvider', {
            onEventHandler: customResourceLambda,
        });
        new core_1.CustomResource(this, 'ConfigureAWSWAFLogs', {
            serviceToken: customResourceProvider.serviceToken,
            resourceType: 'Custom::ConfigureAWSWAFLogs',
            properties: {
                WAFWebACLArn: wafweb.attrArn,
                DeliveryStreamArn: firehoseStream.attrArn,
            },
        });
        new core_1.CustomResource(this, 'ConfigureAppAccessLogBucket', {
            serviceToken: customResourceProvider.serviceToken,
            resourceType: 'Custom::ConfigureAppAccessLogBucket',
            properties: {
                Region: cdk.Aws.REGION,
                AppAccessLogBucket: accessLogBucket.bucketName,
                LogParser: logParserLambda.functionArn,
                ScannersProbesAthenaLogParser: 'yes',
                MoveS3LogsForPartition: moveLogToPartitionLambda.functionName,
            },
        });
        new core_1.CustomResource(this, 'ConfigureWafLogBucket', {
            serviceToken: customResourceProvider.serviceToken,
            resourceType: 'Custom::ConfigureWafLogBucket',
            properties: {
                Region: cdk.Aws.REGION,
                WafLogBucket: wafLogBucket.bucketName,
                LogParser: logParserLambda.functionArn,
                HttpFloodAthenaLogParser: 'yes',
            },
        });
        new core_1.CustomResource(this, 'ConfigureWebAcl', {
            serviceToken: customResourceProvider.serviceToken,
            resourceType: 'Custom::ConfigureWebAcl',
            properties: {
                WAFWhitelistSetIPV4: whitelistIpSetV4.attrId,
                WAFWhitelistSetIPV4Name: whitelistIpSetV4.name,
            },
        });
        new core_1.CustomResource(this, 'GenerateAppLogParserConfFile', {
            serviceToken: customResourceProvider.serviceToken,
            resourceType: 'Custom::GenerateAppLogParserConfFile',
            properties: {
                StackName: stack.stackName,
                ErrorThreshold: errorThreshold,
                WAFBlockPeriod: blockPeriod,
                AppAccessLogBucket: accessLogBucket.bucketName,
            },
        });
        new core_1.CustomResource(this, 'GenerateWafLogParserConfFile', {
            serviceToken: customResourceProvider.serviceToken,
            resourceType: 'Custom::GenerateWafLogParserConfFile',
            properties: {
                StackName: stack.stackName,
                RequestThreshold: requestThreshold,
                WAFBlockPeriod: blockPeriod,
                WafAccessLogBucket: wafLogBucket.bucketName,
            },
        });
        new core_1.CustomResource(this, 'AddAthenaPartitions', {
            serviceToken: customResourceProvider.serviceToken,
            resourceType: 'Custom::AddAthenaPartitions',
            properties: {
                ResourceType: 'Custom::AddAthenaPartitions',
                AddAthenaPartitionsLambda: addAthenaPartitionsLambda.functionName,
                GlueAccessLogsDatabase: glueAccessLogsDatabase.databaseName,
                AppAccessLogBucket: accessLogBucket.bucketName,
                GlueAppAccessLogsTable: 'app_access_logs',
                GlueWafAccessLogsTable: glueWafAccessLogsTable.tableName,
                WafLogBucket: wafLogBucket.bucketName,
                AthenaWorkGroup: addPartitionAthenaQueryWorkGroup.name,
            },
        });
        new cdk.CfnOutput(this, 'AppAccessLogBucketName', {
            value: accessLogBucket.bucketName,
        });
        new cdk.CfnOutput(this, 'WafLogBucketName', {
            value: wafLogBucket.bucketName,
        });
        new cdk.CfnOutput(this, 'WAFWebACLName', { value: wafweb.attrId });
        new cdk.CfnOutput(this, 'WAFWebACLArn', { value: wafweb.attrArn });
    }
    /**
     * @experimental
     */
    validateResourceNamingPrefix(resourceNamingPrefix) {
        return /^[a-zA-Z]+[a-zA-Z0-9_]+$/.test(resourceNamingPrefix);
    }
}
exports.AutomatedWaf = AutomatedWaf;
_a = JSII_RTTI_SYMBOL_1;
AutomatedWaf[_a] = { fqn: "@ikala-cloud/aws-waf-solution.AutomatedWaf", version: "1.0.26" };
//# sourceMappingURL=data:application/json;base64,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