# -*- coding: utf-8 -*-
import os
import nose
from nose import tools
import scipy as sp
import numpy as np
from numpy import testing

# dynfunconn
from dyconnmap.fc import (
    aec,
    #   biplv,
    coherence,
    Coherence,
    dpli,
    esc,
    glm,
    icoherence,
    iplv,
    iplv_fast,
    # mi,
    mutual_information,
    nesc,
    pac,
    pec,
    pli,
    plv,
    PLV,
    plv_fast,
    rho_index,
    cos,
    # sl,
    wpli,
    dwpli,
    corr,
    Corr,
    crosscorr,
    partcorr,
)


def test_aec():
    data = np.load("../examples/data/eeg_32chans_10secs.npy")

    corr = aec(data, [1.0, 4.0], [20.0, 45.0], 128)

    # np.save("data/test_aec.npy", corr)

    expected = np.load("data/test_aec.npy")
    np.testing.assert_array_almost_equal(corr, expected)


def test_coherence():
    data = np.load("../examples/data/eeg_32chans_10secs.npy")

    csdparams = {"NFFT": 256, "noverlap": 256 / 2.0}

    coh = coherence(data, [1.0, 4.0], 128.0, **csdparams)

    # np.save("data/test_coherence.npy", coh)

    expected = np.load("data/test_coherence.npy")
    # np.testing.assert_array_equal(coh, expected)
    np.testing.assert_array_almost_equal(coh, expected)


def test_coherence_class():
    data = np.load("../examples/data/eeg_32chans_10secs.npy")

    csdparams = {"NFFT": 256, "noverlap": 256 / 2.0}

    coh = Coherence([1.0, 4.0], 128.0, **csdparams)
    pp_data = coh.preprocess(data)
    _, avg = coh.estimate(pp_data)

    expected = np.load("data/test_coherence.npy")

    avg = avg + avg.T
    np.fill_diagonal(avg, 1.0)

    # np.save("data/test_coherence_class.npy", avg)

    # np.testing.assert_array_equal(coh, expected)
    np.testing.assert_array_almost_equal(avg, expected)


def test_dpli():
    data = np.load("../examples/data/eeg_32chans_10secs.npy")

    dpliv = dpli(data, [1.0, 4.0], 128.0)
    expected = np.load("data/test_dpli.npy")
    np.testing.assert_array_equal(dpliv, expected)


def test_esc():
    data = np.load("../examples/data/eeg_32chans_10secs.npy")

    escv = esc(data, [1.0, 4.0], [20.0, 45.0], 128.0)

    expected = np.load("data/test_esc.npy")
    np.testing.assert_array_almost_equal(escv, expected)


def test_glm():
    data = np.load("../examples/data/eeg_32chans_10secs.npy")

    num_ts, ts_len = np.shape(data)
    window_size = ts_len / 2.0

    fb_lo = [4.0, 8.0]
    fb_hi = [25.0, 40.0]
    fs = 128.0

    ts, ts_avg = glm(data, fb_lo, fb_hi, fs, pairs=None, window_size=window_size)

    expected = np.load("data/test_glm_ts.npy")
    np.testing.assert_array_equal(ts, expected)

    expected = np.load("data/test_glm_avg.npy")
    np.testing.assert_array_equal(ts_avg, expected)


def test_icoherence():
    data = np.load("../examples/data/eeg_32chans_10secs.npy")

    csdparams = {"NFFT": 256, "noverlap": 256 / 2.0}

    icoh = icoherence(data, [1.0, 4.0], 128.0, **csdparams)

    # np.save("data/test_icoherence.npy", icoh)

    expected = np.load("data/test_icoherence.npy")
    # np.testing.assert_array_equal(icoh, expected)
    np.testing.assert_array_almost_equal(icoh, expected)


def test_iplv():
    data = np.load("../examples/data/eeg_32chans_10secs.npy")
    ts, avg = iplv(data, [1.0, 4.0], 128.0)

    ts = np.float32(ts)
    avg = np.float32(avg)

    # np.save("data/test_iplv_ts.npy", ts)
    # np.save("data/test_iplv_avg.npy", avg)

    expected_ts = np.load("data/test_iplv_ts.npy")
    expected_ts = np.float32(expected_ts)
    np.testing.assert_array_almost_equal(ts, expected_ts)

    expected_avg = np.load("data/test_iplv_avg.npy")
    expected_avg = np.float32(expected_avg)
    np.testing.assert_array_almost_equal(avg, expected_avg)


def test_iplv_nofilter():
    data = np.load("../examples/data/rois39_samples100.npy")
    ts, avg = iplv(data)

    if "TRAVIS" in os.environ:
        # We have to use the following to make the test work on Travis
        expected_ts = np.load("data/test_iplv_nofilter_ts.npy")
        np.testing.assert_array_almost_equal(ts, expected_ts)

        expected_avg = np.load("data/test_iplv_nofilter_avg.npy")
        np.testing.assert_array_almost_equal(avg, expected_avg)
    else:
        # The following tests pass locally; but they fail on Travis o_O
        expected_ts = np.load("data/test_iplv_nofilter_ts.npy")
        np.testing.assert_array_almost_equal(ts, expected_ts)
        expected_avg = np.load("data/test_iplv_nofilter_avg.npy")
        np.testing.assert_array_almost_equal(avg, expected_avg)


def test_fast_iplv_nofilter():
    data = np.load("../examples/data/rois39_samples100.npy")
    avg = iplv_fast(data)

    # iPLV returns a fully symmetrical matrix, so we have to
    # fill with zeros the diagonal and the lower triagular
    np.fill_diagonal(avg, 0.0)
    avg[np.tril_indices_from(avg)] = 0.0

    avg = np.float32(avg)

    expected_avg = np.load("data/test_iplv_nofilter_avg.npy")
    expected_avg = np.float32(expected_avg)
    np.testing.assert_array_equal(avg, expected_avg)


def test_mui():
    pass


def test_nesc():
    data = np.load("../examples/data/eeg_32chans_10secs.npy")
    nescv = nesc(data, [4.0, 7.0], [20.0, 45.0], 128)

    expected = np.load("data/test_nesc.npy")
    np.testing.assert_array_equal(nescv, expected)


def test_pac_one_channel():
    data = np.load("../examples/data/eeg_32chans_10secs.npy")
    data = data[0:1, 0:128]

    fs = 128
    f_lo = [1.0, 4.0]
    f_hi = [20.0, 30.0]

    estimator = PLV(f_lo, fs)
    ts, avg = pac(data, f_lo, f_hi, fs, estimator)

    avg = np.squeeze(np.real(avg))

    expected = 0.468296707219
    nose.tools.assert_almost_equal(avg, expected)


def test_pac_multiple_channels():
    data = np.load("../examples/data/eeg_32chans_10secs.npy")

    fs = 128
    fb_lo = [1.0, 4.0]
    fb_hi = [20.0, 30.0]

    estimator = PLV(fb_lo, fs)
    ts, avg = pac(data, fb_lo, fb_hi, 128, estimator)
    ts = np.nan_to_num(ts)
    avg = np.nan_to_num(avg)

    expected_ts = np.load("data/test_pac_plv_ts.npy")
    expected_ts = np.nan_to_num(ts)
    np.testing.assert_array_equal(ts, expected_ts)
    avg = np.float32(avg)

    expected_avg = np.load("data/test_pac_plv_avg.npy")
    expected_avg = np.nan_to_num(expected_avg)
    expected_avg = np.float32(expected_avg)
    np.testing.assert_allclose(avg, expected_avg, rtol=1e-10, atol=0.0)


def test_pec():
    data = np.load("../examples/data/eeg_32chans_10secs.npy")

    fb_lo = [1.0, 4.0]
    fb_hi = [25, 40.0]
    fs = 128

    v = pec(data, fb_lo, fb_hi, fs)

    # np.save("data/test_pec.npy", v)

    expected = np.load("data/test_pec.npy")
    np.testing.assert_array_almost_equal(v, expected)


def test_pli():
    data = np.load("../examples/data/eeg_32chans_10secs.npy")
    n_channels, n_samples = np.shape(data)

    pairs = [
        (r1, r2) for r1 in range(n_channels) for r2 in range(n_channels) if r1 != r2
    ]
    ts, avg = pli(data, [1.0, 4.0], 128.0, pairs)

    # np.save("data/test_pli_ts.npy", ts)
    # np.save("data/test_pli_avg.npy", avg)

    expected_ts = np.load("data/test_pli_ts.npy")
    np.testing.assert_array_almost_equal(ts, expected_ts)

    expected_avg = np.load("data/test_pli_avg.npy")
    np.testing.assert_array_almost_equal(avg, expected_avg)


def test_plv():
    data = np.load("../examples/data/eeg_32chans_10secs.npy")
    ts, avg = plv(data)

    ts = np.float32(ts)
    avg = np.float32(avg)

    # np.save("data/test_plv_ts.npy", ts)
    # np.save("data/test_plv_avg.npy", avg)

    expected_ts = np.load("data/test_plv_ts.npy")
    expected_ts = np.float32(expected_ts)
    np.testing.assert_array_equal(ts, expected_ts)  # , rtol=1e-10, atol=0.0)

    expected_avg = np.load("data/test_plv_avg.npy")
    expected_avg = np.float32(expected_avg)
    np.testing.assert_array_equal(avg, expected_avg)  # , rtol=1e-10, atol=0.0)


def test_fast_plv():
    data = np.load("../examples/data/eeg_32chans_10secs.npy")
    avg = plv_fast(data)

    avg = np.float32(avg)

    # PLV returns a fully symmetrical matrix, so we have to
    # fill with zeros the diagonal and the lower triagular
    np.fill_diagonal(avg, 0.0)
    avg[np.tril_indices_from(avg)] = 0.0

    expected_avg = np.load("data/test_plv_avg.npy")
    np.testing.assert_allclose(avg, expected_avg, rtol=1e-10, atol=0.0)


def test_rho_index():
    data = np.load("../examples/data/eeg_32chans_10secs.npy")

    rho_mtx = rho_index(data, 10, [1.0, 4.0], 128.0)

    expected = np.load("data/test_rho_index.npy")
    np.testing.assert_array_equal(rho_mtx, expected)


def test_sl():
    pass


def test_wpli():
    data = np.load("../examples/data/eeg_32chans_10secs.npy")

    csdparams = {"NFFT": 256, "noverlap": 256 / 2.0}

    wpliv = wpli(data, [1.0, 4.0], 128.0, **csdparams)
    wpliv = np.nan_to_num(wpliv)

    expected = np.load("data/test_wpli.npy")
    expected = np.nan_to_num(expected)
    np.testing.assert_allclose(wpliv, expected, rtol=1e-10, atol=0.0)


def test_dwpli():
    data = np.load("../examples/data/eeg_32chans_10secs.npy")

    csdparams = {"NFFT": 256, "noverlap": 256 / 2.0}

    dwpliv = dwpli(data, [1.0, 4.0], 128.0, **csdparams)
    dwpliv = np.nan_to_num(dwpliv)

    expected = np.load("data/test_dwpli.npy")
    expected = np.nan_to_num(expected)

    np.testing.assert_allclose(dwpliv, expected, rtol=1e-10, atol=0.0)


def test_corr():
    data = np.load("../examples/data/eeg_32chans_10secs.npy")

    r = corr(data, [1.0, 4.0], 128.0)

    expected = np.load("data/test_corr.npy")
    np.testing.assert_array_equal(r, expected)


def test_corr_class():
    data = np.load("../examples/data/eeg_32chans_10secs.npy")

    obj_corr = Corr([1.0, 4.0], 128.0)
    pp_data = obj_corr.preprocess(data)
    r = obj_corr.estimate(pp_data)

    expected = np.load("data/test_corr.npy")
    np.testing.assert_array_equal(r, expected)


def test_crosscorr():
    data = np.load("../examples/data/eeg_32chans_10secs.npy")

    r = crosscorr(data, [1.0, 4.0], 128.0)

    expected = np.load("data/test_crosscorr.npy")
    np.testing.assert_array_equal(r, expected)


def test_partcorr():
    """

    Notes
    -----

    Disable test for now. It fails on Travis but passes in the development
    environment.

    """
    data = np.load("../examples/data/eeg_32chans_10secs.npy")

    r = partcorr(data, [1.0, 4.0], 128.0)

    expected = np.load("data/test_partcorr.npy")

    # if "TRAVIS" in os.environ:
    # np.testing.assert_allclose(r, expected, rtol=1e-10, atol=0.0)
    # else:
    # np.testing.assert_array_equal(r, expected)


def test_cos():
    data = np.load("../examples/data/eeg_32chans_10secs.npy")

    conn = cos(data, [1.0, 4.0], 128.0)

    expected = np.load("data/test_cos.npy")

    np.testing.assert_array_equal(conn, expected)


def test_cos_nofilter():
    data = np.load("../examples/data/eeg_32chans_10secs.npy")

    conn = cos(data)

    expected = np.load("data/test_cos_nofilter.npy")

    np.testing.assert_array_equal(conn, expected)
