"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const cdk = require("@aws-cdk/core");
const sns = require("../lib");
module.exports = {
    'topic tests': {
        'all defaults'(test) {
            const stack = new cdk.Stack();
            new sns.Topic(stack, 'MyTopic');
            assert_1.expect(stack).toMatch({
                'Resources': {
                    'MyTopic86869434': {
                        'Type': 'AWS::SNS::Topic',
                    },
                },
            });
            test.done();
        },
        'specify topicName'(test) {
            const stack = new cdk.Stack();
            new sns.Topic(stack, 'MyTopic', {
                topicName: 'topicName',
            });
            assert_1.expect(stack).toMatch({
                'Resources': {
                    'MyTopic86869434': {
                        'Type': 'AWS::SNS::Topic',
                        'Properties': {
                            'TopicName': 'topicName',
                        },
                    },
                },
            });
            test.done();
        },
        'specify displayName'(test) {
            const stack = new cdk.Stack();
            new sns.Topic(stack, 'MyTopic', {
                displayName: 'displayName',
            });
            assert_1.expect(stack).toMatch({
                'Resources': {
                    'MyTopic86869434': {
                        'Type': 'AWS::SNS::Topic',
                        'Properties': {
                            'DisplayName': 'displayName',
                        },
                    },
                },
            });
            test.done();
        },
        'specify kmsMasterKey'(test) {
            const stack = new cdk.Stack();
            const key = new kms.Key(stack, 'CustomKey');
            new sns.Topic(stack, 'MyTopic', {
                masterKey: key,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::Topic', {
                'KmsMasterKeyId': { 'Ref': 'CustomKey1E6D0D07' },
            }));
            test.done();
        },
        'specify both'(test) {
            const stack = new cdk.Stack();
            new sns.Topic(stack, 'MyTopic', {
                topicName: 'topicName',
                displayName: 'displayName',
            });
            assert_1.expect(stack).toMatch({
                'Resources': {
                    'MyTopic86869434': {
                        'Type': 'AWS::SNS::Topic',
                        'Properties': {
                            'DisplayName': 'displayName',
                            'TopicName': 'topicName',
                        },
                    },
                },
            });
            test.done();
        },
    },
    'can add a policy to the topic'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        topic.addToResourcePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['sns:*'],
            principals: [new iam.ArnPrincipal('arn')],
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::TopicPolicy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        'Sid': '0',
                        'Action': 'sns:*',
                        'Effect': 'Allow',
                        'Principal': { 'AWS': 'arn' },
                        'Resource': '*',
                    }],
            },
        }));
        test.done();
    },
    'give publishing permissions'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        const user = new iam.User(stack, 'User');
        // WHEN
        topic.grantPublish(user);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                Version: '2012-10-17',
                'Statement': [
                    {
                        'Action': 'sns:Publish',
                        'Effect': 'Allow',
                        'Resource': stack.resolve(topic.topicArn),
                    },
                ],
            },
        }));
        test.done();
    },
    'topic resource policy includes unique SIDs'(test) {
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'MyTopic');
        topic.addToResourcePolicy(new iam.PolicyStatement({
            actions: ['service:statement0'],
            principals: [new iam.ArnPrincipal('arn')],
        }));
        topic.addToResourcePolicy(new iam.PolicyStatement({
            actions: ['service:statement1'],
            principals: [new iam.ArnPrincipal('arn')],
        }));
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyTopic86869434': {
                    'Type': 'AWS::SNS::Topic',
                },
                'MyTopicPolicy12A5EC17': {
                    'Type': 'AWS::SNS::TopicPolicy',
                    'Properties': {
                        'PolicyDocument': {
                            'Statement': [
                                {
                                    'Action': 'service:statement0',
                                    'Effect': 'Allow',
                                    'Principal': { 'AWS': 'arn' },
                                    'Sid': '0',
                                },
                                {
                                    'Action': 'service:statement1',
                                    'Effect': 'Allow',
                                    'Principal': { 'AWS': 'arn' },
                                    'Sid': '1',
                                },
                            ],
                            'Version': '2012-10-17',
                        },
                        'Topics': [
                            {
                                'Ref': 'MyTopic86869434',
                            },
                        ],
                    },
                },
            },
        });
        test.done();
    },
    'fromTopicArn'(test) {
        // GIVEN
        const stack2 = new cdk.Stack();
        // WHEN
        const imported = sns.Topic.fromTopicArn(stack2, 'Imported', 'arn:aws:sns:*:123456789012:my_corporate_topic');
        // THEN
        test.deepEqual(imported.topicName, 'my_corporate_topic');
        test.deepEqual(imported.topicArn, 'arn:aws:sns:*:123456789012:my_corporate_topic');
        test.done();
    },
    'test metrics'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // THEN
        test.deepEqual(stack.resolve(topic.metricNumberOfMessagesPublished()), {
            dimensions: { TopicName: { 'Fn::GetAtt': ['TopicBFC7AF6E', 'TopicName'] } },
            namespace: 'AWS/SNS',
            metricName: 'NumberOfMessagesPublished',
            period: cdk.Duration.minutes(5),
            statistic: 'Sum',
        });
        test.deepEqual(stack.resolve(topic.metricPublishSize()), {
            dimensions: { TopicName: { 'Fn::GetAtt': ['TopicBFC7AF6E', 'TopicName'] } },
            namespace: 'AWS/SNS',
            metricName: 'PublishSize',
            period: cdk.Duration.minutes(5),
            statistic: 'Average',
        });
        test.done();
    },
    'subscription is created under the topic scope by default'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        topic.addSubscription({
            bind: () => ({
                protocol: sns.SubscriptionProtocol.HTTP,
                endpoint: 'http://foo/bar',
                subscriberId: 'my-subscription',
            }),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::Subscription'));
        test.done();
    },
    'if "scope" is defined, subscription will be created under that scope'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'A');
        const stack2 = new cdk.Stack(app, 'B');
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        topic.addSubscription({
            bind: () => ({
                protocol: sns.SubscriptionProtocol.HTTP,
                endpoint: 'http://foo/bar',
                subscriberScope: stack2,
                subscriberId: 'subscriberId',
            }),
        });
        // THEN
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::SNS::Subscription'));
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::SNS::Subscription'));
        test.done();
    },
    'fails if topic policy has no actions'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'my-stack');
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        topic.addToResourcePolicy(new iam.PolicyStatement({
            resources: ['*'],
            principals: [new iam.ArnPrincipal('arn')],
        }));
        // THEN
        test.throws(() => app.synth(), /A PolicyStatement must specify at least one \'action\' or \'notAction\'/);
        test.done();
    },
    'fails if topic policy has no IAM principals'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'my-stack');
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        topic.addToResourcePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['sns:*'],
        }));
        // THEN
        test.throws(() => app.synth(), /A PolicyStatement used in a resource-based policy must specify at least one IAM principal/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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