"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TopicBase = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const policy_1 = require("./policy");
const subscription_1 = require("./subscription");
/**
 * Either a new or imported Topic
 */
class TopicBase extends core_1.Resource {
    /**
     * Subscribe some endpoint to this topic
     */
    addSubscription(subscription) {
        const subscriptionConfig = subscription.bind(this);
        const scope = subscriptionConfig.subscriberScope || this;
        let id = subscriptionConfig.subscriberId;
        if (core_1.Token.isUnresolved(subscriptionConfig.subscriberId)) {
            id = this.nextTokenId(scope);
        }
        // We use the subscriber's id as the construct id. There's no meaning
        // to subscribing the same subscriber twice on the same topic.
        if (scope.node.tryFindChild(id)) {
            throw new Error(`A subscription with id "${id}" already exists under the scope ${scope.node.path}`);
        }
        new subscription_1.Subscription(scope, id, {
            topic: this,
            ...subscriptionConfig,
        });
    }
    /**
     * Adds a statement to the IAM resource policy associated with this topic.
     *
     * If this topic was created in this stack (`new Topic`), a topic policy
     * will be automatically created upon the first call to `addToPolicy`. If
     * the topic is imported (`Topic.import`), then this is a no-op.
     */
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.TopicPolicy(this, 'Policy', { topics: [this] });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    validate() {
        var _a;
        const errors = super.validate();
        errors.push(...((_a = this.policy) === null || _a === void 0 ? void 0 : _a.document.validateForResourcePolicy()) || []);
        return errors;
    }
    /**
     * Grant topic publishing permissions to the given identity
     */
    grantPublish(grantee) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: ['sns:Publish'],
            resourceArns: [this.topicArn],
            resource: this,
        });
    }
    nextTokenId(scope) {
        let nextSuffix = 1;
        const re = /TokenSubscription:([\d]*)/gm;
        // Search through the construct and all of its children
        // for previous subscriptions that match our regex pattern
        for (const source of scope.node.findAll()) {
            const m = re.exec(source.node.id); // Use regex to find a match
            if (m !== null) { // if we found a match
                const matchSuffix = parseInt(m[1], 10); // get the suffix for that match (as integer)
                if (matchSuffix >= nextSuffix) { // check if the match suffix is larger or equal to currently proposed suffix
                    nextSuffix = matchSuffix + 1; // increment the suffix
                }
            }
        }
        return `TokenSubscription:${nextSuffix}`;
    }
}
exports.TopicBase = TopicBase;
//# sourceMappingURL=data:application/json;base64,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