#!/usr/bin/env python3


# Copyright © 2019 Broadcom. All rights reserved.
# The term “Broadcom” refers to Broadcom Inc. and/or its subsidiaries.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may also obtain a copy of the License at
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


# lldp_profile_tlv_add.py(pyGen v1.0.0)


"""

:mod:`lldp_profile_tlv_add` - PyFOS util to create for lldp_profile
*******************************************************************************
The :mod:`lldp_profile_tlv_add` PyFOS util to create for lldp_profile


The LLDP profile.

lldp_profile_tlv_add : usage

* Infrastructure Options:
    * -i,--ipaddr=IPADDR: The IP address of the FOS switch.
    * -L,--login=LOGIN: The login name.
    * -P,--password=PASSWORD: The password.
    * -f,--vfid=VFID: The VFID to which the request is directed.
    * -s,--secured=MODE: The HTTPS mode "self" or "CA" [Optional].
    * -v,--verbose: Verbose mode [Optional].

* Util Script Options:
    * --tlv=TLV List of enabled TLVs for the LLDP profile. Empty enabled-tlvs\
      indicates that the tlvs will be derived from lldp global\
      optional-tlvs.
    * --name=NAME The LLDP profile name.
* Output:
    * Python dictionary content with RESTCONF response data.


.. function:: lldp_profile_tlv_add.create_lldp_profile(session,\
enabled_tlvs_tlv, name)

    *Create lldp_profile*

        Example Usage of the Method::

            ret = lldp_profile_tlv_add.create_lldp_profile(session,\
            enabled_tlvs_tlv, name)
            print (ret)

        Details::

            lldp_profileObj = lldp_profile()
            lldp_profileObj.set_enabled_tlvs_tlv(enabled_tlvs_tlv)
            lldp_profileObj.set_name(name)
            print (ret)

        * Input::

            :param session: The session returned by the login.
            :param enabled_tlvs_tlv: List of enabled TLVs for the LLDP\
              profile. Empty enabled-tlvs indicates that the tlvs will be\
              derived from lldp global optional-tlvs.
            :param name: The LLDP profile name.

        * Output:

            :rtype: Dictionary of response

"""


# Start utils imports
import sys
from pyfos import pyfos_auth
from pyfos import pyfos_util
from pyfos.pyfos_brocade_lldp import lldp_profile
from pyfos.utils import brcd_util
# End module imports


def _create_lldp_profile(session, lldp_profileObj):
    return lldp_profileObj.post(session)


def create_lldp_profile(session, enabled_tlvs_tlv=None, name=None):
    lldp_profileObj = lldp_profile()
    lldp_profileObj.set_enabled_tlvs_tlv(enabled_tlvs_tlv)
    lldp_profileObj.set_name(name)
    return _create_lldp_profile(session, lldp_profileObj)


def validate(lldp_profileObj):
    if lldp_profileObj.peek_enabled_tlvs_tlv() == "[]" or\
       lldp_profileObj.peek_name() is None:
        return 1
    return 0


def main(argv):
    filters = ["enabled_tlvs_tlv", "name"]
    inputs = brcd_util.parse(argv, lldp_profile, filters, validate)
    session = brcd_util.getsession(inputs)
    result = _create_lldp_profile(session, inputs['utilobject'])
    pyfos_util.response_print(result)
    pyfos_auth.logout(session)


if __name__ == "__main__":
    main(sys.argv[1:])
