import * as iam from '@aws-cdk/aws-iam';
import * as kms from '@aws-cdk/aws-kms';
import { Construct as CompatConstruct, IResource, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
/**
 * An SSM Parameter reference.
 *
 * @stability stable
 */
export interface IParameter extends IResource {
    /**
     * The ARN of the SSM Parameter resource.
     *
     * @stability stable
     * @attribute true
     */
    readonly parameterArn: string;
    /**
     * The name of the SSM Parameter resource.
     *
     * @stability stable
     * @attribute true
     */
    readonly parameterName: string;
    /**
     * The type of the SSM Parameter resource.
     *
     * @stability stable
     * @attribute true
     */
    readonly parameterType: string;
    /**
     * Grants read (DescribeParameter, GetParameter, GetParameterHistory) permissions on the SSM Parameter.
     *
     * @param grantee the role to be granted read-only access to the parameter.
     * @stability stable
     */
    grantRead(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grants write (PutParameter) permissions on the SSM Parameter.
     *
     * @param grantee the role to be granted write access to the parameter.
     * @stability stable
     */
    grantWrite(grantee: iam.IGrantable): iam.Grant;
}
/**
 * A String SSM Parameter.
 *
 * @stability stable
 */
export interface IStringParameter extends IParameter {
    /**
     * The parameter value.
     *
     * Value must not nest another parameter. Do not use {{}} in the value.
     *
     * @stability stable
     * @attribute Value
     */
    readonly stringValue: string;
}
/**
 * A StringList SSM Parameter.
 *
 * @stability stable
 */
export interface IStringListParameter extends IParameter {
    /**
     * The parameter value.
     *
     * Value must not nest another parameter. Do not use {{}} in the value. Values in the array
     * cannot contain commas (``,``).
     *
     * @stability stable
     * @attribute Value
     */
    readonly stringListValue: string[];
}
/**
 * Properties needed to create a new SSM Parameter.
 *
 * @stability stable
 */
export interface ParameterOptions {
    /**
     * A regular expression used to validate the parameter value.
     *
     * For example, for String types with values restricted to
     * numbers, you can specify the following: ``^\d+$``
     *
     * @default no validation is performed
     * @stability stable
     */
    readonly allowedPattern?: string;
    /**
     * Information about the parameter that you want to add to the system.
     *
     * @default none
     * @stability stable
     */
    readonly description?: string;
    /**
     * The name of the parameter.
     *
     * @default - a name will be generated by CloudFormation
     * @stability stable
     */
    readonly parameterName?: string;
    /**
     * Indicates of the parameter name is a simple name (i.e. does not include "/" separators).
     *
     * This is only required only if `parameterName` is a token, which means we
     * are unable to detect if the name is simple or "path-like" for the purpose
     * of rendering SSM parameter ARNs.
     *
     * If `parameterName` is not specified, `simpleName` must be `true` (or
     * undefined) since the name generated by AWS CloudFormation is always a
     * simple name.
     *
     * @default - auto-detect based on `parameterName`
     * @stability stable
     */
    readonly simpleName?: boolean;
    /**
     * The tier of the string parameter.
     *
     * @default - undefined
     * @stability stable
     */
    readonly tier?: ParameterTier;
}
/**
 * Properties needed to create a String SSM parameter.
 *
 * @stability stable
 */
export interface StringParameterProps extends ParameterOptions {
    /**
     * The value of the parameter.
     *
     * It may not reference another parameter and ``{{}}`` cannot be used in the value.
     *
     * @stability stable
     */
    readonly stringValue: string;
    /**
     * The type of the string parameter.
     *
     * @default ParameterType.STRING
     * @stability stable
     */
    readonly type?: ParameterType;
}
/**
 * Properties needed to create a StringList SSM Parameter.
 *
 * @stability stable
 */
export interface StringListParameterProps extends ParameterOptions {
    /**
     * The values of the parameter.
     *
     * It may not reference another parameter and ``{{}}`` cannot be used in the value.
     *
     * @stability stable
     */
    readonly stringListValue: string[];
}
/**
 * Basic features shared across all types of SSM Parameters.
 */
declare abstract class ParameterBase extends Resource implements IParameter {
    abstract readonly parameterArn: string;
    abstract readonly parameterName: string;
    abstract readonly parameterType: string;
    /**
     * The encryption key that is used to encrypt this parameter.
     *
     * * @default - default master key
     *
     * @stability stable
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * Grants read (DescribeParameter, GetParameter, GetParameterHistory) permissions on the SSM Parameter.
     *
     * @stability stable
     */
    grantRead(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grants write (PutParameter) permissions on the SSM Parameter.
     *
     * @stability stable
     */
    grantWrite(grantee: iam.IGrantable): iam.Grant;
}
/**
 * SSM parameter type.
 *
 * @stability stable
 */
export declare enum ParameterType {
    /**
     * String.
     *
     * @stability stable
     */
    STRING = "String",
    /**
     * Secure String Parameter Store uses an AWS Key Management Service (KMS) customer master key (CMK) to encrypt the parameter value.
     *
     * @stability stable
     */
    SECURE_STRING = "SecureString",
    /**
     * String List.
     *
     * @stability stable
     */
    STRING_LIST = "StringList",
    /**
     * An Amazon EC2 image ID, such as ami-0ff8a91507f77f867.
     *
     * @stability stable
     */
    AWS_EC2_IMAGE_ID = "AWS::EC2::Image::Id"
}
/**
 * SSM parameter tier.
 *
 * @stability stable
 */
export declare enum ParameterTier {
    /**
     * String.
     *
     * @stability stable
     */
    ADVANCED = "Advanced",
    /**
     * String.
     *
     * @stability stable
     */
    INTELLIGENT_TIERING = "Intelligent-Tiering",
    /**
     * String.
     *
     * @stability stable
     */
    STANDARD = "Standard"
}
/**
 * Common attributes for string parameters.
 *
 * @stability stable
 */
export interface CommonStringParameterAttributes {
    /**
     * The name of the parameter store value.
     *
     * This value can be a token or a concrete string. If it is a concrete string
     * and includes "/" it must also be prefixed with a "/" (fully-qualified).
     *
     * @stability stable
     */
    readonly parameterName: string;
    /**
     * Indicates of the parameter name is a simple name (i.e. does not include "/" separators).
     *
     * This is only required only if `parameterName` is a token, which means we
     * are unable to detect if the name is simple or "path-like" for the purpose
     * of rendering SSM parameter ARNs.
     *
     * If `parameterName` is not specified, `simpleName` must be `true` (or
     * undefined) since the name generated by AWS CloudFormation is always a
     * simple name.
     *
     * @default - auto-detect based on `parameterName`
     * @stability stable
     */
    readonly simpleName?: boolean;
}
/**
 * Attributes for parameters of various types of string.
 *
 * @see ParameterType
 * @stability stable
 */
export interface StringParameterAttributes extends CommonStringParameterAttributes {
    /**
     * The version number of the value you wish to retrieve.
     *
     * @default The latest version will be retrieved.
     * @stability stable
     */
    readonly version?: number;
    /**
     * The type of the string parameter.
     *
     * @default ParameterType.STRING
     * @stability stable
     */
    readonly type?: ParameterType;
}
/**
 * Attributes for secure string parameters.
 *
 * @stability stable
 */
export interface SecureStringParameterAttributes extends CommonStringParameterAttributes {
    /**
     * The version number of the value you wish to retrieve.
     *
     * This is required for secure strings.
     *
     * @stability stable
     */
    readonly version: number;
    /**
     * The encryption key that is used to encrypt this parameter.
     *
     * @default - default master key
     * @stability stable
     */
    readonly encryptionKey?: kms.IKey;
}
/**
 * Creates a new String SSM Parameter.
 *
 * @stability stable
 * @resource AWS::SSM::Parameter
 */
export declare class StringParameter extends ParameterBase implements IStringParameter {
    /**
     * Imports an external string parameter by name.
     *
     * @stability stable
     */
    static fromStringParameterName(scope: Construct, id: string, stringParameterName: string): IStringParameter;
    /**
     * Imports an external string parameter with name and optional version.
     *
     * @stability stable
     */
    static fromStringParameterAttributes(scope: Construct, id: string, attrs: StringParameterAttributes): IStringParameter;
    /**
     * Imports a secure string parameter from the SSM parameter store.
     *
     * @stability stable
     */
    static fromSecureStringParameterAttributes(scope: Construct, id: string, attrs: SecureStringParameterAttributes): IStringParameter;
    /**
     * Reads the value of an SSM parameter during synthesis through an environmental context provider.
     *
     * Requires that the stack this scope is defined in will have explicit
     * account/region information. Otherwise, it will fail during synthesis.
     *
     * @stability stable
     */
    static valueFromLookup(scope: CompatConstruct, parameterName: string): string;
    /**
     * Returns a token that will resolve (during deployment) to the string value of an SSM string parameter.
     *
     * @param scope Some scope within a stack.
     * @param parameterName The name of the SSM parameter.
     * @param version The parameter version (recommended in order to ensure that the value won't change during deployment).
     * @stability stable
     */
    static valueForStringParameter(scope: Construct, parameterName: string, version?: number): string;
    /**
     * Returns a token that will resolve (during deployment) to the string value of an SSM string parameter.
     *
     * @param scope Some scope within a stack.
     * @param parameterName The name of the SSM parameter.
     * @param type The type of the SSM parameter.
     * @param version The parameter version (recommended in order to ensure that the value won't change during deployment).
     * @stability stable
     */
    static valueForTypedStringParameter(scope: Construct, parameterName: string, type?: ParameterType, version?: number): string;
    /**
     * Returns a token that will resolve (during deployment).
     *
     * @param scope Some scope within a stack.
     * @param parameterName The name of the SSM parameter.
     * @param version The parameter version (required for secure strings).
     * @stability stable
     */
    static valueForSecureStringParameter(scope: Construct, parameterName: string, version: number): string;
    /**
     * The ARN of the SSM Parameter resource.
     *
     * @stability stable
     */
    readonly parameterArn: string;
    /**
     * The name of the SSM Parameter resource.
     *
     * @stability stable
     */
    readonly parameterName: string;
    /**
     * The type of the SSM Parameter resource.
     *
     * @stability stable
     */
    readonly parameterType: string;
    /**
     * The parameter value.
     *
     * Value must not nest another parameter. Do not use {{}} in the value.
     *
     * @stability stable
     */
    readonly stringValue: string;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: StringParameterProps);
}
/**
 * Creates a new StringList SSM Parameter.
 *
 * @stability stable
 * @resource AWS::SSM::Parameter
 */
export declare class StringListParameter extends ParameterBase implements IStringListParameter {
    /**
     * Imports an external parameter of type string list.
     *
     * Returns a token and should not be parsed.
     *
     * @stability stable
     */
    static fromStringListParameterName(scope: Construct, id: string, stringListParameterName: string): IStringListParameter;
    /**
     * The ARN of the SSM Parameter resource.
     *
     * @stability stable
     */
    readonly parameterArn: string;
    /**
     * The name of the SSM Parameter resource.
     *
     * @stability stable
     */
    readonly parameterName: string;
    /**
     * The type of the SSM Parameter resource.
     *
     * @stability stable
     */
    readonly parameterType: string;
    /**
     * The parameter value.
     *
     * Value must not nest another parameter. Do not use {{}} in the value. Values in the array
     * cannot contain commas (``,``).
     *
     * @stability stable
     */
    readonly stringListValue: string[];
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: StringListParameterProps);
}
export {};
