"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Domain = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const route53 = require("aws-cdk-lib/aws-route53");
const constructs_1 = require("constructs");
const certificate_1 = require("./certificate");
const add_1 = require("./errors/add");
const fqdn_1 = require("./fqdn");
const zone_1 = require("./zone");
class Domain extends constructs_1.Construct {
    /**
     * Initializing a `new Domain` construct instance will lookup the Route53 hosted zone
     * and define ACM DNS-validated certificate.
     *
     * After initialization you must use `assign(alias)` method to to configure `A`/`AAAA` records
     * with the `alias` as the record value.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.assigned = false;
        const { certificate, enableIpv6 = true, region = 'us-east-1', subdomain, zone, } = props;
        this.enableIpv6 = enableIpv6;
        this.zone = zone_1.DomainZone.define(this, { zone });
        this.region = region;
        this.fqdn = new fqdn_1.FQDN(this, 'FQDN', {
            zone: this.zone,
            subdomain,
        }).value;
        this.certificate = certificate_1.DomainCertificate.define(this, {
            certificate,
            region: this.region,
            zone: this.zone,
            fqdn: this.fqdn,
        });
    }
    /**
     * Assign an alias as record target with the fully-qualified domain name.
     * This will create both `A` & `AAAA` DNS records, unless `disableIpV6` was set to `true`
     * during initialization of `Domain` construct (resulting in only `A` record being created).
     *
     * @param alias Route53 alias record target used to assign as A/AAAA record value.
     *
     * @example
     * domain.addTarget(new targets.CloudFrontTarget(distribution))
     */
    addTarget(alias) {
        // prevent accidental reassignment
        if (this.assigned) {
            add_1.addError(this, `Domain already assigned: ${this.fqdn}`);
            return;
        }
        // set assigned state flag
        this.assigned = true;
        // set IPv4 record
        new route53.ARecord(this, 'AliasRecordIpV4', {
            zone: this.zone,
            target: route53.RecordTarget.fromAlias(alias),
        });
        // return early if IPv6 is disabled
        if (!this.enableIpv6)
            return;
        // set IPv6 record
        new route53.AaaaRecord(this, 'AliasRecordIpV6', {
            zone: this.zone,
            target: route53.RecordTarget.fromAlias(alias),
        });
    }
    /**
     * Helper method to configure CloudFront distribution with the domain,
     * certificate and IPv6 support.
     *
     * @returns CloudFront configuration for certificate, domainNames and IPv6
     */
    configureCloudFront() {
        return {
            certificate: this.certificate,
            domainNames: [this.fqdn],
            enableIpv6: this.enableIpv6,
        };
    }
}
exports.Domain = Domain;
_a = JSII_RTTI_SYMBOL_1;
Domain[_a] = { fqn: "@alma-cdk/domain.Domain", version: "0.0.8" };
//# sourceMappingURL=data:application/json;base64,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