"""Modules to collect pathpy objects."""
# !/usr/bin/python -tt
# -*- coding: utf-8 -*-
# =============================================================================
# File      : containers.py -- Base containers for pathpy
# Author    : Jürgen Hackl <hackl@ifi.uzh.ch>
# Time-stamp: <Wed 2020-07-08 04:22 juergen>
#
# Copyright (c) 2016-2019 Pathpy Developers
# =============================================================================
from __future__ import annotations
from typing import Set, Dict, Any


class BaseCollection:
    """Class to store objects"""

    def __init__(self) -> None:
        """Initialize the NodeContainer object."""
        # initialize variables
        self._map: dict = dict()

    def __len__(self) -> int:
        """Returns the number of nodes"""
        return len(self._map)

    def __iter__(self):
        """Iterator over the note set."""
        return self._map.values().__iter__()

    def __str__(self) -> str:
        """Print the BaseContainer object"""
        return set(self._map.values()).__str__()

    def __eq__(self, other) -> bool:
        """Return if two collections are equal"""
        return self._map == other._map

    def __contains__(self, item: Any) -> bool:
        """Returns if item is in nodes."""
        _contain: bool = False
        if item in self._map.values():
            _contain = True
        elif isinstance(item, str) and item in self._map:
            _contain = True
        return _contain

    def __getitem__(self, key: Any) -> Any:
        """Returns a node object."""
        return self._map[key]

    def __setitem__(self, key: Any, value: Any) -> None:
        """set a object"""
        self._map[key] = value

    def items(self):
        """Return a new view of the container’s items ((key, value) pairs)."""
        return self._map.items()

    def keys(self):
        """Return a new view of the container’s keys. """
        return self._map.keys()

    def values(self):
        """Return a new view of the container’s values."""
        return self._map.values()

    def pop(self, key, default: Any = KeyError) -> Any:
        """Pop item form dict"""
        self._map.pop(key, default)

    @property
    def uids(self) -> Set[str]:
        """Returns a set of object uids"""
        return set(self._map)

    @property
    def index(self) -> Dict[str, int]:
        """Returns a dictionary that maps object uids to  integer indices.

        The indices of nodes correspond to the row/column ordering of objects
        in any list/array/matrix representation generated by pathpy, e.g. for
        degrees.sequence or adjacency_matrix.

        Returns
        -------
        dict
            maps node uids to zero-based integer index

        """
        return dict(zip(self._map, range(len(self))))

    @property
    def dict(self) -> Dict:
        """Returns a dictionary of node objects."""
        return self._map

    def contain(self, other: Any) -> bool:
        """Returns true if node is available."""
        boolean: bool
        if other not in self and other.uid not in self.uids:
            boolean = False
        else:
            boolean = True
        return boolean


# =============================================================================
# eof
#
# Local Variables:
# mode: python
# mode: linum
# mode: auto-fill
# fill-column: 79
# End:
