"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.flatten = exports.GitHubWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cx_api_1 = require("aws-cdk-lib/cx-api");
const pipelines_1 = require("aws-cdk-lib/pipelines");
const helpers_internal_1 = require("aws-cdk-lib/pipelines/lib/helpers-internal");
const decamelize = require("decamelize");
const YAML = require("yaml");
const aws_credentials_1 = require("./private/aws-credentials");
const github = require("./workflows-model");
const CDKOUT_ARTIFACT = 'cdk.out';
const RUNS_ON = 'ubuntu-latest';
const ASSET_HASH_NAME = 'asset-hash';
/**
 * CDK Pipelines for GitHub workflows.
 */
class GitHubWorkflow extends pipelines_1.PipelineBase {
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j;
        super(scope, id, props);
        this.jobOutputs = {};
        this.assetHashMap = {};
        this.cdkCliVersion = props.cdkCliVersion;
        this.preSynthed = (_b = props.preSynthed) !== null && _b !== void 0 ? _b : false;
        this.buildContainer = props.buildContainer;
        this.preBuildSteps = (_c = props.preBuildSteps) !== null && _c !== void 0 ? _c : [];
        this.postBuildSteps = (_d = props.postBuildSteps) !== null && _d !== void 0 ? _d : [];
        this.gitHubActionRoleArn = props.gitHubActionRoleArn;
        this.useGitHubActionRole = this.gitHubActionRoleArn ? true : false;
        this.awsCredentials = (_e = props.awsCredentials) !== null && _e !== void 0 ? _e : {
            accessKeyId: 'AWS_ACCESS_KEY_ID',
            secretAccessKey: 'AWS_SECRET_ACCESS_KEY',
        };
        this.dockerCredentials = (_f = props.dockerCredentials) !== null && _f !== void 0 ? _f : [];
        this.workflowPath = (_g = props.workflowPath) !== null && _g !== void 0 ? _g : '.github/workflows/deploy.yml';
        if (!this.workflowPath.endsWith('.yml') && !this.workflowPath.endsWith('.yaml')) {
            throw new Error('workflow file is expected to be a yaml file');
        }
        if (!this.workflowPath.includes('.github/workflows/')) {
            throw new Error('workflow files must be stored in the \'.github/workflows\' directory of your repository');
        }
        this.workflowName = (_h = props.workflowName) !== null && _h !== void 0 ? _h : 'deploy';
        this.workflowTriggers = (_j = props.workflowTriggers) !== null && _j !== void 0 ? _j : {
            push: { branches: ['main'] },
            workflowDispatch: {},
        };
    }
    doBuildPipeline() {
        var _b;
        const app = aws_cdk_lib_1.Stage.of(this);
        if (!app) {
            throw new Error('The GitHub Workflow must be defined in the scope of an App');
        }
        const cdkoutDir = app.outdir;
        const jobs = new Array();
        const structure = new helpers_internal_1.PipelineGraph(this, {
            selfMutation: false,
            publishTemplate: true,
            prepareStep: false,
        });
        for (const stageNode of flatten(structure.graph.sortedChildren())) {
            if (!helpers_internal_1.isGraph(stageNode)) {
                throw new Error(`Top-level children must be graphs, got '${stageNode}'`);
            }
            const tranches = stageNode.sortedLeaves();
            for (const tranche of tranches) {
                for (const node of tranche) {
                    const job = this.jobForNode(node, {
                        assemblyDir: cdkoutDir,
                        structure,
                    });
                    if (job) {
                        jobs.push(job);
                    }
                }
            }
        }
        // convert jobs to a map and make sure there are no duplicates
        const jobmap = {};
        for (const job of jobs) {
            if (job.id in jobmap) {
                throw new Error(`duplicate job id ${job.id}`);
            }
            jobmap[job.id] = snakeCaseKeys(job.definition);
        }
        // Update jobs with late-bound output requests
        this.insertJobOutputs(jobmap);
        const workflow = {
            name: this.workflowName,
            on: snakeCaseKeys(this.workflowTriggers, '_'),
            jobs: jobmap,
        };
        // write as a yaml file
        const yaml = YAML.stringify(workflow, {
            indent: 2,
        });
        // create directory if it does not exist
        fs_1.mkdirSync(path.dirname(this.workflowPath), { recursive: true });
        // GITHUB_WORKFLOW is set when GitHub Actions is running the workflow.
        // see: https://docs.github.com/en/actions/learn-github-actions/environment-variables#default-environment-variables
        const diffProtection = (_b = this.node.tryGetContext('cdk-pipelines-github:diffProtection')) !== null && _b !== void 0 ? _b : true;
        if (diffProtection && process.env.GITHUB_WORKFLOW === this.workflowName) {
            // check if workflow file has changed
            if (!fs_1.existsSync(this.workflowPath) || yaml !== fs_1.readFileSync(this.workflowPath, 'utf8')) {
                throw new Error(`Please commit the updated workflow file ${path.relative(__dirname, this.workflowPath)} when you change your pipeline definition.`);
            }
        }
        // eslint-disable-next-line no-console
        console.log(`writing ${this.workflowPath}`);
        fs_1.writeFileSync(this.workflowPath, yaml);
    }
    insertJobOutputs(jobmap) {
        for (const [jobId, jobOutputs] of Object.entries(this.jobOutputs)) {
            jobmap[jobId] = {
                ...jobmap[jobId],
                outputs: {
                    ...jobmap[jobId].outputs,
                    ...this.renderJobOutputs(jobOutputs),
                },
            };
        }
    }
    renderJobOutputs(outputs) {
        const renderedOutputs = {};
        for (const output of outputs) {
            renderedOutputs[output.outputName] = `\${{ steps.${output.stepId}.outputs.${output.outputName} }}`;
        }
        return renderedOutputs;
    }
    /**
     * Make an action from the given node and/or step
     */
    jobForNode(node, options) {
        var _b, _c;
        switch ((_b = node.data) === null || _b === void 0 ? void 0 : _b.type) {
            // Nothing for these, they are groupings (shouldn't even have popped up here)
            case 'group':
            case 'stack-group':
            case undefined:
                throw new Error(`jobForNode: did not expect to get group nodes: ${(_c = node.data) === null || _c === void 0 ? void 0 : _c.type}`);
            case 'self-update':
                throw new Error('GitHub Workflows does not support self mutation');
            case 'publish-assets':
                return this.jobForAssetPublish(node, node.data.assets, options);
            case 'prepare':
                throw new Error('"prepare" is not supported by GitHub Worflows');
            case 'execute':
                return this.jobForDeploy(node, node.data.stack, node.data.captureOutputs);
            case 'step':
                if (node.data.isBuildStep) {
                    return this.jobForBuildStep(node, node.data.step);
                }
                else if (node.data.step instanceof pipelines_1.ShellStep) {
                    return this.jobForScriptStep(node, node.data.step);
                }
                else {
                    throw new Error(`unsupported step type: ${node.data.step.constructor.name}`);
                }
        }
    }
    jobForAssetPublish(node, assets, options) {
        if (assets.length === 0) {
            throw new Error('Asset Publish step must have at least 1 asset');
        }
        const installSuffix = this.cdkCliVersion ? `@${this.cdkCliVersion}` : '';
        const cdkoutDir = options.assemblyDir;
        const jobId = node.uniqueId;
        const assetId = assets[0].assetId;
        // check if asset is docker asset and if we have docker credentials
        const dockerLoginSteps = [];
        if (node.uniqueId.includes('DockerAsset') && this.dockerCredentials.length > 0) {
            for (const creds of this.dockerCredentials) {
                dockerLoginSteps.push(...this.stepsToConfigureDocker(creds));
            }
        }
        // create one file and make one step
        const relativeToAssembly = (p) => path.posix.join(cdkoutDir, path.relative(path.resolve(cdkoutDir), p));
        const fileContents = ['set -x'].concat(assets.map((asset) => {
            return `npx cdk-assets --path "${relativeToAssembly(asset.assetManifestPath)}" --verbose publish "${asset.assetSelector}"`;
        }));
        // we need the jobId to reference the outputs later
        this.assetHashMap[assetId] = jobId;
        fileContents.push(`echo '::set-output name=${ASSET_HASH_NAME}::${assetId}'`);
        const publishStepFile = path.join(cdkoutDir, `publish-${jobId}-step.sh`);
        fs_1.mkdirSync(path.dirname(publishStepFile), { recursive: true });
        fs_1.writeFileSync(publishStepFile, fileContents.join('\n'), { encoding: 'utf-8' });
        const publishStep = {
            id: 'Publish',
            name: `Publish ${jobId}`,
            run: `/bin/bash ./cdk.out/${path.relative(cdkoutDir, publishStepFile)}`,
        };
        return {
            id: jobId,
            definition: {
                name: `Publish Assets ${jobId}`,
                needs: this.renderDependencies(node),
                permissions: {
                    contents: github.JobPermission.READ,
                    idToken: this.useGitHubActionRole ? github.JobPermission.WRITE : github.JobPermission.NONE,
                },
                runsOn: RUNS_ON,
                outputs: {
                    [ASSET_HASH_NAME]: `\${{ steps.Publish.outputs.${ASSET_HASH_NAME} }}`,
                },
                steps: [
                    ...this.stepsToDownloadAssembly(cdkoutDir),
                    {
                        name: 'Install',
                        run: `npm install --no-save cdk-assets${installSuffix}`,
                    },
                    ...this.stepsToConfigureAws(this.useGitHubActionRole, { region: 'us-west-2' }),
                    ...dockerLoginSteps,
                    publishStep,
                ],
            },
        };
    }
    jobForDeploy(node, stack, _captureOutputs) {
        const region = stack.region;
        const account = stack.account;
        if (!region || !account) {
            throw new Error('"account" and "region" are required');
        }
        if (!stack.templateUrl) {
            throw new Error(`unable to determine template URL for stack ${stack.stackArtifactId}`);
        }
        const resolve = (s) => {
            return cx_api_1.EnvironmentPlaceholders.replace(s, {
                accountId: account,
                region: region,
                partition: 'aws',
            });
        };
        const replaceAssetHash = (template) => {
            var _b;
            const hash = path.parse((_b = template.split('/').pop()) !== null && _b !== void 0 ? _b : '').name;
            if (this.assetHashMap[hash] === undefined) {
                throw new Error(`Template asset hash ${hash} not found.`);
            }
            return template.replace(hash, `\${{ needs.${this.assetHashMap[hash]}.outputs.${ASSET_HASH_NAME} }}`);
        };
        const params = {
            'name': stack.stackName,
            'template': replaceAssetHash(resolve(stack.templateUrl)),
            'no-fail-on-empty-changeset': '1',
        };
        if (stack.executionRoleArn) {
            params['role-arn'] = resolve(stack.executionRoleArn);
        }
        const assumeRoleArn = stack.assumeRoleArn ? resolve(stack.assumeRoleArn) : undefined;
        return {
            id: node.uniqueId,
            definition: {
                name: `Deploy ${stack.stackArtifactId}`,
                permissions: {
                    contents: github.JobPermission.READ,
                    idToken: this.useGitHubActionRole ? github.JobPermission.WRITE : github.JobPermission.NONE,
                },
                needs: this.renderDependencies(node),
                runsOn: RUNS_ON,
                steps: [
                    ...this.stepsToConfigureAws(this.useGitHubActionRole, { region, assumeRoleArn }),
                    {
                        id: 'Deploy',
                        uses: 'aws-actions/aws-cloudformation-github-deploy@v1',
                        with: params,
                    },
                ],
            },
        };
    }
    jobForBuildStep(node, step) {
        if (!(step instanceof pipelines_1.ShellStep)) {
            throw new Error('synthStep must be a ScriptStep');
        }
        if (step.inputs.length > 0) {
            throw new Error('synthStep cannot have inputs');
        }
        if (step.outputs.length > 1) {
            throw new Error('synthStep must have a single output');
        }
        if (!step.primaryOutput) {
            throw new Error('synthStep requires a primaryOutput which contains cdk.out');
        }
        const cdkOut = step.outputs[0];
        const installSteps = step.installCommands.length > 0 ? [{
                name: 'Install',
                run: step.installCommands.join('\n'),
            }] : [];
        return {
            id: node.uniqueId,
            definition: {
                name: 'Synthesize',
                permissions: {
                    contents: github.JobPermission.READ,
                },
                runsOn: RUNS_ON,
                needs: this.renderDependencies(node),
                env: step.env,
                container: this.buildContainer,
                steps: [
                    ...this.stepsToCheckout(),
                    ...this.preBuildSteps,
                    ...installSteps,
                    {
                        name: 'Build',
                        run: step.commands.join('\n'),
                    },
                    ...this.postBuildSteps,
                    ...this.stepsToUploadAssembly(cdkOut.directory),
                ],
            },
        };
    }
    /**
     * Searches for the stack that produced the output via the current
     * job's dependencies.
     *
     * This function should always find a stack, since it is guaranteed
     * that a CfnOutput comes from a referenced stack.
     */
    findStackOfOutput(ref, node) {
        var _b;
        for (const dep of node.allDeps) {
            if (((_b = dep.data) === null || _b === void 0 ? void 0 : _b.type) === 'execute' && ref.isProducedBy(dep.data.stack)) {
                return dep.uniqueId;
            }
        }
        // Should never happen
        throw new Error(`The output ${ref.outputName} is not referenced by any of the dependent stacks!`);
    }
    addJobOutput(jobId, output) {
        if (this.jobOutputs[jobId] === undefined) {
            this.jobOutputs[jobId] = [output];
        }
        else {
            this.jobOutputs[jobId].push(output);
        }
    }
    jobForScriptStep(node, step) {
        const envVariables = {};
        for (const [envName, ref] of Object.entries(step.envFromCfnOutputs)) {
            const jobId = this.findStackOfOutput(ref, node);
            this.addJobOutput(jobId, {
                outputName: ref.outputName,
                stepId: 'Deploy',
            });
            envVariables[envName] = `\${{ needs.${jobId}.outputs.${ref.outputName} }}`;
        }
        const downloadInputs = new Array();
        const uploadOutputs = new Array();
        for (const input of step.inputs) {
            downloadInputs.push({
                uses: 'actions/download-artifact@v2',
                with: {
                    name: input.fileSet.id,
                    path: input.directory,
                },
            });
        }
        for (const output of step.outputs) {
            uploadOutputs.push({
                uses: 'actions/upload-artifact@v2.1.1',
                with: {
                    name: output.fileSet.id,
                    path: output.directory,
                },
            });
        }
        const installSteps = step.installCommands.length > 0 ? [{
                name: 'Install',
                run: step.installCommands.join('\n'),
            }] : [];
        return {
            id: node.uniqueId,
            definition: {
                name: step.id,
                permissions: {
                    contents: github.JobPermission.READ,
                },
                runsOn: RUNS_ON,
                needs: this.renderDependencies(node),
                env: {
                    ...step.env,
                    ...envVariables,
                },
                steps: [
                    ...downloadInputs,
                    ...installSteps,
                    { run: step.commands.join('\n') },
                    ...uploadOutputs,
                ],
            },
        };
    }
    stepsToConfigureAws(openId, { region, assumeRoleArn }) {
        function getDeployRole(arn) {
            return arn.replace('cfn-exec', 'deploy');
        }
        let steps = [];
        if (openId) {
            steps.push(aws_credentials_1.awsCredentialStep('Authenticate Via OIDC Role', {
                region,
                gitHubActionRoleArn: this.gitHubActionRoleArn,
            }));
            if (assumeRoleArn) {
                // Result of initial credentials with GitHub Action role are these environment variables
                steps.push(aws_credentials_1.awsCredentialStep('Assume CDK Deploy Role', {
                    region,
                    accessKeyId: '${{ env.AWS_ACCESS_KEY_ID }}',
                    secretAccessKey: '${{ env.AWS_SECRET_ACCESS_KEY }}',
                    sessionToken: '${{ env.AWS_SESSION_TOKEN }}',
                    roleToAssume: getDeployRole(assumeRoleArn),
                }));
            }
        }
        else {
            steps.push(aws_credentials_1.awsCredentialStep('Authenticate Via GitHub Secrets', {
                region,
                accessKeyId: `\${{ secrets.${this.awsCredentials.accessKeyId} }}`,
                secretAccessKey: `\${{ secrets.${this.awsCredentials.secretAccessKey} }}`,
                sessionToken: `\${{ secrets.${this.awsCredentials.sessionToken} }}`,
                roleToAssume: assumeRoleArn,
            }));
        }
        return steps;
    }
    stepsToConfigureDocker(dockerCredential) {
        let params;
        if (dockerCredential.name === 'docker') {
            params = {
                username: `\${{ secrets.${dockerCredential.usernameKey} }}`,
                password: `\${{ secrets.${dockerCredential.passwordKey} }}`,
            };
        }
        else if (dockerCredential.name === 'ecr') {
            params = {
                registry: dockerCredential.registry,
            };
        }
        else {
            params = {
                registry: dockerCredential.registry,
                username: `\${{ secrets.${dockerCredential.usernameKey} }}`,
                password: `\${{ secrets.${dockerCredential.passwordKey} }}`,
            };
        }
        return [
            {
                uses: 'docker/login-action@v1',
                with: params,
            },
        ];
    }
    stepsToDownloadAssembly(targetDir) {
        if (this.preSynthed) {
            return this.stepsToCheckout();
        }
        return [{
                name: `Download ${CDKOUT_ARTIFACT}`,
                uses: 'actions/download-artifact@v2',
                with: {
                    name: CDKOUT_ARTIFACT,
                    path: targetDir,
                },
            }];
    }
    stepsToCheckout() {
        return [{
                name: 'Checkout',
                uses: 'actions/checkout@v2',
            }];
    }
    stepsToUploadAssembly(dir) {
        if (this.preSynthed) {
            return [];
        }
        return [{
                name: `Upload ${CDKOUT_ARTIFACT}`,
                uses: 'actions/upload-artifact@v2.1.1',
                with: {
                    name: CDKOUT_ARTIFACT,
                    path: dir,
                },
            }];
    }
    renderDependencies(node) {
        const deps = new Array();
        for (const d of node.allDeps) {
            if (d instanceof helpers_internal_1.Graph) {
                deps.push(...d.allLeaves().nodes);
            }
            else {
                deps.push(d);
            }
        }
        return deps.map(x => x.uniqueId);
    }
}
exports.GitHubWorkflow = GitHubWorkflow;
_a = JSII_RTTI_SYMBOL_1;
GitHubWorkflow[_a] = { fqn: "cdk-pipelines-github.GitHubWorkflow", version: "0.1.11" };
function snakeCaseKeys(obj, sep = '-') {
    if (typeof obj !== 'object' || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(o => snakeCaseKeys(o, sep));
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        // we don't want to snake case environment variables
        if (k !== 'env' && typeof v === 'object' && v != null) {
            v = snakeCaseKeys(v);
        }
        result[decamelize(k, { separator: sep })] = v;
    }
    return result;
}
function* flatten(xs) {
    for (const x of xs) {
        for (const y of x) {
            yield y;
        }
    }
}
exports.flatten = flatten;
//# sourceMappingURL=data:application/json;base64,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