"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrowApi = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const lambda = require("aws-cdk-lib/aws-lambda");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const logs = require("aws-cdk-lib/aws-logs");
/**
 * For copying shared code to all paths
 */
const fse = require("fs-extra");
/**
 * @experimental
 */
class CrowApi extends constructs_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Pulling out props
        const { sourceDirectory = 'src', sharedDirectory = 'shared', useAuthorizerLambda = false, authorizerDirectory = 'authorizer', authorizerLambdaConfiguration = {}, tokenAuthorizerConfiguration = {}, createApiKey = false, logRetention = logs.RetentionDays.ONE_WEEK, apiGatewayConfiguration = {}, apiGatewayName = 'crow-api', lambdaConfigurations = {}, models = [], requestValidators = [], methodConfigurations = {}, } = props;
        // Initializing constants
        const LAMBDA_RUNTIME = lambda.Runtime.NODEJS_14_X;
        const SPECIAL_DIRECTORIES = [
            sharedDirectory,
            authorizerDirectory,
        ];
        // Helpers functions for constructor
        // Prepares default Lambda props and overrides them with user input
        function bundleLambdaProps(codePath, userConfiguration, sharedLayer) {
            let layers;
            if (sharedLayer) {
                const { layers: userLayers = [], } = userConfiguration;
                layers = [sharedLayer, ...userLayers];
            }
            const defaultProps = {
                runtime: LAMBDA_RUNTIME,
                code: lambda.Code.fromAsset(codePath),
                handler: 'index.handler',
                logRetention,
            };
            const lambdaProps = {
                ...defaultProps,
                ...userConfiguration,
                layers,
            };
            return lambdaProps;
        }
        // Returns child directories given the path of a parent
        function getDirectoryChildren(parentDirectory) {
            try {
                const directories = fse.readdirSync(parentDirectory, { withFileTypes: true })
                    .filter((dirent) => dirent.isDirectory())
                    .map((dirent) => dirent.name);
                return directories;
            }
            catch {
                /**
                 * The only time I have run into this was when the src/ directory
                 * was empty.
                 * If it is empty, let CDK tree validation tell user that the
                 * REST API does not have any methods.
                 */
            }
            return [];
        }
        // API Gateway log group
        const gatewayLogGroup = new logs.LogGroup(this, 'api-access-logs', {
            retention: logs.RetentionDays.ONE_WEEK,
        });
        // The API Gateway itself
        const gateway = new apigateway.RestApi(this, apiGatewayName, {
            deploy: true,
            deployOptions: {
                loggingLevel: apigateway.MethodLoggingLevel.ERROR,
                accessLogDestination: new apigateway.LogGroupLogDestination(gatewayLogGroup),
            },
            apiKeySourceType: createApiKey ? apigateway.ApiKeySourceType.HEADER : undefined,
            ...apiGatewayConfiguration,
        });
        const createdModels = {};
        models.forEach((model) => {
            // modelName is used as ID and can now be used for referencing model in method options
            createdModels[model.modelName] = gateway.addModel(model.modelName, model);
        });
        const createdRequestValidators = {};
        requestValidators.forEach((requestValidator) => {
            // requestValidatorName is used as ID and can now be used for referencing model in method options
            createdRequestValidators[requestValidator.requestValidatorName] = gateway.addRequestValidator(requestValidator.requestValidatorName, requestValidator);
        });
        // Create API key if desired
        if (createApiKey) {
            const apiKey = gateway.addApiKey('api-key');
            const usagePlan = new apigateway.UsagePlan(this, 'usage-plan', {
                throttle: {
                    burstLimit: 5000,
                    rateLimit: 10000,
                },
                apiStages: [
                    {
                        api: gateway,
                        stage: gateway.deploymentStage,
                    },
                ],
            });
            usagePlan.addApiKey(apiKey);
        }
        // Create Lambda layer out of shared directory if it exists
        const sourceSharedDirectory = `${sourceDirectory}/${sharedDirectory}`;
        let sharedLayer;
        if (fse.existsSync(sourceSharedDirectory)) {
            sharedLayer = new lambda.LayerVersion(this, 'shared-layer', {
                code: lambda.Code.fromAsset(sourceSharedDirectory),
                compatibleRuntimes: [LAMBDA_RUNTIME],
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
            this.lambdaLayer = sharedLayer;
        }
        // Create Lambda authorizer to be used in subsequent Methods
        let tokenAuthorizer;
        if (useAuthorizerLambda) {
            const fullAuthorizerDirectory = `${sourceDirectory}/${authorizerDirectory}`;
            const authorizerLambdaProps = bundleLambdaProps(fullAuthorizerDirectory, authorizerLambdaConfiguration, sharedLayer);
            const authorizerLambda = new lambda.Function(this, 'authorizer-lambda', authorizerLambdaProps);
            this.authorizerLambda = authorizerLambda;
            const bundledTokenAuthConfig = {
                handler: authorizerLambda,
                resultsCacheTtl: cdk.Duration.seconds(3600),
                ...tokenAuthorizerConfiguration,
            };
            tokenAuthorizer = new apigateway.TokenAuthorizer(this, 'token-authorizer', bundledTokenAuthConfig);
        }
        // Time to start walking the directories
        const root = sourceDirectory;
        const verbs = ['get', 'post', 'put', 'delete'];
        const graph = {};
        const lambdasByPath = {};
        // Initialize with root
        graph['/'] = {
            resource: gateway.root,
            path: root,
            paths: [],
            verbs: [],
        };
        // First element in tuple is directory path, second is API path
        const nodes = [[root, '/']];
        // BFS that creates API Gateway structure using addMethod
        while (nodes.length) {
            // The `|| ['type', 'script']` piece is needed or TS throws a fit
            const [directoryPath, apiPath] = nodes.shift() || ['type', 'script'];
            const children = getDirectoryChildren(directoryPath);
            // For debugging purposes
            // console.log(`${apiPath}'s children are: ${children}`);
            // Don't have to worry about previously visited nodes
            // since this is a file structure
            // ...unless there are symlinks? Haven't run into that
            children.forEach((child) => {
                const newDirectoryPath = `${directoryPath}/${child}`;
                // If we're on the root path, don't separate with a slash (/)
                //   because it ends up looking like //child-path
                const newApiPath = apiPath === '/' ? `/${child}` : `${apiPath}/${child}`;
                if (verbs.includes(child)) {
                    // If directory is a verb, we don't traverse it anymore
                    //   and need to create an API Gateway method and Lambda
                    const userLambdaConfiguration = lambdaConfigurations[newApiPath]
                        || {};
                    const lambdaProps = bundleLambdaProps(newDirectoryPath, userLambdaConfiguration, sharedLayer);
                    const newLambda = new lambda.Function(this, newDirectoryPath, lambdaProps);
                    // Pull out useAuthorizerLambda value and the tweaked model values
                    const { useAuthorizerLambda: authorizerLambdaConfigured = false, requestModels: crowRequestModels, methodResponses: crowMethodResponses, ...userMethodConfiguration } = methodConfigurations[newApiPath] || {};
                    // Map models
                    const requestModels = {};
                    if (crowRequestModels) {
                        Object.entries(crowRequestModels).forEach(([contentType, modelName]) => {
                            requestModels[contentType] = createdModels[modelName];
                        });
                    }
                    const methodResponses = [];
                    if (crowMethodResponses && crowMethodResponses.length > 0) {
                        crowMethodResponses.forEach((crowMethodResponse) => {
                            const responseModels = {};
                            if (crowMethodResponse.responseModels) {
                                const crowResponseModels = crowMethodResponse.responseModels;
                                Object.entries(crowResponseModels).forEach(([contentType, modelName]) => {
                                    responseModels[contentType] = createdModels[modelName];
                                });
                            }
                            const { statusCode, responseParameters, } = crowMethodResponse;
                            methodResponses.push({
                                statusCode,
                                responseParameters,
                                responseModels,
                            });
                        });
                    }
                    let methodConfiguration = {
                        ...userMethodConfiguration,
                        requestModels,
                        methodResponses,
                    };
                    // If this method should be behind an authorizer Lambda
                    //   construct the methodConfiguration object as such
                    if (authorizerLambdaConfigured && useAuthorizerLambda) {
                        methodConfiguration = {
                            ...userMethodConfiguration,
                            authorizationType: apigateway.AuthorizationType.CUSTOM,
                            authorizer: tokenAuthorizer,
                            requestModels,
                            methodResponses,
                        };
                    }
                    graph[apiPath].resource.addMethod(child.toUpperCase(), new apigateway.LambdaIntegration(newLambda), methodConfiguration);
                    graph[apiPath].verbs.push(child);
                    lambdasByPath[newApiPath] = newLambda;
                }
                else if (SPECIAL_DIRECTORIES.includes(child)) {
                    // The special directories should not result in an API path
                    // This means the API also cannot have a resource with the
                    //   same name
                }
                else {
                    // If directory is not a verb, create new API Gateway resource
                    //   for use by verb directory later
                    const newResource = graph[apiPath].resource
                        .resourceForPath(child);
                    nodes.push([newDirectoryPath, newApiPath]);
                    // Add child to parent's paths
                    graph[apiPath].paths.push(child);
                    // Initialize graph node to include child
                    graph[newApiPath] = {
                        resource: newResource,
                        path: newDirectoryPath,
                        paths: [],
                        verbs: [],
                    };
                }
            });
        }
        // For debugging purposes
        // console.log(graph);
        // Expose API Gateway
        this.gateway = gateway;
        this.lambdaFunctions = lambdasByPath;
    }
}
exports.CrowApi = CrowApi;
_a = JSII_RTTI_SYMBOL_1;
CrowApi[_a] = { fqn: "crow-api.CrowApi", version: "2.2.0" };
//# sourceMappingURL=data:application/json;base64,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