"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.checkTemplateForCyclicDependencies = void 0;
/**
 * Check a template for cyclic dependencies
 *
 * This will make sure that we don't happily validate templates
 * in unit tests that wouldn't deploy to CloudFormation anyway.
 */
function checkTemplateForCyclicDependencies(template) {
    var _a, _b, _c;
    const logicalIds = new Set(Object.keys((_a = template.Resources) !== null && _a !== void 0 ? _a : {}));
    const dependencies = new Map();
    for (const [logicalId, resource] of Object.entries((_b = template.Resources) !== null && _b !== void 0 ? _b : {})) {
        dependencies.set(logicalId, intersect(findResourceDependencies(resource), logicalIds));
    }
    // We will now progressively remove entries from the map of 'dependencies' that have
    // 0 elements in them. If we can't do that anymore and the map isn't empty, we
    // have a cyclic dependency.
    while (dependencies.size > 0) {
        const free = Array.from(dependencies.entries()).filter(([_, deps]) => deps.size === 0);
        if (free.length === 0) {
            // Oops!
            const cycle = findCycle(dependencies);
            const cycleResources = {};
            for (const logicalId of cycle) {
                cycleResources[logicalId] = (_c = template.Resources) === null || _c === void 0 ? void 0 : _c[logicalId];
            }
            throw new Error(`Template is undeployable, these resources have a dependency cycle: ${cycle.join(' -> ')}:\n\n${JSON.stringify(cycleResources, undefined, 2)}`);
        }
        for (const [logicalId, _] of free) {
            for (const deps of dependencies.values()) {
                deps.delete(logicalId);
            }
            dependencies.delete(logicalId);
        }
    }
}
exports.checkTemplateForCyclicDependencies = checkTemplateForCyclicDependencies;
function findResourceDependencies(res) {
    var _a;
    return new Set([
        ...toArray((_a = res.DependsOn) !== null && _a !== void 0 ? _a : []),
        ...findExpressionDependencies(res.Properties),
    ]);
}
function toArray(x) {
    return Array.isArray(x) ? x : [x];
}
function findExpressionDependencies(obj) {
    const ret = new Set();
    recurse(obj);
    return ret;
    function recurse(x) {
        if (!x) {
            return;
        }
        if (Array.isArray(x)) {
            x.forEach(recurse);
        }
        if (typeof x === 'object') {
            const keys = Object.keys(x);
            if (keys.length === 1 && keys[0] === 'Ref') {
                ret.add(x[keys[0]]);
            }
            else if (keys.length === 1 && keys[0] === 'Fn::GetAtt') {
                ret.add(x[keys[0]][0]);
            }
            else if (keys.length === 1 && keys[0] === 'Fn::Sub') {
                const argument = x[keys[0]];
                const pattern = Array.isArray(argument) ? argument[0] : argument;
                // pattern should always be a string, but we've encountered some cases in which
                // it isn't. Better safeguard.
                if (typeof pattern === 'string') {
                    for (const logId of logicalIdsInSubString(pattern)) {
                        ret.add(logId);
                    }
                }
                const contextDict = Array.isArray(argument) ? argument[1] : undefined;
                if (contextDict && typeof contextDict === 'object') {
                    Object.values(contextDict).forEach(recurse);
                }
            }
            else {
                Object.values(x).forEach(recurse);
            }
        }
    }
}
/**
 * Return the logical IDs found in a {Fn::Sub} format string
 */
function logicalIdsInSubString(x) {
    return analyzeSubPattern(x).flatMap((fragment) => {
        switch (fragment.type) {
            case 'getatt':
            case 'ref':
                return [fragment.logicalId];
            case 'literal':
                return [];
        }
    });
}
function analyzeSubPattern(pattern) {
    const ret = [];
    let start = 0;
    let ph0 = pattern.indexOf('${', start);
    while (ph0 > -1) {
        if (pattern[ph0 + 2] === '!') {
            // "${!" means "don't actually substitute"
            start = ph0 + 3;
            ph0 = pattern.indexOf('${', start);
            continue;
        }
        const ph1 = pattern.indexOf('}', ph0 + 2);
        if (ph1 === -1) {
            break;
        }
        const placeholder = pattern.substring(ph0 + 2, ph1);
        if (ph0 > start) {
            ret.push({ type: 'literal', content: pattern.substring(start, ph0) });
        }
        if (placeholder.includes('.')) {
            const [logicalId, attr] = placeholder.split('.');
            ret.push({ type: 'getatt', logicalId: logicalId, attr: attr });
        }
        else {
            ret.push({ type: 'ref', logicalId: placeholder });
        }
        start = ph1 + 1;
        ph0 = pattern.indexOf('${', start);
    }
    if (start < pattern.length - 1) {
        ret.push({ type: 'literal', content: pattern.slice(start) });
    }
    return ret;
}
function intersect(xs, ys) {
    return new Set(Array.from(xs).filter(x => ys.has(x)));
}
/**
 * Find cycles in a graph
 *
 * Not the fastest, but effective and should be rare
 */
function findCycle(deps) {
    for (const node of deps.keys()) {
        const cycle = recurse(node, [node]);
        if (cycle) {
            return cycle;
        }
    }
    throw new Error('No cycle found. Assertion failure!');
    function recurse(node, path) {
        var _a;
        for (const dep of (_a = deps.get(node)) !== null && _a !== void 0 ? _a : []) {
            if (dep === path[0]) {
                return [...path, dep];
            }
            const cycle = recurse(dep, [...path, dep]);
            if (cycle) {
                return cycle;
            }
        }
        return undefined;
    }
}
//# sourceMappingURL=data:application/json;base64,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