"""!The Python Ionic SDK handles errors through exceptions rather than error return codes.
"""

import ionicsdk._util as _private
import ionicsdk
from ctypes import *

class CServerResponse(Structure):
    __slots__ = [
            'nHttpResponseCode',
            'nServerErrorCode',
            'pszServerErrorMessage',
            'pszServerErrorDataJson',
            'pszConversationId',
            ]
    _fields_ = [
            ('nHttpResponseCode', c_int),
            ('nServerErrorCode', c_int),
            ('pszServerErrorMessage', c_char_p),
            ('pszServerErrorDataJson', c_char_p),
            ('pszConversationId', c_char_p),
            ]

class IonicException(Exception):
    """!Exception for Ionic SDK client logic.
    """
    def __init__(self, message, code):
        """!Constructs an exception instance with the provided values

        @param
            message (string) The error text message
        @param
            code (int) The error code that caused the exception. See errors.py for more information.
        """
        super(IonicException, self).__init__(message)
        ##(int) The error code that caused the exception. See errors.py for more information.
        self.code = code
        ##(string) The error text message
        self.message = message
    
    def __str__(self):
        message = super(IonicException, self).__str__()
        message += " - Error code: " + str(self.code)
        message += "\r\nVersion: " + str(ionicsdk.__version__) + ", commit ID: " + str(ionicsdk.__commit__)
        return message

class ServerResponse(object):
    """!Server response object returned from most Ionic server calls.
    """
    def __init__(self, httpcode, servercode, servermessage, serverdatajson, conversationid):
        """!Constructs a server response instance with the provided values

        @param
            httpcode (int) The HTTP response code
        @param
            servercode (int) The internal server error code or zero if none was provided by the server
        @param
            servermessage (string) The internal server error data JSON or empty if none was provided by the server
        @param
            serverdatajson (string) The internal server error data JSON or empty if none was provided by the server
        @param
            conversationid (string) The conversation ID generated by the client for use in the server request
        """

        ## (int) The HTTP response code
        self.httpcode = httpcode
        ## (int) The internal server error code or zero if none was provided by the server
        self.servercode = servercode
        ##(string) The internal server error message or empty if none was provided by the server
        self.servermessage = servermessage
        ##(string) The internal server error data JSON or empty if none was provided by the server
        self.serverdatajson = serverdatajson
        ##(string) The conversation ID generated by the client for use in the server request
        self.conversationid = conversationid
    
    @staticmethod
    def _marshalFromC(cServerResponseLP):
        if cServerResponseLP is None:
            return None
        cServerResponse = cServerResponseLP.contents
        return ServerResponse(cServerResponse.nHttpResponseCode,
                              cServerResponse.nServerErrorCode,
                              _private.CMarshalUtil.stringFromC(cServerResponse.pszServerErrorMessage),
                              _private.CMarshalUtil.stringFromC(cServerResponse.pszServerErrorDataJson),
                              _private.CMarshalUtil.stringFromC(cServerResponse.pszConversationId))
            
    @staticmethod
    def _marshalToC(serverException):
        if serverException is None:
            return None

        return CServerResponse( serverException.httpcode,
                                serverException.servercode,
                                _private.CMarshalUtil.stringToC(serverException.servermessage),
                                _private.CMarshalUtil.stringToC(serverException.serverdatajson),
                                _private.CMarshalUtil.stringToC(serverException.conversationid))

# Exception for server responses
class IonicServerException(IonicException, ServerResponse):
    """!Exception for Ionic server errors.
    """
    def __init__(self, message, code, httpcode, servercode, servermessage, serverdatajson, conversationid):
        """!Constructs a server exception instance with the provided values

        @param
            message (string) The error text message
        @param
            code (int) The error code that caused the exception. See errors.py for more information.
        @param
            httpcode (int) The HTTP response code
        @param
            servercode (int) The internal server error code or zero if none was provided by the server
        @param
            servermessage (string) The internal server error data JSON or empty if none was provided by the server
        @param
            serverdatajson (string) The internal server error data JSON or empty if none was provided by the server
        @param
            conversationid (string) The conversation ID generated by the client for use in the server request
        """
        IonicException.__init__(self, message, code)
        ServerResponse.__init__(self, httpcode, servercode, servermessage, serverdatajson, conversationid)
    
    def __str__(self):
        message = super(IonicServerException, self).__str__()
        if self.httpcode:
            message += "\r\nHTTP code: " + str(self.httpcode)
        if self.servercode:
            message += "\r\nServer error code: " + str(self.servercode)
        if self.servermessage:
            message += "\r\nServer error message: " + str(self.servermessage)
        if self.conversationid:
            message += "\r\nConversation ID: " + str(self.conversationid)
        return message
    
    @staticmethod
    def _marshalFromC(message, code, cServerResponse):
        if cServerResponse is None:
            return None
        return IonicServerException(
            message,
            code,
            cServerResponse.nHttpResponseCode,
            cServerResponse.nServerErrorCode,
            _private.CMarshalUtil.stringFromC(cServerResponse.pszServerErrorMessage),
            _private.CMarshalUtil.stringFromC(cServerResponse.pszServerErrorDataJson),
            _private.CMarshalUtil.stringFromC(cServerResponse.pszConversationId))
        
