# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pyroots']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'pyroots',
    'version': '0.4.0',
    'description': 'Pure python single variable function solvers',
    'long_description': '.. image:: https://travis-ci.org/pmav99/pyroots.svg?branch=master\n    :target: https://travis-ci.org/pmav99/pyroots\n\npyroots\n=======\n\nAbstract\n--------\n\nA Python library implementing various root finding methods for single-variable\nfunctions.\n\nCurrently the following methods have been implemented::\n\n* The `bisect <http://en.wikipedia.org/wiki/Bisection_method>`_ method.\n* The `ridder <http://en.wikipedia.org/wiki/Ridders%27_method>`_ method.\n* The `brent <http://en.wikipedia.org/wiki/Brent%27s_method>`_ method.\n\nWith regard to ``Brent``\'s method, there are two implementations, the first one\nuses inverse quadratic extrapolation (``Brentq``) while the other ones uses\nhyperbolic extrapolation (``Brenth``).\n\nIf you don\'t know which method to use, you should probably use ``Brentq``.  That\nbeing said, ``Bisect`` method is safe and slow.\n\nExample\n-------\n\n::\n\n    # define the function whose root you are searching\n    def f(x, a):\n        return x ** 2 - a + 1\n\n    # Create the Solver object (instead of Brentq you could also import Brenth/Ridder/Bisect)\n    from pyroots import Brentq\n    brent = Brentq(epsilon=1e-5)\n\n    # solve the function in `[-3, 0]` while `a` is equal to 2\n    result = brent(f, -3, 0, a=2)\n    print(result)\n\nwill output::\n\n         converged : True\n           message : Solution converged.\n         iteration :   6\n        func calls :   9\n                x0 :    -1.0000000748530762\n              xtol :     0.0000000000000002\n             f(x0) :     0.0000001497061579\n           epsilon :     0.0000100000000000\n\nRationale\n---------\n\nThe functionality of ``pyroots`` is already implemented in ``scipy``, so the\nnatural question is why rediscover the wheel?\n\nWell, the main reason is that ``scipy`` is a huge dependency.  ``Pyroots`` on\nthe other hand is just a single package that is easily installed and that you\ncan easily bundle with ``py2exe`` or similar projects.  It doesn\'t even need to\nget installed, just throw the ``pyroots`` folder in your project and you are\nready to go.\n\nApart from that, the API used by ``scipy``\'s functions is not very\nuser-friendly. For example you can\'t use keyword arguments for your functions.\nMoreover, in ``scipy`` there is no reliable way to define how many digits of\naccuracy you want in the obtained root.  For example, you may ask for 6 digits,\nbut scipy may calculate up to 14 (or 12 or whatever) digits.  The main\nimplication of this "glitch" is that scipy\'s method may evaluate the function\nmore times than those really needed. If the function calculates something\ntrivial like the functions in the following examples, then these extra function\ncalls are no big deal, but if your functions take significant time to evaluate\n,e.g. more than seconds, then this can quickly become annoying, or even, simply\nunacceptable, e.g. the function takes some minutes to return a value.\n\nInstallation\n------------\n\nwith pip::\n\n    pip install pyroots\n\nor from source::\n\n    python setup.py install\n\nUsage\n-----\n\nAll the solvers share the same API, so you can easily switch between the\nvarious methods.\n\nFunction\n++++++++\n\nThe function whose root you are searching must take at least a single argument\nand return a single number.  This first argument is also the dependent variable\nand, apart from that, the function can also take any number of\npositional/keyword arguments. For example the following functions are totally\nvalid ones::\n\n    def f(x, a):\n        return x ** 2 - a + 1\n\n    def g(x, a, b, c=3):\n        return x ** 2 + a ** b - c\n\nSolver Objects\n--------------\n\nThe first thing you have to do is to create a ``Solver`` object for the method\nyou want to use::\n\n    from pyroots import Brentq\n\n    brent = Brentq()\n\nWhen you create the ``Solver`` object, you can specify several parameters\nthat will affect the convergence. The most important are:\n\n* `epsilon` which specifies the number of digits that will be taken under\n  consideration when checking for convergence. It defaults to `1e-6`.\n* `raise_on_fail` which will raise an exception if convergence failed. It\n  defaults to `True`.\n\nUsing the above function definitions, in order to find the root of ``f`` you\nmust first define an interval that contains the root. Let\'s say that this\ninterval is defined as ``[xa, xb]``.  In this case you will call the solver\nlike this::\n\n    def f(x, a):\n        return x ** 2 - a + 1\n\n    solver = Brentq()\n    result = solver(f, xa, xb, a=3)\n\nResult Objects\n--------------\n\nAll the methods return a ``Result`` object that has the following attributes::\n\n    result.x0               # the root\n    result.fx0              # the value of ``f(x0)`\n    result.convergence      # True/False\n    result.iterations       # the number of iterations\n    result.func_calls       # the number of function evaluations.\n    result.msg              # a descriptive message regarding the convergence (or the failure of convergence)\n\nIf, for some reason, convergence cannot be achieved, then a ``ConvergenceError``\nis raised.  If you don\'t want that to happen, then you have to pass ``False`` as\nthe value of ``raise_on_fail`` argument::\n\n    def f(x):\n        return x ** 2 - 1\n\n    result = brent(f, xa=-10, xb=-5, raise_on_fail=False)\n    print(result)\n\nAPI\n---\n\nEach solver factory has the following signature::\n\n    SolverFactory(epsilon=1e-6, xtol=EPS, max_iter=500, raise_on_fail=True, debug_precision=10)\n\nwhere:\n\n* ``epsilon`` is the required precision of the solution, i.e. a solution is\n  achieved when ``|f(x0)|`` is smaller than ``epsilon``.\n* ``max_iter`` is the maximum allowed number of iterations.\n* ``raise_on_fail`` is a boolean flag indicating whether or not an exception\n  should be raised if convergence fails. It defaults to True\n\nEach solver object has the following signature::\n\n    solver_object(f, xa, xb, *args, **kwargs)\n\nwhere:\n\n* ``f`` is the function whose root we are searching.\n* ``xa`` is the lower bracket of the interval of the solution we search.\n* ``xb`` is the upper bracket of the interval of the solution we search.\n* ``*args`` are passed as positional arguments when ``f`` is evaluated.\n* ``**kwargs`` are passed as keyword arguments when ``f`` is evaluated.\n\nDocumentation\n-------------\n\nFor the time being documentation is not yet ready, but the examples in the\nREADME should be enough to get your feet wet.\n\nThe source code repository of pyroots can be found at: https://github.com/pmav99/pyroots\n\nFeedback and contributions are greatly appreciated.\n\npmav99 <gmail>\n',
    'author': 'Panos Mavrogiorgos',
    'author_email': 'pmav99@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/pmav99/pyroots',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=2.7, !=3.0.*, !=3.1.*, !=3.2.*, !=3.3.*',
}


setup(**setup_kwargs)
