from dataclasses import dataclass, field
from enum import Enum
from typing import Any, Dict, Optional


class SentinelStatus(Enum):
    """Sentinel operational status."""
    STOPPED = "stopped"
    RUNNING = "running"
    PAUSED = "paused"
    ERROR = "error"


class AlertSeverity(Enum):
    """Alert severity levels."""
    INFO = "info"
    WARNING = "warning"
    CRITICAL = "critical"


@dataclass
class HealthCheck:
    """Definition of a health check."""
    name: str
    target: str  # Host or service to check
    check_type: str  # "ping", "port", "http", "custom"
    interval_seconds: int = 60
    timeout_seconds: int = 10
    threshold_failures: int = 3  # Consecutive failures to trigger alert
    parameters: Dict[str, Any] = field(default_factory=dict)
    enabled: bool = True


@dataclass
class CheckResult:
    """Result of a health check."""
    check: HealthCheck
    success: bool
    response_time_ms: float
    timestamp: str
    error: Optional[str] = None
    details: Dict[str, Any] = field(default_factory=dict)


@dataclass
class Alert:
    """An alert generated by the sentinel."""
    id: str
    check_name: str
    target: str
    severity: AlertSeverity
    message: str
    timestamp: str
    consecutive_failures: int
    incident_id: Optional[str] = None  # If incident was created
    acknowledged: bool = False
