from typing import Dict, Type, Optional, cast

import oaas
from grpc import Channel
from oaas import ClientDefinition
from oaas.client_provider import T
from oaas_grpc.client.grpc_proxy_instance_handler import GrpcProxyInstanceHandler
from oaas_grpc.client.proxy import GrpcCallProxy
from oaas_registry_api.rpc.registry_pb2 import (
    OaasServiceDefinition,
)


def as_service_definition(client_definition: ClientDefinition) -> OaasServiceDefinition:
    return OaasServiceDefinition(
        namespace=client_definition.namespace,
        name=client_definition.name,
        version=client_definition.version,
        tags=client_definition.tags,
    )


class OaasGrpcClient(oaas.ClientMiddleware):
    def __init__(self) -> None:
        self._channels: Dict[str, Channel] = dict()

    def create_client(self, client_definition: ClientDefinition) -> T:
        """
        In the default setup, the
        :param client_definition:
        :return:
        """
        return self.create_grpc_client(
            namespace=client_definition.namespace,
            name=client_definition.name,
            version=client_definition.version,
            tags=client_definition.tags,
            code=cast(Type[T], client_definition.code),
        )

    def create_grpc_client(
        self,
        *,
        namespace: str = "default",
        name: str,
        version: str = "1",
        tags: Optional[Dict[str, str]] = None,
        code: Type[T]
    ) -> T:
        return cast(
            T,
            GrpcCallProxy(
                instance_handler=GrpcProxyInstanceHandler(
                    namespace=namespace,
                    name=name,
                    version=version,
                    tags=tags,
                    code=code,
                )
            ),
        )

    def can_handle(self, client_definition: ClientDefinition) -> bool:
        """
        We only accept services that end with Stub, that are
        probably generated by gRPC.
        """
        name = client_definition.code.__name__

        return name.endswith("Stub")
