# Copyright 2024 Broadcom. All Rights Reserved.
import logging
import re
from typing import Any
from typing import Dict
from typing import List
from typing import Tuple

from pyVmomi import vim  # pylint: disable=E0401

from config_modules_vmware.controllers.base_controller import BaseController
from config_modules_vmware.controllers.vcenter.utils import vc_users_groups_roles_utils
from config_modules_vmware.framework.auth.contexts.base_context import BaseContext
from config_modules_vmware.framework.auth.contexts.vc_context import VcenterContext
from config_modules_vmware.framework.clients.common import consts
from config_modules_vmware.framework.logging.logger_adapter import LoggerAdapter
from config_modules_vmware.framework.models.controller_models.metadata import ControllerMetadata
from config_modules_vmware.framework.models.output_models.compliance_response import ComplianceStatus
from config_modules_vmware.framework.models.output_models.remediate_response import RemediateStatus

logger = LoggerAdapter(logging.getLogger(__name__))

NAME = "name"
TYPE = "type"
ROLE = "role"
USER = "USER"
GROUP = "GROUP"
PROPAGATE = "propagate"
ROLE_ID = "role_id"
GLOBAL_CONFIG = "global"
VCENTER_CONFIG = "vcenter"
EXCLUDE_USER_PATTERNS = "exclude_user_patterns"
TO_ADD = "+"
TO_DELETE = "-"


class UsersGroupsRolesConfig(BaseController):
    """Class for UsersGroupsRolesSettings config with get and set methods.
    | ConfigId - 415
    | ConfigTitle - The vCenter Server users must have the correct roles assigned.
    """

    def __init__(self):
        # Initialize instance key with "name" and comparator option with Identifier based comparison.
        super().__init__()
        self.domain_alias_to_name_map = None

    metadata = ControllerMetadata(
        name="users_groups_roles",  # controller name
        path_in_schema="compliance_config.vcenter.users_groups_roles",
        # path in the schema to this controller's definition.
        configuration_id="415",  # configuration id as defined in compliance kit.
        title="The vCenter Server users must have the correct roles assigned.",  # controller title as defined in compliance kit.
        tags=[],  # controller tags for future querying and filtering
        version="1.0.0",  # version of the controller implementation.
        since="",  # version when the controller was first introduced in the compliance kit.
        products=[BaseContext.ProductEnum.VCENTER],  # product from enum in BaseContext.
        components=[],  # subcomponent within the product if applicable.
        status=ControllerMetadata.ControllerStatus.ENABLED,  # used to enable/disable a controller
        impact=ControllerMetadata.RemediationImpact.REMEDIATION_SKIPPED,  # from enum in ControllerMetadata.RemediationImpact.
        scope="",  # any information or limitations about how the controller operates. i.e. runs as a CLI on VCSA.
    )

    def _load_user_exclude_patterns(self, exclude_user_patterns: List):
        """
        Load exclude user patterns.
        :param exclude_user_patterns: patterns input from desired state to exclude certain users such as the
                                      ones automatically generated by system..
        :type exclude_user_patterns: List
        """
        patterns = []
        for pattern in exclude_user_patterns:
            pattern = "^" + pattern
            pattern = pattern.replace("\\", "\\\\")
            regex_pattern = re.compile(pattern, re.IGNORECASE)
            logger.debug(f"Regex pattern: {regex_pattern}")
            patterns.append(regex_pattern)
        logger.debug(f"Regex patterns: {patterns}")
        return patterns

    def _create_alias_domain_name_mapping(self, context: VcenterContext):
        """
        Create domain alias to domain name mapping for existing aliases.
        :param context: Product context instance.
        :type context: VcenterContext
        """
        if self.domain_alias_to_name_map is None:
            vc_vmomi_sso_client = context.vc_vmomi_sso_client()
            all_domains = vc_vmomi_sso_client.get_all_domains()
            logger.info(f"All domains in VC {all_domains}")
            self.domain_alias_to_name_map = {}
            default_domain = getattr(all_domains, "systemDomainName", "")
            if default_domain:
                self.domain_alias_to_name_map[default_domain.lower()] = default_domain.lower()
            external_domains = getattr(all_domains, "externalDomains", [])
            for external_domain in external_domains:
                if external_domain.alias:
                    self.domain_alias_to_name_map[external_domain.alias.lower()] = external_domain.name.lower()
        logger.info(f"domain alias mapping {self.domain_alias_to_name_map}")

    def _should_filter(self, name: str, patterns) -> bool:
        """Checks if a given name belongs to local domain."""
        for pattern in patterns:
            if pattern.match(name):
                return True
        return False

    def _normalize_domain(self, name: str) -> str:
        """Replaces alias with canonical domain if alias exists, otherwise returns name as is."""
        if "\\" in name:
            domain_or_alias, user_or_group = name.split("\\", 1)
            domain_or_alias = domain_or_alias.lower()
            # Replace alias with domain name if an alias is configured, otherwise keep the original
            domain_name = self.domain_alias_to_name_map.get(domain_or_alias, domain_or_alias)
            return f"{domain_name}\\{user_or_group.lower()}"
        else:
            return name.lower()

    def _is_global_defined_permission(self, permission: Dict, global_users_roles: List) -> bool:
        """
        Check if a permission retrieved with vCenter method is global defined permission.
        :param permission: vCenter retrieved permission.
        :type permission: Dict
        :param global_users_roles: global permissions.
        :type global_users_roles: List
        :return: True if it is global defined permission otherwise it is vCenter local permission
        :rtype: bool
        """
        for global_permission in global_users_roles:
            if (
                self._normalize_domain(global_permission[NAME]) == self._normalize_domain(permission[NAME])
                and global_permission[ROLE] == permission[ROLE]
                and global_permission[TYPE] == permission[TYPE]
                and global_permission[PROPAGATE] == permission[PROPAGATE]
                and permission[PROPAGATE]
            ):
                return True
        return False

    def _get_global_permissions(self, context: VcenterContext) -> List:
        """Get global permissions through interbal APIs.

        :param context: Product context instance.
        :type context: VcenterContext
        :return: List of dictionary (with keys-'role', 'name', 'type', and 'propagate') objects.
        :rtype: List
        """
        content = context.vc_vmomi_client().content
        authorization_manager = content.authorizationManager
        role_collection = authorization_manager.roleList
        role_id_name_map = {role.roleId: role.name for role in role_collection}

        global_users_roles = []
        vc_invsvc_mob3_client = context.vc_invsvc_mob3_client()
        global_permissions = vc_users_groups_roles_utils.get_global_permissions(vc_invsvc_mob3_client)
        for global_permission in global_permissions:
            global_permission[ROLE] = role_id_name_map[global_permission.get(ROLE_ID)]
            del global_permission[ROLE_ID]
            global_users_roles.append(global_permission)
        return global_users_roles

    def _get_vcenter_permissions(self, context: VcenterContext, global_users_roles: List) -> List:
        """Get vcenter permissions through pyvmomi APIs.

        :param context: Product context instance.
        :type context: VcenterContext
        :param global_users_roles: a list of global permissions.
        :type global_users_roles: VcenterContext
        :return: List of dictionary (with keys-'role', 'name', 'type', and 'propagate') objects.
        :rtype: List
        """
        content = context.vc_vmomi_client().content
        authorization_manager = content.authorizationManager
        role_collection = authorization_manager.roleList

        vc_users_roles = []
        unique_entries = set()  # Set to store unique (name, role, type)
        for role in role_collection:
            permissions = authorization_manager.RetrieveRolePermissions(role.roleId)
            for permission in permissions:
                # skip permissions not in vcenter level
                # exclude permission that is not defined on this vcenter.
                name = permission.principal
                if permission.entity != content.rootFolder:
                    logger.debug(f"Skip this permission, name: {name}, entity: {permission.entity}")
                    continue
                vc_permission = {
                    ROLE: role.name,
                    NAME: name,
                    TYPE: USER if not permission.group else GROUP,
                    PROPAGATE: permission.propagate,
                }
                # if a permission is defined in both global and vcenter level with exact same
                # name, type, role and propagate attribute, we will see two exactly same entries
                # in vcenter permission, one defined in global should be excluded.
                key_tuple = (name.lower(), role.name, USER if not permission.group else GROUP, permission.propagate)
                if key_tuple not in unique_entries:
                    unique_entries.add(key_tuple)
                    # check if this permission is global permission, if yes, exclude
                    # from compliance checking for vcenter portion
                    if self._is_global_defined_permission(vc_permission, global_users_roles):
                        logger.debug(f"Skip this global permission: {vc_permission}")
                        continue
                vc_users_roles.append(vc_permission)
        return vc_users_roles

    def get(self, context: VcenterContext) -> Tuple[Dict[str, list], List[Any]]:
        """Get roles for users and groups.

        :param context: Product context instance.
        :type context: VcenterContext
        :return: Tuple of list of dictionary (with keys-'role', 'name', 'type', and 'propagate') objects and a list of error messages.
        :rtype: Tuple
        """

        errors = []
        result = {}
        self._create_alias_domain_name_mapping(context)

        try:
            # get global permissions
            global_users_roles = self._get_global_permissions(context)
            if global_users_roles:
                result[GLOBAL_CONFIG] = global_users_roles
            logger.debug(f"Get global permissions: {global_users_roles}")

            # get vcenter level permissions
            vc_users_roles = self._get_vcenter_permissions(context, global_users_roles)
            if vc_users_roles:
                result[VCENTER_CONFIG] = vc_users_roles
            logger.debug(f"Vcenter permissions: {vc_users_roles}")

        except Exception as e:
            logger.exception(f"An error occurred: {e}")
            errors.append(str(e))
        return result, errors

    def _set_global_permissions(
        self, context: VcenterContext, global_remediate_configs: List[Dict], role_name_id_map: Dict
    ) -> None:
        """Set global permissions for users and groups.

        :param context: Product context instance.
        :type context: VcenterContext
        :param global_remediate_configs: remediate configs for global permissions.
        :type global_remediate_configs: List
        :param role_name_id_map: role name to id map.
        :type role_name_id_map: Dict
        :return: None
        :rtype: None
        """

        logger.debug(f"Remediate drifts for global permissions: {global_remediate_configs}")
        # remediate global permissions
        vc_invsvc_mob3_client = context.vc_invsvc_mob3_client()
        for op, entry in global_remediate_configs:
            user = entry[NAME]
            role_id = role_name_id_map[entry[ROLE]]
            group = True if entry[TYPE] == GROUP else False
            propagate = entry[PROPAGATE]
            if op == TO_ADD:
                logger.debug(
                    f"Adding Permission for Entry: user - {user}, role id: {role_id}, group - {group}, propagate - {propagate}"
                )
                vc_users_groups_roles_utils.add_global_permission(
                    vc_invsvc_mob3_client, user, role_id, group, propagate
                )
            else:  # op == TO_DELETE
                logger.debug(f"Removing Permission for Entry: {user}, group: {group}")
                vc_users_groups_roles_utils.remove_global_permission(vc_invsvc_mob3_client, user, group)
            logger.debug(f"Op: {op} for Global Permission Entry: {entry}")

    def _set_vcenter_permissions(
        self, context: VcenterContext, vc_remediate_configs: List[Dict], role_name_id_map: Dict
    ) -> None:
        """Set vcenter permissions for users and groups.

        :param context: Product context instance.
        :type context: VcenterContext
        :param vc_remediate_configs: remediate configs for vcenter permissions.
        :type vc_remediate_configs: List
        :param role_name_id_map: role name to id map.
        :type role_name_id_map: Dict
        :return: None
        :rtype: None
        """

        content = context.vc_vmomi_client().content
        authorization_manager = content.authorizationManager

        logger.debug(f"Remediate drifts for vCenter permission: {vc_remediate_configs}")
        # get vcenter MOID.
        vc_moid = content.rootFolder
        for op, entry in vc_remediate_configs:
            user = entry[NAME]
            group = entry[TYPE] == GROUP
            if op == TO_ADD:
                propagate = entry[PROPAGATE]
                perm = vim.AuthorizationManager.Permission()
                perm.principal = user
                perm.roleId = role_name_id_map[entry[ROLE]]
                perm.group = group
                perm.propagate = propagate
                logger.debug(f"Permission: {perm}")
                authorization_manager.SetEntityPermissions(vc_moid, [perm])
            else:  # op == TO_DELETE
                authorization_manager.RemoveEntityPermission(vc_moid, user, group)
            logger.debug(f"Op: {op} for vCenter permission Entry: {entry}")

    def set(self, context: VcenterContext, desired_values: Dict) -> Tuple:
        """
        Set method to remediate drifts found in users, roles and permissions.
        """
        status = RemediateStatus.SUCCESS
        errors = []
        self._create_alias_domain_name_mapping(context)

        try:
            content = context.vc_vmomi_client().content
            authorization_manager = content.authorizationManager
            role_collection = authorization_manager.roleList
            role_name_id_map = {role.name: role.roleId for role in role_collection}

            # remediate global permissions.
            global_remediate_configs = desired_values.get(GLOBAL_CONFIG)
            if global_remediate_configs:
                self._set_global_permissions(context, global_remediate_configs, role_name_id_map)
                logger.info("Global permissions updated successfully")

            # remediate vcenter permissions.
            vc_remediate_configs = desired_values.get(VCENTER_CONFIG)
            if vc_remediate_configs:
                self._set_vcenter_permissions(context, vc_remediate_configs, role_name_id_map)
                logger.info("vCenter permissions updated successfully")

        except Exception as e:
            logger.exception(f"An error occurred: {e}")
            errors.append(str(e))
            status = RemediateStatus.FAILED

        return status, errors

    def _find_drift(self, current: List, desired: List) -> List:
        """Find the drift between current and desired users_groups_roles.

        :param current: Current list of users,groups,roles.
        :type current: List
        :param desired: Desired list of users,groups,roles.
        :type desired: list
        :return: List of drifts dict for compliance check or remediation
        :rtype: List
        """

        # user name (principal) and type (user or group) are two keys to differentiate
        # a permission entry.
        current_dict = {(entry[NAME], entry[TYPE]): entry for entry in (current or [])}
        desired_dict = {(entry[NAME], entry[TYPE]): entry for entry in (desired or [])}
        config_drift = []
        all_keys = set(current_dict.keys()).union(set(desired_dict.keys()))

        for key in all_keys:
            current_entry = current_dict.get(key)
            desired_entry = desired_dict.get(key)
            logger.debug(f"Current Entry: {current_entry}, Desired Entry: {desired_entry}")

            if current_entry and desired_entry:
                # Both exist: check for mismatches
                if current_entry != desired_entry:
                    # to modify an entry, remove old one and add new one based on desired state.
                    config_drift.append((TO_DELETE, current_entry))
                    config_drift.append((TO_ADD, desired_entry))
            elif current_entry:
                # Extra in current
                config_drift.append((TO_DELETE, current_entry))
            elif desired_entry:
                # Extra in desired
                config_drift.append((TO_ADD, desired_entry))

        return config_drift

    def _find_drifts(self, current: Dict, desired: Dict) -> Tuple[Dict, List]:
        """Find the drift between current and desired users_groups_roles.

        :param current: Current configs of users,groups,roles.
        :type current: List
        :param desired: Desired configs of users,groups,roles.
        :type desired: list
        :return: Tuple of config drifts and a list of errors if any
        :rtype: Tuple
        """

        errors = []
        config_drifts = {}
        try:
            current = self._normalize_all_domains(current)
            desired = self._normalize_all_domains(desired)

            for config_type in [GLOBAL_CONFIG, VCENTER_CONFIG]:
                current_values = current.get(config_type)
                desired_values = desired.get(config_type)
                drift = self._find_drift(current_values, desired_values)
                if drift:
                    config_drifts[config_type] = drift
            logger.debug(f"Configuration drifts: {config_drifts}")

        except Exception as e:
            logger.exception(f"An error occurred: {e}")
            errors.append(str(e))

        return config_drifts, errors

    def _normalize_all_domains(self, config_values):
        normalized_values = {}

        for config_type in [GLOBAL_CONFIG, VCENTER_CONFIG]:
            values = config_values.get(config_type)
            if values:
                normalized_values[config_type] = [
                    {
                        ROLE: item[ROLE],
                        NAME: self._normalize_domain(item[NAME]),
                        TYPE: item[TYPE].upper(),
                        PROPAGATE: item[PROPAGATE],
                    }
                    for item in values
                ]

        return normalized_values

    def _exclude_users(self, config_values, patterns):
        exclude_local_configs = {}
        for config_type in [GLOBAL_CONFIG, VCENTER_CONFIG]:
            values = config_values.get(config_type, [])
            for item in values:
                if not self._should_filter(item.get(NAME), patterns):
                    exclude_local_configs.setdefault(config_type, []).append(item)
        return exclude_local_configs

    def check_compliance(self, context: VcenterContext, desired_values: Dict) -> Dict:
        """Check compliance of current configuration against provided desired values.

        :param context: Product context instance.
        :type context: VcenterContext
        :param desired_values: Desired list of users,groups,roles on vCenter.
        :type desired_values: list
        :return: Dict of status and current/desired value(for non_compliant) or errors (for failure).
        :rtype: Dict
        """

        logger.info("Running check compliance for vCenter users_groups_roles control")
        current_values, errors = self.get(context=context)
        if errors:
            return {consts.STATUS: ComplianceStatus.FAILED, consts.ERRORS: errors}

        # check if need to exclude local users.
        exclude_user_patterns = desired_values.get(EXCLUDE_USER_PATTERNS, [])
        if exclude_user_patterns:
            patterns = self._load_user_exclude_patterns(exclude_user_patterns)
            current_values = self._exclude_users(current_values, patterns)

        # Compare the current and desired value. If not same, return "NON_COMPLIANT" with values.
        # Otherwise, return "COMPLIANT".
        non_compliant_configs, errors = self._find_drifts(current_values, desired_values)
        if errors:
            return {consts.STATUS: ComplianceStatus.FAILED, consts.ERRORS: errors}

        if non_compliant_configs:
            if EXCLUDE_USER_PATTERNS in desired_values:
                del desired_values[EXCLUDE_USER_PATTERNS]
            return {
                consts.STATUS: ComplianceStatus.NON_COMPLIANT,
                consts.CURRENT: current_values,
                consts.DESIRED: desired_values,
            }
        return {consts.STATUS: ComplianceStatus.COMPLIANT}

    def remediate(self, context: VcenterContext, desired_values: Dict) -> Dict:
        """
        Remediate configuration drifts by applying desired values.

        | Sample desired state

        .. code-block:: json

            {
              "global": [
                {
                  "name": "VSPHERE.LOCAL\\SyncUsers",
                  "type": "GROUP",
                  "role": "SyncUsers",
                  "propagate": True
                },
                {
                  "name": "VSPHERE.LOCAL\\Administrator",
                  "type": "USER",
                  "role": "Admin",
                  "propagate": True
                }
              ],
              "vcenter": [
                {
                  "name": "VSPHERE.LOCAL\\Administrator",
                  "type": "USER",
                  "role": "Admin",
                  "propagate": True
                },
                {
                  "name": "VSPHERE.LOCAL\\Administrators",
                  "type": "USER",
                  "role": "Admin",
                  "propagate": False
                }
              ]
            }

        :param context: Product context instance.
        :type context: VcenterContext
        :param desired_values: Desired values for VM migrate encryption policy
        :type desired_values: Dict
        :return: Dict of status and current/desired value(for non_compliant) or errors (for failure).
        :rtype: Dict
        """
        logger.info("Running remediation")
        current_values, errors = self.get(context=context)

        if errors:
            return {consts.STATUS: RemediateStatus.FAILED, consts.ERRORS: errors}

        # check if need to exclude local users.
        exclude_user_patterns = desired_values.get(EXCLUDE_USER_PATTERNS, [])
        if exclude_user_patterns:
            patterns = self._load_user_exclude_patterns(exclude_user_patterns)
            current_values = self._exclude_users(current_values, patterns)
        remediate_configs, errors = self._find_drifts(current_values, desired_values)
        logger.debug(f"Current configs: {current_values}, Desired configs: {desired_values}")
        logger.debug(f"Drift for remediation: {remediate_configs}")
        if not errors and not remediate_configs:
            return {consts.STATUS: RemediateStatus.SKIPPED, consts.ERRORS: [consts.CONTROL_ALREADY_COMPLIANT]}
        elif errors:
            return {consts.STATUS: RemediateStatus.FAILED, consts.ERRORS: errors}

        status, errors = self.set(context, remediate_configs)
        if not errors:
            if EXCLUDE_USER_PATTERNS in desired_values:
                del desired_values[EXCLUDE_USER_PATTERNS]
            return {consts.STATUS: status, consts.OLD: current_values, consts.NEW: desired_values}
        return {consts.STATUS: status, consts.ERRORS: errors}
