# Copyright 2024 Broadcom. All Rights Reserved.
import logging
from typing import Any
from typing import Dict
from typing import List
from typing import Tuple

from config_modules_vmware.controllers.base_controller import BaseController
from config_modules_vmware.controllers.vcenter.utils.sso_member_utils import filter_member_configs
from config_modules_vmware.framework.auth.contexts.base_context import BaseContext
from config_modules_vmware.framework.auth.contexts.vc_context import VcenterContext
from config_modules_vmware.framework.clients.common import consts
from config_modules_vmware.framework.clients.vcenter.vc_vmomi_sso_client import VcVmomiSSOClient
from config_modules_vmware.framework.logging.logger_adapter import LoggerAdapter
from config_modules_vmware.framework.models.controller_models.metadata import ControllerMetadata
from config_modules_vmware.framework.models.output_models.compliance_response import ComplianceStatus
from config_modules_vmware.framework.models.output_models.remediate_response import RemediateStatus
from config_modules_vmware.framework.utils.comparator import Comparator

logger = LoggerAdapter(logging.getLogger(__name__))

BASH_SHELL_ADMINISTRATOR_GROUP_KEY = "SystemConfiguration.BashShellAdministrators"
ID_KEY = "id"
DOMAIN_KEY = "domain"
NAME_KEY = "name"
MEMBER_TYPE_KEY = "member_type"
MEMBER_TYPE_USER = "USER"
MEMBER_TYPE_GROUP = "GROUP"


class SSOBashShellAuthorizedMembersConfig(BaseController):
    """Manage authorized members in the SystemConfiguration.BashShellAdministrators group with get and set methods.

    | Config Id - 1216
    | Config Title - vCenter must limit membership to the SystemConfiguration.BashShellAdministrators SSO group.

    """

    metadata = ControllerMetadata(
        name="sso_bash_shell_authorized_members",  # controller name
        path_in_schema="compliance_config.vcenter.sso_bash_shell_authorized_members",
        # path in the schema to this controller's definition.
        configuration_id="1216",  # configuration id as defined in compliance kit.
        title="vCenter must limit membership to the SystemConfiguration.BashShellAdministrators SSO group.",
        # controller title as defined in compliance kit.
        tags=[],  # controller tags for future querying and filtering
        version="1.0.0",  # version of the controller implementation.
        since="",  # version when the controller was first introduced in the compliance kit.
        products=[BaseContext.ProductEnum.VCENTER],  # product from enum in BaseContext.
        components=[],  # subcomponent within the product if applicable.
        status=ControllerMetadata.ControllerStatus.ENABLED,  # used to enable/disable a controller
        impact=ControllerMetadata.RemediationImpact.REMEDIATION_SKIPPED,
        # from enum in ControllerMetadata.RemediationImpact.
        scope="",  # any information or limitations about how the controller operates. i.e. runs as a CLI on VCSA.
    )

    def get(self, context: VcenterContext) -> Tuple[List, List[Any]]:
        """Get authorized members in the SystemConfiguration.BashShellAdministrators group.

        | We limit our traversal to the first level of groups because a group can have subgroups, which in turn can
            contain groups with users. This approach is consistent with the behavior of the dir-cli command:

        .. code-block:: shell

            /usr/lib/vmware-vmafd/bin/dir-cli group list --name <group_name>

        | Sample get output

        .. code-block:: json

            [
              {
                "name": "user-1",
                "domain": "vmware.com",
                "member_type": "USER"
              },
              {
                "name": "user-2",
                "domain": "vmware.com",
                "member_type": "USER"
              },
              {
                "name": "devops",
                "domain": "vsphere.local",
                "member_type": "GROUP"
              }
            ]

        :param context: Product context instance.
        :type context: VcenterContext
        :return: Tuple of List of dictionaries containing user and groups belonging to the BashShellAdministrators
         group and a list of error messages.
        :rtype: Tuple
        """
        vc_vmomi_sso_client = context.vc_vmomi_sso_client()
        errors = []
        try:
            result = self.__get_all_members_of_bash_shell_administrator_group(vc_vmomi_sso_client)
        except Exception as e:
            logger.exception(f"An error occurred while retrieving members of bash shell admin group: {e}")
            errors.append(str(e))
            result = []
        return result, errors

    def __get_all_members_of_bash_shell_administrator_group(self, sso_client: VcVmomiSSOClient) -> List[Dict]:
        """Retrieve all members in SystemConfiguration.BashShellAdministrators group.

        :param sso_client: VcVmomiSSOClient instance
        :type sso_client: VcVmomiSSOClient
        :return: List of dictionaries containing user and groups belonging to the BashShellAdministrators group.
        :rtype: List[Dict]
        """
        members_in_bash_shell_administrators_group = []

        system_domain = sso_client.get_system_domain()
        logger.info(f"Retrieved system domain - {system_domain}")

        bash_shell_admin_group = sso_client._get_group(BASH_SHELL_ADMINISTRATOR_GROUP_KEY, system_domain)
        logger.info(f"Retrieved bash shell admin group - {bash_shell_admin_group}")

        if bash_shell_admin_group and hasattr(bash_shell_admin_group, ID_KEY):
            group_id = bash_shell_admin_group.id
            users_in_group = sso_client.find_users_in_group(group_id)
            logger.debug(f"Users in group - {users_in_group}")
            groups_in_group = sso_client.find_groups_in_group(group_id)
            logger.debug(f"Groups in group - {groups_in_group}")

            for user in users_in_group:
                if hasattr(user, ID_KEY):
                    user_principal_id = user.id
                    user_name = getattr(user_principal_id, NAME_KEY)
                    domain = getattr(user_principal_id, DOMAIN_KEY)

                    if user_name and domain:
                        user = {NAME_KEY: user_name, DOMAIN_KEY: domain, MEMBER_TYPE_KEY: MEMBER_TYPE_USER}
                        members_in_bash_shell_administrators_group.append(user)

            for group in groups_in_group:
                if hasattr(group, ID_KEY):
                    group_principal_id = group.id
                    group_name = getattr(group_principal_id, NAME_KEY)
                    domain = getattr(group_principal_id, DOMAIN_KEY)

                    if group_name and domain:
                        group = {NAME_KEY: group_name, DOMAIN_KEY: domain, MEMBER_TYPE_KEY: MEMBER_TYPE_GROUP}
                        members_in_bash_shell_administrators_group.append(group)

        logger.info(
            f"Retrieved all members in SystemConfiguration.BashShellAdministrators"
            f" group {members_in_bash_shell_administrators_group}"
        )
        return members_in_bash_shell_administrators_group

    def set(self, context: VcenterContext, desired_values: List) -> Tuple[str, List[Any]]:
        """Remediation has not been implemented for this control. It's possible that a customer may legitimately add
         a new user and forget to update the control accordingly. Remediating the control could lead to the removal
         of these users, with potential unknown implications.

        :param context: Product context instance.
        :type context: VcenterContext
        :param desired_values: List of objects containing users and groups details with name, domain and member_type.
        :type desired_values: List
        :return: Tuple of "status" and list of error messages.
        :rtype: Tuple
        """
        errors = [consts.REMEDIATION_SKIPPED_MESSAGE]
        status = RemediateStatus.SKIPPED
        return status, errors

    def check_compliance(self, context: VcenterContext, desired_values: Dict) -> Dict:
        """
        Check compliance of authorized members.

        :param context: Product context instance.
        :type context: VcenterContext
        :param desired_values: Desired values for authorized members.
        :type desired_values: Dict
        :return: Dict of status and current/desired value(for non_compliant) or errors (for failure).
        :rtype: Dict
        """
        logger.info("Checking compliance for sso bash shell authorized member config")
        all_member_configs, errors = self.get(context=context)

        if errors:
            return {consts.STATUS: ComplianceStatus.FAILED, consts.ERRORS: errors}

        # check if need to exclude any members.
        exclude_user_patterns = desired_values.get("exclude_user_patterns", [])
        if exclude_user_patterns:
            logger.debug(f"User input name patterns to be excluded from compliance check: {exclude_user_patterns}")
            all_member_configs = filter_member_configs(all_member_configs, exclude_user_patterns)

        non_compliant_configs, desired_configs = Comparator.get_non_compliant_configs(
            all_member_configs, desired_values.get("members", [])
        )

        if non_compliant_configs:
            result = {
                consts.STATUS: ComplianceStatus.NON_COMPLIANT,
                consts.CURRENT: non_compliant_configs,
                consts.DESIRED: desired_configs,
            }
        else:
            result = {consts.STATUS: ComplianceStatus.COMPLIANT}
        return result
