import os
import pandas as pd
import sys
from tabulate import tabulate
import click
import pandasql as psql
import numpy
from pandasql.sqldf import PandaSQLException


def get_filename(filepath):
    return os.path.splitext(filepath)[0]


def get_file_extension(filepath):
    return os.path.splitext(filepath)[1]


def assert_param(param, **kwargs):
    assert param in kwargs.keys(), f"Missing param: {param}!"


def check_path(original_function):
    def wrapper(*args, **kwargs):

        assert_param('filepath', **kwargs)

        # check if filepath exists
        if os.path.exists(kwargs['filepath']):
            return original_function(*args, **kwargs)
        else:
            click.echo(f"\nOuch! Could not find '{kwargs['filepath']}'")
            sys.exit(0)

    return wrapper


def validate_format(format):
    if not isinstance(format, str) or format not in['csv', 'excel', 'parquet']:
        click.echo("Ouch! Invalid format choice. You need to choose between 'csv', 'excel' or 'parquet'.")
        sys.exit()


def get_filepath_without_extension(filepath):
    current_file_ext = get_file_extension(filepath=filepath)
    return filepath[:-len(current_file_ext)]


def get_new_filepath(filepath, desired_format):

    validate_format(desired_format)

    filepath_wo_ext = get_filepath_without_extension(filepath=filepath)

    if desired_format == 'csv':
        return filepath_wo_ext + ".csv"

    elif desired_format == 'excel':
        return filepath_wo_ext + ".xlsx"

    elif desired_format == 'parquet':
        return filepath_wo_ext + ".parquet"


def output_to_file(*args, **kwargs):
    """
    Outputs a DataFrame in the desired format
    :param df: Pandas DataFrame
    :param filepath: path where the output file will be stored example '/Users/john/Downloads/mydir/myfile.csv'
    :param delimiter: (optional) csv delimiter if you want to output in csv with other than comma
    :param desired_format: has to be 'csv', 'excel' or 'parquet'
    :return: true if write operation went through, false otherwise

    """

    assert_param('df', **kwargs)
    assert_param('filepath', **kwargs)
    assert_param('desired_format', **kwargs)

    validate_format(kwargs['desired_format'])

    if kwargs['desired_format'] == 'csv':
        if 'delimiter' in kwargs.keys() and kwargs['delimiter'] is not None:
            try:
                kwargs['df'].to_csv(kwargs['filepath'], sep=kwargs['delimiter'], index=False)
                return True
            except Exception:
                return False
        else:
            try:
                kwargs['df'].to_csv(kwargs['filepath'], index=False)
                return True
            except Exception:
                return False

    elif kwargs['desired_format'] == 'excel':
        try:
            # pandas only supports writing on xlsx
            excel_filename = get_filename(kwargs['filepath']) + '.xlsx'
            kwargs['df'].to_excel(excel_filename, index=False)
            return True
        except Exception:
            return False

    elif kwargs['desired_format'] == 'parquet':
        try:
            kwargs['df'].to_parquet(kwargs['filepath'], index=False)
            return True
        except Exception:
            return False


@check_path
def delete_local_file(filepath):
    os.remove(filepath)


def pd_tabulate(my_df, headers='keys'):
    return tabulate(my_df, headers=headers, tablefmt='fancy_grid')


def get_col_list(col_string):

    # convert input str into list of columns
    col_list = col_string.split(',')

    # discard any potential trailing and leading white spaces generated by user input
    col_list = [x.strip(' ') for x in col_list].copy()

    return col_list


def get_format_from_file_extension(file_extension):

    if file_extension == '.csv':
        format = 'csv'

    elif file_extension == '.parquet':
        format = 'parquet'

    elif file_extension in ['.xlsx', '.xls']:
        format = 'excel'

    else:
        click.echo("Ouch! csvcli can only process CSV, excel and parquet files")
        sys.exit(0)

    return format


def guess_delimiter(filepath):

    for delim in [',', '\t', ';', '|']:
        if is_valid_delimiter(delimiter=delim, filepath=filepath):
            return delim

    return None


def is_valid_delimiter(delimiter, filepath):

    if not len(delimiter) == 1:
        click.echo("\nOuch! CSV delimiter must be a 1-character string")
        sys.exit(0)

    try:
        df = pd.read_csv(filepath, delimiter=delimiter, engine='c')
        if df.shape[1] > 1:
            return True
        else:
            return False

    except Exception:
        try:
            df = pd.read_csv(filepath, delimiter=delimiter, engine='python')
            if df.shape[1] > 1:
                return True
            else:
                return False

        except Exception:
            return False


@check_path
def read_file_to_df(filepath, format, delimiter):

    data = None

    if format == 'csv':
        data = pd.read_csv(filepath, delimiter=delimiter)

    elif format == 'parquet':
        data = pd.read_parquet(filepath)

    elif format == 'excel':
        data = pd.read_excel(filepath)

    return data


def assert_col_in_df(df, column):
    if column not in df.columns:
        click.echo(f"Ouch! Column '{column}' does not seem to exist...")
        sys.exit(0)


def filter_df(df, head=False, n=None, columns=None, sort_by=None, order='ASC'):

    output_df = df.copy()

    if columns is not None:

        col_list = get_col_list(col_string=columns)

        for col in col_list:
            assert_col_in_df(df=output_df, column=col)

        output_df = df[col_list].copy()

    if sort_by is not None:
        assert_col_in_df(df=output_df, column=sort_by)
        if order.lower() in ['asc', 'ascending']:
            output_df = output_df.sort_values(by=sort_by, ascending=True)
        elif order.lower() in ['desc', 'descending']:
            output_df = output_df.sort_values(by=sort_by, ascending=False)
        else:
            click.echo("Ouch! Wrong order value, you can choose either ASC for ascending or DESC for descending")
            sys.exit(0)

    if head:
        if n is not None:
            output_df = output_df.head(n)
        else:
            output_df = output_df.head()

    return output_df


def display_df(df, headers='keys'):
    click.echo(pd_tabulate(df, headers=headers))


def paginated_display(df, filename, chunksize=25000):

    page_count = df.shape[0] // chunksize
    if df.shape[0] % chunksize != 0:
        page_count += 1

    offset = 0
    limit = offset + chunksize

    for page in range(1, page_count+1):

        partial_df = df[offset:limit]
        display_df(df=partial_df)
        click.echo(f"\nFilename: {filename} - Total number of rows: {df.shape[0]}\n")
        offset += chunksize
        limit = offset + chunksize


def get_dtype(series, pretty=False):
    series = series[series.notna()].copy()

    if series.shape[0] > 0:

        if pretty:
            return type(series.head(1).values[0]).__name__

        else:
            return type(series.head(1).values[0])

    else:
        return None


def get_dtypes(df, pretty=False):

    type_dict_list = [{"column_name": col, 'data_type': get_dtype(df[col], pretty)} for col in df.columns]

    return pd.DataFrame(type_dict_list)


def get_summary_stats(df):

    return df.describe()


def get_null_columns(df):

    null_df = df.isna().sum().reset_index().rename(columns={'index': 'column_name', 0: 'count_of_nulls'})

    return null_df


def get_df_casted_to_supported_types(df):

    # pandasql uses SQLite syntax
    # SQLite does not support arrays
    # In order to not have the query crash whenever an array is in the query result we need to cast arrays into strings
    # We also cast dictionaries and dates into str to be on the safe side

    not_supported_types = [numpy.ndarray, list, tuple, set, dict, numpy.datetime64]

    bad_columns = [col for col in df.columns if get_dtype(df[col]) in not_supported_types]

    for col in bad_columns:
        df[col] = df[col].astype(str)

    return df


def filter_df_by_query(df, query):

    file = get_df_casted_to_supported_types(df)

    # if the column names contain spaces, convert them to underscores so you can still query them
    file.columns = [col.replace(' ', '_') for col in file.columns]

    try:
        result_df = psql.sqldf(query, {**locals(), **globals()})

    except PandaSQLException as e:

        error = e.__str__().split('\n')[0]

        click.echo(f"Ouch! Your SQL query failed: {error}")

        sys.exit(0)

    return result_df


def get_value_counts(df, column):

    result = pd.DataFrame(df[column].value_counts()).reset_index()
    result.rename(columns={'index': 'unique_value', column: f'count'}, inplace=True)

    return result


