import pprint
from urllib.error import HTTPError
import click
import os
import json
import webbrowser
from typing import Literal
from pydantic.v1 import ValidationError
from eas2cli.models import Tokens
from eas2cli.core import readtokens
from eas2cli.core import _tok2dict
from time import sleep
from urllib.request import urlopen
from urllib.parse import urlencode


@click.group()
def eas2cli():
    pass


@eas2cli.command()
@click.option('--tokenfile', default='~/.eidajwt',
              help='File where the tokens are stored. Usually "~/.eidajwt".')
def logout(tokenfile: str = '~/.eidajwt'):
    """Remove the file with tokens

    The user will need to manually log in again to be able to get an access token"""
    if click.confirm('If you do this you will need to manually login again to get an access token.\nDo you really want to logout?'):
        try:
            os.remove(os.path.expanduser(tokenfile))
            click.echo('You have been successfully logged out')
        except FileNotFoundError:
            click.echo(message=click.style('Error: ', fg='red'), err=True, nl=False)
            click.echo(message='No tokens have been found.', err=True)
        except Exception as e:
            click.echo(message=click.style('Error: ', fg='red'), err=True, nl=False)
            click.echo(message=str(e), err=True)
    return


@eas2cli.command()
@click.option('--tokenfile', default='~/.eidajwt',
              help='File where the tokens are stored. Usually "~/.eidajwt".')
def login(tokenfile: str = '~/.eidajwt'):
    """Open a webpage to allow the user to login

    After successful login the user can enter the code returned to get the tokens"""
    url = 'https://geofon.gfz.de/eas2/device/code'
    urltoken = 'https://geofon.gfz.de/eas2/token'
    postdata = {'client_id': 'eas2cli'}
    with urlopen(url, data=urlencode(postdata).encode()) as newdev:
        data = json.loads(newdev.read())
    click.echo('Enter user code "%s" at %s' % (data['user_code'], data['verification_uri']))
    webbrowser.open(data['verification_uri'])

    # Prepare request
    postdata = {'client_id': 'eas2cli',
                'device_code': data['device_code'],
                'grant_type': 'urn:ietf:params:oauth:grant-type:device_code'}

    # Sleep for some seconds
    sleep(data['interval'])
    iterations = range(data['interval'], data['expires_in'], data['interval'])
    with click.progressbar(iterations, label='Time remaining to get a token') as bar:
        for it in bar:
            # Sleep for some seconds
            sleep(data['interval'])
            # Try to get the token
            try:
                with urlopen(urltoken, data=urlencode(postdata).encode()) as fin:
                    rawtoken = fin.read().decode()
                    # Verify data
                    datatoken = json.loads(rawtoken)
                    Tokens(**datatoken)
                with open(os.path.expanduser(tokenfile), 'wt') as fout:
                    fout.write(rawtoken)
                break
            except Exception:
                pass
        else:
            click.echo(click.style('No token could be retrieved! Please, try again.', fg='red'))
            return
    click.echo('Token saved in default location!')
    return


@eas2cli.command()
@click.option('--endpoint', default='https://geofon.gfz.de/eas2/token',
              help='URL where a new token can obtained.')
@click.option('--tokenfile', default='~/.eidajwt',
              help='File where the tokens are stored. Usually "~/.eidajwt".')
def refresh(tokenfile: str = '~/.eidajwt', endpoint: str = 'https://geofon.gfz.de/eas2/token'):
    """Refresh the access and id tokens stored locally"""
    try:
        tokens = readtokens(tokenfile)
    except Exception as e:
        click.echo(message=click.style('Error: ', fg='red'), err=True, nl=False)
        click.echo(message=str(e) + ' Try to log in again.', err=True)
        return

    postdata = {
        'grant_type': 'refresh_token',
        'refresh_token': tokens.refresh_token
    }
    try:
        with urlopen(endpoint, data=urlencode(postdata).encode()) as newtok:
            newtokbytes = newtok.read().decode()
            try:
                newtokens = Tokens(**json.loads(newtokbytes))
            except ValidationError:
                click.echo(message=click.style('Error: ', fg='red'), err=True, nl=False)
                click.echo("The tokens received could not be properly read. Wrong format?\n%s" % newtokbytes, err=True)
                return
    except HTTPError as e:
        if e.status == 400:
            err = e.read().decode()
            errjson = json.loads(err)
            # print(errjson)
            click.echo(message=click.style('Error: ', fg='red'), err=True, nl=False)
            click.echo(pprint.pformat(errjson), err=True)
            return

    at = _tok2dict(tokens.access_token)
    # Check that the new access token is valid
    try:
        atnew = _tok2dict(newtokens.access_token, validate=True)
    except Exception as e:
        click.echo(message=click.style('Error: ', fg='red'), err=True, nl=False)
        click.echo("There was an error validating the new access token. %s" % (str(e),))
        return

    # Check that the new ID token is valid and better than the one we already have
    try:
        idnew = _tok2dict(newtokens.id_token, validate=True)
    except Exception as e:
        click.echo(message=click.style('Error: ', fg='red'), err=True, nl=False)
        click.echo("There was an error validating the new ID token. %s" % (str(e),))
        return

    with open(os.path.expanduser(tokenfile), 'wt') as fout:
        fout.write(newtokbytes)


@eas2cli.command()
@click.option('--tokenfile', default='~/.eidajwt',
              help='File where the tokens are stored. Usually "~/.eidajwt".')
@click.option('--validate/--no-validate', default=False,
              help='Declare if tokens must be validated when showing them')
@click.argument('token', default='all', )
def show(token: Literal['access', 'refresh', 'id', 'all'], tokenfile: str = '~/.eidajwt', validate: bool = False):
    """Show the tokens stored locally"""
    try:
        nt = readtokens(tokenfile)
    except Exception as e:
        click.echo(message=click.style('Error: ', fg='red'), err=True, nl=False)
        click.echo(message=str(e) + ' Try to log in again.', err=True)
        return

    if token == 'all':
        for att, value in nt:
            if att == 'refresh_token' or not att.endswith('_token'):
                click.echo('%s: %s' % (click.style(att.removesuffix('_token'), fg='green'), value))
            else:
                click.echo('%s: %s' % (click.style(att.removesuffix('_token'), fg='green'),
                                       _tok2dict(value, validate=validate)))
    elif token == 'refresh':
        click.echo('%s: %s' % (click.style('refresh', fg='green'), nt.refresh_token))
    else:
        click.echo('%s: %s' % (click.style(token, fg='green'),
                               _tok2dict(getattr(nt, token + '_token'), validate=validate)))


if __name__ == '__main__':
    eas2cli()
