"""
Trading-Technical-Indicators (tti) python library

File name: _accumulation_distribution_line.py
    Implements the Accumulation Distribution Line technical indicator.
"""

import pandas as pd

from ._technical_indicator import TechnicalIndicator
from ..utils.constants import TRADE_SIGNALS


class AccumulationDistributionLine(TechnicalIndicator):
    """
    Accumulation Distribution Line Technical Indicator class implementation.

    Args:
        input_data (pandas.DataFrame): The input data. Required input columns
            are ``high``, ``low``, ``close``, ``volume``. The index is of type
            ``pandas.DatetimeIndex``.

        fill_missing_values (bool, default=True): If set to True, missing
            values in the input data are being filled.

    Attributes:
        _input_data (pandas.DataFrame): The ``input_data`` after preprocessing.

        _ti_data (pandas.DataFrame): The calculated indicator. Index is of type
            ``pandas.DatetimeIndex``. It contains one column, the ``adl``.

        _properties (dict): Indicator properties.

        _calling_instance (str): The name of the class.

    Raises:
        WrongTypeForInputParameter: Input argument has wrong type.
        WrongValueForInputParameter: Unsupported value for input argument.
        NotEnoughInputData: Not enough data for calculating the indicator.
        TypeError: Type error occurred when validating the ``input_data``.
        ValueError: Value error occurred when validating the ``input_data``.
    """
    def __init__(self, input_data, fill_missing_values=True):

        # Control is passing to the parent class
        super().__init__(calling_instance=self.__class__.__name__,
                         input_data=input_data,
                         fill_missing_values=fill_missing_values)

    def _calculateTi(self):
        """
        Calculates the technical indicator for the given input data.

        Returns:
            pandas.DataFrame: The calculated indicator. Index is of type
            ``pandas.DatetimeIndex``. It contains one column, the ``adl``.
        """

        adl = pd.DataFrame(index=self._input_data.index, columns=['adl'],
                           data=0, dtype='int64')

        adl['adl'] = self._input_data['volume'] * (
                (self._input_data['close'] - self._input_data['low']) -
                (self._input_data['high'] - self._input_data['close'])
        ) / (self._input_data['high'] - self._input_data['low'])

        for i in range(1, len(adl.index)):
            adl['adl'].iat[i] += adl['adl'].iat[i - 1]

        return adl.astype(dtype='int64', errors='ignore')

    def getTiSignal(self):
        """
        Calculates and returns the trading signal for the calculated technical
        indicator.

        Returns:
            {('hold', 0), ('buy', -1), ('sell', 1)}: The calculated trading
            signal.
        """

        # Trading signals Divergences calculated in 2-days period

        # Not enough data for calculating trading signal
        if len(self._ti_data.index) < 3:
            return TRADE_SIGNALS['hold']

        # Warning for a upward breakout
        if self._ti_data['adl'].iat[-3] > self._ti_data['adl'].iat[-2] > \
                self._ti_data['adl'].iat[-1]:
            return TRADE_SIGNALS['buy']

        # Warning for a downward breakout
        elif self._ti_data['adl'].iat[-3] < self._ti_data['adl'].iat[-2] < \
                self._ti_data['adl'].iat[-1]:
            return TRADE_SIGNALS['sell']

        else:
            return TRADE_SIGNALS['hold']
