"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToKinesisStreams = void 0;
// Imports
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
/**
 * @summary The ApiGatewayToKinesisStreams class.
 */
class ApiGatewayToKinesisStreams extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the ApiGatewayToKinesisStreams class.
     * @since 1.62.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Setup the Kinesis stream
        this.kinesisStream = defaults.buildKinesisStream(scope, {
            existingStreamObj: props.existingStreamObj,
            kinesisStreamProps: props.kinesisStreamProps
        });
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole, this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, props.apiGatewayProps);
        // Setup the API Gateway role
        this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
        });
        this.kinesisStream.grantWrite(this.apiGatewayRole);
        // Setup API Gateway methods
        const requestValidator = this.apiGateway.addRequestValidator('request-validator', {
            requestValidatorName: 'request-body-validator',
            validateRequestBody: true
        });
        // PutRecord
        const putRecordResource = this.apiGateway.root.addResource('record');
        defaults.addProxyMethodToApiResource({
            service: 'kinesis',
            action: 'PutRecord',
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'POST',
            apiResource: putRecordResource,
            requestTemplate: this.getPutRecordTemplate(props.putRecordRequestTemplate),
            contentType: "'x-amz-json-1.1'",
            requestValidator,
            requestModel: { 'application/json': this.getPutRecordModel(props.putRecordRequestModel) }
        });
        // PutRecords
        const putRecordsResource = this.apiGateway.root.addResource('records');
        defaults.addProxyMethodToApiResource({
            service: 'kinesis',
            action: 'PutRecords',
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'POST',
            apiResource: putRecordsResource,
            requestTemplate: this.getPutRecordsTemplate(props.putRecordsRequestTemplate),
            contentType: "'x-amz-json-1.1'",
            requestValidator,
            requestModel: { 'application/json': this.getPutRecordsModel(props.putRecordsRequestModel) }
        });
    }
    getPutRecordTemplate(putRecordTemplate) {
        if (putRecordTemplate !== undefined) {
            return putRecordTemplate.replace("${StreamName}", this.kinesisStream.streamName);
        }
        return `{ "StreamName": "${this.kinesisStream.streamName}", "Data": "$util.base64Encode($input.json('$.data'))", "PartitionKey": "$input.path('$.partitionKey')" }`;
    }
    getPutRecordModel(putRecordModel) {
        let modelProps;
        if (putRecordModel !== undefined) {
            modelProps = putRecordModel;
        }
        else {
            modelProps = {
                contentType: 'application/json',
                modelName: 'PutRecordModel',
                description: 'PutRecord proxy single-record payload',
                schema: {
                    schema: api.JsonSchemaVersion.DRAFT4,
                    title: 'PutRecord proxy single-record payload',
                    type: api.JsonSchemaType.OBJECT,
                    required: ['data', 'partitionKey'],
                    properties: {
                        data: { type: api.JsonSchemaType.STRING },
                        partitionKey: { type: api.JsonSchemaType.STRING }
                    }
                }
            };
        }
        return this.apiGateway.addModel('PutRecordModel', modelProps);
    }
    getPutRecordsTemplate(putRecordsTemplate) {
        if (putRecordsTemplate !== undefined) {
            return putRecordsTemplate.replace("${StreamName}", this.kinesisStream.streamName);
        }
        return `{ "StreamName": "${this.kinesisStream.streamName}", "Records": [ #foreach($elem in $input.path('$.records')) { "Data": "$util.base64Encode($elem.data)", "PartitionKey": "$elem.partitionKey"}#if($foreach.hasNext),#end #end ] }`;
    }
    getPutRecordsModel(putRecordsModel) {
        let modelProps;
        if (putRecordsModel !== undefined) {
            modelProps = putRecordsModel;
        }
        else {
            modelProps = {
                contentType: 'application/json',
                modelName: 'PutRecordsModel',
                description: 'PutRecords proxy payload data',
                schema: {
                    schema: api.JsonSchemaVersion.DRAFT4,
                    title: 'PutRecords proxy payload data',
                    type: api.JsonSchemaType.OBJECT,
                    required: ['records'],
                    properties: {
                        records: {
                            type: api.JsonSchemaType.ARRAY,
                            items: {
                                type: api.JsonSchemaType.OBJECT,
                                required: ['data', 'partitionKey'],
                                properties: {
                                    data: { type: api.JsonSchemaType.STRING },
                                    partitionKey: { type: api.JsonSchemaType.STRING }
                                }
                            }
                        }
                    }
                }
            };
        }
        return this.apiGateway.addModel('PutRecordsModel', modelProps);
    }
}
exports.ApiGatewayToKinesisStreams = ApiGatewayToKinesisStreams;
//# sourceMappingURL=data:application/json;base64,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