# nmrtoolbox

## Introduction
This is a simple utility that provides modules for working with NMRPipe peak tables and performing a receiver operator characteristic (ROC) analysis to quantify the quality of the "recovered" peaks relative to a control set of "injected" peaks.  The modules in this package are as follows:
- `nmrtoolbox.peak`: classes for reading in peak tables (currently supports both synthetically generated and recovered peak tables from NMRPipe)
- `nmrtoolbox.roc`: perform receiver operator characteristic (ROC) analysis of a recovered peak table relative to an injected peak table
- `nmrtoolbox.mask`: define regions of a spectrum that contain signal or are empty
- `nmrtoolbox.util`: various supporting utilities used by other modules

## Applications

### Example #1 - Formal Workflow
The tools in this package are utilized by the [NUScon](https://nuscon.org/home) software package.  You can access NUScon on the NMRbox platform (free for academic, government, and non-profit users).  Running `nuscon -h` will provide instructions on how to run the NUScon evaluation workflow, which directly utilizes the tools presented here in the `nmrtoolbox` package.

### Example #2 - Kick the Tires

```commandline
from nmrtoolbox.peak import PeakTablePipeRec, PeakTablePipeInj
from nmrtoolbox.roc import roc

# read a peak table generated by NMRPipe peak picker
pt_rec = PeakTablePipeRec(<file-recovered.tab>)
# remove peaks not in the specified NMRPipe classification
pt.reduce(cluster_type=1)
# remove peaks whose linewidths are outliers
pt.reduce(chi2prob=.75)

# read a peak table generated by NMRPipe>genSimTab.tcl 
pt_inj = PeakTablePipeInj(<file-injected.tab>)

# perform ROC analysis
my_roc = roc(
    recovered_table=pt_rec,
    injected_table=pt_inj,
)

# show and plot results
my_roc.print_stats()
my_roc.plot_roc()
my_roc.plot_peaks()
my_roc.plot_outliers()
```

The `nmrtoolbox.peak.reduce()` function supports the following filter criteria:
- number
- height
- abs_height
- roi
- index
- cluster_type
- chi2prob
- mask

Note: Filtering by `mask` requires the external use of NMRPipe to generate a mask file indicating where the spectrum is empty.  This binary data is converted by [Connjur Spectrum Translator](https://nmrbox.nmrhub.org/software/spectrum-translator) into a "tabular" format file (i.e. plain text) which is then read in by `nmrtoolbox.mask`.


## Changelog

v7
- addition of `roc` module
- addition of `mask` module

v6
- rename package as `nmrtoolbox`
- use subclasses to handle NMRPipe peak tables coming from genSimTab or from the peak picker

v5
- new Axis class for containing metadata from peak table header
