"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const appscaling = require("@aws-cdk/aws-applicationautoscaling");
const aws_cloudformation_1 = require("@aws-cdk/aws-cloudformation");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const dynamodb_generated_1 = require("./dynamodb.generated");
const replica_provider_1 = require("./replica-provider");
const scalable_table_attribute_1 = require("./scalable-table-attribute");
const HASH_KEY_TYPE = 'HASH';
const RANGE_KEY_TYPE = 'RANGE';
// https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html#limits-secondary-indexes
const MAX_LOCAL_SECONDARY_INDEX_COUNT = 5;
const READ_DATA_ACTIONS = [
    'dynamodb:BatchGetItem',
    'dynamodb:GetRecords',
    'dynamodb:GetShardIterator',
    'dynamodb:Query',
    'dynamodb:GetItem',
    'dynamodb:Scan'
];
const READ_STREAM_DATA_ACTIONS = [
    'dynamodb:DescribeStream',
    'dynamodb:GetRecords',
    'dynamodb:GetShardIterator',
];
const WRITE_DATA_ACTIONS = [
    'dynamodb:BatchWriteItem',
    'dynamodb:PutItem',
    'dynamodb:UpdateItem',
    'dynamodb:DeleteItem'
];
class TableBase extends core_1.Resource {
    /**
     * Adds an IAM policy statement associated with this table to an IAM
     * principal's policy.
     * @param grantee The principal (no-op if undefined)
     * @param actions The set of actions to allow (i.e. "dynamodb:PutItem", "dynamodb:GetItem", ...)
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [
                this.tableArn,
                core_1.Lazy.stringValue({ produce: () => this.hasIndex ? `${this.tableArn}/index/*` : core_1.Aws.NO_VALUE })
            ],
            scope: this,
        });
    }
    /**
     * Adds an IAM policy statement associated with this table's stream to an
     * IAM principal's policy.
     * @param grantee The principal (no-op if undefined)
     * @param actions The set of actions to allow (i.e. "dynamodb:DescribeStream", "dynamodb:GetRecords", ...)
     */
    grantStream(grantee, ...actions) {
        if (!this.tableStreamArn) {
            throw new Error(`DynamoDB Streams must be enabled on the table ${this.node.path}`);
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.tableStreamArn],
            scope: this,
        });
    }
    /**
     * Permits an IAM principal all data read operations from this table:
     * BatchGetItem, GetRecords, GetShardIterator, Query, GetItem, Scan.
     * @param grantee The principal to grant access to
     */
    grantReadData(grantee) {
        return this.grant(grantee, ...READ_DATA_ACTIONS);
    }
    /**
     * Permits an IAM Principal to list streams attached to current dynamodb table.
     *
     * @param grantee The principal (no-op if undefined)
     */
    grantTableListStreams(grantee) {
        if (!this.tableStreamArn) {
            throw new Error(`DynamoDB Streams must be enabled on the table ${this.node.path}`);
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['dynamodb:ListStreams'],
            resourceArns: [
                core_1.Lazy.stringValue({ produce: () => `${this.tableArn}/stream/*` })
            ],
        });
    }
    /**
     * Permits an IAM principal all stream data read operations for this
     * table's stream:
     * DescribeStream, GetRecords, GetShardIterator, ListStreams.
     * @param grantee The principal to grant access to
     */
    grantStreamRead(grantee) {
        this.grantTableListStreams(grantee);
        return this.grantStream(grantee, ...READ_STREAM_DATA_ACTIONS);
    }
    /**
     * Permits an IAM principal all data write operations to this table:
     * BatchWriteItem, PutItem, UpdateItem, DeleteItem.
     * @param grantee The principal to grant access to
     */
    grantWriteData(grantee) {
        return this.grant(grantee, ...WRITE_DATA_ACTIONS);
    }
    /**
     * Permits an IAM principal to all data read/write operations to this table.
     * BatchGetItem, GetRecords, GetShardIterator, Query, GetItem, Scan,
     * BatchWriteItem, PutItem, UpdateItem, DeleteItem
     * @param grantee The principal to grant access to
     */
    grantReadWriteData(grantee) {
        return this.grant(grantee, ...READ_DATA_ACTIONS, ...WRITE_DATA_ACTIONS);
    }
    /**
     * Permits all DynamoDB operations ("dynamodb:*") to an IAM principal.
     * @param grantee The principal to grant access to
     */
    grantFullAccess(grantee) {
        return this.grant(grantee, 'dynamodb:*');
    }
    /**
     * Return the given named metric for this Table
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/DynamoDB',
            metricName,
            dimensions: {
                TableName: this.tableName,
            },
            ...props
        });
    }
    /**
     * Metric for the consumed read capacity units this table
     *
     * @default sum over a minute
     */
    metricConsumedReadCapacityUnits(props) {
        return this.metric('ConsumedReadCapacityUnits', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the consumed write capacity units this table
     *
     * @default sum over a minute
     */
    metricConsumedWriteCapacityUnits(props) {
        return this.metric('ConsumedWriteCapacityUnits', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the system errors this table
     *
     * @default sum over a minute
     */
    metricSystemErrors(props) {
        return this.metric('SystemErrors', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the user errors this table
     *
     * @default sum over a minute
     */
    metricUserErrors(props) {
        return this.metric('UserErrors', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the conditional check failed requests this table
     *
     * @default sum over a minute
     */
    metricConditionalCheckFailedRequests(props) {
        return this.metric('ConditionalCheckFailedRequests', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the successful request latency this table
     *
     * @default avg over a minute
     */
    metricSuccessfulRequestLatency(props) {
        return this.metric('SuccessfulRequestLatency', { statistic: 'avg', ...props });
    }
}
/**
 * Provides a DynamoDB table.
 */
class Table extends TableBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.tableName,
        });
        this.keySchema = new Array();
        this.attributeDefinitions = new Array();
        this.globalSecondaryIndexes = new Array();
        this.localSecondaryIndexes = new Array();
        this.secondaryIndexNames = new Set();
        this.nonKeyAttributes = new Set();
        this.tableScaling = {};
        this.indexScaling = new Map();
        this.billingMode = props.billingMode || BillingMode.PROVISIONED;
        this.validateProvisioning(props);
        let streamSpecification;
        if (props.replicationRegions) {
            if (props.stream && props.stream !== StreamViewType.NEW_AND_OLD_IMAGES) {
                throw new Error('`stream` must be set to `NEW_AND_OLD_IMAGES` when specifying `replicationRegions`');
            }
            streamSpecification = { streamViewType: StreamViewType.NEW_AND_OLD_IMAGES };
            if (props.billingMode && props.billingMode !== BillingMode.PAY_PER_REQUEST) {
                throw new Error('The `PAY_PER_REQUEST` billing mode must be used when specifying `replicationRegions`');
            }
            this.billingMode = BillingMode.PAY_PER_REQUEST;
        }
        else if (props.stream) {
            streamSpecification = { streamViewType: props.stream };
        }
        this.table = new dynamodb_generated_1.CfnTable(this, 'Resource', {
            tableName: this.physicalName,
            keySchema: this.keySchema,
            attributeDefinitions: this.attributeDefinitions,
            globalSecondaryIndexes: core_1.Lazy.anyValue({ produce: () => this.globalSecondaryIndexes }, { omitEmptyArray: true }),
            localSecondaryIndexes: core_1.Lazy.anyValue({ produce: () => this.localSecondaryIndexes }, { omitEmptyArray: true }),
            pointInTimeRecoverySpecification: props.pointInTimeRecovery ? { pointInTimeRecoveryEnabled: props.pointInTimeRecovery } : undefined,
            billingMode: this.billingMode === BillingMode.PAY_PER_REQUEST ? this.billingMode : undefined,
            provisionedThroughput: this.billingMode === BillingMode.PAY_PER_REQUEST ? undefined : {
                readCapacityUnits: props.readCapacity || 5,
                writeCapacityUnits: props.writeCapacity || 5
            },
            sseSpecification: props.serverSideEncryption ? { sseEnabled: props.serverSideEncryption } : undefined,
            streamSpecification,
            timeToLiveSpecification: props.timeToLiveAttribute ? { attributeName: props.timeToLiveAttribute, enabled: true } : undefined
        });
        this.table.applyRemovalPolicy(props.removalPolicy);
        if (props.tableName) {
            this.node.addMetadata('aws:cdk:hasPhysicalName', props.tableName);
        }
        this.tableArn = this.getResourceArnAttribute(this.table.attrArn, {
            service: 'dynamodb',
            resource: 'table',
            resourceName: this.physicalName,
        });
        this.tableName = this.getResourceNameAttribute(this.table.ref);
        this.tableStreamArn = streamSpecification ? this.table.attrStreamArn : undefined;
        this.scalingRole = this.makeScalingRole();
        this.addKey(props.partitionKey, HASH_KEY_TYPE);
        this.tablePartitionKey = props.partitionKey;
        if (props.sortKey) {
            this.addKey(props.sortKey, RANGE_KEY_TYPE);
            this.tableSortKey = props.sortKey;
        }
        if (props.replicationRegions) {
            this.createReplicaTables(props.replicationRegions);
        }
    }
    /**
     * Permits an IAM Principal to list all DynamoDB Streams.
     * @deprecated Use {@link #grantTableListStreams} for more granular permission
     * @param grantee The principal (no-op if undefined)
     */
    static grantListStreams(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['dynamodb:ListStreams'],
            resourceArns: ['*'],
        });
    }
    /**
     * Creates a Table construct that represents an external table via table name.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param tableName The table's name.
     */
    static fromTableName(scope, id, tableName) {
        return Table.fromTableAttributes(scope, id, { tableName });
    }
    /**
     * Creates a Table construct that represents an external table via table arn.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param tableArn The table's ARN.
     */
    static fromTableArn(scope, id, tableArn) {
        return Table.fromTableAttributes(scope, id, { tableArn });
    }
    /**
     * Creates a Table construct that represents an external table.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `TableAttributes` object.
     */
    static fromTableAttributes(scope, id, attrs) {
        class Import extends TableBase {
            constructor(_tableArn, tableName, tableStreamArn) {
                super(scope, id);
                this.tableArn = _tableArn;
                this.tableName = tableName;
                this.tableStreamArn = tableStreamArn;
            }
            get hasIndex() {
                return false;
            }
        }
        let name;
        let arn;
        const stack = core_1.Stack.of(scope);
        if (!attrs.tableName) {
            if (!attrs.tableArn) {
                throw new Error('One of tableName or tableArn is required!');
            }
            arn = attrs.tableArn;
            const maybeTableName = stack.parseArn(attrs.tableArn).resourceName;
            if (!maybeTableName) {
                throw new Error('ARN for DynamoDB table must be in the form: ...');
            }
            name = maybeTableName;
        }
        else {
            if (attrs.tableArn) {
                throw new Error('Only one of tableArn or tableName can be provided');
            }
            name = attrs.tableName;
            arn = stack.formatArn({
                service: 'dynamodb',
                resource: 'table',
                resourceName: attrs.tableName,
            });
        }
        return new Import(arn, name, attrs.tableStreamArn);
    }
    /**
     * Add a global secondary index of table.
     *
     * @param props the property of global secondary index
     */
    addGlobalSecondaryIndex(props) {
        this.validateProvisioning(props);
        this.validateIndexName(props.indexName);
        // build key schema and projection for index
        const gsiKeySchema = this.buildIndexKeySchema(props.partitionKey, props.sortKey);
        const gsiProjection = this.buildIndexProjection(props);
        this.secondaryIndexNames.add(props.indexName);
        this.globalSecondaryIndexes.push({
            indexName: props.indexName,
            keySchema: gsiKeySchema,
            projection: gsiProjection,
            provisionedThroughput: this.billingMode === BillingMode.PAY_PER_REQUEST ? undefined : {
                readCapacityUnits: props.readCapacity || 5,
                writeCapacityUnits: props.writeCapacity || 5
            }
        });
        this.indexScaling.set(props.indexName, {});
    }
    /**
     * Add a local secondary index of table.
     *
     * @param props the property of local secondary index
     */
    addLocalSecondaryIndex(props) {
        // https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html#limits-secondary-indexes
        if (this.localSecondaryIndexes.length >= MAX_LOCAL_SECONDARY_INDEX_COUNT) {
            throw new RangeError(`a maximum number of local secondary index per table is ${MAX_LOCAL_SECONDARY_INDEX_COUNT}`);
        }
        this.validateIndexName(props.indexName);
        // build key schema and projection for index
        const lsiKeySchema = this.buildIndexKeySchema(this.tablePartitionKey, props.sortKey);
        const lsiProjection = this.buildIndexProjection(props);
        this.secondaryIndexNames.add(props.indexName);
        this.localSecondaryIndexes.push({
            indexName: props.indexName,
            keySchema: lsiKeySchema,
            projection: lsiProjection
        });
    }
    /**
     * Enable read capacity scaling for this table
     *
     * @returns An object to configure additional AutoScaling settings
     */
    autoScaleReadCapacity(props) {
        if (this.tableScaling.scalableReadAttribute) {
            throw new Error('Read AutoScaling already enabled for this table');
        }
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        return this.tableScaling.scalableReadAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, 'ReadScaling', {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}`,
            dimension: 'dynamodb:table:ReadCapacityUnits',
            role: this.scalingRole,
            ...props
        });
    }
    /**
     * Enable write capacity scaling for this table
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     */
    autoScaleWriteCapacity(props) {
        if (this.tableScaling.scalableWriteAttribute) {
            throw new Error('Write AutoScaling already enabled for this table');
        }
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        return this.tableScaling.scalableWriteAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, 'WriteScaling', {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}`,
            dimension: 'dynamodb:table:WriteCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
    }
    /**
     * Enable read capacity scaling for the given GSI
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     */
    autoScaleGlobalSecondaryIndexReadCapacity(indexName, props) {
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        const attributePair = this.indexScaling.get(indexName);
        if (!attributePair) {
            throw new Error(`No global secondary index with name ${indexName}`);
        }
        if (attributePair.scalableReadAttribute) {
            throw new Error('Read AutoScaling already enabled for this index');
        }
        return attributePair.scalableReadAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, `${indexName}ReadScaling`, {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}/index/${indexName}`,
            dimension: 'dynamodb:index:ReadCapacityUnits',
            role: this.scalingRole,
            ...props
        });
    }
    /**
     * Enable write capacity scaling for the given GSI
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     */
    autoScaleGlobalSecondaryIndexWriteCapacity(indexName, props) {
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        const attributePair = this.indexScaling.get(indexName);
        if (!attributePair) {
            throw new Error(`No global secondary index with name ${indexName}`);
        }
        if (attributePair.scalableWriteAttribute) {
            throw new Error('Write AutoScaling already enabled for this index');
        }
        return attributePair.scalableWriteAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, `${indexName}WriteScaling`, {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}/index/${indexName}`,
            dimension: 'dynamodb:index:WriteCapacityUnits',
            role: this.scalingRole,
            ...props
        });
    }
    /**
     * Validate the table construct.
     *
     * @returns an array of validation error message
     */
    validate() {
        const errors = new Array();
        if (!this.tablePartitionKey) {
            errors.push('a partition key must be specified');
        }
        if (this.localSecondaryIndexes.length > 0 && !this.tableSortKey) {
            errors.push('a sort key of the table must be specified to add local secondary indexes');
        }
        return errors;
    }
    /**
     * Validate read and write capacity are not specified for on-demand tables (billing mode PAY_PER_REQUEST).
     *
     * @param props read and write capacity properties
     */
    validateProvisioning(props) {
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            if (props.readCapacity !== undefined || props.writeCapacity !== undefined) {
                throw new Error('you cannot provision read and write capacity for a table with PAY_PER_REQUEST billing mode');
            }
        }
    }
    /**
     * Validate index name to check if a duplicate name already exists.
     *
     * @param indexName a name of global or local secondary index
     */
    validateIndexName(indexName) {
        if (this.secondaryIndexNames.has(indexName)) {
            // a duplicate index name causes validation exception, status code 400, while trying to create CFN stack
            throw new Error(`a duplicate index name, ${indexName}, is not allowed`);
        }
        this.secondaryIndexNames.add(indexName);
    }
    /**
     * Validate non-key attributes by checking limits within secondary index, which may vary in future.
     *
     * @param nonKeyAttributes a list of non-key attribute names
     */
    validateNonKeyAttributes(nonKeyAttributes) {
        if (this.nonKeyAttributes.size + nonKeyAttributes.length > 20) {
            // https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html#limits-secondary-indexes
            throw new RangeError('a maximum number of nonKeyAttributes across all of secondary indexes is 20');
        }
        // store all non-key attributes
        nonKeyAttributes.forEach(att => this.nonKeyAttributes.add(att));
    }
    buildIndexKeySchema(partitionKey, sortKey) {
        this.registerAttribute(partitionKey);
        const indexKeySchema = [
            { attributeName: partitionKey.name, keyType: HASH_KEY_TYPE }
        ];
        if (sortKey) {
            this.registerAttribute(sortKey);
            indexKeySchema.push({ attributeName: sortKey.name, keyType: RANGE_KEY_TYPE });
        }
        return indexKeySchema;
    }
    buildIndexProjection(props) {
        if (props.projectionType === ProjectionType.INCLUDE && !props.nonKeyAttributes) {
            // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-dynamodb-projectionobject.html
            throw new Error(`non-key attributes should be specified when using ${ProjectionType.INCLUDE} projection type`);
        }
        if (props.projectionType !== ProjectionType.INCLUDE && props.nonKeyAttributes) {
            // this combination causes validation exception, status code 400, while trying to create CFN stack
            throw new Error(`non-key attributes should not be specified when not using ${ProjectionType.INCLUDE} projection type`);
        }
        if (props.nonKeyAttributes) {
            this.validateNonKeyAttributes(props.nonKeyAttributes);
        }
        return {
            projectionType: props.projectionType ? props.projectionType : ProjectionType.ALL,
            nonKeyAttributes: props.nonKeyAttributes ? props.nonKeyAttributes : undefined
        };
    }
    findKey(keyType) {
        return this.keySchema.find(prop => prop.keyType === keyType);
    }
    addKey(attribute, keyType) {
        const existingProp = this.findKey(keyType);
        if (existingProp) {
            throw new Error(`Unable to set ${attribute.name} as a ${keyType} key, because ${existingProp.attributeName} is a ${keyType} key`);
        }
        this.registerAttribute(attribute);
        this.keySchema.push({
            attributeName: attribute.name,
            keyType
        });
        return this;
    }
    /**
     * Register the key attribute of table or secondary index to assemble attribute definitions of TableResourceProps.
     *
     * @param attribute the key attribute of table or secondary index
     */
    registerAttribute(attribute) {
        const { name, type } = attribute;
        const existingDef = this.attributeDefinitions.find(def => def.attributeName === name);
        if (existingDef && existingDef.attributeType !== type) {
            throw new Error(`Unable to specify ${name} as ${type} because it was already defined as ${existingDef.attributeType}`);
        }
        if (!existingDef) {
            this.attributeDefinitions.push({
                attributeName: name,
                attributeType: type
            });
        }
    }
    /**
     * Return the role that will be used for AutoScaling
     */
    makeScalingRole() {
        // Use a Service Linked Role.
        // https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-service-linked-roles.html
        return iam.Role.fromRoleArn(this, 'ScalingRole', core_1.Stack.of(this).formatArn({
            service: 'iam',
            region: '',
            resource: 'role/aws-service-role/dynamodb.application-autoscaling.amazonaws.com',
            resourceName: 'AWSServiceRoleForApplicationAutoScaling_DynamoDBTable'
        }));
    }
    /**
     * Creates replica tables
     *
     * @param regions regions where to create tables
     */
    createReplicaTables(regions) {
        const stack = core_1.Stack.of(this);
        if (!core_1.Token.isUnresolved(stack.region) && regions.includes(stack.region)) {
            throw new Error('`replicationRegions` cannot include the region where this stack is deployed.');
        }
        const provider = replica_provider_1.ReplicaProvider.getOrCreate(this);
        // Documentation at https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/V2gt_IAM.html
        // is currently incorrect. AWS Support recommends `dynamodb:*` in both source and destination regions
        // Permissions in the source region
        this.grant(provider.onEventHandler, 'dynamodb:*');
        this.grant(provider.isCompleteHandler, 'dynamodb:DescribeTable');
        // Permissions in the destination regions
        provider.onEventHandler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['dynamodb:*'],
            resources: regions.map(region => stack.formatArn({
                region,
                service: 'dynamodb',
                resource: 'table',
                resourceName: this.tableName
            }))
        }));
        let previousRegion;
        for (const region of new Set(regions)) { // Remove duplicates
            // Use multiple custom resources because multiple create/delete
            // updates cannot be combined in a single API call.
            const currentRegion = new aws_cloudformation_1.CustomResource(this, `Replica${region}`, {
                provider: provider.provider,
                resourceType: 'Custom::DynamoDBReplica',
                properties: {
                    TableName: this.tableName,
                    Region: region,
                }
            });
            // Deploy time check to prevent from creating a replica in the region
            // where this stack is deployed. Only needed for environment agnostic
            // stacks.
            if (core_1.Token.isUnresolved(stack.region)) {
                const createReplica = new core_1.CfnCondition(this, `StackRegionNotEquals${region}`, {
                    expression: core_1.Fn.conditionNot(core_1.Fn.conditionEquals(region, core_1.Aws.REGION))
                });
                const cfnCustomResource = currentRegion.node.defaultChild;
                cfnCustomResource.cfnOptions.condition = createReplica;
            }
            // We need to create/delete regions sequentially because we cannot
            // have multiple table updates at the same time. The `isCompleteHandler`
            // of the provider waits until the replica is an ACTIVE state.
            if (previousRegion) {
                currentRegion.node.addDependency(previousRegion);
            }
            previousRegion = currentRegion;
        }
    }
    /**
     * Whether this table has indexes
     */
    get hasIndex() {
        return this.globalSecondaryIndexes.length + this.localSecondaryIndexes.length > 0;
    }
}
exports.Table = Table;
var AttributeType;
(function (AttributeType) {
    /** Up to 400KiB of binary data (which must be encoded as base64 before sending to DynamoDB) */
    AttributeType["BINARY"] = "B";
    /** Numeric values made of up to 38 digits (positive, negative or zero) */
    AttributeType["NUMBER"] = "N";
    /** Up to 400KiB of UTF-8 encoded text */
    AttributeType["STRING"] = "S";
})(AttributeType = exports.AttributeType || (exports.AttributeType = {}));
/**
 * DyanmoDB's Read/Write capacity modes.
 */
var BillingMode;
(function (BillingMode) {
    /**
     * Pay only for what you use. You don't configure Read/Write capacity units.
     */
    BillingMode["PAY_PER_REQUEST"] = "PAY_PER_REQUEST";
    /**
     * Explicitly specified Read/Write capacity units.
     */
    BillingMode["PROVISIONED"] = "PROVISIONED";
})(BillingMode = exports.BillingMode || (exports.BillingMode = {}));
var ProjectionType;
(function (ProjectionType) {
    /** Only the index and primary keys are projected into the index. */
    ProjectionType["KEYS_ONLY"] = "KEYS_ONLY";
    /** Only the specified table attributes are projected into the index. The list of projected attributes is in `nonKeyAttributes`. */
    ProjectionType["INCLUDE"] = "INCLUDE";
    /** All of the table attributes are projected into the index. */
    ProjectionType["ALL"] = "ALL";
})(ProjectionType = exports.ProjectionType || (exports.ProjectionType = {}));
/**
 * When an item in the table is modified, StreamViewType determines what information
 * is written to the stream for this table.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_StreamSpecification.html
 */
var StreamViewType;
(function (StreamViewType) {
    /** The entire item, as it appears after it was modified, is written to the stream. */
    StreamViewType["NEW_IMAGE"] = "NEW_IMAGE";
    /** The entire item, as it appeared before it was modified, is written to the stream. */
    StreamViewType["OLD_IMAGE"] = "OLD_IMAGE";
    /** Both the new and the old item images of the item are written to the stream. */
    StreamViewType["NEW_AND_OLD_IMAGES"] = "NEW_AND_OLD_IMAGES";
    /** Only the key attributes of the modified item are written to the stream. */
    StreamViewType["KEYS_ONLY"] = "KEYS_ONLY";
})(StreamViewType = exports.StreamViewType || (exports.StreamViewType = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGFibGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0YWJsZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLGtFQUFrRTtBQUNsRSxvRUFBZ0Y7QUFDaEYsc0RBQXNEO0FBQ3RELHdDQUF3QztBQUN4Qyx3Q0FBeUg7QUFDekgsNkRBQWdEO0FBQ2hELHlEQUFxRDtBQUVyRCx5RUFBb0U7QUFFcEUsTUFBTSxhQUFhLEdBQUcsTUFBTSxDQUFDO0FBQzdCLE1BQU0sY0FBYyxHQUFHLE9BQU8sQ0FBQztBQUUvQix3R0FBd0c7QUFDeEcsTUFBTSwrQkFBK0IsR0FBRyxDQUFDLENBQUM7QUFFMUMsTUFBTSxpQkFBaUIsR0FBRztJQUN4Qix1QkFBdUI7SUFDdkIscUJBQXFCO0lBQ3JCLDJCQUEyQjtJQUMzQixnQkFBZ0I7SUFDaEIsa0JBQWtCO0lBQ2xCLGVBQWU7Q0FDaEIsQ0FBQztBQUVGLE1BQU0sd0JBQXdCLEdBQUc7SUFDL0IseUJBQXlCO0lBQ3pCLHFCQUFxQjtJQUNyQiwyQkFBMkI7Q0FDNUIsQ0FBQztBQUVGLE1BQU0sa0JBQWtCLEdBQUc7SUFDekIseUJBQXlCO0lBQ3pCLGtCQUFrQjtJQUNsQixxQkFBcUI7SUFDckIscUJBQXFCO0NBQ3RCLENBQUM7QUEwU0YsTUFBZSxTQUFVLFNBQVEsZUFBUTtJQWdCdkM7Ozs7O09BS0c7SUFDSSxLQUFLLENBQUMsT0FBdUIsRUFBRSxHQUFHLE9BQWlCO1FBQ3hELE9BQU8sR0FBRyxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUM7WUFDOUIsT0FBTztZQUNQLE9BQU87WUFDUCxZQUFZLEVBQUU7Z0JBQ1osSUFBSSxDQUFDLFFBQVE7Z0JBQ2IsV0FBSSxDQUFDLFdBQVcsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxHQUFHLElBQUksQ0FBQyxRQUFRLFVBQVUsQ0FBQyxDQUFDLENBQUMsVUFBRyxDQUFDLFFBQVEsRUFBRSxDQUFDO2FBQy9GO1lBQ0QsS0FBSyxFQUFFLElBQUk7U0FDWixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSxXQUFXLENBQUMsT0FBdUIsRUFBRSxHQUFHLE9BQWlCO1FBQzlELElBQUksQ0FBQyxJQUFJLENBQUMsY0FBYyxFQUFFO1lBQ3hCLE1BQU0sSUFBSSxLQUFLLENBQUMsaURBQWlELElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQztTQUNwRjtRQUVELE9BQU8sR0FBRyxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUM7WUFDOUIsT0FBTztZQUNQLE9BQU87WUFDUCxZQUFZLEVBQUUsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDO1lBQ25DLEtBQUssRUFBRSxJQUFJO1NBQ1osQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxhQUFhLENBQUMsT0FBdUI7UUFDMUMsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRSxHQUFHLGlCQUFpQixDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxxQkFBcUIsQ0FBQyxPQUF1QjtRQUNsRCxJQUFJLENBQUMsSUFBSSxDQUFDLGNBQWMsRUFBRTtZQUN4QixNQUFNLElBQUksS0FBSyxDQUFDLGlEQUFpRCxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDLENBQUM7U0FDcEY7UUFDRCxPQUFPLEdBQUcsQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDO1lBQzlCLE9BQU87WUFDUCxPQUFPLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQztZQUNqQyxZQUFZLEVBQUU7Z0JBQ1osV0FBSSxDQUFDLFdBQVcsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxHQUFHLElBQUksQ0FBQyxRQUFRLFdBQVcsRUFBRSxDQUFDO2FBQ2pFO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksZUFBZSxDQUFDLE9BQXVCO1FBQzVDLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQyxXQUFXLENBQUMsT0FBTyxFQUFFLEdBQUcsd0JBQXdCLENBQUMsQ0FBQztJQUNoRSxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLGNBQWMsQ0FBQyxPQUF1QjtRQUMzQyxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxFQUFFLEdBQUcsa0JBQWtCLENBQUMsQ0FBQztJQUNwRCxDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSxrQkFBa0IsQ0FBQyxPQUF1QjtRQUMvQyxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxFQUFFLEdBQUcsaUJBQWlCLEVBQUUsR0FBRyxrQkFBa0IsQ0FBQyxDQUFDO0lBQzFFLENBQUM7SUFFRDs7O09BR0c7SUFDSSxlQUFlLENBQUMsT0FBdUI7UUFDNUMsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRSxZQUFZLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsVUFBa0IsRUFBRSxLQUFnQztRQUNoRSxPQUFPLElBQUksVUFBVSxDQUFDLE1BQU0sQ0FBQztZQUMzQixTQUFTLEVBQUUsY0FBYztZQUN6QixVQUFVO1lBQ1YsVUFBVSxFQUFFO2dCQUNWLFNBQVMsRUFBRSxJQUFJLENBQUMsU0FBUzthQUMxQjtZQUNELEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksK0JBQStCLENBQUMsS0FBZ0M7UUFDckUsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLDJCQUEyQixFQUFFLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxHQUFHLEtBQUssRUFBQyxDQUFDLENBQUM7SUFDakYsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxnQ0FBZ0MsQ0FBQyxLQUFnQztRQUN0RSxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsNEJBQTRCLEVBQUUsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLEdBQUcsS0FBSyxFQUFDLENBQUMsQ0FBQztJQUNsRixDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLGtCQUFrQixDQUFDLEtBQWdDO1FBQ3hELE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxjQUFjLEVBQUUsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLEdBQUcsS0FBSyxFQUFDLENBQUMsQ0FBQztJQUNwRSxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLGdCQUFnQixDQUFDLEtBQWdDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxZQUFZLEVBQUUsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLEdBQUcsS0FBSyxFQUFDLENBQUMsQ0FBQztJQUNsRSxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLG9DQUFvQyxDQUFDLEtBQWdDO1FBQzFFLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxnQ0FBZ0MsRUFBRSxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsR0FBRyxLQUFLLEVBQUMsQ0FBQyxDQUFDO0lBQ3RGLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksOEJBQThCLENBQUMsS0FBZ0M7UUFDcEUsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLDBCQUEwQixFQUFFLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxHQUFHLEtBQUssRUFBQyxDQUFDLENBQUM7SUFDaEYsQ0FBQztDQUdGO0FBRUQ7O0dBRUc7QUFDSCxNQUFhLEtBQU0sU0FBUSxTQUFTO0lBdUhsQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQWlCO1FBQ3pELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2YsWUFBWSxFQUFFLEtBQUssQ0FBQyxTQUFTO1NBQzlCLENBQUMsQ0FBQztRQW5CWSxjQUFTLEdBQUcsSUFBSSxLQUFLLEVBQThCLENBQUM7UUFDcEQseUJBQW9CLEdBQUcsSUFBSSxLQUFLLEVBQXdDLENBQUM7UUFDekUsMkJBQXNCLEdBQUcsSUFBSSxLQUFLLEVBQXlDLENBQUM7UUFDNUUsMEJBQXFCLEdBQUcsSUFBSSxLQUFLLEVBQXdDLENBQUM7UUFFMUUsd0JBQW1CLEdBQUcsSUFBSSxHQUFHLEVBQVUsQ0FBQztRQUN4QyxxQkFBZ0IsR0FBRyxJQUFJLEdBQUcsRUFBVSxDQUFDO1FBTXJDLGlCQUFZLEdBQTBCLEVBQUUsQ0FBQztRQUN6QyxpQkFBWSxHQUFHLElBQUksR0FBRyxFQUFpQyxDQUFDO1FBUXZFLElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsSUFBSSxXQUFXLENBQUMsV0FBVyxDQUFDO1FBQ2hFLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUVqQyxJQUFJLG1CQUFxRSxDQUFDO1FBQzFFLElBQUksS0FBSyxDQUFDLGtCQUFrQixFQUFFO1lBQzVCLElBQUksS0FBSyxDQUFDLE1BQU0sSUFBSSxLQUFLLENBQUMsTUFBTSxLQUFLLGNBQWMsQ0FBQyxrQkFBa0IsRUFBRTtnQkFDdEUsTUFBTSxJQUFJLEtBQUssQ0FBQyxtRkFBbUYsQ0FBQyxDQUFDO2FBQ3RHO1lBQ0QsbUJBQW1CLEdBQUcsRUFBRSxjQUFjLEVBQUUsY0FBYyxDQUFDLGtCQUFrQixFQUFFLENBQUM7WUFFNUUsSUFBSSxLQUFLLENBQUMsV0FBVyxJQUFJLEtBQUssQ0FBQyxXQUFXLEtBQUssV0FBVyxDQUFDLGVBQWUsRUFBRTtnQkFDMUUsTUFBTSxJQUFJLEtBQUssQ0FBQyxzRkFBc0YsQ0FBQyxDQUFDO2FBQ3pHO1lBQ0QsSUFBSSxDQUFDLFdBQVcsR0FBRyxXQUFXLENBQUMsZUFBZSxDQUFDO1NBQ2hEO2FBQU0sSUFBSSxLQUFLLENBQUMsTUFBTSxFQUFFO1lBQ3ZCLG1CQUFtQixHQUFHLEVBQUUsY0FBYyxFQUFHLEtBQUssQ0FBQyxNQUFNLEVBQUUsQ0FBQztTQUN6RDtRQUVELElBQUksQ0FBQyxLQUFLLEdBQUcsSUFBSSw2QkFBUSxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDMUMsU0FBUyxFQUFFLElBQUksQ0FBQyxZQUFZO1lBQzVCLFNBQVMsRUFBRSxJQUFJLENBQUMsU0FBUztZQUN6QixvQkFBb0IsRUFBRSxJQUFJLENBQUMsb0JBQW9CO1lBQy9DLHNCQUFzQixFQUFFLFdBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLHNCQUFzQixFQUFFLEVBQUUsRUFBRSxjQUFjLEVBQUUsSUFBSSxFQUFFLENBQUM7WUFDL0cscUJBQXFCLEVBQUUsV0FBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMscUJBQXFCLEVBQUUsRUFBRSxFQUFFLGNBQWMsRUFBRSxJQUFJLEVBQUUsQ0FBQztZQUM3RyxnQ0FBZ0MsRUFBRSxLQUFLLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxDQUFDLEVBQUUsMEJBQTBCLEVBQUUsS0FBSyxDQUFDLG1CQUFtQixFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDbkksV0FBVyxFQUFFLElBQUksQ0FBQyxXQUFXLEtBQUssV0FBVyxDQUFDLGVBQWUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUM1RixxQkFBcUIsRUFBRSxJQUFJLENBQUMsV0FBVyxLQUFLLFdBQVcsQ0FBQyxlQUFlLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUM7Z0JBQ3BGLGlCQUFpQixFQUFFLEtBQUssQ0FBQyxZQUFZLElBQUksQ0FBQztnQkFDMUMsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLGFBQWEsSUFBSSxDQUFDO2FBQzdDO1lBQ0QsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLG9CQUFvQixDQUFDLENBQUMsQ0FBQyxFQUFFLFVBQVUsRUFBRSxLQUFLLENBQUMsb0JBQW9CLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUNyRyxtQkFBbUI7WUFDbkIsdUJBQXVCLEVBQUUsS0FBSyxDQUFDLG1CQUFtQixDQUFDLENBQUMsQ0FBQyxFQUFFLGFBQWEsRUFBRSxLQUFLLENBQUMsbUJBQW1CLEVBQUUsT0FBTyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTO1NBQzdILENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxLQUFLLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBRW5ELElBQUksS0FBSyxDQUFDLFNBQVMsRUFBRTtZQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLHlCQUF5QixFQUFFLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQztTQUFFO1FBRTNGLElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxDQUFDLHVCQUF1QixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxFQUFFO1lBQy9ELE9BQU8sRUFBRSxVQUFVO1lBQ25CLFFBQVEsRUFBRSxPQUFPO1lBQ2pCLFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWTtTQUNoQyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBRS9ELElBQUksQ0FBQyxjQUFjLEdBQUcsbUJBQW1CLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFFakYsSUFBSSxDQUFDLFdBQVcsR0FBRyxJQUFJLENBQUMsZUFBZSxFQUFFLENBQUM7UUFFMUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsWUFBWSxFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBQy9DLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxLQUFLLENBQUMsWUFBWSxDQUFDO1FBRTVDLElBQUksS0FBSyxDQUFDLE9BQU8sRUFBRTtZQUNqQixJQUFJLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUUsY0FBYyxDQUFDLENBQUM7WUFDM0MsSUFBSSxDQUFDLFlBQVksR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1NBQ25DO1FBRUQsSUFBSSxLQUFLLENBQUMsa0JBQWtCLEVBQUU7WUFDNUIsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1NBQ3BEO0lBQ0gsQ0FBQztJQXZMRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLGdCQUFnQixDQUFDLE9BQXVCO1FBQ3BELE9BQU8sR0FBRyxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUM7WUFDOUIsT0FBTztZQUNQLE9BQU8sRUFBRSxDQUFDLHNCQUFzQixDQUFDO1lBQ2pDLFlBQVksRUFBRSxDQUFDLEdBQUcsQ0FBQztTQUNwQixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLGFBQWEsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxTQUFpQjtRQUN6RSxPQUFPLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEVBQUUsU0FBUyxFQUFFLENBQUMsQ0FBQztJQUM3RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLFlBQVksQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUFnQjtRQUN2RSxPQUFPLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEVBQUUsUUFBUSxFQUFFLENBQUMsQ0FBQztJQUM1RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLG1CQUFtQixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXNCO1FBRXBGLE1BQU0sTUFBTyxTQUFRLFNBQVM7WUFNNUIsWUFBWSxTQUFpQixFQUFFLFNBQWlCLEVBQUUsY0FBdUI7Z0JBQ3ZFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7Z0JBQ2pCLElBQUksQ0FBQyxRQUFRLEdBQUcsU0FBUyxDQUFDO2dCQUMxQixJQUFJLENBQUMsU0FBUyxHQUFHLFNBQVMsQ0FBQztnQkFDM0IsSUFBSSxDQUFDLGNBQWMsR0FBRyxjQUFjLENBQUM7WUFDdkMsQ0FBQztZQUVELElBQWMsUUFBUTtnQkFDcEIsT0FBTyxLQUFLLENBQUM7WUFDZixDQUFDO1NBQ0Y7UUFFRCxJQUFJLElBQVksQ0FBQztRQUNqQixJQUFJLEdBQVcsQ0FBQztRQUNoQixNQUFNLEtBQUssR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzlCLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFO1lBQ3BCLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxFQUFFO2dCQUFFLE1BQU0sSUFBSSxLQUFLLENBQUMsMkNBQTJDLENBQUMsQ0FBQzthQUFFO1lBRXRGLEdBQUcsR0FBRyxLQUFLLENBQUMsUUFBUSxDQUFDO1lBQ3JCLE1BQU0sY0FBYyxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDLFlBQVksQ0FBQztZQUNuRSxJQUFJLENBQUMsY0FBYyxFQUFFO2dCQUFFLE1BQU0sSUFBSSxLQUFLLENBQUMsaURBQWlELENBQUMsQ0FBQzthQUFFO1lBQzVGLElBQUksR0FBRyxjQUFjLENBQUM7U0FDdkI7YUFBTTtZQUNMLElBQUksS0FBSyxDQUFDLFFBQVEsRUFBRTtnQkFBRSxNQUFNLElBQUksS0FBSyxDQUFDLG1EQUFtRCxDQUFDLENBQUM7YUFBRTtZQUM3RixJQUFJLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztZQUN2QixHQUFHLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztnQkFDcEIsT0FBTyxFQUFFLFVBQVU7Z0JBQ25CLFFBQVEsRUFBRSxPQUFPO2dCQUNqQixZQUFZLEVBQUUsS0FBSyxDQUFDLFNBQVM7YUFDOUIsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxPQUFPLElBQUksTUFBTSxDQUFDLEdBQUcsRUFBRSxJQUFJLEVBQUUsS0FBSyxDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBQ3JELENBQUM7SUFzR0Q7Ozs7T0FJRztJQUNJLHVCQUF1QixDQUFDLEtBQWdDO1FBQzdELElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNqQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBRXhDLDRDQUE0QztRQUM1QyxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsS0FBSyxDQUFDLFlBQVksRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDakYsTUFBTSxhQUFhLEdBQUcsSUFBSSxDQUFDLG9CQUFvQixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRXZELElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQzlDLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLENBQUM7WUFDL0IsU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTO1lBQzFCLFNBQVMsRUFBRSxZQUFZO1lBQ3ZCLFVBQVUsRUFBRSxhQUFhO1lBQ3pCLHFCQUFxQixFQUFFLElBQUksQ0FBQyxXQUFXLEtBQUssV0FBVyxDQUFDLGVBQWUsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQztnQkFDcEYsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLFlBQVksSUFBSSxDQUFDO2dCQUMxQyxrQkFBa0IsRUFBRSxLQUFLLENBQUMsYUFBYSxJQUFJLENBQUM7YUFDN0M7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksc0JBQXNCLENBQUMsS0FBK0I7UUFDM0Qsd0dBQXdHO1FBQ3hHLElBQUksSUFBSSxDQUFDLHFCQUFxQixDQUFDLE1BQU0sSUFBSSwrQkFBK0IsRUFBRTtZQUN4RSxNQUFNLElBQUksVUFBVSxDQUFDLDBEQUEwRCwrQkFBK0IsRUFBRSxDQUFDLENBQUM7U0FDbkg7UUFFRCxJQUFJLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBRXhDLDRDQUE0QztRQUM1QyxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsSUFBSSxDQUFDLGlCQUFpQixFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUNyRixNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsb0JBQW9CLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFdkQsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDOUMsSUFBSSxDQUFDLHFCQUFxQixDQUFDLElBQUksQ0FBQztZQUM5QixTQUFTLEVBQUUsS0FBSyxDQUFDLFNBQVM7WUFDMUIsU0FBUyxFQUFFLFlBQVk7WUFDdkIsVUFBVSxFQUFFLGFBQWE7U0FDMUIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxxQkFBcUIsQ0FBQyxLQUF5QjtRQUNwRCxJQUFJLElBQUksQ0FBQyxZQUFZLENBQUMscUJBQXFCLEVBQUU7WUFDM0MsTUFBTSxJQUFJLEtBQUssQ0FBQyxpREFBaUQsQ0FBQyxDQUFDO1NBQ3BFO1FBQ0QsSUFBSSxJQUFJLENBQUMsV0FBVyxLQUFLLFdBQVcsQ0FBQyxlQUFlLEVBQUU7WUFDcEQsTUFBTSxJQUFJLEtBQUssQ0FBQywyRUFBMkUsQ0FBQyxDQUFDO1NBQzlGO1FBRUQsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLHFCQUFxQixHQUFHLElBQUksaURBQXNCLENBQUMsSUFBSSxFQUFFLGFBQWEsRUFBRTtZQUMvRixnQkFBZ0IsRUFBRSxVQUFVLENBQUMsZ0JBQWdCLENBQUMsUUFBUTtZQUN0RCxVQUFVLEVBQUUsU0FBUyxJQUFJLENBQUMsU0FBUyxFQUFFO1lBQ3JDLFNBQVMsRUFBRSxrQ0FBa0M7WUFDN0MsSUFBSSxFQUFFLElBQUksQ0FBQyxXQUFXO1lBQ3RCLEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksc0JBQXNCLENBQUMsS0FBeUI7UUFDckQsSUFBSSxJQUFJLENBQUMsWUFBWSxDQUFDLHNCQUFzQixFQUFFO1lBQzVDLE1BQU0sSUFBSSxLQUFLLENBQUMsa0RBQWtELENBQUMsQ0FBQztTQUNyRTtRQUNELElBQUksSUFBSSxDQUFDLFdBQVcsS0FBSyxXQUFXLENBQUMsZUFBZSxFQUFFO1lBQ3BELE1BQU0sSUFBSSxLQUFLLENBQUMsMkVBQTJFLENBQUMsQ0FBQztTQUM5RjtRQUVELE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxzQkFBc0IsR0FBRyxJQUFJLGlEQUFzQixDQUFDLElBQUksRUFBRSxjQUFjLEVBQUU7WUFDakcsZ0JBQWdCLEVBQUUsVUFBVSxDQUFDLGdCQUFnQixDQUFDLFFBQVE7WUFDdEQsVUFBVSxFQUFFLFNBQVMsSUFBSSxDQUFDLFNBQVMsRUFBRTtZQUNyQyxTQUFTLEVBQUUsbUNBQW1DO1lBQzlDLElBQUksRUFBRSxJQUFJLENBQUMsV0FBVztZQUN0QixHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLHlDQUF5QyxDQUFDLFNBQWlCLEVBQUUsS0FBeUI7UUFDM0YsSUFBSSxJQUFJLENBQUMsV0FBVyxLQUFLLFdBQVcsQ0FBQyxlQUFlLEVBQUU7WUFDcEQsTUFBTSxJQUFJLEtBQUssQ0FBQywyRUFBMkUsQ0FBQyxDQUFDO1NBQzlGO1FBQ0QsTUFBTSxhQUFhLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDdkQsSUFBSSxDQUFDLGFBQWEsRUFBRTtZQUNsQixNQUFNLElBQUksS0FBSyxDQUFDLHVDQUF1QyxTQUFTLEVBQUUsQ0FBQyxDQUFDO1NBQ3JFO1FBQ0QsSUFBSSxhQUFhLENBQUMscUJBQXFCLEVBQUU7WUFDdkMsTUFBTSxJQUFJLEtBQUssQ0FBQyxpREFBaUQsQ0FBQyxDQUFDO1NBQ3BFO1FBRUQsT0FBTyxhQUFhLENBQUMscUJBQXFCLEdBQUcsSUFBSSxpREFBc0IsQ0FBQyxJQUFJLEVBQUUsR0FBRyxTQUFTLGFBQWEsRUFBRTtZQUN2RyxnQkFBZ0IsRUFBRSxVQUFVLENBQUMsZ0JBQWdCLENBQUMsUUFBUTtZQUN0RCxVQUFVLEVBQUUsU0FBUyxJQUFJLENBQUMsU0FBUyxVQUFVLFNBQVMsRUFBRTtZQUN4RCxTQUFTLEVBQUUsa0NBQWtDO1lBQzdDLElBQUksRUFBRSxJQUFJLENBQUMsV0FBVztZQUN0QixHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLDBDQUEwQyxDQUFDLFNBQWlCLEVBQUUsS0FBeUI7UUFDNUYsSUFBSSxJQUFJLENBQUMsV0FBVyxLQUFLLFdBQVcsQ0FBQyxlQUFlLEVBQUU7WUFDcEQsTUFBTSxJQUFJLEtBQUssQ0FBQywyRUFBMkUsQ0FBQyxDQUFDO1NBQzlGO1FBQ0QsTUFBTSxhQUFhLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDdkQsSUFBSSxDQUFDLGFBQWEsRUFBRTtZQUNsQixNQUFNLElBQUksS0FBSyxDQUFDLHVDQUF1QyxTQUFTLEVBQUUsQ0FBQyxDQUFDO1NBQ3JFO1FBQ0QsSUFBSSxhQUFhLENBQUMsc0JBQXNCLEVBQUU7WUFDeEMsTUFBTSxJQUFJLEtBQUssQ0FBQyxrREFBa0QsQ0FBQyxDQUFDO1NBQ3JFO1FBRUQsT0FBTyxhQUFhLENBQUMsc0JBQXNCLEdBQUcsSUFBSSxpREFBc0IsQ0FBQyxJQUFJLEVBQUUsR0FBRyxTQUFTLGNBQWMsRUFBRTtZQUN6RyxnQkFBZ0IsRUFBRSxVQUFVLENBQUMsZ0JBQWdCLENBQUMsUUFBUTtZQUN0RCxVQUFVLEVBQUUsU0FBUyxJQUFJLENBQUMsU0FBUyxVQUFVLFNBQVMsRUFBRTtZQUN4RCxTQUFTLEVBQUUsbUNBQW1DO1lBQzlDLElBQUksRUFBRSxJQUFJLENBQUMsV0FBVztZQUN0QixHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNPLFFBQVE7UUFDaEIsTUFBTSxNQUFNLEdBQUcsSUFBSSxLQUFLLEVBQVUsQ0FBQztRQUVuQyxJQUFJLENBQUMsSUFBSSxDQUFDLGlCQUFpQixFQUFFO1lBQzNCLE1BQU0sQ0FBQyxJQUFJLENBQUMsbUNBQW1DLENBQUMsQ0FBQztTQUNsRDtRQUNELElBQUksSUFBSSxDQUFDLHFCQUFxQixDQUFDLE1BQU0sR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxFQUFFO1lBQy9ELE1BQU0sQ0FBQyxJQUFJLENBQUMsMEVBQTBFLENBQUMsQ0FBQztTQUN6RjtRQUVELE9BQU8sTUFBTSxDQUFDO0lBQ2hCLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssb0JBQW9CLENBQUMsS0FBd0Q7UUFDbkYsSUFBSSxJQUFJLENBQUMsV0FBVyxLQUFLLFdBQVcsQ0FBQyxlQUFlLEVBQUU7WUFDcEQsSUFBSSxLQUFLLENBQUMsWUFBWSxLQUFLLFNBQVMsSUFBSSxLQUFLLENBQUMsYUFBYSxLQUFLLFNBQVMsRUFBRTtnQkFDekUsTUFBTSxJQUFJLEtBQUssQ0FBQyw0RkFBNEYsQ0FBQyxDQUFDO2FBQy9HO1NBQ0Y7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLGlCQUFpQixDQUFDLFNBQWlCO1FBQ3pDLElBQUksSUFBSSxDQUFDLG1CQUFtQixDQUFDLEdBQUcsQ0FBQyxTQUFTLENBQUMsRUFBRTtZQUMzQyx3R0FBd0c7WUFDeEcsTUFBTSxJQUFJLEtBQUssQ0FBQywyQkFBMkIsU0FBUyxrQkFBa0IsQ0FBQyxDQUFDO1NBQ3pFO1FBQ0QsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEdBQUcsQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLHdCQUF3QixDQUFDLGdCQUEwQjtRQUN6RCxJQUFJLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLEdBQUcsZ0JBQWdCLENBQUMsTUFBTSxHQUFHLEVBQUUsRUFBRTtZQUM3RCx3R0FBd0c7WUFDeEcsTUFBTSxJQUFJLFVBQVUsQ0FBQyw0RUFBNEUsQ0FBQyxDQUFDO1NBQ3BHO1FBRUQsK0JBQStCO1FBQy9CLGdCQUFnQixDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztJQUNsRSxDQUFDO0lBRU8sbUJBQW1CLENBQUMsWUFBdUIsRUFBRSxPQUFtQjtRQUN0RSxJQUFJLENBQUMsaUJBQWlCLENBQUMsWUFBWSxDQUFDLENBQUM7UUFDckMsTUFBTSxjQUFjLEdBQWlDO1lBQ25ELEVBQUUsYUFBYSxFQUFFLFlBQVksQ0FBQyxJQUFJLEVBQUUsT0FBTyxFQUFFLGFBQWEsRUFBRTtTQUM3RCxDQUFDO1FBRUYsSUFBSSxPQUFPLEVBQUU7WUFDWCxJQUFJLENBQUMsaUJBQWlCLENBQUMsT0FBTyxDQUFDLENBQUM7WUFDaEMsY0FBYyxDQUFDLElBQUksQ0FBQyxFQUFFLGFBQWEsRUFBRSxPQUFPLENBQUMsSUFBSSxFQUFFLE9BQU8sRUFBRSxjQUFjLEVBQUUsQ0FBQyxDQUFDO1NBQy9FO1FBRUQsT0FBTyxjQUFjLENBQUM7SUFDeEIsQ0FBQztJQUVPLG9CQUFvQixDQUFDLEtBQTBCO1FBQ3JELElBQUksS0FBSyxDQUFDLGNBQWMsS0FBSyxjQUFjLENBQUMsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLGdCQUFnQixFQUFFO1lBQzlFLCtHQUErRztZQUMvRyxNQUFNLElBQUksS0FBSyxDQUFDLHFEQUFxRCxjQUFjLENBQUMsT0FBTyxrQkFBa0IsQ0FBQyxDQUFDO1NBQ2hIO1FBRUQsSUFBSSxLQUFLLENBQUMsY0FBYyxLQUFLLGNBQWMsQ0FBQyxPQUFPLElBQUksS0FBSyxDQUFDLGdCQUFnQixFQUFFO1lBQzdFLGtHQUFrRztZQUNsRyxNQUFNLElBQUksS0FBSyxDQUFDLDZEQUE2RCxjQUFjLENBQUMsT0FBTyxrQkFBa0IsQ0FBQyxDQUFDO1NBQ3hIO1FBRUQsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEVBQUU7WUFDMUIsSUFBSSxDQUFDLHdCQUF3QixDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1NBQ3ZEO1FBRUQsT0FBTztZQUNMLGNBQWMsRUFBRSxLQUFLLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsR0FBRztZQUNoRixnQkFBZ0IsRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsU0FBUztTQUM5RSxDQUFDO0lBQ0osQ0FBQztJQUVPLE9BQU8sQ0FBQyxPQUFlO1FBQzdCLE9BQU8sSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsT0FBTyxLQUFLLE9BQU8sQ0FBQyxDQUFDO0lBQy9ELENBQUM7SUFFTyxNQUFNLENBQUMsU0FBb0IsRUFBRSxPQUFlO1FBQ2xELE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDM0MsSUFBSSxZQUFZLEVBQUU7WUFDaEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxpQkFBaUIsU0FBUyxDQUFDLElBQUksU0FBUyxPQUFPLGlCQUFpQixZQUFZLENBQUMsYUFBYSxTQUFTLE9BQU8sTUFBTSxDQUFDLENBQUM7U0FDbkk7UUFDRCxJQUFJLENBQUMsaUJBQWlCLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDbEMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUM7WUFDbEIsYUFBYSxFQUFFLFNBQVMsQ0FBQyxJQUFJO1lBQzdCLE9BQU87U0FDUixDQUFDLENBQUM7UUFDSCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssaUJBQWlCLENBQUMsU0FBb0I7UUFDNUMsTUFBTSxFQUFFLElBQUksRUFBRSxJQUFJLEVBQUUsR0FBRyxTQUFTLENBQUM7UUFDakMsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLG9CQUFvQixDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxhQUFhLEtBQUssSUFBSSxDQUFDLENBQUM7UUFDdEYsSUFBSSxXQUFXLElBQUksV0FBVyxDQUFDLGFBQWEsS0FBSyxJQUFJLEVBQUU7WUFDckQsTUFBTSxJQUFJLEtBQUssQ0FBQyxxQkFBcUIsSUFBSSxPQUFPLElBQUksc0NBQXNDLFdBQVcsQ0FBQyxhQUFhLEVBQUUsQ0FBQyxDQUFDO1NBQ3hIO1FBQ0QsSUFBSSxDQUFDLFdBQVcsRUFBRTtZQUNoQixJQUFJLENBQUMsb0JBQW9CLENBQUMsSUFBSSxDQUFDO2dCQUM3QixhQUFhLEVBQUUsSUFBSTtnQkFDbkIsYUFBYSxFQUFFLElBQUk7YUFDcEIsQ0FBQyxDQUFDO1NBQ0o7SUFDSCxDQUFDO0lBRUQ7O09BRUc7SUFDSyxlQUFlO1FBQ3JCLDZCQUE2QjtRQUM3QixtSEFBbUg7UUFDbkgsT0FBTyxHQUFHLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFLFlBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsU0FBUyxDQUFDO1lBQ3hFLE9BQU8sRUFBRSxLQUFLO1lBQ2QsTUFBTSxFQUFFLEVBQUU7WUFDVixRQUFRLEVBQUUsc0VBQXNFO1lBQ2hGLFlBQVksRUFBRSx1REFBdUQ7U0FDdEUsQ0FBQyxDQUFDLENBQUM7SUFDTixDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLG1CQUFtQixDQUFDLE9BQWlCO1FBQzNDLE1BQU0sS0FBSyxHQUFHLFlBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFN0IsSUFBSSxDQUFDLFlBQUssQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxJQUFJLE9BQU8sQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxFQUFFO1lBQ3ZFLE1BQU0sSUFBSSxLQUFLLENBQUMsOEVBQThFLENBQUMsQ0FBQztTQUNqRztRQUVELE1BQU0sUUFBUSxHQUFHLGtDQUFlLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRW5ELGtHQUFrRztRQUNsRyxxR0FBcUc7UUFFckcsbUNBQW1DO1FBQ25DLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLGNBQWMsRUFBRSxZQUFZLENBQUMsQ0FBQztRQUNsRCxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxpQkFBaUIsRUFBRSx3QkFBd0IsQ0FBQyxDQUFDO1FBRWpFLHlDQUF5QztRQUN6QyxRQUFRLENBQUMsY0FBYyxDQUFDLGVBQWUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDOUQsT0FBTyxFQUFFLENBQUMsWUFBWSxDQUFDO1lBQ3ZCLFNBQVMsRUFBRSxPQUFPLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQztnQkFDL0MsTUFBTTtnQkFDTixPQUFPLEVBQUUsVUFBVTtnQkFDbkIsUUFBUSxFQUFFLE9BQU87Z0JBQ2pCLFlBQVksRUFBRSxJQUFJLENBQUMsU0FBUzthQUM3QixDQUFDLENBQUM7U0FDSixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksY0FBYyxDQUFDO1FBQ25CLEtBQUssTUFBTSxNQUFNLElBQUksSUFBSSxHQUFHLENBQUMsT0FBTyxDQUFDLEVBQUUsRUFBRSxvQkFBb0I7WUFDM0QsK0RBQStEO1lBQy9ELG1EQUFtRDtZQUNuRCxNQUFNLGFBQWEsR0FBRyxJQUFJLG1DQUFjLENBQUMsSUFBSSxFQUFFLFVBQVUsTUFBTSxFQUFFLEVBQUU7Z0JBQ2pFLFFBQVEsRUFBRSxRQUFRLENBQUMsUUFBUTtnQkFDM0IsWUFBWSxFQUFFLHlCQUF5QjtnQkFDdkMsVUFBVSxFQUFFO29CQUNWLFNBQVMsRUFBRSxJQUFJLENBQUMsU0FBUztvQkFDekIsTUFBTSxFQUFFLE1BQU07aUJBQ2Y7YUFDRixDQUFDLENBQUM7WUFFSCxxRUFBcUU7WUFDckUscUVBQXFFO1lBQ3JFLFVBQVU7WUFDVixJQUFJLFlBQUssQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxFQUFFO2dCQUNwQyxNQUFNLGFBQWEsR0FBRyxJQUFJLG1CQUFZLENBQUMsSUFBSSxFQUFFLHVCQUF1QixNQUFNLEVBQUUsRUFBRTtvQkFDNUUsVUFBVSxFQUFFLFNBQUUsQ0FBQyxZQUFZLENBQUMsU0FBRSxDQUFDLGVBQWUsQ0FBQyxNQUFNLEVBQUUsVUFBRyxDQUFDLE1BQU0sQ0FBQyxDQUFDO2lCQUNwRSxDQUFDLENBQUM7Z0JBQ0gsTUFBTSxpQkFBaUIsR0FBRyxhQUFhLENBQUMsSUFBSSxDQUFDLFlBQWlDLENBQUM7Z0JBQy9FLGlCQUFpQixDQUFDLFVBQVUsQ0FBQyxTQUFTLEdBQUcsYUFBYSxDQUFDO2FBQ3hEO1lBRUQsa0VBQWtFO1lBQ2xFLHdFQUF3RTtZQUN4RSw4REFBOEQ7WUFDOUQsSUFBSSxjQUFjLEVBQUU7Z0JBQ2xCLGFBQWEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLGNBQWMsQ0FBQyxDQUFDO2FBQ2xEO1lBQ0QsY0FBYyxHQUFHLGFBQWEsQ0FBQztTQUNoQztJQUNILENBQUM7SUFFRDs7T0FFRztJQUNILElBQWMsUUFBUTtRQUNwQixPQUFPLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxNQUFNLEdBQUcsSUFBSSxDQUFDLHFCQUFxQixDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUM7SUFDcEYsQ0FBQztDQUNGO0FBdGlCRCxzQkFzaUJDO0FBRUQsSUFBWSxhQU9YO0FBUEQsV0FBWSxhQUFhO0lBQ3ZCLCtGQUErRjtJQUMvRiw2QkFBWSxDQUFBO0lBQ1osMEVBQTBFO0lBQzFFLDZCQUFZLENBQUE7SUFDWix5Q0FBeUM7SUFDekMsNkJBQVksQ0FBQTtBQUNkLENBQUMsRUFQVyxhQUFhLEdBQWIscUJBQWEsS0FBYixxQkFBYSxRQU94QjtBQUVEOztHQUVHO0FBQ0gsSUFBWSxXQVNYO0FBVEQsV0FBWSxXQUFXO0lBQ3JCOztPQUVHO0lBQ0gsa0RBQW1DLENBQUE7SUFDbkM7O09BRUc7SUFDSCwwQ0FBMkIsQ0FBQTtBQUM3QixDQUFDLEVBVFcsV0FBVyxHQUFYLG1CQUFXLEtBQVgsbUJBQVcsUUFTdEI7QUFFRCxJQUFZLGNBT1g7QUFQRCxXQUFZLGNBQWM7SUFDeEIsb0VBQW9FO0lBQ3BFLHlDQUF1QixDQUFBO0lBQ3ZCLG1JQUFtSTtJQUNuSSxxQ0FBbUIsQ0FBQTtJQUNuQixnRUFBZ0U7SUFDaEUsNkJBQVcsQ0FBQTtBQUNiLENBQUMsRUFQVyxjQUFjLEdBQWQsc0JBQWMsS0FBZCxzQkFBYyxRQU96QjtBQUVEOzs7OztHQUtHO0FBQ0gsSUFBWSxjQVNYO0FBVEQsV0FBWSxjQUFjO0lBQ3hCLHNGQUFzRjtJQUN0Rix5Q0FBdUIsQ0FBQTtJQUN2Qix3RkFBd0Y7SUFDeEYseUNBQXVCLENBQUE7SUFDdkIsa0ZBQWtGO0lBQ2xGLDJEQUF5QyxDQUFBO0lBQ3pDLDhFQUE4RTtJQUM5RSx5Q0FBdUIsQ0FBQTtBQUN6QixDQUFDLEVBVFcsY0FBYyxHQUFkLHNCQUFjLEtBQWQsc0JBQWMsUUFTekIiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBhcHBzY2FsaW5nIGZyb20gJ0Bhd3MtY2RrL2F3cy1hcHBsaWNhdGlvbmF1dG9zY2FsaW5nJztcbmltcG9ydCB7IENmbkN1c3RvbVJlc291cmNlLCBDdXN0b21SZXNvdXJjZSB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZGZvcm1hdGlvbic7XG5pbXBvcnQgKiBhcyBjbG91ZHdhdGNoIGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZHdhdGNoJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCB7IEF3cywgQ2ZuQ29uZGl0aW9uLCBDb25zdHJ1Y3QsIEZuLCBJUmVzb3VyY2UsIExhenksIFJlbW92YWxQb2xpY3ksIFJlc291cmNlLCBTdGFjaywgVG9rZW4gfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IENmblRhYmxlIH0gZnJvbSAnLi9keW5hbW9kYi5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgUmVwbGljYVByb3ZpZGVyIH0gZnJvbSAnLi9yZXBsaWNhLXByb3ZpZGVyJztcbmltcG9ydCB7IEVuYWJsZVNjYWxpbmdQcm9wcywgSVNjYWxhYmxlVGFibGVBdHRyaWJ1dGUgfSBmcm9tICcuL3NjYWxhYmxlLWF0dHJpYnV0ZS1hcGknO1xuaW1wb3J0IHsgU2NhbGFibGVUYWJsZUF0dHJpYnV0ZSB9IGZyb20gJy4vc2NhbGFibGUtdGFibGUtYXR0cmlidXRlJztcblxuY29uc3QgSEFTSF9LRVlfVFlQRSA9ICdIQVNIJztcbmNvbnN0IFJBTkdFX0tFWV9UWVBFID0gJ1JBTkdFJztcblxuLy8gaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9MaW1pdHMuaHRtbCNsaW1pdHMtc2Vjb25kYXJ5LWluZGV4ZXNcbmNvbnN0IE1BWF9MT0NBTF9TRUNPTkRBUllfSU5ERVhfQ09VTlQgPSA1O1xuXG5jb25zdCBSRUFEX0RBVEFfQUNUSU9OUyA9IFtcbiAgJ2R5bmFtb2RiOkJhdGNoR2V0SXRlbScsXG4gICdkeW5hbW9kYjpHZXRSZWNvcmRzJyxcbiAgJ2R5bmFtb2RiOkdldFNoYXJkSXRlcmF0b3InLFxuICAnZHluYW1vZGI6UXVlcnknLFxuICAnZHluYW1vZGI6R2V0SXRlbScsXG4gICdkeW5hbW9kYjpTY2FuJ1xuXTtcblxuY29uc3QgUkVBRF9TVFJFQU1fREFUQV9BQ1RJT05TID0gW1xuICAnZHluYW1vZGI6RGVzY3JpYmVTdHJlYW0nLFxuICAnZHluYW1vZGI6R2V0UmVjb3JkcycsXG4gICdkeW5hbW9kYjpHZXRTaGFyZEl0ZXJhdG9yJyxcbl07XG5cbmNvbnN0IFdSSVRFX0RBVEFfQUNUSU9OUyA9IFtcbiAgJ2R5bmFtb2RiOkJhdGNoV3JpdGVJdGVtJyxcbiAgJ2R5bmFtb2RiOlB1dEl0ZW0nLFxuICAnZHluYW1vZGI6VXBkYXRlSXRlbScsXG4gICdkeW5hbW9kYjpEZWxldGVJdGVtJ1xuXTtcblxuZXhwb3J0IGludGVyZmFjZSBBdHRyaWJ1dGUge1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgYW4gYXR0cmlidXRlLlxuICAgKi9cbiAgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgZGF0YSB0eXBlIG9mIGFuIGF0dHJpYnV0ZS5cbiAgICovXG4gIHJlYWRvbmx5IHR5cGU6IEF0dHJpYnV0ZVR5cGU7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgVGFibGVPcHRpb25zIHtcbiAgLyoqXG4gICAqIFBhcnRpdGlvbiBrZXkgYXR0cmlidXRlIGRlZmluaXRpb24uXG4gICAqL1xuICByZWFkb25seSBwYXJ0aXRpb25LZXk6IEF0dHJpYnV0ZTtcblxuICAvKipcbiAgICogVGFibGUgc29ydCBrZXkgYXR0cmlidXRlIGRlZmluaXRpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IG5vIHNvcnQga2V5XG4gICAqL1xuICByZWFkb25seSBzb3J0S2V5PzogQXR0cmlidXRlO1xuXG4gIC8qKlxuICAgKiBUaGUgcmVhZCBjYXBhY2l0eSBmb3IgdGhlIHRhYmxlLiBDYXJlZnVsIGlmIHlvdSBhZGQgR2xvYmFsIFNlY29uZGFyeSBJbmRleGVzLCBhc1xuICAgKiB0aG9zZSB3aWxsIHNoYXJlIHRoZSB0YWJsZSdzIHByb3Zpc2lvbmVkIHRocm91Z2hwdXQuXG4gICAqXG4gICAqIENhbiBvbmx5IGJlIHByb3ZpZGVkIGlmIGJpbGxpbmdNb2RlIGlzIFByb3Zpc2lvbmVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCA1XG4gICAqL1xuICByZWFkb25seSByZWFkQ2FwYWNpdHk/OiBudW1iZXI7XG4gIC8qKlxuICAgKiBUaGUgd3JpdGUgY2FwYWNpdHkgZm9yIHRoZSB0YWJsZS4gQ2FyZWZ1bCBpZiB5b3UgYWRkIEdsb2JhbCBTZWNvbmRhcnkgSW5kZXhlcywgYXNcbiAgICogdGhvc2Ugd2lsbCBzaGFyZSB0aGUgdGFibGUncyBwcm92aXNpb25lZCB0aHJvdWdocHV0LlxuICAgKlxuICAgKiBDYW4gb25seSBiZSBwcm92aWRlZCBpZiBiaWxsaW5nTW9kZSBpcyBQcm92aXNpb25lZC5cbiAgICpcbiAgICogQGRlZmF1bHQgNVxuICAgKi9cbiAgcmVhZG9ubHkgd3JpdGVDYXBhY2l0eT86IG51bWJlcjtcblxuICAvKipcbiAgICogU3BlY2lmeSBob3cgeW91IGFyZSBjaGFyZ2VkIGZvciByZWFkIGFuZCB3cml0ZSB0aHJvdWdocHV0IGFuZCBob3cgeW91IG1hbmFnZSBjYXBhY2l0eS5cbiAgICpcbiAgICogQGRlZmF1bHQgUFJPVklTSU9ORUQgaWYgYHJlcGxpY2F0aW9uUmVnaW9uc2AgaXMgbm90IHNwZWNpZmllZCwgUEFZX1BFUl9SRVFVRVNUIG90aGVyd2lzZVxuICAgKi9cbiAgcmVhZG9ubHkgYmlsbGluZ01vZGU/OiBCaWxsaW5nTW9kZTtcblxuICAvKipcbiAgICogV2hldGhlciBwb2ludC1pbi10aW1lIHJlY292ZXJ5IGlzIGVuYWJsZWQuXG4gICAqIEBkZWZhdWx0IC0gcG9pbnQtaW4tdGltZSByZWNvdmVyeSBpcyBkaXNhYmxlZFxuICAgKi9cbiAgcmVhZG9ubHkgcG9pbnRJblRpbWVSZWNvdmVyeT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgc2VydmVyLXNpZGUgZW5jcnlwdGlvbiB3aXRoIGFuIEFXUyBtYW5hZ2VkIGN1c3RvbWVyIG1hc3RlciBrZXkgaXMgZW5hYmxlZC5cbiAgICogQGRlZmF1bHQgLSBzZXJ2ZXItc2lkZSBlbmNyeXB0aW9uIGlzIGVuYWJsZWQgd2l0aCBhbiBBV1Mgb3duZWQgY3VzdG9tZXIgbWFzdGVyIGtleVxuICAgKi9cbiAgcmVhZG9ubHkgc2VydmVyU2lkZUVuY3J5cHRpb24/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiBUVEwgYXR0cmlidXRlLlxuICAgKiBAZGVmYXVsdCAtIFRUTCBpcyBkaXNhYmxlZFxuICAgKi9cbiAgcmVhZG9ubHkgdGltZVRvTGl2ZUF0dHJpYnV0ZT86IHN0cmluZztcblxuICAvKipcbiAgICogV2hlbiBhbiBpdGVtIGluIHRoZSB0YWJsZSBpcyBtb2RpZmllZCwgU3RyZWFtVmlld1R5cGUgZGV0ZXJtaW5lcyB3aGF0IGluZm9ybWF0aW9uXG4gICAqIGlzIHdyaXR0ZW4gdG8gdGhlIHN0cmVhbSBmb3IgdGhpcyB0YWJsZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBzdHJlYW1zIGFyZSBkaXNhYmxlZCB1bmxlc3MgYHJlcGxpY2F0aW9uUmVnaW9uc2AgaXMgc3BlY2lmaWVkXG4gICAqL1xuICByZWFkb25seSBzdHJlYW0/OiBTdHJlYW1WaWV3VHlwZTtcblxuICAvKipcbiAgICogVGhlIHJlbW92YWwgcG9saWN5IHRvIGFwcGx5IHRvIHRoZSBEeW5hbW9EQiBUYWJsZS5cbiAgICpcbiAgICogQGRlZmF1bHQgUmVtb3ZhbFBvbGljeS5SRVRBSU5cbiAgICovXG4gIHJlYWRvbmx5IHJlbW92YWxQb2xpY3k/OiBSZW1vdmFsUG9saWN5O1xuXG4gIC8qKlxuICAgKiBSZWdpb25zIHdoZXJlIHJlcGxpY2EgdGFibGVzIHdpbGwgYmUgY3JlYXRlZFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIHJlcGxpY2EgdGFibGVzIGFyZSBjcmVhdGVkXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIHJlYWRvbmx5IHJlcGxpY2F0aW9uUmVnaW9ucz86IHN0cmluZ1tdO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIFRhYmxlUHJvcHMgZXh0ZW5kcyBUYWJsZU9wdGlvbnMge1xuICAvKipcbiAgICogRW5mb3JjZXMgYSBwYXJ0aWN1bGFyIHBoeXNpY2FsIHRhYmxlIG5hbWUuXG4gICAqIEBkZWZhdWx0IDxnZW5lcmF0ZWQ+XG4gICAqL1xuICByZWFkb25seSB0YWJsZU5hbWU/OiBzdHJpbmc7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgU2Vjb25kYXJ5SW5kZXhQcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgc2Vjb25kYXJ5IGluZGV4LlxuICAgKi9cbiAgcmVhZG9ubHkgaW5kZXhOYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBzZXQgb2YgYXR0cmlidXRlcyB0aGF0IGFyZSBwcm9qZWN0ZWQgaW50byB0aGUgc2Vjb25kYXJ5IGluZGV4LlxuICAgKiBAZGVmYXVsdCBBTExcbiAgICovXG4gIHJlYWRvbmx5IHByb2plY3Rpb25UeXBlPzogUHJvamVjdGlvblR5cGU7XG5cbiAgLyoqXG4gICAqIFRoZSBub24ta2V5IGF0dHJpYnV0ZXMgdGhhdCBhcmUgcHJvamVjdGVkIGludG8gdGhlIHNlY29uZGFyeSBpbmRleC5cbiAgICogQGRlZmF1bHQgLSBObyBhZGRpdGlvbmFsIGF0dHJpYnV0ZXNcbiAgICovXG4gIHJlYWRvbmx5IG5vbktleUF0dHJpYnV0ZXM/OiBzdHJpbmdbXTtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBHbG9iYWxTZWNvbmRhcnlJbmRleFByb3BzIGV4dGVuZHMgU2Vjb25kYXJ5SW5kZXhQcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgYXR0cmlidXRlIG9mIGEgcGFydGl0aW9uIGtleSBmb3IgdGhlIGdsb2JhbCBzZWNvbmRhcnkgaW5kZXguXG4gICAqL1xuICByZWFkb25seSBwYXJ0aXRpb25LZXk6IEF0dHJpYnV0ZTtcblxuICAvKipcbiAgICogVGhlIGF0dHJpYnV0ZSBvZiBhIHNvcnQga2V5IGZvciB0aGUgZ2xvYmFsIHNlY29uZGFyeSBpbmRleC5cbiAgICogQGRlZmF1bHQgLSBObyBzb3J0IGtleVxuICAgKi9cbiAgcmVhZG9ubHkgc29ydEtleT86IEF0dHJpYnV0ZTtcblxuICAvKipcbiAgICogVGhlIHJlYWQgY2FwYWNpdHkgZm9yIHRoZSBnbG9iYWwgc2Vjb25kYXJ5IGluZGV4LlxuICAgKlxuICAgKiBDYW4gb25seSBiZSBwcm92aWRlZCBpZiB0YWJsZSBiaWxsaW5nTW9kZSBpcyBQcm92aXNpb25lZCBvciB1bmRlZmluZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IDVcbiAgICovXG4gIHJlYWRvbmx5IHJlYWRDYXBhY2l0eT86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIHdyaXRlIGNhcGFjaXR5IGZvciB0aGUgZ2xvYmFsIHNlY29uZGFyeSBpbmRleC5cbiAgICpcbiAgICogQ2FuIG9ubHkgYmUgcHJvdmlkZWQgaWYgdGFibGUgYmlsbGluZ01vZGUgaXMgUHJvdmlzaW9uZWQgb3IgdW5kZWZpbmVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCA1XG4gICAqL1xuICByZWFkb25seSB3cml0ZUNhcGFjaXR5PzogbnVtYmVyO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIExvY2FsU2Vjb25kYXJ5SW5kZXhQcm9wcyBleHRlbmRzIFNlY29uZGFyeUluZGV4UHJvcHMge1xuICAvKipcbiAgICogVGhlIGF0dHJpYnV0ZSBvZiBhIHNvcnQga2V5IGZvciB0aGUgbG9jYWwgc2Vjb25kYXJ5IGluZGV4LlxuICAgKi9cbiAgcmVhZG9ubHkgc29ydEtleTogQXR0cmlidXRlO1xufVxuXG4vKipcbiAqIEFuIGludGVyZmFjZSB0aGF0IHJlcHJlc2VudHMgYSBEeW5hbW9EQiBUYWJsZSAtIGVpdGhlciBjcmVhdGVkIHdpdGggdGhlIENESywgb3IgYW4gZXhpc3Rpbmcgb25lLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElUYWJsZSBleHRlbmRzIElSZXNvdXJjZSB7XG4gIC8qKlxuICAgKiBBcm4gb2YgdGhlIGR5bmFtb2RiIHRhYmxlLlxuICAgKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSB0YWJsZUFybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUYWJsZSBuYW1lIG9mIHRoZSBkeW5hbW9kYiB0YWJsZS5cbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgdGFibGVOYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEFSTiBvZiB0aGUgdGFibGUncyBzdHJlYW0sIGlmIHRoZXJlIGlzIG9uZS5cbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgdGFibGVTdHJlYW1Bcm4/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFBlcm1pdHMgYW4gSUFNIHByaW5jaXBhbCBhbGwgZGF0YSByZWFkIG9wZXJhdGlvbnMgZnJvbSB0aGlzIHRhYmxlOlxuICAgKiBCYXRjaEdldEl0ZW0sIEdldFJlY29yZHMsIEdldFNoYXJkSXRlcmF0b3IsIFF1ZXJ5LCBHZXRJdGVtLCBTY2FuLlxuICAgKiBAcGFyYW0gZ3JhbnRlZSBUaGUgcHJpbmNpcGFsIHRvIGdyYW50IGFjY2VzcyB0b1xuICAgKi9cbiAgZ3JhbnRSZWFkRGF0YShncmFudGVlOiBpYW0uSUdyYW50YWJsZSk6IGlhbS5HcmFudDtcblxuICAvKipcbiAgICogUGVybWl0cyBhbiBJQU0gUHJpbmNpcGFsIHRvIGxpc3Qgc3RyZWFtcyBhdHRhY2hlZCB0byBjdXJyZW50IGR5bmFtb2RiIHRhYmxlLlxuICAgKlxuICAgKiBAcGFyYW0gZ3JhbnRlZSBUaGUgcHJpbmNpcGFsIChuby1vcCBpZiB1bmRlZmluZWQpXG4gICAqL1xuICBncmFudFRhYmxlTGlzdFN0cmVhbXMoZ3JhbnRlZTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQ7XG5cbiAgLyoqXG4gICAqIFBlcm1pdHMgYW4gSUFNIHByaW5jaXBhbCBhbGwgc3RyZWFtIGRhdGEgcmVhZCBvcGVyYXRpb25zIGZvciB0aGlzXG4gICAqIHRhYmxlJ3Mgc3RyZWFtOlxuICAgKiBEZXNjcmliZVN0cmVhbSwgR2V0UmVjb3JkcywgR2V0U2hhcmRJdGVyYXRvciwgTGlzdFN0cmVhbXMuXG4gICAqIEBwYXJhbSBncmFudGVlIFRoZSBwcmluY2lwYWwgdG8gZ3JhbnQgYWNjZXNzIHRvXG4gICAqL1xuICBncmFudFN0cmVhbVJlYWQoZ3JhbnRlZTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQ7XG5cbiAgLyoqXG4gICAqIFBlcm1pdHMgYW4gSUFNIHByaW5jaXBhbCBhbGwgZGF0YSB3cml0ZSBvcGVyYXRpb25zIHRvIHRoaXMgdGFibGU6XG4gICAqIEJhdGNoV3JpdGVJdGVtLCBQdXRJdGVtLCBVcGRhdGVJdGVtLCBEZWxldGVJdGVtLlxuICAgKiBAcGFyYW0gZ3JhbnRlZSBUaGUgcHJpbmNpcGFsIHRvIGdyYW50IGFjY2VzcyB0b1xuICAgKi9cbiAgZ3JhbnRXcml0ZURhdGEoZ3JhbnRlZTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQ7XG5cbiAgLyoqXG4gICAqIFBlcm1pdHMgYW4gSUFNIHByaW5jaXBhbCB0byBhbGwgZGF0YSByZWFkL3dyaXRlIG9wZXJhdGlvbnMgdG8gdGhpcyB0YWJsZS5cbiAgICogQmF0Y2hHZXRJdGVtLCBHZXRSZWNvcmRzLCBHZXRTaGFyZEl0ZXJhdG9yLCBRdWVyeSwgR2V0SXRlbSwgU2NhbixcbiAgICogQmF0Y2hXcml0ZUl0ZW0sIFB1dEl0ZW0sIFVwZGF0ZUl0ZW0sIERlbGV0ZUl0ZW1cbiAgICogQHBhcmFtIGdyYW50ZWUgVGhlIHByaW5jaXBhbCB0byBncmFudCBhY2Nlc3MgdG9cbiAgICovXG4gIGdyYW50UmVhZFdyaXRlRGF0YShncmFudGVlOiBpYW0uSUdyYW50YWJsZSk6IGlhbS5HcmFudDtcblxuICAvKipcbiAgICogTWV0cmljIGZvciB0aGUgbnVtYmVyIG9mIEVycm9ycyBleGVjdXRpbmcgYWxsIExhbWJkYXNcbiAgICovXG4gIG1ldHJpYyhtZXRyaWNOYW1lOiBzdHJpbmcsIHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWM7XG5cbiAgLyoqXG4gICAqIE1ldHJpYyBmb3IgdGhlIGNvbnN1bWVkIHJlYWQgY2FwYWNpdHkgdW5pdHNcbiAgICpcbiAgICogQHBhcmFtIHByb3BzIHByb3BlcnRpZXMgb2YgYSBtZXRyaWNcbiAgICovXG4gIG1ldHJpY0NvbnN1bWVkUmVhZENhcGFjaXR5VW5pdHMocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpOiBjbG91ZHdhdGNoLk1ldHJpYztcblxuICAvKipcbiAgICogTWV0cmljIGZvciB0aGUgY29uc3VtZWQgd3JpdGUgY2FwYWNpdHkgdW5pdHNcbiAgICpcbiAgICogQHBhcmFtIHByb3BzIHByb3BlcnRpZXMgb2YgYSBtZXRyaWNcbiAgICovXG4gIG1ldHJpY0NvbnN1bWVkV3JpdGVDYXBhY2l0eVVuaXRzKHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWM7XG5cbiAgLyoqXG4gICAqIE1ldHJpYyBmb3IgdGhlIHN5c3RlbSBlcnJvcnNcbiAgICpcbiAgICogQHBhcmFtIHByb3BzIHByb3BlcnRpZXMgb2YgYSBtZXRyaWNcbiAgICovXG4gIG1ldHJpY1N5c3RlbUVycm9ycyhwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljO1xuXG4gIC8qKlxuICAgKiBNZXRyaWMgZm9yIHRoZSB1c2VyIGVycm9yc1xuICAgKlxuICAgKiBAcGFyYW0gcHJvcHMgcHJvcGVydGllcyBvZiBhIG1ldHJpY1xuICAgKi9cbiAgbWV0cmljVXNlckVycm9ycyhwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljO1xuXG4gIC8qKlxuICAgKiBNZXRyaWMgZm9yIHRoZSBjb25kaXRpb25hbCBjaGVjayBmYWlsZWQgcmVxdWVzdHNcbiAgICpcbiAgICogQHBhcmFtIHByb3BzIHByb3BlcnRpZXMgb2YgYSBtZXRyaWNcbiAgICovXG4gIG1ldHJpY0NvbmRpdGlvbmFsQ2hlY2tGYWlsZWRSZXF1ZXN0cyhwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljO1xuXG4gIC8qKlxuICAgKiBNZXRyaWMgZm9yIHRoZSBzdWNjZXNzZnVsIHJlcXVlc3QgbGF0ZW5jeVxuICAgKlxuICAgKiBAcGFyYW0gcHJvcHMgcHJvcGVydGllcyBvZiBhIG1ldHJpY1xuICAgKi9cbiAgbWV0cmljU3VjY2Vzc2Z1bFJlcXVlc3RMYXRlbmN5KHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWM7XG59XG5cbi8qKlxuICogUmVmZXJlbmNlIHRvIGEgZHluYW1vZGIgdGFibGUuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVGFibGVBdHRyaWJ1dGVzIHtcbiAgLyoqXG4gICAqIFRoZSBBUk4gb2YgdGhlIGR5bmFtb2RiIHRhYmxlLlxuICAgKiBPbmUgb2YgdGhpcywgb3Ige0BsaW5rIHRhYmVOYW1lfSwgaXMgcmVxdWlyZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gdGFibGUgYXJuXG4gICAqL1xuICByZWFkb25seSB0YWJsZUFybj86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHRhYmxlIG5hbWUgb2YgdGhlIGR5bmFtb2RiIHRhYmxlLlxuICAgKiBPbmUgb2YgdGhpcywgb3Ige0BsaW5rIHRhYmVBcm59LCBpcyByZXF1aXJlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyB0YWJsZSBuYW1lXG4gICAqL1xuICByZWFkb25seSB0YWJsZU5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBBUk4gb2YgdGhlIHRhYmxlJ3Mgc3RyZWFtLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIHRhYmxlIHN0cmVhbVxuICAgKi9cbiAgcmVhZG9ubHkgdGFibGVTdHJlYW1Bcm4/OiBzdHJpbmc7XG59XG5cbmFic3RyYWN0IGNsYXNzIFRhYmxlQmFzZSBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSVRhYmxlIHtcbiAgLyoqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSB0YWJsZUFybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgdGFibGVOYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSB0YWJsZVN0cmVhbUFybj86IHN0cmluZztcblxuICAvKipcbiAgICogQWRkcyBhbiBJQU0gcG9saWN5IHN0YXRlbWVudCBhc3NvY2lhdGVkIHdpdGggdGhpcyB0YWJsZSB0byBhbiBJQU1cbiAgICogcHJpbmNpcGFsJ3MgcG9saWN5LlxuICAgKiBAcGFyYW0gZ3JhbnRlZSBUaGUgcHJpbmNpcGFsIChuby1vcCBpZiB1bmRlZmluZWQpXG4gICAqIEBwYXJhbSBhY3Rpb25zIFRoZSBzZXQgb2YgYWN0aW9ucyB0byBhbGxvdyAoaS5lLiBcImR5bmFtb2RiOlB1dEl0ZW1cIiwgXCJkeW5hbW9kYjpHZXRJdGVtXCIsIC4uLilcbiAgICovXG4gIHB1YmxpYyBncmFudChncmFudGVlOiBpYW0uSUdyYW50YWJsZSwgLi4uYWN0aW9uczogc3RyaW5nW10pOiBpYW0uR3JhbnQge1xuICAgIHJldHVybiBpYW0uR3JhbnQuYWRkVG9QcmluY2lwYWwoe1xuICAgICAgZ3JhbnRlZSxcbiAgICAgIGFjdGlvbnMsXG4gICAgICByZXNvdXJjZUFybnM6IFtcbiAgICAgICAgdGhpcy50YWJsZUFybixcbiAgICAgICAgTGF6eS5zdHJpbmdWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IHRoaXMuaGFzSW5kZXggPyBgJHt0aGlzLnRhYmxlQXJufS9pbmRleC8qYCA6IEF3cy5OT19WQUxVRSB9KVxuICAgICAgXSxcbiAgICAgIHNjb3BlOiB0aGlzLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYW4gSUFNIHBvbGljeSBzdGF0ZW1lbnQgYXNzb2NpYXRlZCB3aXRoIHRoaXMgdGFibGUncyBzdHJlYW0gdG8gYW5cbiAgICogSUFNIHByaW5jaXBhbCdzIHBvbGljeS5cbiAgICogQHBhcmFtIGdyYW50ZWUgVGhlIHByaW5jaXBhbCAobm8tb3AgaWYgdW5kZWZpbmVkKVxuICAgKiBAcGFyYW0gYWN0aW9ucyBUaGUgc2V0IG9mIGFjdGlvbnMgdG8gYWxsb3cgKGkuZS4gXCJkeW5hbW9kYjpEZXNjcmliZVN0cmVhbVwiLCBcImR5bmFtb2RiOkdldFJlY29yZHNcIiwgLi4uKVxuICAgKi9cbiAgcHVibGljIGdyYW50U3RyZWFtKGdyYW50ZWU6IGlhbS5JR3JhbnRhYmxlLCAuLi5hY3Rpb25zOiBzdHJpbmdbXSk6IGlhbS5HcmFudCB7XG4gICAgaWYgKCF0aGlzLnRhYmxlU3RyZWFtQXJuKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYER5bmFtb0RCIFN0cmVhbXMgbXVzdCBiZSBlbmFibGVkIG9uIHRoZSB0YWJsZSAke3RoaXMubm9kZS5wYXRofWApO1xuICAgIH1cblxuICAgIHJldHVybiBpYW0uR3JhbnQuYWRkVG9QcmluY2lwYWwoe1xuICAgICAgZ3JhbnRlZSxcbiAgICAgIGFjdGlvbnMsXG4gICAgICByZXNvdXJjZUFybnM6IFt0aGlzLnRhYmxlU3RyZWFtQXJuXSxcbiAgICAgIHNjb3BlOiB0aGlzLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFBlcm1pdHMgYW4gSUFNIHByaW5jaXBhbCBhbGwgZGF0YSByZWFkIG9wZXJhdGlvbnMgZnJvbSB0aGlzIHRhYmxlOlxuICAgKiBCYXRjaEdldEl0ZW0sIEdldFJlY29yZHMsIEdldFNoYXJkSXRlcmF0b3IsIFF1ZXJ5LCBHZXRJdGVtLCBTY2FuLlxuICAgKiBAcGFyYW0gZ3JhbnRlZSBUaGUgcHJpbmNpcGFsIHRvIGdyYW50IGFjY2VzcyB0b1xuICAgKi9cbiAgcHVibGljIGdyYW50UmVhZERhdGEoZ3JhbnRlZTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQge1xuICAgIHJldHVybiB0aGlzLmdyYW50KGdyYW50ZWUsIC4uLlJFQURfREFUQV9BQ1RJT05TKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBQZXJtaXRzIGFuIElBTSBQcmluY2lwYWwgdG8gbGlzdCBzdHJlYW1zIGF0dGFjaGVkIHRvIGN1cnJlbnQgZHluYW1vZGIgdGFibGUuXG4gICAqXG4gICAqIEBwYXJhbSBncmFudGVlIFRoZSBwcmluY2lwYWwgKG5vLW9wIGlmIHVuZGVmaW5lZClcbiAgICovXG4gIHB1YmxpYyBncmFudFRhYmxlTGlzdFN0cmVhbXMoZ3JhbnRlZTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQge1xuICAgIGlmICghdGhpcy50YWJsZVN0cmVhbUFybikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBEeW5hbW9EQiBTdHJlYW1zIG11c3QgYmUgZW5hYmxlZCBvbiB0aGUgdGFibGUgJHt0aGlzLm5vZGUucGF0aH1gKTtcbiAgICB9XG4gICAgcmV0dXJuIGlhbS5HcmFudC5hZGRUb1ByaW5jaXBhbCh7XG4gICAgICBncmFudGVlLFxuICAgICAgYWN0aW9uczogWydkeW5hbW9kYjpMaXN0U3RyZWFtcyddLFxuICAgICAgcmVzb3VyY2VBcm5zOiBbXG4gICAgICAgIExhenkuc3RyaW5nVmFsdWUoeyBwcm9kdWNlOiAoKSA9PiBgJHt0aGlzLnRhYmxlQXJufS9zdHJlYW0vKmAgfSlcbiAgICAgIF0sXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogUGVybWl0cyBhbiBJQU0gcHJpbmNpcGFsIGFsbCBzdHJlYW0gZGF0YSByZWFkIG9wZXJhdGlvbnMgZm9yIHRoaXNcbiAgICogdGFibGUncyBzdHJlYW06XG4gICAqIERlc2NyaWJlU3RyZWFtLCBHZXRSZWNvcmRzLCBHZXRTaGFyZEl0ZXJhdG9yLCBMaXN0U3RyZWFtcy5cbiAgICogQHBhcmFtIGdyYW50ZWUgVGhlIHByaW5jaXBhbCB0byBncmFudCBhY2Nlc3MgdG9cbiAgICovXG4gIHB1YmxpYyBncmFudFN0cmVhbVJlYWQoZ3JhbnRlZTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQge1xuICAgIHRoaXMuZ3JhbnRUYWJsZUxpc3RTdHJlYW1zKGdyYW50ZWUpO1xuICAgIHJldHVybiB0aGlzLmdyYW50U3RyZWFtKGdyYW50ZWUsIC4uLlJFQURfU1RSRUFNX0RBVEFfQUNUSU9OUyk7XG4gIH1cblxuICAvKipcbiAgICogUGVybWl0cyBhbiBJQU0gcHJpbmNpcGFsIGFsbCBkYXRhIHdyaXRlIG9wZXJhdGlvbnMgdG8gdGhpcyB0YWJsZTpcbiAgICogQmF0Y2hXcml0ZUl0ZW0sIFB1dEl0ZW0sIFVwZGF0ZUl0ZW0sIERlbGV0ZUl0ZW0uXG4gICAqIEBwYXJhbSBncmFudGVlIFRoZSBwcmluY2lwYWwgdG8gZ3JhbnQgYWNjZXNzIHRvXG4gICAqL1xuICBwdWJsaWMgZ3JhbnRXcml0ZURhdGEoZ3JhbnRlZTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQge1xuICAgIHJldHVybiB0aGlzLmdyYW50KGdyYW50ZWUsIC4uLldSSVRFX0RBVEFfQUNUSU9OUyk7XG4gIH1cblxuICAvKipcbiAgICogUGVybWl0cyBhbiBJQU0gcHJpbmNpcGFsIHRvIGFsbCBkYXRhIHJlYWQvd3JpdGUgb3BlcmF0aW9ucyB0byB0aGlzIHRhYmxlLlxuICAgKiBCYXRjaEdldEl0ZW0sIEdldFJlY29yZHMsIEdldFNoYXJkSXRlcmF0b3IsIFF1ZXJ5LCBHZXRJdGVtLCBTY2FuLFxuICAgKiBCYXRjaFdyaXRlSXRlbSwgUHV0SXRlbSwgVXBkYXRlSXRlbSwgRGVsZXRlSXRlbVxuICAgKiBAcGFyYW0gZ3JhbnRlZSBUaGUgcHJpbmNpcGFsIHRvIGdyYW50IGFjY2VzcyB0b1xuICAgKi9cbiAgcHVibGljIGdyYW50UmVhZFdyaXRlRGF0YShncmFudGVlOiBpYW0uSUdyYW50YWJsZSk6IGlhbS5HcmFudCB7XG4gICAgcmV0dXJuIHRoaXMuZ3JhbnQoZ3JhbnRlZSwgLi4uUkVBRF9EQVRBX0FDVElPTlMsIC4uLldSSVRFX0RBVEFfQUNUSU9OUyk7XG4gIH1cblxuICAvKipcbiAgICogUGVybWl0cyBhbGwgRHluYW1vREIgb3BlcmF0aW9ucyAoXCJkeW5hbW9kYjoqXCIpIHRvIGFuIElBTSBwcmluY2lwYWwuXG4gICAqIEBwYXJhbSBncmFudGVlIFRoZSBwcmluY2lwYWwgdG8gZ3JhbnQgYWNjZXNzIHRvXG4gICAqL1xuICBwdWJsaWMgZ3JhbnRGdWxsQWNjZXNzKGdyYW50ZWU6IGlhbS5JR3JhbnRhYmxlKSB7XG4gICAgcmV0dXJuIHRoaXMuZ3JhbnQoZ3JhbnRlZSwgJ2R5bmFtb2RiOionKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gdGhlIGdpdmVuIG5hbWVkIG1ldHJpYyBmb3IgdGhpcyBUYWJsZVxuICAgKi9cbiAgcHVibGljIG1ldHJpYyhtZXRyaWNOYW1lOiBzdHJpbmcsIHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWMge1xuICAgIHJldHVybiBuZXcgY2xvdWR3YXRjaC5NZXRyaWMoe1xuICAgICAgbmFtZXNwYWNlOiAnQVdTL0R5bmFtb0RCJyxcbiAgICAgIG1ldHJpY05hbWUsXG4gICAgICBkaW1lbnNpb25zOiB7XG4gICAgICAgIFRhYmxlTmFtZTogdGhpcy50YWJsZU5hbWUsXG4gICAgICB9LFxuICAgICAgLi4ucHJvcHNcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBNZXRyaWMgZm9yIHRoZSBjb25zdW1lZCByZWFkIGNhcGFjaXR5IHVuaXRzIHRoaXMgdGFibGVcbiAgICpcbiAgICogQGRlZmF1bHQgc3VtIG92ZXIgYSBtaW51dGVcbiAgICovXG4gIHB1YmxpYyBtZXRyaWNDb25zdW1lZFJlYWRDYXBhY2l0eVVuaXRzKHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWMge1xuICAgIHJldHVybiB0aGlzLm1ldHJpYygnQ29uc3VtZWRSZWFkQ2FwYWNpdHlVbml0cycsIHsgc3RhdGlzdGljOiAnc3VtJywgLi4ucHJvcHN9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBNZXRyaWMgZm9yIHRoZSBjb25zdW1lZCB3cml0ZSBjYXBhY2l0eSB1bml0cyB0aGlzIHRhYmxlXG4gICAqXG4gICAqIEBkZWZhdWx0IHN1bSBvdmVyIGEgbWludXRlXG4gICAqL1xuICBwdWJsaWMgbWV0cmljQ29uc3VtZWRXcml0ZUNhcGFjaXR5VW5pdHMocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpOiBjbG91ZHdhdGNoLk1ldHJpYyB7XG4gICAgcmV0dXJuIHRoaXMubWV0cmljKCdDb25zdW1lZFdyaXRlQ2FwYWNpdHlVbml0cycsIHsgc3RhdGlzdGljOiAnc3VtJywgLi4ucHJvcHN9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBNZXRyaWMgZm9yIHRoZSBzeXN0ZW0gZXJyb3JzIHRoaXMgdGFibGVcbiAgICpcbiAgICogQGRlZmF1bHQgc3VtIG92ZXIgYSBtaW51dGVcbiAgICovXG4gIHB1YmxpYyBtZXRyaWNTeXN0ZW1FcnJvcnMocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpOiBjbG91ZHdhdGNoLk1ldHJpYyB7XG4gICAgcmV0dXJuIHRoaXMubWV0cmljKCdTeXN0ZW1FcnJvcnMnLCB7IHN0YXRpc3RpYzogJ3N1bScsIC4uLnByb3BzfSk7XG4gIH1cblxuICAvKipcbiAgICogTWV0cmljIGZvciB0aGUgdXNlciBlcnJvcnMgdGhpcyB0YWJsZVxuICAgKlxuICAgKiBAZGVmYXVsdCBzdW0gb3ZlciBhIG1pbnV0ZVxuICAgKi9cbiAgcHVibGljIG1ldHJpY1VzZXJFcnJvcnMocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpOiBjbG91ZHdhdGNoLk1ldHJpYyB7XG4gICAgcmV0dXJuIHRoaXMubWV0cmljKCdVc2VyRXJyb3JzJywgeyBzdGF0aXN0aWM6ICdzdW0nLCAuLi5wcm9wc30pO1xuICB9XG5cbiAgLyoqXG4gICAqIE1ldHJpYyBmb3IgdGhlIGNvbmRpdGlvbmFsIGNoZWNrIGZhaWxlZCByZXF1ZXN0cyB0aGlzIHRhYmxlXG4gICAqXG4gICAqIEBkZWZhdWx0IHN1bSBvdmVyIGEgbWludXRlXG4gICAqL1xuICBwdWJsaWMgbWV0cmljQ29uZGl0aW9uYWxDaGVja0ZhaWxlZFJlcXVlc3RzKHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWMge1xuICAgIHJldHVybiB0aGlzLm1ldHJpYygnQ29uZGl0aW9uYWxDaGVja0ZhaWxlZFJlcXVlc3RzJywgeyBzdGF0aXN0aWM6ICdzdW0nLCAuLi5wcm9wc30pO1xuICB9XG5cbiAgLyoqXG4gICAqIE1ldHJpYyBmb3IgdGhlIHN1Y2Nlc3NmdWwgcmVxdWVzdCBsYXRlbmN5IHRoaXMgdGFibGVcbiAgICpcbiAgICogQGRlZmF1bHQgYXZnIG92ZXIgYSBtaW51dGVcbiAgICovXG4gIHB1YmxpYyBtZXRyaWNTdWNjZXNzZnVsUmVxdWVzdExhdGVuY3kocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpOiBjbG91ZHdhdGNoLk1ldHJpYyB7XG4gICAgcmV0dXJuIHRoaXMubWV0cmljKCdTdWNjZXNzZnVsUmVxdWVzdExhdGVuY3knLCB7IHN0YXRpc3RpYzogJ2F2ZycsIC4uLnByb3BzfSk7XG4gIH1cblxuICBwcm90ZWN0ZWQgYWJzdHJhY3QgZ2V0IGhhc0luZGV4KCk6IGJvb2xlYW47XG59XG5cbi8qKlxuICogUHJvdmlkZXMgYSBEeW5hbW9EQiB0YWJsZS5cbiAqL1xuZXhwb3J0IGNsYXNzIFRhYmxlIGV4dGVuZHMgVGFibGVCYXNlIHtcbiAgLyoqXG4gICAqIFBlcm1pdHMgYW4gSUFNIFByaW5jaXBhbCB0byBsaXN0IGFsbCBEeW5hbW9EQiBTdHJlYW1zLlxuICAgKiBAZGVwcmVjYXRlZCBVc2Uge0BsaW5rICNncmFudFRhYmxlTGlzdFN0cmVhbXN9IGZvciBtb3JlIGdyYW51bGFyIHBlcm1pc3Npb25cbiAgICogQHBhcmFtIGdyYW50ZWUgVGhlIHByaW5jaXBhbCAobm8tb3AgaWYgdW5kZWZpbmVkKVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBncmFudExpc3RTdHJlYW1zKGdyYW50ZWU6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50IHtcbiAgICByZXR1cm4gaWFtLkdyYW50LmFkZFRvUHJpbmNpcGFsKHtcbiAgICAgIGdyYW50ZWUsXG4gICAgICBhY3Rpb25zOiBbJ2R5bmFtb2RiOkxpc3RTdHJlYW1zJ10sXG4gICAgICByZXNvdXJjZUFybnM6IFsnKiddLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBUYWJsZSBjb25zdHJ1Y3QgdGhhdCByZXByZXNlbnRzIGFuIGV4dGVybmFsIHRhYmxlIHZpYSB0YWJsZSBuYW1lLlxuICAgKlxuICAgKiBAcGFyYW0gc2NvcGUgVGhlIHBhcmVudCBjcmVhdGluZyBjb25zdHJ1Y3QgKHVzdWFsbHkgYHRoaXNgKS5cbiAgICogQHBhcmFtIGlkIFRoZSBjb25zdHJ1Y3QncyBuYW1lLlxuICAgKiBAcGFyYW0gdGFibGVOYW1lIFRoZSB0YWJsZSdzIG5hbWUuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21UYWJsZU5hbWUoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgdGFibGVOYW1lOiBzdHJpbmcpOiBJVGFibGUge1xuICAgIHJldHVybiBUYWJsZS5mcm9tVGFibGVBdHRyaWJ1dGVzKHNjb3BlLCBpZCwgeyB0YWJsZU5hbWUgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIFRhYmxlIGNvbnN0cnVjdCB0aGF0IHJlcHJlc2VudHMgYW4gZXh0ZXJuYWwgdGFibGUgdmlhIHRhYmxlIGFybi5cbiAgICpcbiAgICogQHBhcmFtIHNjb3BlIFRoZSBwYXJlbnQgY3JlYXRpbmcgY29uc3RydWN0ICh1c3VhbGx5IGB0aGlzYCkuXG4gICAqIEBwYXJhbSBpZCBUaGUgY29uc3RydWN0J3MgbmFtZS5cbiAgICogQHBhcmFtIHRhYmxlQXJuIFRoZSB0YWJsZSdzIEFSTi5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVRhYmxlQXJuKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHRhYmxlQXJuOiBzdHJpbmcpOiBJVGFibGUge1xuICAgIHJldHVybiBUYWJsZS5mcm9tVGFibGVBdHRyaWJ1dGVzKHNjb3BlLCBpZCwgeyB0YWJsZUFybiB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgVGFibGUgY29uc3RydWN0IHRoYXQgcmVwcmVzZW50cyBhbiBleHRlcm5hbCB0YWJsZS5cbiAgICpcbiAgICogQHBhcmFtIHNjb3BlIFRoZSBwYXJlbnQgY3JlYXRpbmcgY29uc3RydWN0ICh1c3VhbGx5IGB0aGlzYCkuXG4gICAqIEBwYXJhbSBpZCBUaGUgY29uc3RydWN0J3MgbmFtZS5cbiAgICogQHBhcmFtIGF0dHJzIEEgYFRhYmxlQXR0cmlidXRlc2Agb2JqZWN0LlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tVGFibGVBdHRyaWJ1dGVzKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGF0dHJzOiBUYWJsZUF0dHJpYnV0ZXMpOiBJVGFibGUge1xuXG4gICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgVGFibGVCYXNlIHtcblxuICAgICAgcHVibGljIHJlYWRvbmx5IHRhYmxlTmFtZTogc3RyaW5nO1xuICAgICAgcHVibGljIHJlYWRvbmx5IHRhYmxlQXJuOiBzdHJpbmc7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgdGFibGVTdHJlYW1Bcm4/OiBzdHJpbmc7XG5cbiAgICAgIGNvbnN0cnVjdG9yKF90YWJsZUFybjogc3RyaW5nLCB0YWJsZU5hbWU6IHN0cmluZywgdGFibGVTdHJlYW1Bcm4/OiBzdHJpbmcpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICAgICAgdGhpcy50YWJsZUFybiA9IF90YWJsZUFybjtcbiAgICAgICAgdGhpcy50YWJsZU5hbWUgPSB0YWJsZU5hbWU7XG4gICAgICAgIHRoaXMudGFibGVTdHJlYW1Bcm4gPSB0YWJsZVN0cmVhbUFybjtcbiAgICAgIH1cblxuICAgICAgcHJvdGVjdGVkIGdldCBoYXNJbmRleCgpOiBib29sZWFuIHtcbiAgICAgICAgcmV0dXJuIGZhbHNlO1xuICAgICAgfVxuICAgIH1cblxuICAgIGxldCBuYW1lOiBzdHJpbmc7XG4gICAgbGV0IGFybjogc3RyaW5nO1xuICAgIGNvbnN0IHN0YWNrID0gU3RhY2sub2Yoc2NvcGUpO1xuICAgIGlmICghYXR0cnMudGFibGVOYW1lKSB7XG4gICAgICBpZiAoIWF0dHJzLnRhYmxlQXJuKSB7IHRocm93IG5ldyBFcnJvcignT25lIG9mIHRhYmxlTmFtZSBvciB0YWJsZUFybiBpcyByZXF1aXJlZCEnKTsgfVxuXG4gICAgICBhcm4gPSBhdHRycy50YWJsZUFybjtcbiAgICAgIGNvbnN0IG1heWJlVGFibGVOYW1lID0gc3RhY2sucGFyc2VBcm4oYXR0cnMudGFibGVBcm4pLnJlc291cmNlTmFtZTtcbiAgICAgIGlmICghbWF5YmVUYWJsZU5hbWUpIHsgdGhyb3cgbmV3IEVycm9yKCdBUk4gZm9yIER5bmFtb0RCIHRhYmxlIG11c3QgYmUgaW4gdGhlIGZvcm06IC4uLicpOyB9XG4gICAgICBuYW1lID0gbWF5YmVUYWJsZU5hbWU7XG4gICAgfSBlbHNlIHtcbiAgICAgIGlmIChhdHRycy50YWJsZUFybikgeyB0aHJvdyBuZXcgRXJyb3IoJ09ubHkgb25lIG9mIHRhYmxlQXJuIG9yIHRhYmxlTmFtZSBjYW4gYmUgcHJvdmlkZWQnKTsgfVxuICAgICAgbmFtZSA9IGF0dHJzLnRhYmxlTmFtZTtcbiAgICAgIGFybiA9IHN0YWNrLmZvcm1hdEFybih7XG4gICAgICAgIHNlcnZpY2U6ICdkeW5hbW9kYicsXG4gICAgICAgIHJlc291cmNlOiAndGFibGUnLFxuICAgICAgICByZXNvdXJjZU5hbWU6IGF0dHJzLnRhYmxlTmFtZSxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHJldHVybiBuZXcgSW1wb3J0KGFybiwgbmFtZSwgYXR0cnMudGFibGVTdHJlYW1Bcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB0YWJsZUFybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgdGFibGVOYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB0YWJsZVN0cmVhbUFybjogc3RyaW5nIHwgdW5kZWZpbmVkO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgdGFibGU6IENmblRhYmxlO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkga2V5U2NoZW1hID0gbmV3IEFycmF5PENmblRhYmxlLktleVNjaGVtYVByb3BlcnR5PigpO1xuICBwcml2YXRlIHJlYWRvbmx5IGF0dHJpYnV0ZURlZmluaXRpb25zID0gbmV3IEFycmF5PENmblRhYmxlLkF0dHJpYnV0ZURlZmluaXRpb25Qcm9wZXJ0eT4oKTtcbiAgcHJpdmF0ZSByZWFkb25seSBnbG9iYWxTZWNvbmRhcnlJbmRleGVzID0gbmV3IEFycmF5PENmblRhYmxlLkdsb2JhbFNlY29uZGFyeUluZGV4UHJvcGVydHk+KCk7XG4gIHByaXZhdGUgcmVhZG9ubHkgbG9jYWxTZWNvbmRhcnlJbmRleGVzID0gbmV3IEFycmF5PENmblRhYmxlLkxvY2FsU2Vjb25kYXJ5SW5kZXhQcm9wZXJ0eT4oKTtcblxuICBwcml2YXRlIHJlYWRvbmx5IHNlY29uZGFyeUluZGV4TmFtZXMgPSBuZXcgU2V0PHN0cmluZz4oKTtcbiAgcHJpdmF0ZSByZWFkb25seSBub25LZXlBdHRyaWJ1dGVzID0gbmV3IFNldDxzdHJpbmc+KCk7XG5cbiAgcHJpdmF0ZSByZWFkb25seSB0YWJsZVBhcnRpdGlvbktleTogQXR0cmlidXRlO1xuICBwcml2YXRlIHJlYWRvbmx5IHRhYmxlU29ydEtleT86IEF0dHJpYnV0ZTtcblxuICBwcml2YXRlIHJlYWRvbmx5IGJpbGxpbmdNb2RlOiBCaWxsaW5nTW9kZTtcbiAgcHJpdmF0ZSByZWFkb25seSB0YWJsZVNjYWxpbmc6IFNjYWxhYmxlQXR0cmlidXRlUGFpciA9IHt9O1xuICBwcml2YXRlIHJlYWRvbmx5IGluZGV4U2NhbGluZyA9IG5ldyBNYXA8c3RyaW5nLCBTY2FsYWJsZUF0dHJpYnV0ZVBhaXI+KCk7XG4gIHByaXZhdGUgcmVhZG9ubHkgc2NhbGluZ1JvbGU6IGlhbS5JUm9sZTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogVGFibGVQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgcGh5c2ljYWxOYW1lOiBwcm9wcy50YWJsZU5hbWUsXG4gICAgfSk7XG5cbiAgICB0aGlzLmJpbGxpbmdNb2RlID0gcHJvcHMuYmlsbGluZ01vZGUgfHwgQmlsbGluZ01vZGUuUFJPVklTSU9ORUQ7XG4gICAgdGhpcy52YWxpZGF0ZVByb3Zpc2lvbmluZyhwcm9wcyk7XG5cbiAgICBsZXQgc3RyZWFtU3BlY2lmaWNhdGlvbjogQ2ZuVGFibGUuU3RyZWFtU3BlY2lmaWNhdGlvblByb3BlcnR5IHwgdW5kZWZpbmVkO1xuICAgIGlmIChwcm9wcy5yZXBsaWNhdGlvblJlZ2lvbnMpIHtcbiAgICAgIGlmIChwcm9wcy5zdHJlYW0gJiYgcHJvcHMuc3RyZWFtICE9PSBTdHJlYW1WaWV3VHlwZS5ORVdfQU5EX09MRF9JTUFHRVMpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdgc3RyZWFtYCBtdXN0IGJlIHNldCB0byBgTkVXX0FORF9PTERfSU1BR0VTYCB3aGVuIHNwZWNpZnlpbmcgYHJlcGxpY2F0aW9uUmVnaW9uc2AnKTtcbiAgICAgIH1cbiAgICAgIHN0cmVhbVNwZWNpZmljYXRpb24gPSB7IHN0cmVhbVZpZXdUeXBlOiBTdHJlYW1WaWV3VHlwZS5ORVdfQU5EX09MRF9JTUFHRVMgfTtcblxuICAgICAgaWYgKHByb3BzLmJpbGxpbmdNb2RlICYmIHByb3BzLmJpbGxpbmdNb2RlICE9PSBCaWxsaW5nTW9kZS5QQVlfUEVSX1JFUVVFU1QpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdUaGUgYFBBWV9QRVJfUkVRVUVTVGAgYmlsbGluZyBtb2RlIG11c3QgYmUgdXNlZCB3aGVuIHNwZWNpZnlpbmcgYHJlcGxpY2F0aW9uUmVnaW9uc2AnKTtcbiAgICAgIH1cbiAgICAgIHRoaXMuYmlsbGluZ01vZGUgPSBCaWxsaW5nTW9kZS5QQVlfUEVSX1JFUVVFU1Q7XG4gICAgfSBlbHNlIGlmIChwcm9wcy5zdHJlYW0pIHtcbiAgICAgIHN0cmVhbVNwZWNpZmljYXRpb24gPSB7IHN0cmVhbVZpZXdUeXBlIDogcHJvcHMuc3RyZWFtIH07XG4gICAgfVxuXG4gICAgdGhpcy50YWJsZSA9IG5ldyBDZm5UYWJsZSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICB0YWJsZU5hbWU6IHRoaXMucGh5c2ljYWxOYW1lLFxuICAgICAga2V5U2NoZW1hOiB0aGlzLmtleVNjaGVtYSxcbiAgICAgIGF0dHJpYnV0ZURlZmluaXRpb25zOiB0aGlzLmF0dHJpYnV0ZURlZmluaXRpb25zLFxuICAgICAgZ2xvYmFsU2Vjb25kYXJ5SW5kZXhlczogTGF6eS5hbnlWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IHRoaXMuZ2xvYmFsU2Vjb25kYXJ5SW5kZXhlcyB9LCB7IG9taXRFbXB0eUFycmF5OiB0cnVlIH0pLFxuICAgICAgbG9jYWxTZWNvbmRhcnlJbmRleGVzOiBMYXp5LmFueVZhbHVlKHsgcHJvZHVjZTogKCkgPT4gdGhpcy5sb2NhbFNlY29uZGFyeUluZGV4ZXMgfSwgeyBvbWl0RW1wdHlBcnJheTogdHJ1ZSB9KSxcbiAgICAgIHBvaW50SW5UaW1lUmVjb3ZlcnlTcGVjaWZpY2F0aW9uOiBwcm9wcy5wb2ludEluVGltZVJlY292ZXJ5ID8geyBwb2ludEluVGltZVJlY292ZXJ5RW5hYmxlZDogcHJvcHMucG9pbnRJblRpbWVSZWNvdmVyeSB9IDogdW5kZWZpbmVkLFxuICAgICAgYmlsbGluZ01vZGU6IHRoaXMuYmlsbGluZ01vZGUgPT09IEJpbGxpbmdNb2RlLlBBWV9QRVJfUkVRVUVTVCA/IHRoaXMuYmlsbGluZ01vZGUgOiB1bmRlZmluZWQsXG4gICAgICBwcm92aXNpb25lZFRocm91Z2hwdXQ6IHRoaXMuYmlsbGluZ01vZGUgPT09IEJpbGxpbmdNb2RlLlBBWV9QRVJfUkVRVUVTVCA/IHVuZGVmaW5lZCA6IHtcbiAgICAgICAgcmVhZENhcGFjaXR5VW5pdHM6IHByb3BzLnJlYWRDYXBhY2l0eSB8fCA1LFxuICAgICAgICB3cml0ZUNhcGFjaXR5VW5pdHM6IHByb3BzLndyaXRlQ2FwYWNpdHkgfHwgNVxuICAgICAgfSxcbiAgICAgIHNzZVNwZWNpZmljYXRpb246IHByb3BzLnNlcnZlclNpZGVFbmNyeXB0aW9uID8geyBzc2VFbmFibGVkOiBwcm9wcy5zZXJ2ZXJTaWRlRW5jcnlwdGlvbiB9IDogdW5kZWZpbmVkLFxuICAgICAgc3RyZWFtU3BlY2lmaWNhdGlvbixcbiAgICAgIHRpbWVUb0xpdmVTcGVjaWZpY2F0aW9uOiBwcm9wcy50aW1lVG9MaXZlQXR0cmlidXRlID8geyBhdHRyaWJ1dGVOYW1lOiBwcm9wcy50aW1lVG9MaXZlQXR0cmlidXRlLCBlbmFibGVkOiB0cnVlIH0gOiB1bmRlZmluZWRcbiAgICB9KTtcbiAgICB0aGlzLnRhYmxlLmFwcGx5UmVtb3ZhbFBvbGljeShwcm9wcy5yZW1vdmFsUG9saWN5KTtcblxuICAgIGlmIChwcm9wcy50YWJsZU5hbWUpIHsgdGhpcy5ub2RlLmFkZE1ldGFkYXRhKCdhd3M6Y2RrOmhhc1BoeXNpY2FsTmFtZScsIHByb3BzLnRhYmxlTmFtZSk7IH1cblxuICAgIHRoaXMudGFibGVBcm4gPSB0aGlzLmdldFJlc291cmNlQXJuQXR0cmlidXRlKHRoaXMudGFibGUuYXR0ckFybiwge1xuICAgICAgc2VydmljZTogJ2R5bmFtb2RiJyxcbiAgICAgIHJlc291cmNlOiAndGFibGUnLFxuICAgICAgcmVzb3VyY2VOYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICB9KTtcbiAgICB0aGlzLnRhYmxlTmFtZSA9IHRoaXMuZ2V0UmVzb3VyY2VOYW1lQXR0cmlidXRlKHRoaXMudGFibGUucmVmKTtcblxuICAgIHRoaXMudGFibGVTdHJlYW1Bcm4gPSBzdHJlYW1TcGVjaWZpY2F0aW9uID8gdGhpcy50YWJsZS5hdHRyU3RyZWFtQXJuIDogdW5kZWZpbmVkO1xuXG4gICAgdGhpcy5zY2FsaW5nUm9sZSA9IHRoaXMubWFrZVNjYWxpbmdSb2xlKCk7XG5cbiAgICB0aGlzLmFkZEtleShwcm9wcy5wYXJ0aXRpb25LZXksIEhBU0hfS0VZX1RZUEUpO1xuICAgIHRoaXMudGFibGVQYXJ0aXRpb25LZXkgPSBwcm9wcy5wYXJ0aXRpb25LZXk7XG5cbiAgICBpZiAocHJvcHMuc29ydEtleSkge1xuICAgICAgdGhpcy5hZGRLZXkocHJvcHMuc29ydEtleSwgUkFOR0VfS0VZX1RZUEUpO1xuICAgICAgdGhpcy50YWJsZVNvcnRLZXkgPSBwcm9wcy5zb3J0S2V5O1xuICAgIH1cblxuICAgIGlmIChwcm9wcy5yZXBsaWNhdGlvblJlZ2lvbnMpIHtcbiAgICAgIHRoaXMuY3JlYXRlUmVwbGljYVRhYmxlcyhwcm9wcy5yZXBsaWNhdGlvblJlZ2lvbnMpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBnbG9iYWwgc2Vjb25kYXJ5IGluZGV4IG9mIHRhYmxlLlxuICAgKlxuICAgKiBAcGFyYW0gcHJvcHMgdGhlIHByb3BlcnR5IG9mIGdsb2JhbCBzZWNvbmRhcnkgaW5kZXhcbiAgICovXG4gIHB1YmxpYyBhZGRHbG9iYWxTZWNvbmRhcnlJbmRleChwcm9wczogR2xvYmFsU2Vjb25kYXJ5SW5kZXhQcm9wcykge1xuICAgIHRoaXMudmFsaWRhdGVQcm92aXNpb25pbmcocHJvcHMpO1xuICAgIHRoaXMudmFsaWRhdGVJbmRleE5hbWUocHJvcHMuaW5kZXhOYW1lKTtcblxuICAgIC8vIGJ1aWxkIGtleSBzY2hlbWEgYW5kIHByb2plY3Rpb24gZm9yIGluZGV4XG4gICAgY29uc3QgZ3NpS2V5U2NoZW1hID0gdGhpcy5idWlsZEluZGV4S2V5U2NoZW1hKHByb3BzLnBhcnRpdGlvbktleSwgcHJvcHMuc29ydEtleSk7XG4gICAgY29uc3QgZ3NpUHJvamVjdGlvbiA9IHRoaXMuYnVpbGRJbmRleFByb2plY3Rpb24ocHJvcHMpO1xuXG4gICAgdGhpcy5zZWNvbmRhcnlJbmRleE5hbWVzLmFkZChwcm9wcy5pbmRleE5hbWUpO1xuICAgIHRoaXMuZ2xvYmFsU2Vjb25kYXJ5SW5kZXhlcy5wdXNoKHtcbiAgICAgIGluZGV4TmFtZTogcHJvcHMuaW5kZXhOYW1lLFxuICAgICAga2V5U2NoZW1hOiBnc2lLZXlTY2hlbWEsXG4gICAgICBwcm9qZWN0aW9uOiBnc2lQcm9qZWN0aW9uLFxuICAgICAgcHJvdmlzaW9uZWRUaHJvdWdocHV0OiB0aGlzLmJpbGxpbmdNb2RlID09PSBCaWxsaW5nTW9kZS5QQVlfUEVSX1JFUVVFU1QgPyB1bmRlZmluZWQgOiB7XG4gICAgICAgIHJlYWRDYXBhY2l0eVVuaXRzOiBwcm9wcy5yZWFkQ2FwYWNpdHkgfHwgNSxcbiAgICAgICAgd3JpdGVDYXBhY2l0eVVuaXRzOiBwcm9wcy53cml0ZUNhcGFjaXR5IHx8IDVcbiAgICAgIH1cbiAgICB9KTtcblxuICAgIHRoaXMuaW5kZXhTY2FsaW5nLnNldChwcm9wcy5pbmRleE5hbWUsIHt9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBsb2NhbCBzZWNvbmRhcnkgaW5kZXggb2YgdGFibGUuXG4gICAqXG4gICAqIEBwYXJhbSBwcm9wcyB0aGUgcHJvcGVydHkgb2YgbG9jYWwgc2Vjb25kYXJ5IGluZGV4XG4gICAqL1xuICBwdWJsaWMgYWRkTG9jYWxTZWNvbmRhcnlJbmRleChwcm9wczogTG9jYWxTZWNvbmRhcnlJbmRleFByb3BzKSB7XG4gICAgLy8gaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9MaW1pdHMuaHRtbCNsaW1pdHMtc2Vjb25kYXJ5LWluZGV4ZXNcbiAgICBpZiAodGhpcy5sb2NhbFNlY29uZGFyeUluZGV4ZXMubGVuZ3RoID49IE1BWF9MT0NBTF9TRUNPTkRBUllfSU5ERVhfQ09VTlQpIHtcbiAgICAgIHRocm93IG5ldyBSYW5nZUVycm9yKGBhIG1heGltdW0gbnVtYmVyIG9mIGxvY2FsIHNlY29uZGFyeSBpbmRleCBwZXIgdGFibGUgaXMgJHtNQVhfTE9DQUxfU0VDT05EQVJZX0lOREVYX0NPVU5UfWApO1xuICAgIH1cblxuICAgIHRoaXMudmFsaWRhdGVJbmRleE5hbWUocHJvcHMuaW5kZXhOYW1lKTtcblxuICAgIC8vIGJ1aWxkIGtleSBzY2hlbWEgYW5kIHByb2plY3Rpb24gZm9yIGluZGV4XG4gICAgY29uc3QgbHNpS2V5U2NoZW1hID0gdGhpcy5idWlsZEluZGV4S2V5U2NoZW1hKHRoaXMudGFibGVQYXJ0aXRpb25LZXksIHByb3BzLnNvcnRLZXkpO1xuICAgIGNvbnN0IGxzaVByb2plY3Rpb24gPSB0aGlzLmJ1aWxkSW5kZXhQcm9qZWN0aW9uKHByb3BzKTtcblxuICAgIHRoaXMuc2Vjb25kYXJ5SW5kZXhOYW1lcy5hZGQocHJvcHMuaW5kZXhOYW1lKTtcbiAgICB0aGlzLmxvY2FsU2Vjb25kYXJ5SW5kZXhlcy5wdXNoKHtcbiAgICAgIGluZGV4TmFtZTogcHJvcHMuaW5kZXhOYW1lLFxuICAgICAga2V5U2NoZW1hOiBsc2lLZXlTY2hlbWEsXG4gICAgICBwcm9qZWN0aW9uOiBsc2lQcm9qZWN0aW9uXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogRW5hYmxlIHJlYWQgY2FwYWNpdHkgc2NhbGluZyBmb3IgdGhpcyB0YWJsZVxuICAgKlxuICAgKiBAcmV0dXJucyBBbiBvYmplY3QgdG8gY29uZmlndXJlIGFkZGl0aW9uYWwgQXV0b1NjYWxpbmcgc2V0dGluZ3NcbiAgICovXG4gIHB1YmxpYyBhdXRvU2NhbGVSZWFkQ2FwYWNpdHkocHJvcHM6IEVuYWJsZVNjYWxpbmdQcm9wcyk6IElTY2FsYWJsZVRhYmxlQXR0cmlidXRlIHtcbiAgICBpZiAodGhpcy50YWJsZVNjYWxpbmcuc2NhbGFibGVSZWFkQXR0cmlidXRlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1JlYWQgQXV0b1NjYWxpbmcgYWxyZWFkeSBlbmFibGVkIGZvciB0aGlzIHRhYmxlJyk7XG4gICAgfVxuICAgIGlmICh0aGlzLmJpbGxpbmdNb2RlID09PSBCaWxsaW5nTW9kZS5QQVlfUEVSX1JFUVVFU1QpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQXV0b1NjYWxpbmcgaXMgbm90IGF2YWlsYWJsZSBmb3IgdGFibGVzIHdpdGggUEFZX1BFUl9SRVFVRVNUIGJpbGxpbmcgbW9kZScpO1xuICAgIH1cblxuICAgIHJldHVybiB0aGlzLnRhYmxlU2NhbGluZy5zY2FsYWJsZVJlYWRBdHRyaWJ1dGUgPSBuZXcgU2NhbGFibGVUYWJsZUF0dHJpYnV0ZSh0aGlzLCAnUmVhZFNjYWxpbmcnLCB7XG4gICAgICBzZXJ2aWNlTmFtZXNwYWNlOiBhcHBzY2FsaW5nLlNlcnZpY2VOYW1lc3BhY2UuRFlOQU1PREIsXG4gICAgICByZXNvdXJjZUlkOiBgdGFibGUvJHt0aGlzLnRhYmxlTmFtZX1gLFxuICAgICAgZGltZW5zaW9uOiAnZHluYW1vZGI6dGFibGU6UmVhZENhcGFjaXR5VW5pdHMnLFxuICAgICAgcm9sZTogdGhpcy5zY2FsaW5nUm9sZSxcbiAgICAgIC4uLnByb3BzXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogRW5hYmxlIHdyaXRlIGNhcGFjaXR5IHNjYWxpbmcgZm9yIHRoaXMgdGFibGVcbiAgICpcbiAgICogQHJldHVybnMgQW4gb2JqZWN0IHRvIGNvbmZpZ3VyZSBhZGRpdGlvbmFsIEF1dG9TY2FsaW5nIHNldHRpbmdzIGZvciB0aGlzIGF0dHJpYnV0ZVxuICAgKi9cbiAgcHVibGljIGF1dG9TY2FsZVdyaXRlQ2FwYWNpdHkocHJvcHM6IEVuYWJsZVNjYWxpbmdQcm9wcyk6IElTY2FsYWJsZVRhYmxlQXR0cmlidXRlIHtcbiAgICBpZiAodGhpcy50YWJsZVNjYWxpbmcuc2NhbGFibGVXcml0ZUF0dHJpYnV0ZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdXcml0ZSBBdXRvU2NhbGluZyBhbHJlYWR5IGVuYWJsZWQgZm9yIHRoaXMgdGFibGUnKTtcbiAgICB9XG4gICAgaWYgKHRoaXMuYmlsbGluZ01vZGUgPT09IEJpbGxpbmdNb2RlLlBBWV9QRVJfUkVRVUVTVCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdBdXRvU2NhbGluZyBpcyBub3QgYXZhaWxhYmxlIGZvciB0YWJsZXMgd2l0aCBQQVlfUEVSX1JFUVVFU1QgYmlsbGluZyBtb2RlJyk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHRoaXMudGFibGVTY2FsaW5nLnNjYWxhYmxlV3JpdGVBdHRyaWJ1dGUgPSBuZXcgU2NhbGFibGVUYWJsZUF0dHJpYnV0ZSh0aGlzLCAnV3JpdGVTY2FsaW5nJywge1xuICAgICAgc2VydmljZU5hbWVzcGFjZTogYXBwc2NhbGluZy5TZXJ2aWNlTmFtZXNwYWNlLkRZTkFNT0RCLFxuICAgICAgcmVzb3VyY2VJZDogYHRhYmxlLyR7dGhpcy50YWJsZU5hbWV9YCxcbiAgICAgIGRpbWVuc2lvbjogJ2R5bmFtb2RiOnRhYmxlOldyaXRlQ2FwYWNpdHlVbml0cycsXG4gICAgICByb2xlOiB0aGlzLnNjYWxpbmdSb2xlLFxuICAgICAgLi4ucHJvcHMsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogRW5hYmxlIHJlYWQgY2FwYWNpdHkgc2NhbGluZyBmb3IgdGhlIGdpdmVuIEdTSVxuICAgKlxuICAgKiBAcmV0dXJucyBBbiBvYmplY3QgdG8gY29uZmlndXJlIGFkZGl0aW9uYWwgQXV0b1NjYWxpbmcgc2V0dGluZ3MgZm9yIHRoaXMgYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgYXV0b1NjYWxlR2xvYmFsU2Vjb25kYXJ5SW5kZXhSZWFkQ2FwYWNpdHkoaW5kZXhOYW1lOiBzdHJpbmcsIHByb3BzOiBFbmFibGVTY2FsaW5nUHJvcHMpOiBJU2NhbGFibGVUYWJsZUF0dHJpYnV0ZSB7XG4gICAgaWYgKHRoaXMuYmlsbGluZ01vZGUgPT09IEJpbGxpbmdNb2RlLlBBWV9QRVJfUkVRVUVTVCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdBdXRvU2NhbGluZyBpcyBub3QgYXZhaWxhYmxlIGZvciB0YWJsZXMgd2l0aCBQQVlfUEVSX1JFUVVFU1QgYmlsbGluZyBtb2RlJyk7XG4gICAgfVxuICAgIGNvbnN0IGF0dHJpYnV0ZVBhaXIgPSB0aGlzLmluZGV4U2NhbGluZy5nZXQoaW5kZXhOYW1lKTtcbiAgICBpZiAoIWF0dHJpYnV0ZVBhaXIpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgTm8gZ2xvYmFsIHNlY29uZGFyeSBpbmRleCB3aXRoIG5hbWUgJHtpbmRleE5hbWV9YCk7XG4gICAgfVxuICAgIGlmIChhdHRyaWJ1dGVQYWlyLnNjYWxhYmxlUmVhZEF0dHJpYnV0ZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdSZWFkIEF1dG9TY2FsaW5nIGFscmVhZHkgZW5hYmxlZCBmb3IgdGhpcyBpbmRleCcpO1xuICAgIH1cblxuICAgIHJldHVybiBhdHRyaWJ1dGVQYWlyLnNjYWxhYmxlUmVhZEF0dHJpYnV0ZSA9IG5ldyBTY2FsYWJsZVRhYmxlQXR0cmlidXRlKHRoaXMsIGAke2luZGV4TmFtZX1SZWFkU2NhbGluZ2AsIHtcbiAgICAgIHNlcnZpY2VOYW1lc3BhY2U6IGFwcHNjYWxpbmcuU2VydmljZU5hbWVzcGFjZS5EWU5BTU9EQixcbiAgICAgIHJlc291cmNlSWQ6IGB0YWJsZS8ke3RoaXMudGFibGVOYW1lfS9pbmRleC8ke2luZGV4TmFtZX1gLFxuICAgICAgZGltZW5zaW9uOiAnZHluYW1vZGI6aW5kZXg6UmVhZENhcGFjaXR5VW5pdHMnLFxuICAgICAgcm9sZTogdGhpcy5zY2FsaW5nUm9sZSxcbiAgICAgIC4uLnByb3BzXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogRW5hYmxlIHdyaXRlIGNhcGFjaXR5IHNjYWxpbmcgZm9yIHRoZSBnaXZlbiBHU0lcbiAgICpcbiAgICogQHJldHVybnMgQW4gb2JqZWN0IHRvIGNvbmZpZ3VyZSBhZGRpdGlvbmFsIEF1dG9TY2FsaW5nIHNldHRpbmdzIGZvciB0aGlzIGF0dHJpYnV0ZVxuICAgKi9cbiAgcHVibGljIGF1dG9TY2FsZUdsb2JhbFNlY29uZGFyeUluZGV4V3JpdGVDYXBhY2l0eShpbmRleE5hbWU6IHN0cmluZywgcHJvcHM6IEVuYWJsZVNjYWxpbmdQcm9wcyk6IElTY2FsYWJsZVRhYmxlQXR0cmlidXRlIHtcbiAgICBpZiAodGhpcy5iaWxsaW5nTW9kZSA9PT0gQmlsbGluZ01vZGUuUEFZX1BFUl9SRVFVRVNUKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0F1dG9TY2FsaW5nIGlzIG5vdCBhdmFpbGFibGUgZm9yIHRhYmxlcyB3aXRoIFBBWV9QRVJfUkVRVUVTVCBiaWxsaW5nIG1vZGUnKTtcbiAgICB9XG4gICAgY29uc3QgYXR0cmlidXRlUGFpciA9IHRoaXMuaW5kZXhTY2FsaW5nLmdldChpbmRleE5hbWUpO1xuICAgIGlmICghYXR0cmlidXRlUGFpcikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBObyBnbG9iYWwgc2Vjb25kYXJ5IGluZGV4IHdpdGggbmFtZSAke2luZGV4TmFtZX1gKTtcbiAgICB9XG4gICAgaWYgKGF0dHJpYnV0ZVBhaXIuc2NhbGFibGVXcml0ZUF0dHJpYnV0ZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdXcml0ZSBBdXRvU2NhbGluZyBhbHJlYWR5IGVuYWJsZWQgZm9yIHRoaXMgaW5kZXgnKTtcbiAgICB9XG5cbiAgICByZXR1cm4gYXR0cmlidXRlUGFpci5zY2FsYWJsZVdyaXRlQXR0cmlidXRlID0gbmV3IFNjYWxhYmxlVGFibGVBdHRyaWJ1dGUodGhpcywgYCR7aW5kZXhOYW1lfVdyaXRlU2NhbGluZ2AsIHtcbiAgICAgIHNlcnZpY2VOYW1lc3BhY2U6IGFwcHNjYWxpbmcuU2VydmljZU5hbWVzcGFjZS5EWU5BTU9EQixcbiAgICAgIHJlc291cmNlSWQ6IGB0YWJsZS8ke3RoaXMudGFibGVOYW1lfS9pbmRleC8ke2luZGV4TmFtZX1gLFxuICAgICAgZGltZW5zaW9uOiAnZHluYW1vZGI6aW5kZXg6V3JpdGVDYXBhY2l0eVVuaXRzJyxcbiAgICAgIHJvbGU6IHRoaXMuc2NhbGluZ1JvbGUsXG4gICAgICAuLi5wcm9wc1xuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFZhbGlkYXRlIHRoZSB0YWJsZSBjb25zdHJ1Y3QuXG4gICAqXG4gICAqIEByZXR1cm5zIGFuIGFycmF5IG9mIHZhbGlkYXRpb24gZXJyb3IgbWVzc2FnZVxuICAgKi9cbiAgcHJvdGVjdGVkIHZhbGlkYXRlKCk6IHN0cmluZ1tdIHtcbiAgICBjb25zdCBlcnJvcnMgPSBuZXcgQXJyYXk8c3RyaW5nPigpO1xuXG4gICAgaWYgKCF0aGlzLnRhYmxlUGFydGl0aW9uS2V5KSB7XG4gICAgICBlcnJvcnMucHVzaCgnYSBwYXJ0aXRpb24ga2V5IG11c3QgYmUgc3BlY2lmaWVkJyk7XG4gICAgfVxuICAgIGlmICh0aGlzLmxvY2FsU2Vjb25kYXJ5SW5kZXhlcy5sZW5ndGggPiAwICYmICF0aGlzLnRhYmxlU29ydEtleSkge1xuICAgICAgZXJyb3JzLnB1c2goJ2Egc29ydCBrZXkgb2YgdGhlIHRhYmxlIG11c3QgYmUgc3BlY2lmaWVkIHRvIGFkZCBsb2NhbCBzZWNvbmRhcnkgaW5kZXhlcycpO1xuICAgIH1cblxuICAgIHJldHVybiBlcnJvcnM7XG4gIH1cblxuICAvKipcbiAgICogVmFsaWRhdGUgcmVhZCBhbmQgd3JpdGUgY2FwYWNpdHkgYXJlIG5vdCBzcGVjaWZpZWQgZm9yIG9uLWRlbWFuZCB0YWJsZXMgKGJpbGxpbmcgbW9kZSBQQVlfUEVSX1JFUVVFU1QpLlxuICAgKlxuICAgKiBAcGFyYW0gcHJvcHMgcmVhZCBhbmQgd3JpdGUgY2FwYWNpdHkgcHJvcGVydGllc1xuICAgKi9cbiAgcHJpdmF0ZSB2YWxpZGF0ZVByb3Zpc2lvbmluZyhwcm9wczogeyByZWFkQ2FwYWNpdHk/OiBudW1iZXIsIHdyaXRlQ2FwYWNpdHk/OiBudW1iZXIgfSk6IHZvaWQge1xuICAgIGlmICh0aGlzLmJpbGxpbmdNb2RlID09PSBCaWxsaW5nTW9kZS5QQVlfUEVSX1JFUVVFU1QpIHtcbiAgICAgIGlmIChwcm9wcy5yZWFkQ2FwYWNpdHkgIT09IHVuZGVmaW5lZCB8fCBwcm9wcy53cml0ZUNhcGFjaXR5ICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCd5b3UgY2Fubm90IHByb3Zpc2lvbiByZWFkIGFuZCB3cml0ZSBjYXBhY2l0eSBmb3IgYSB0YWJsZSB3aXRoIFBBWV9QRVJfUkVRVUVTVCBiaWxsaW5nIG1vZGUnKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogVmFsaWRhdGUgaW5kZXggbmFtZSB0byBjaGVjayBpZiBhIGR1cGxpY2F0ZSBuYW1lIGFscmVhZHkgZXhpc3RzLlxuICAgKlxuICAgKiBAcGFyYW0gaW5kZXhOYW1lIGEgbmFtZSBvZiBnbG9iYWwgb3IgbG9jYWwgc2Vjb25kYXJ5IGluZGV4XG4gICAqL1xuICBwcml2YXRlIHZhbGlkYXRlSW5kZXhOYW1lKGluZGV4TmFtZTogc3RyaW5nKSB7XG4gICAgaWYgKHRoaXMuc2Vjb25kYXJ5SW5kZXhOYW1lcy5oYXMoaW5kZXhOYW1lKSkge1xuICAgICAgLy8gYSBkdXBsaWNhdGUgaW5kZXggbmFtZSBjYXVzZXMgdmFsaWRhdGlvbiBleGNlcHRpb24sIHN0YXR1cyBjb2RlIDQwMCwgd2hpbGUgdHJ5aW5nIHRvIGNyZWF0ZSBDRk4gc3RhY2tcbiAgICAgIHRocm93IG5ldyBFcnJvcihgYSBkdXBsaWNhdGUgaW5kZXggbmFtZSwgJHtpbmRleE5hbWV9LCBpcyBub3QgYWxsb3dlZGApO1xuICAgIH1cbiAgICB0aGlzLnNlY29uZGFyeUluZGV4TmFtZXMuYWRkKGluZGV4TmFtZSk7XG4gIH1cblxuICAvKipcbiAgICogVmFsaWRhdGUgbm9uLWtleSBhdHRyaWJ1dGVzIGJ5IGNoZWNraW5nIGxpbWl0cyB3aXRoaW4gc2Vjb25kYXJ5IGluZGV4LCB3aGljaCBtYXkgdmFyeSBpbiBmdXR1cmUuXG4gICAqXG4gICAqIEBwYXJhbSBub25LZXlBdHRyaWJ1dGVzIGEgbGlzdCBvZiBub24ta2V5IGF0dHJpYnV0ZSBuYW1lc1xuICAgKi9cbiAgcHJpdmF0ZSB2YWxpZGF0ZU5vbktleUF0dHJpYnV0ZXMobm9uS2V5QXR0cmlidXRlczogc3RyaW5nW10pIHtcbiAgICBpZiAodGhpcy5ub25LZXlBdHRyaWJ1dGVzLnNpemUgKyBub25LZXlBdHRyaWJ1dGVzLmxlbmd0aCA+IDIwKSB7XG4gICAgICAvLyBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1hem9uZHluYW1vZGIvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL0xpbWl0cy5odG1sI2xpbWl0cy1zZWNvbmRhcnktaW5kZXhlc1xuICAgICAgdGhyb3cgbmV3IFJhbmdlRXJyb3IoJ2EgbWF4aW11bSBudW1iZXIgb2Ygbm9uS2V5QXR0cmlidXRlcyBhY3Jvc3MgYWxsIG9mIHNlY29uZGFyeSBpbmRleGVzIGlzIDIwJyk7XG4gICAgfVxuXG4gICAgLy8gc3RvcmUgYWxsIG5vbi1rZXkgYXR0cmlidXRlc1xuICAgIG5vbktleUF0dHJpYnV0ZXMuZm9yRWFjaChhdHQgPT4gdGhpcy5ub25LZXlBdHRyaWJ1dGVzLmFkZChhdHQpKTtcbiAgfVxuXG4gIHByaXZhdGUgYnVpbGRJbmRleEtleVNjaGVtYShwYXJ0aXRpb25LZXk6IEF0dHJpYnV0ZSwgc29ydEtleT86IEF0dHJpYnV0ZSk6IENmblRhYmxlLktleVNjaGVtYVByb3BlcnR5W10ge1xuICAgIHRoaXMucmVnaXN0ZXJBdHRyaWJ1dGUocGFydGl0aW9uS2V5KTtcbiAgICBjb25zdCBpbmRleEtleVNjaGVtYTogQ2ZuVGFibGUuS2V5U2NoZW1hUHJvcGVydHlbXSA9IFtcbiAgICAgIHsgYXR0cmlidXRlTmFtZTogcGFydGl0aW9uS2V5Lm5hbWUsIGtleVR5cGU6IEhBU0hfS0VZX1RZUEUgfVxuICAgIF07XG5cbiAgICBpZiAoc29ydEtleSkge1xuICAgICAgdGhpcy5yZWdpc3RlckF0dHJpYnV0ZShzb3J0S2V5KTtcbiAgICAgIGluZGV4S2V5U2NoZW1hLnB1c2goeyBhdHRyaWJ1dGVOYW1lOiBzb3J0S2V5Lm5hbWUsIGtleVR5cGU6IFJBTkdFX0tFWV9UWVBFIH0pO1xuICAgIH1cblxuICAgIHJldHVybiBpbmRleEtleVNjaGVtYTtcbiAgfVxuXG4gIHByaXZhdGUgYnVpbGRJbmRleFByb2plY3Rpb24ocHJvcHM6IFNlY29uZGFyeUluZGV4UHJvcHMpOiBDZm5UYWJsZS5Qcm9qZWN0aW9uUHJvcGVydHkge1xuICAgIGlmIChwcm9wcy5wcm9qZWN0aW9uVHlwZSA9PT0gUHJvamVjdGlvblR5cGUuSU5DTFVERSAmJiAhcHJvcHMubm9uS2V5QXR0cmlidXRlcykge1xuICAgICAgLy8gaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXByb3BlcnRpZXMtZHluYW1vZGItcHJvamVjdGlvbm9iamVjdC5odG1sXG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYG5vbi1rZXkgYXR0cmlidXRlcyBzaG91bGQgYmUgc3BlY2lmaWVkIHdoZW4gdXNpbmcgJHtQcm9qZWN0aW9uVHlwZS5JTkNMVURFfSBwcm9qZWN0aW9uIHR5cGVgKTtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMucHJvamVjdGlvblR5cGUgIT09IFByb2plY3Rpb25UeXBlLklOQ0xVREUgJiYgcHJvcHMubm9uS2V5QXR0cmlidXRlcykge1xuICAgICAgLy8gdGhpcyBjb21iaW5hdGlvbiBjYXVzZXMgdmFsaWRhdGlvbiBleGNlcHRpb24sIHN0YXR1cyBjb2RlIDQwMCwgd2hpbGUgdHJ5aW5nIHRvIGNyZWF0ZSBDRk4gc3RhY2tcbiAgICAgIHRocm93IG5ldyBFcnJvcihgbm9uLWtleSBhdHRyaWJ1dGVzIHNob3VsZCBub3QgYmUgc3BlY2lmaWVkIHdoZW4gbm90IHVzaW5nICR7UHJvamVjdGlvblR5cGUuSU5DTFVERX0gcHJvamVjdGlvbiB0eXBlYCk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLm5vbktleUF0dHJpYnV0ZXMpIHtcbiAgICAgIHRoaXMudmFsaWRhdGVOb25LZXlBdHRyaWJ1dGVzKHByb3BzLm5vbktleUF0dHJpYnV0ZXMpO1xuICAgIH1cblxuICAgIHJldHVybiB7XG4gICAgICBwcm9qZWN0aW9uVHlwZTogcHJvcHMucHJvamVjdGlvblR5cGUgPyBwcm9wcy5wcm9qZWN0aW9uVHlwZSA6IFByb2plY3Rpb25UeXBlLkFMTCxcbiAgICAgIG5vbktleUF0dHJpYnV0ZXM6IHByb3BzLm5vbktleUF0dHJpYnV0ZXMgPyBwcm9wcy5ub25LZXlBdHRyaWJ1dGVzIDogdW5kZWZpbmVkXG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgZmluZEtleShrZXlUeXBlOiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5rZXlTY2hlbWEuZmluZChwcm9wID0+IHByb3Aua2V5VHlwZSA9PT0ga2V5VHlwZSk7XG4gIH1cblxuICBwcml2YXRlIGFkZEtleShhdHRyaWJ1dGU6IEF0dHJpYnV0ZSwga2V5VHlwZTogc3RyaW5nKSB7XG4gICAgY29uc3QgZXhpc3RpbmdQcm9wID0gdGhpcy5maW5kS2V5KGtleVR5cGUpO1xuICAgIGlmIChleGlzdGluZ1Byb3ApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgVW5hYmxlIHRvIHNldCAke2F0dHJpYnV0ZS5uYW1lfSBhcyBhICR7a2V5VHlwZX0ga2V5LCBiZWNhdXNlICR7ZXhpc3RpbmdQcm9wLmF0dHJpYnV0ZU5hbWV9IGlzIGEgJHtrZXlUeXBlfSBrZXlgKTtcbiAgICB9XG4gICAgdGhpcy5yZWdpc3RlckF0dHJpYnV0ZShhdHRyaWJ1dGUpO1xuICAgIHRoaXMua2V5U2NoZW1hLnB1c2goe1xuICAgICAgYXR0cmlidXRlTmFtZTogYXR0cmlidXRlLm5hbWUsXG4gICAgICBrZXlUeXBlXG4gICAgfSk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmVnaXN0ZXIgdGhlIGtleSBhdHRyaWJ1dGUgb2YgdGFibGUgb3Igc2Vjb25kYXJ5IGluZGV4IHRvIGFzc2VtYmxlIGF0dHJpYnV0ZSBkZWZpbml0aW9ucyBvZiBUYWJsZVJlc291cmNlUHJvcHMuXG4gICAqXG4gICAqIEBwYXJhbSBhdHRyaWJ1dGUgdGhlIGtleSBhdHRyaWJ1dGUgb2YgdGFibGUgb3Igc2Vjb25kYXJ5IGluZGV4XG4gICAqL1xuICBwcml2YXRlIHJlZ2lzdGVyQXR0cmlidXRlKGF0dHJpYnV0ZTogQXR0cmlidXRlKSB7XG4gICAgY29uc3QgeyBuYW1lLCB0eXBlIH0gPSBhdHRyaWJ1dGU7XG4gICAgY29uc3QgZXhpc3RpbmdEZWYgPSB0aGlzLmF0dHJpYnV0ZURlZmluaXRpb25zLmZpbmQoZGVmID0+IGRlZi5hdHRyaWJ1dGVOYW1lID09PSBuYW1lKTtcbiAgICBpZiAoZXhpc3RpbmdEZWYgJiYgZXhpc3RpbmdEZWYuYXR0cmlidXRlVHlwZSAhPT0gdHlwZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbmFibGUgdG8gc3BlY2lmeSAke25hbWV9IGFzICR7dHlwZX0gYmVjYXVzZSBpdCB3YXMgYWxyZWFkeSBkZWZpbmVkIGFzICR7ZXhpc3RpbmdEZWYuYXR0cmlidXRlVHlwZX1gKTtcbiAgICB9XG4gICAgaWYgKCFleGlzdGluZ0RlZikge1xuICAgICAgdGhpcy5hdHRyaWJ1dGVEZWZpbml0aW9ucy5wdXNoKHtcbiAgICAgICAgYXR0cmlidXRlTmFtZTogbmFtZSxcbiAgICAgICAgYXR0cmlidXRlVHlwZTogdHlwZVxuICAgICAgfSk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybiB0aGUgcm9sZSB0aGF0IHdpbGwgYmUgdXNlZCBmb3IgQXV0b1NjYWxpbmdcbiAgICovXG4gIHByaXZhdGUgbWFrZVNjYWxpbmdSb2xlKCk6IGlhbS5JUm9sZSB7XG4gICAgLy8gVXNlIGEgU2VydmljZSBMaW5rZWQgUm9sZS5cbiAgICAvLyBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXV0b3NjYWxpbmcvYXBwbGljYXRpb24vdXNlcmd1aWRlL2FwcGxpY2F0aW9uLWF1dG8tc2NhbGluZy1zZXJ2aWNlLWxpbmtlZC1yb2xlcy5odG1sXG4gICAgcmV0dXJuIGlhbS5Sb2xlLmZyb21Sb2xlQXJuKHRoaXMsICdTY2FsaW5nUm9sZScsIFN0YWNrLm9mKHRoaXMpLmZvcm1hdEFybih7XG4gICAgICBzZXJ2aWNlOiAnaWFtJyxcbiAgICAgIHJlZ2lvbjogJycsXG4gICAgICByZXNvdXJjZTogJ3JvbGUvYXdzLXNlcnZpY2Utcm9sZS9keW5hbW9kYi5hcHBsaWNhdGlvbi1hdXRvc2NhbGluZy5hbWF6b25hd3MuY29tJyxcbiAgICAgIHJlc291cmNlTmFtZTogJ0FXU1NlcnZpY2VSb2xlRm9yQXBwbGljYXRpb25BdXRvU2NhbGluZ19EeW5hbW9EQlRhYmxlJ1xuICAgIH0pKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIHJlcGxpY2EgdGFibGVzXG4gICAqXG4gICAqIEBwYXJhbSByZWdpb25zIHJlZ2lvbnMgd2hlcmUgdG8gY3JlYXRlIHRhYmxlc1xuICAgKi9cbiAgcHJpdmF0ZSBjcmVhdGVSZXBsaWNhVGFibGVzKHJlZ2lvbnM6IHN0cmluZ1tdKSB7XG4gICAgY29uc3Qgc3RhY2sgPSBTdGFjay5vZih0aGlzKTtcblxuICAgIGlmICghVG9rZW4uaXNVbnJlc29sdmVkKHN0YWNrLnJlZ2lvbikgJiYgcmVnaW9ucy5pbmNsdWRlcyhzdGFjay5yZWdpb24pKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ2ByZXBsaWNhdGlvblJlZ2lvbnNgIGNhbm5vdCBpbmNsdWRlIHRoZSByZWdpb24gd2hlcmUgdGhpcyBzdGFjayBpcyBkZXBsb3llZC4nKTtcbiAgICB9XG5cbiAgICBjb25zdCBwcm92aWRlciA9IFJlcGxpY2FQcm92aWRlci5nZXRPckNyZWF0ZSh0aGlzKTtcblxuICAgIC8vIERvY3VtZW50YXRpb24gYXQgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtYXpvbmR5bmFtb2RiL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9WMmd0X0lBTS5odG1sXG4gICAgLy8gaXMgY3VycmVudGx5IGluY29ycmVjdC4gQVdTIFN1cHBvcnQgcmVjb21tZW5kcyBgZHluYW1vZGI6KmAgaW4gYm90aCBzb3VyY2UgYW5kIGRlc3RpbmF0aW9uIHJlZ2lvbnNcblxuICAgIC8vIFBlcm1pc3Npb25zIGluIHRoZSBzb3VyY2UgcmVnaW9uXG4gICAgdGhpcy5ncmFudChwcm92aWRlci5vbkV2ZW50SGFuZGxlciwgJ2R5bmFtb2RiOionKTtcbiAgICB0aGlzLmdyYW50KHByb3ZpZGVyLmlzQ29tcGxldGVIYW5kbGVyLCAnZHluYW1vZGI6RGVzY3JpYmVUYWJsZScpO1xuXG4gICAgLy8gUGVybWlzc2lvbnMgaW4gdGhlIGRlc3RpbmF0aW9uIHJlZ2lvbnNcbiAgICBwcm92aWRlci5vbkV2ZW50SGFuZGxlci5hZGRUb1JvbGVQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgYWN0aW9uczogWydkeW5hbW9kYjoqJ10sXG4gICAgICByZXNvdXJjZXM6IHJlZ2lvbnMubWFwKHJlZ2lvbiA9PiBzdGFjay5mb3JtYXRBcm4oe1xuICAgICAgICByZWdpb24sXG4gICAgICAgIHNlcnZpY2U6ICdkeW5hbW9kYicsXG4gICAgICAgIHJlc291cmNlOiAndGFibGUnLFxuICAgICAgICByZXNvdXJjZU5hbWU6IHRoaXMudGFibGVOYW1lXG4gICAgICB9KSlcbiAgICB9KSk7XG5cbiAgICBsZXQgcHJldmlvdXNSZWdpb247XG4gICAgZm9yIChjb25zdCByZWdpb24gb2YgbmV3IFNldChyZWdpb25zKSkgeyAvLyBSZW1vdmUgZHVwbGljYXRlc1xuICAgICAgLy8gVXNlIG11bHRpcGxlIGN1c3RvbSByZXNvdXJjZXMgYmVjYXVzZSBtdWx0aXBsZSBjcmVhdGUvZGVsZXRlXG4gICAgICAvLyB1cGRhdGVzIGNhbm5vdCBiZSBjb21iaW5lZCBpbiBhIHNpbmdsZSBBUEkgY2FsbC5cbiAgICAgIGNvbnN0IGN1cnJlbnRSZWdpb24gPSBuZXcgQ3VzdG9tUmVzb3VyY2UodGhpcywgYFJlcGxpY2Eke3JlZ2lvbn1gLCB7XG4gICAgICAgIHByb3ZpZGVyOiBwcm92aWRlci5wcm92aWRlcixcbiAgICAgICAgcmVzb3VyY2VUeXBlOiAnQ3VzdG9tOjpEeW5hbW9EQlJlcGxpY2EnLFxuICAgICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgICAgVGFibGVOYW1lOiB0aGlzLnRhYmxlTmFtZSxcbiAgICAgICAgICBSZWdpb246IHJlZ2lvbixcbiAgICAgICAgfVxuICAgICAgfSk7XG5cbiAgICAgIC8vIERlcGxveSB0aW1lIGNoZWNrIHRvIHByZXZlbnQgZnJvbSBjcmVhdGluZyBhIHJlcGxpY2EgaW4gdGhlIHJlZ2lvblxuICAgICAgLy8gd2hlcmUgdGhpcyBzdGFjayBpcyBkZXBsb3llZC4gT25seSBuZWVkZWQgZm9yIGVudmlyb25tZW50IGFnbm9zdGljXG4gICAgICAvLyBzdGFja3MuXG4gICAgICBpZiAoVG9rZW4uaXNVbnJlc29sdmVkKHN0YWNrLnJlZ2lvbikpIHtcbiAgICAgICAgY29uc3QgY3JlYXRlUmVwbGljYSA9IG5ldyBDZm5Db25kaXRpb24odGhpcywgYFN0YWNrUmVnaW9uTm90RXF1YWxzJHtyZWdpb259YCwge1xuICAgICAgICAgIGV4cHJlc3Npb246IEZuLmNvbmRpdGlvbk5vdChGbi5jb25kaXRpb25FcXVhbHMocmVnaW9uLCBBd3MuUkVHSU9OKSlcbiAgICAgICAgfSk7XG4gICAgICAgIGNvbnN0IGNmbkN1c3RvbVJlc291cmNlID0gY3VycmVudFJlZ2lvbi5ub2RlLmRlZmF1bHRDaGlsZCBhcyBDZm5DdXN0b21SZXNvdXJjZTtcbiAgICAgICAgY2ZuQ3VzdG9tUmVzb3VyY2UuY2ZuT3B0aW9ucy5jb25kaXRpb24gPSBjcmVhdGVSZXBsaWNhO1xuICAgICAgfVxuXG4gICAgICAvLyBXZSBuZWVkIHRvIGNyZWF0ZS9kZWxldGUgcmVnaW9ucyBzZXF1ZW50aWFsbHkgYmVjYXVzZSB3ZSBjYW5ub3RcbiAgICAgIC8vIGhhdmUgbXVsdGlwbGUgdGFibGUgdXBkYXRlcyBhdCB0aGUgc2FtZSB0aW1lLiBUaGUgYGlzQ29tcGxldGVIYW5kbGVyYFxuICAgICAgLy8gb2YgdGhlIHByb3ZpZGVyIHdhaXRzIHVudGlsIHRoZSByZXBsaWNhIGlzIGFuIEFDVElWRSBzdGF0ZS5cbiAgICAgIGlmIChwcmV2aW91c1JlZ2lvbikge1xuICAgICAgICBjdXJyZW50UmVnaW9uLm5vZGUuYWRkRGVwZW5kZW5jeShwcmV2aW91c1JlZ2lvbik7XG4gICAgICB9XG4gICAgICBwcmV2aW91c1JlZ2lvbiA9IGN1cnJlbnRSZWdpb247XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgdGhpcyB0YWJsZSBoYXMgaW5kZXhlc1xuICAgKi9cbiAgcHJvdGVjdGVkIGdldCBoYXNJbmRleCgpOiBib29sZWFuIHtcbiAgICByZXR1cm4gdGhpcy5nbG9iYWxTZWNvbmRhcnlJbmRleGVzLmxlbmd0aCArIHRoaXMubG9jYWxTZWNvbmRhcnlJbmRleGVzLmxlbmd0aCA+IDA7XG4gIH1cbn1cblxuZXhwb3J0IGVudW0gQXR0cmlidXRlVHlwZSB7XG4gIC8qKiBVcCB0byA0MDBLaUIgb2YgYmluYXJ5IGRhdGEgKHdoaWNoIG11c3QgYmUgZW5jb2RlZCBhcyBiYXNlNjQgYmVmb3JlIHNlbmRpbmcgdG8gRHluYW1vREIpICovXG4gIEJJTkFSWSA9ICdCJyxcbiAgLyoqIE51bWVyaWMgdmFsdWVzIG1hZGUgb2YgdXAgdG8gMzggZGlnaXRzIChwb3NpdGl2ZSwgbmVnYXRpdmUgb3IgemVybykgKi9cbiAgTlVNQkVSID0gJ04nLFxuICAvKiogVXAgdG8gNDAwS2lCIG9mIFVURi04IGVuY29kZWQgdGV4dCAqL1xuICBTVFJJTkcgPSAnUycsXG59XG5cbi8qKlxuICogRHlhbm1vREIncyBSZWFkL1dyaXRlIGNhcGFjaXR5IG1vZGVzLlxuICovXG5leHBvcnQgZW51bSBCaWxsaW5nTW9kZSB7XG4gIC8qKlxuICAgKiBQYXkgb25seSBmb3Igd2hhdCB5b3UgdXNlLiBZb3UgZG9uJ3QgY29uZmlndXJlIFJlYWQvV3JpdGUgY2FwYWNpdHkgdW5pdHMuXG4gICAqL1xuICBQQVlfUEVSX1JFUVVFU1QgPSAnUEFZX1BFUl9SRVFVRVNUJyxcbiAgLyoqXG4gICAqIEV4cGxpY2l0bHkgc3BlY2lmaWVkIFJlYWQvV3JpdGUgY2FwYWNpdHkgdW5pdHMuXG4gICAqL1xuICBQUk9WSVNJT05FRCA9ICdQUk9WSVNJT05FRCcsXG59XG5cbmV4cG9ydCBlbnVtIFByb2plY3Rpb25UeXBlIHtcbiAgLyoqIE9ubHkgdGhlIGluZGV4IGFuZCBwcmltYXJ5IGtleXMgYXJlIHByb2plY3RlZCBpbnRvIHRoZSBpbmRleC4gKi9cbiAgS0VZU19PTkxZID0gJ0tFWVNfT05MWScsXG4gIC8qKiBPbmx5IHRoZSBzcGVjaWZpZWQgdGFibGUgYXR0cmlidXRlcyBhcmUgcHJvamVjdGVkIGludG8gdGhlIGluZGV4LiBUaGUgbGlzdCBvZiBwcm9qZWN0ZWQgYXR0cmlidXRlcyBpcyBpbiBgbm9uS2V5QXR0cmlidXRlc2AuICovXG4gIElOQ0xVREUgPSAnSU5DTFVERScsXG4gIC8qKiBBbGwgb2YgdGhlIHRhYmxlIGF0dHJpYnV0ZXMgYXJlIHByb2plY3RlZCBpbnRvIHRoZSBpbmRleC4gKi9cbiAgQUxMID0gJ0FMTCdcbn1cblxuLyoqXG4gKiBXaGVuIGFuIGl0ZW0gaW4gdGhlIHRhYmxlIGlzIG1vZGlmaWVkLCBTdHJlYW1WaWV3VHlwZSBkZXRlcm1pbmVzIHdoYXQgaW5mb3JtYXRpb25cbiAqIGlzIHdyaXR0ZW4gdG8gdGhlIHN0cmVhbSBmb3IgdGhpcyB0YWJsZS5cbiAqXG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbWF6b25keW5hbW9kYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdHJlYW1TcGVjaWZpY2F0aW9uLmh0bWxcbiAqL1xuZXhwb3J0IGVudW0gU3RyZWFtVmlld1R5cGUge1xuICAvKiogVGhlIGVudGlyZSBpdGVtLCBhcyBpdCBhcHBlYXJzIGFmdGVyIGl0IHdhcyBtb2RpZmllZCwgaXMgd3JpdHRlbiB0byB0aGUgc3RyZWFtLiAqL1xuICBORVdfSU1BR0UgPSAnTkVXX0lNQUdFJyxcbiAgLyoqIFRoZSBlbnRpcmUgaXRlbSwgYXMgaXQgYXBwZWFyZWQgYmVmb3JlIGl0IHdhcyBtb2RpZmllZCwgaXMgd3JpdHRlbiB0byB0aGUgc3RyZWFtLiAqL1xuICBPTERfSU1BR0UgPSAnT0xEX0lNQUdFJyxcbiAgLyoqIEJvdGggdGhlIG5ldyBhbmQgdGhlIG9sZCBpdGVtIGltYWdlcyBvZiB0aGUgaXRlbSBhcmUgd3JpdHRlbiB0byB0aGUgc3RyZWFtLiAqL1xuICBORVdfQU5EX09MRF9JTUFHRVMgPSAnTkVXX0FORF9PTERfSU1BR0VTJyxcbiAgLyoqIE9ubHkgdGhlIGtleSBhdHRyaWJ1dGVzIG9mIHRoZSBtb2RpZmllZCBpdGVtIGFyZSB3cml0dGVuIHRvIHRoZSBzdHJlYW0uICovXG4gIEtFWVNfT05MWSA9ICdLRVlTX09OTFknXG59XG5cbi8qKlxuICogSnVzdCBhIGNvbnZlbmllbnQgd2F5IHRvIGtlZXAgdHJhY2sgb2YgYm90aCBhdHRyaWJ1dGVzXG4gKi9cbmludGVyZmFjZSBTY2FsYWJsZUF0dHJpYnV0ZVBhaXIge1xuICBzY2FsYWJsZVJlYWRBdHRyaWJ1dGU/OiBTY2FsYWJsZVRhYmxlQXR0cmlidXRlO1xuICBzY2FsYWJsZVdyaXRlQXR0cmlidXRlPzogU2NhbGFibGVUYWJsZUF0dHJpYnV0ZTtcbn1cbiJdfQ==