from typing import Optional, Sequence

import sqlalchemy as sa
from sqlalchemy.schema import CreateTable

from dql.data_storage.sqlite import compile_statement


class ColumnMeta(type):
    def __getattr__(cls, name: str):
        return cls(name)


class Column(sa.ColumnClause, metaclass=ColumnMeta):  # pylint: disable=abstract-method
    inherit_cache: Optional[bool] = True

    def __init__(self, text: str) -> None:
        self.name = text
        super().__init__(text)

    def glob(self, glob_str):
        return self.op("GLOB")(glob_str)


def create_udf_table(
    conn, name: str, custom_columns: Sequence["sa.Column"] = ()
) -> "sa.Table":
    """Create a temporary table for storing custom signals generated by a UDF."""
    tbl = sa.Table(
        name,
        sa.MetaData(),
        sa.Column("id", sa.Integer, primary_key=True),
        *custom_columns,
        prefixes=["TEMPORARY"],
    )
    q = CreateTable(
        tbl,
        if_not_exists=True,
    )
    conn.execute(*compile_statement(q))
    return tbl


C = Column
