"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InterfaceVpcEndpoint = exports.InterfaceVpcEndpointAwsService = exports.InterfaceVpcEndpointService = exports.GatewayVpcEndpoint = exports.GatewayVpcEndpointAwsService = exports.VpcEndpointType = exports.VpcEndpoint = void 0;
const iam = require("@aws-cdk/aws-iam");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const peer_1 = require("./peer");
const port_1 = require("./port");
const security_group_1 = require("./security-group");
const util_1 = require("./util");
/**
 * @stability stable
 */
class VpcEndpoint extends core_1.Resource {
    /**
     * Adds a statement to the policy document of the VPC endpoint. The statement must have a Principal.
     *
     * Not all interface VPC endpoints support policy. For more information
     * see https://docs.aws.amazon.com/vpc/latest/userguide/vpce-interface.html
     *
     * @param statement the IAM statement to add.
     * @stability stable
     */
    addToPolicy(statement) {
        if (!statement.hasPrincipal) {
            throw new Error('Statement must have a `Principal`.');
        }
        if (!this.policyDocument) {
            this.policyDocument = new iam.PolicyDocument();
        }
        this.policyDocument.addStatements(statement);
    }
}
exports.VpcEndpoint = VpcEndpoint;
/**
 * The type of VPC endpoint.
 *
 * @stability stable
 */
var VpcEndpointType;
(function (VpcEndpointType) {
    VpcEndpointType["INTERFACE"] = "Interface";
    VpcEndpointType["GATEWAY"] = "Gateway";
})(VpcEndpointType = exports.VpcEndpointType || (exports.VpcEndpointType = {}));
/**
 * An AWS service for a gateway VPC endpoint.
 *
 * @stability stable
 */
class GatewayVpcEndpointAwsService {
    /**
     * @stability stable
     */
    constructor(name, prefix) {
        this.name = `${prefix || 'com.amazonaws'}.${core_1.Aws.REGION}.${name}`;
    }
}
exports.GatewayVpcEndpointAwsService = GatewayVpcEndpointAwsService;
/**
 * @stability stable
 */
GatewayVpcEndpointAwsService.DYNAMODB = new GatewayVpcEndpointAwsService('dynamodb');
/**
 * @stability stable
 */
GatewayVpcEndpointAwsService.S3 = new GatewayVpcEndpointAwsService('s3');
/**
 * A gateway VPC endpoint.
 *
 * @stability stable
 * @resource AWS::EC2::VPCEndpoint
 */
class GatewayVpcEndpoint extends VpcEndpoint {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        const subnets = props.subnets
            ? util_1.flatten(props.subnets.map(s => props.vpc.selectSubnets(s).subnets))
            : [...props.vpc.privateSubnets, ...props.vpc.publicSubnets, ...props.vpc.isolatedSubnets];
        const routeTableIds = util_1.allRouteTableIds(subnets);
        if (routeTableIds.length === 0) {
            throw new Error('Can\'t add a gateway endpoint to VPC; route table IDs are not available');
        }
        const endpoint = new ec2_generated_1.CfnVPCEndpoint(this, 'Resource', {
            policyDocument: core_1.Lazy.any({ produce: () => this.policyDocument }),
            routeTableIds,
            serviceName: props.service.name,
            vpcEndpointType: VpcEndpointType.GATEWAY,
            vpcId: props.vpc.vpcId,
        });
        this.vpcEndpointId = endpoint.ref;
        this.vpcEndpointCreationTimestamp = endpoint.attrCreationTimestamp;
        this.vpcEndpointDnsEntries = endpoint.attrDnsEntries;
        this.vpcEndpointNetworkInterfaceIds = endpoint.attrNetworkInterfaceIds;
    }
    /**
     * @stability stable
     */
    static fromGatewayVpcEndpointId(scope, id, gatewayVpcEndpointId) {
        class Import extends VpcEndpoint {
            constructor() {
                super(...arguments);
                this.vpcEndpointId = gatewayVpcEndpointId;
            }
        }
        return new Import(scope, id);
    }
}
exports.GatewayVpcEndpoint = GatewayVpcEndpoint;
/**
 * A custom-hosted service for an interface VPC endpoint.
 *
 * @stability stable
 */
class InterfaceVpcEndpointService {
    /**
     * @stability stable
     */
    constructor(name, port) {
        /**
         * Whether Private DNS is supported by default.
         *
         * @stability stable
         */
        this.privateDnsDefault = false;
        this.name = name;
        this.port = port || 443;
    }
}
exports.InterfaceVpcEndpointService = InterfaceVpcEndpointService;
/**
 * An AWS service for an interface VPC endpoint.
 *
 * @stability stable
 */
class InterfaceVpcEndpointAwsService {
    /**
     * @stability stable
     */
    constructor(name, prefix, port) {
        /**
         * Whether Private DNS is supported by default.
         *
         * @stability stable
         */
        this.privateDnsDefault = true;
        const region = core_1.Lazy.uncachedString({
            produce: (context) => core_1.Stack.of(context.scope).region,
        });
        this.name = `${prefix || 'com.amazonaws'}.${region}.${name}`;
        this.port = port || 443;
    }
}
exports.InterfaceVpcEndpointAwsService = InterfaceVpcEndpointAwsService;
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.SAGEMAKER_NOTEBOOK = new InterfaceVpcEndpointAwsService('notebook', 'aws.sagemaker');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.ATHENA = new InterfaceVpcEndpointAwsService('athena');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.CLOUDFORMATION = new InterfaceVpcEndpointAwsService('cloudformation');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.CLOUDTRAIL = new InterfaceVpcEndpointAwsService('cloudtrail');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.CODEBUILD = new InterfaceVpcEndpointAwsService('codebuild');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.CODEBUILD_FIPS = new InterfaceVpcEndpointAwsService('codebuild-fips');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.CODECOMMIT = new InterfaceVpcEndpointAwsService('codecommit');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.CODECOMMIT_FIPS = new InterfaceVpcEndpointAwsService('codecommit-fips');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.CODEPIPELINE = new InterfaceVpcEndpointAwsService('codepipeline');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.CONFIG = new InterfaceVpcEndpointAwsService('config');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.EC2 = new InterfaceVpcEndpointAwsService('ec2');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.EC2_MESSAGES = new InterfaceVpcEndpointAwsService('ec2messages');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.ECR = new InterfaceVpcEndpointAwsService('ecr.api');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.ECR_DOCKER = new InterfaceVpcEndpointAwsService('ecr.dkr');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.ECS = new InterfaceVpcEndpointAwsService('ecs');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.ECS_AGENT = new InterfaceVpcEndpointAwsService('ecs-agent');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.ECS_TELEMETRY = new InterfaceVpcEndpointAwsService('ecs-telemetry');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.ELASTIC_FILESYSTEM = new InterfaceVpcEndpointAwsService('elasticfilesystem');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.ELASTIC_FILESYSTEM_FIPS = new InterfaceVpcEndpointAwsService('elasticfilesystem-fips');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.ELASTIC_INFERENCE_RUNTIME = new InterfaceVpcEndpointAwsService('elastic-inference.runtime');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.ELASTIC_LOAD_BALANCING = new InterfaceVpcEndpointAwsService('elasticloadbalancing');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.CLOUDWATCH_EVENTS = new InterfaceVpcEndpointAwsService('events');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.APIGATEWAY = new InterfaceVpcEndpointAwsService('execute-api');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.CODECOMMIT_GIT = new InterfaceVpcEndpointAwsService('git-codecommit');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.CODECOMMIT_GIT_FIPS = new InterfaceVpcEndpointAwsService('git-codecommit-fips');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.GLUE = new InterfaceVpcEndpointAwsService('glue');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.KINESIS_STREAMS = new InterfaceVpcEndpointAwsService('kinesis-streams');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.KINESIS_FIREHOSE = new InterfaceVpcEndpointAwsService('kinesis-firehose');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.KMS = new InterfaceVpcEndpointAwsService('kms');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.CLOUDWATCH_LOGS = new InterfaceVpcEndpointAwsService('logs');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.CLOUDWATCH = new InterfaceVpcEndpointAwsService('monitoring');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.SAGEMAKER_API = new InterfaceVpcEndpointAwsService('sagemaker.api');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME = new InterfaceVpcEndpointAwsService('sagemaker.runtime');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME_FIPS = new InterfaceVpcEndpointAwsService('sagemaker.runtime-fips');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.SECRETS_MANAGER = new InterfaceVpcEndpointAwsService('secretsmanager');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.SERVICE_CATALOG = new InterfaceVpcEndpointAwsService('servicecatalog');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.SNS = new InterfaceVpcEndpointAwsService('sns');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.SQS = new InterfaceVpcEndpointAwsService('sqs');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.SSM = new InterfaceVpcEndpointAwsService('ssm');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.SSM_MESSAGES = new InterfaceVpcEndpointAwsService('ssmmessages');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.STS = new InterfaceVpcEndpointAwsService('sts');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.TRANSFER = new InterfaceVpcEndpointAwsService('transfer.server');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.STORAGE_GATEWAY = new InterfaceVpcEndpointAwsService('storagegateway');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.REKOGNITION = new InterfaceVpcEndpointAwsService('rekognition');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.REKOGNITION_FIPS = new InterfaceVpcEndpointAwsService('rekognition-fips');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.STEP_FUNCTIONS = new InterfaceVpcEndpointAwsService('states');
/**
 * @stability stable
 */
InterfaceVpcEndpointAwsService.LAMBDA = new InterfaceVpcEndpointAwsService('lambda');
/**
 * A interface VPC endpoint.
 *
 * @stability stable
 * @resource AWS::EC2::VPCEndpoint
 */
class InterfaceVpcEndpoint extends VpcEndpoint {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const securityGroups = props.securityGroups || [new security_group_1.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
            })];
        this.securityGroupId = securityGroups[0].securityGroupId;
        this.connections = new connections_1.Connections({
            defaultPort: port_1.Port.tcp(props.service.port),
            securityGroups,
        });
        if (props.open !== false) {
            this.connections.allowDefaultPortFrom(peer_1.Peer.ipv4(props.vpc.vpcCidrBlock));
        }
        // Determine which subnets to place the endpoint in
        const subnetIds = this.endpointSubnets(props);
        const endpoint = new ec2_generated_1.CfnVPCEndpoint(this, 'Resource', {
            privateDnsEnabled: (_b = (_a = props.privateDnsEnabled) !== null && _a !== void 0 ? _a : props.service.privateDnsDefault) !== null && _b !== void 0 ? _b : true,
            policyDocument: core_1.Lazy.any({ produce: () => this.policyDocument }),
            securityGroupIds: securityGroups.map(s => s.securityGroupId),
            serviceName: props.service.name,
            vpcEndpointType: VpcEndpointType.INTERFACE,
            subnetIds,
            vpcId: props.vpc.vpcId,
        });
        this.vpcEndpointId = endpoint.ref;
        this.vpcEndpointCreationTimestamp = endpoint.attrCreationTimestamp;
        this.vpcEndpointDnsEntries = endpoint.attrDnsEntries;
        this.vpcEndpointNetworkInterfaceIds = endpoint.attrNetworkInterfaceIds;
    }
    /**
     * Imports an existing interface VPC endpoint.
     *
     * @stability stable
     */
    static fromInterfaceVpcEndpointAttributes(scope, id, attrs) {
        const securityGroups = attrs.securityGroupId
            ? [security_group_1.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', attrs.securityGroupId)]
            : attrs.securityGroups;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.vpcEndpointId = attrs.vpcEndpointId;
                this.connections = new connections_1.Connections({
                    defaultPort: port_1.Port.tcp(attrs.port),
                    securityGroups,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Determine which subnets to place the endpoint in. This is in its own function
     * because there's a lot of code.
     */
    endpointSubnets(props) {
        var _a;
        const lookupSupportedAzs = (_a = props.lookupSupportedAzs) !== null && _a !== void 0 ? _a : false;
        const subnetSelection = props.vpc.selectSubnets({ ...props.subnets, onePerAz: true });
        // If we don't have an account/region, we will not be able to do filtering on AZs since
        // they will be undefined
        const agnosticAcct = core_1.Token.isUnresolved(this.stack.account);
        const agnosticRegion = core_1.Token.isUnresolved(this.stack.region);
        // Some service names, such as AWS service name references, use Tokens to automatically
        // fill in the region
        // If it is an InterfaceVpcEndpointAwsService, then the reference will be resolvable since
        // only references the region
        const isAwsService = core_1.Token.isUnresolved(props.service.name) && props.service instanceof InterfaceVpcEndpointAwsService;
        // Determine what name we pass to the context provider, either the verbatim name
        // or a resolved version if it is an AWS service reference
        let lookupServiceName = props.service.name;
        if (isAwsService && !agnosticRegion) {
            lookupServiceName = core_1.Stack.of(this).resolve(props.service.name);
        }
        else {
            // It's an agnostic service and we don't know how to resolve it.
            // This is ok if the stack is region agnostic and we're not looking up
            // AZs
            lookupServiceName = props.service.name;
        }
        // Check if lookup is impossible and throw an appropriate error
        // Context provider cannot make an AWS call without an account/region
        if ((agnosticAcct || agnosticRegion) && lookupSupportedAzs) {
            throw new Error('Cannot look up VPC endpoint availability zones if account/region are not specified');
        }
        // Context provider doesn't know the name of the service if there is a Token
        // in the name
        const agnosticService = core_1.Token.isUnresolved(lookupServiceName);
        if (agnosticService && lookupSupportedAzs) {
            throw new Error(`Cannot lookup AZs for a service name with a Token: ${props.service.name}`);
        }
        // Here we do the actual lookup for AZs, if told to do so
        let subnets;
        if (lookupSupportedAzs) {
            const availableAZs = this.availableAvailabilityZones(lookupServiceName);
            subnets = subnetSelection.subnets.filter(s => availableAZs.includes(s.availabilityZone));
        }
        else {
            subnets = subnetSelection.subnets;
        }
        const subnetIds = subnets.map(s => s.subnetId);
        return subnetIds;
    }
    availableAvailabilityZones(serviceName) {
        // Here we check what AZs the endpoint service is available in
        // If for whatever reason we can't retrieve the AZs, and no context is set,
        // we will fall back to all AZs
        const availableAZs = core_1.ContextProvider.getValue(this, {
            provider: cxschema.ContextProvider.ENDPOINT_SERVICE_AVAILABILITY_ZONE_PROVIDER,
            dummyValue: this.stack.availabilityZones,
            props: { serviceName },
        }).value;
        if (!Array.isArray(availableAZs)) {
            throw new Error(`Discovered AZs for endpoint service ${serviceName} must be an array`);
        }
        return availableAZs;
    }
}
exports.InterfaceVpcEndpoint = InterfaceVpcEndpoint;
//# sourceMappingURL=data:application/json;base64,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