import { IBucket } from '@aws-cdk/aws-s3';
import { Resource } from '@aws-cdk/core';
import { OperatingSystemType } from './machine-image';
/**
 * Options when constructing UserData for Linux.
 *
 * @stability stable
 */
export interface LinuxUserDataOptions {
    /**
     * Shebang for the UserData script.
     *
     * @default "#!/bin/bash"
     * @stability stable
     */
    readonly shebang?: string;
}
/**
 * Options when downloading files from S3.
 *
 * @stability stable
 */
export interface S3DownloadOptions {
    /**
     * Name of the S3 bucket to download from.
     *
     * @stability stable
     */
    readonly bucket: IBucket;
    /**
     * The key of the file to download.
     *
     * @stability stable
     */
    readonly bucketKey: string;
    /**
     * The name of the local file.
     *
     * @default Linux   - /tmp/bucketKey
     * Windows - %TEMP%/bucketKey
     * @stability stable
     */
    readonly localFile?: string;
}
/**
 * Options when executing a file.
 *
 * @stability stable
 */
export interface ExecuteFileOptions {
    /**
     * The path to the file.
     *
     * @stability stable
     */
    readonly filePath: string;
    /**
     * The arguments to be passed to the file.
     *
     * @default No arguments are passed to the file.
     * @stability stable
     */
    readonly arguments?: string;
}
/**
 * Instance User Data.
 *
 * @stability stable
 */
export declare abstract class UserData {
    /**
     * Create a userdata object for Linux hosts.
     *
     * @stability stable
     */
    static forLinux(options?: LinuxUserDataOptions): UserData;
    /**
     * Create a userdata object for Windows hosts.
     *
     * @stability stable
     */
    static forWindows(): UserData;
    /**
     * Create a userdata object with custom content.
     *
     * @stability stable
     */
    static custom(content: string): UserData;
    /**
     * @stability stable
     */
    static forOperatingSystem(os: OperatingSystemType): UserData;
    /**
     * Add one or more commands to the user data.
     *
     * @stability stable
     */
    abstract addCommands(...commands: string[]): void;
    /**
     * Add one or more commands to the user data that will run when the script exits.
     *
     * @stability stable
     */
    abstract addOnExitCommands(...commands: string[]): void;
    /**
     * Render the UserData for use in a construct.
     *
     * @stability stable
     */
    abstract render(): string;
    /**
     * Adds commands to download a file from S3.
     *
     * @returns : The local path that the file will be downloaded to
     * @stability stable
     */
    abstract addS3DownloadCommand(params: S3DownloadOptions): string;
    /**
     * Adds commands to execute a file.
     *
     * @stability stable
     */
    abstract addExecuteFileCommand(params: ExecuteFileOptions): void;
    /**
     * Adds a command which will send a cfn-signal when the user data script ends.
     *
     * @stability stable
     */
    abstract addSignalOnExitCommand(resource: Resource): void;
}
