"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecurityGroup = void 0;
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const SECURITY_GROUP_SYMBOL = Symbol.for('@aws-cdk/iam.SecurityGroup');
/**
 * A SecurityGroup that is not created in this template
 */
class SecurityGroupBase extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, props);
        /**
         * Whether the rule can be inlined into a SecurityGroup or not.
         *
         * @stability stable
         */
        this.canInlineRule = false;
        /**
         * @stability stable
         */
        this.connections = new connections_1.Connections({ securityGroups: [this] });
        Object.defineProperty(this, SECURITY_GROUP_SYMBOL, { value: true });
    }
    /**
     * Return whether the indicated object is a security group.
     *
     * @stability stable
     */
    static isSecurityGroup(x) {
        return SECURITY_GROUP_SYMBOL in x;
    }
    /**
     * A unique identifier for this connection peer.
     *
     * @stability stable
     */
    get uniqueId() {
        return core_1.Names.nodeUniqueId(this.node);
    }
    addIngressRule(peer, connection, description, remoteRule) {
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        const [scope, id] = determineRuleScope(this, peer, connection, 'from', remoteRule);
        // Skip duplicates
        if (scope.node.tryFindChild(id) === undefined) {
            new ec2_generated_1.CfnSecurityGroupIngress(scope, id, {
                groupId: this.securityGroupId,
                ...peer.toIngressRuleConfig(),
                ...connection.toRuleJson(),
                description,
            });
        }
    }
    addEgressRule(peer, connection, description, remoteRule) {
        if (description === undefined) {
            description = `to ${peer.uniqueId}:${connection}`;
        }
        const [scope, id] = determineRuleScope(this, peer, connection, 'to', remoteRule);
        // Skip duplicates
        if (scope.node.tryFindChild(id) === undefined) {
            new ec2_generated_1.CfnSecurityGroupEgress(scope, id, {
                groupId: this.securityGroupId,
                ...peer.toEgressRuleConfig(),
                ...connection.toRuleJson(),
                description,
            });
        }
    }
    /**
     * Produce the ingress rule JSON for the given connection.
     *
     * @stability stable
     */
    toIngressRuleConfig() {
        return { sourceSecurityGroupId: this.securityGroupId };
    }
    /**
     * Produce the egress rule JSON for the given connection.
     *
     * @stability stable
     */
    toEgressRuleConfig() {
        return { destinationSecurityGroupId: this.securityGroupId };
    }
}
/**
 * Determine where to parent a new ingress/egress rule
 *
 * A SecurityGroup rule is parented under the group it's related to, UNLESS
 * we're in a cross-stack scenario with another Security Group. In that case,
 * we respect the 'remoteRule' flag and will parent under the other security
 * group.
 *
 * This is necessary to avoid cyclic dependencies between stacks, since both
 * ingress and egress rules will reference both security groups, and a naive
 * parenting will lead to the following situation:
 *
 *   ╔════════════════════╗         ╔════════════════════╗
 *   ║  ┌───────────┐     ║         ║    ┌───────────┐   ║
 *   ║  │  GroupA   │◀────╬─┐   ┌───╬───▶│  GroupB   │   ║
 *   ║  └───────────┘     ║ │   │   ║    └───────────┘   ║
 *   ║        ▲           ║ │   │   ║          ▲         ║
 *   ║        │           ║ │   │   ║          │         ║
 *   ║        │           ║ │   │   ║          │         ║
 *   ║  ┌───────────┐     ║ └───┼───╬────┌───────────┐   ║
 *   ║  │  EgressA  │─────╬─────┘   ║    │ IngressB  │   ║
 *   ║  └───────────┘     ║         ║    └───────────┘   ║
 *   ║                    ║         ║                    ║
 *   ╚════════════════════╝         ╚════════════════════╝
 *
 * By having the ability to switch the parent, we avoid the cyclic reference by
 * keeping all rules in a single stack.
 *
 * If this happens, we also have to change the construct ID, because
 * otherwise we might have two objects with the same ID if we have
 * multiple reversed security group relationships.
 *
 *   ╔═══════════════════════════════════╗
 *   ║┌───────────┐                      ║
 *   ║│  GroupB   │                      ║
 *   ║└───────────┘                      ║
 *   ║      ▲                            ║
 *   ║      │              ┌───────────┐ ║
 *   ║      ├────"from A"──│ IngressB  │ ║
 *   ║      │              └───────────┘ ║
 *   ║      │              ┌───────────┐ ║
 *   ║      ├─────"to B"───│  EgressA  │ ║
 *   ║      │              └───────────┘ ║
 *   ║      │              ┌───────────┐ ║
 *   ║      └─────"to B"───│  EgressC  │ ║  <-- oops
 *   ║                     └───────────┘ ║
 *   ╚═══════════════════════════════════╝
 */
function determineRuleScope(group, peer, connection, fromTo, remoteRule) {
    if (remoteRule && SecurityGroupBase.isSecurityGroup(peer) && differentStacks(group, peer)) {
        // Reversed
        const reversedFromTo = fromTo === 'from' ? 'to' : 'from';
        return [peer, `${group.uniqueId}:${connection} ${reversedFromTo}`];
    }
    else {
        // Regular (do old ID escaping to in order to not disturb existing deployments)
        return [group, `${fromTo} ${renderPeer(peer)}:${connection}`.replace('/', '_')];
    }
}
function renderPeer(peer) {
    return core_1.Token.isUnresolved(peer.uniqueId) ? '{IndirectPeer}' : peer.uniqueId;
}
function differentStacks(group1, group2) {
    return core_1.Stack.of(group1) !== core_1.Stack.of(group2);
}
/**
 * Creates an Amazon EC2 security group within a VPC.
 *
 * Security Groups act like a firewall with a set of rules, and are associated
 * with any AWS resource that has or creates Elastic Network Interfaces (ENIs).
 * A typical example of a resource that has a security group is an Instance (or
 * Auto Scaling Group of instances)
 *
 * If you are defining new infrastructure in CDK, there is a good chance you
 * won't have to interact with this class at all. Like IAM Roles, Security
 * Groups need to exist to control access between AWS resources, but CDK will
 * automatically generate and populate them with least-privilege permissions
 * for you so you can concentrate on your business logic.
 *
 * All Constructs that require Security Groups will create one for you if you
 * don't specify one at construction. After construction, you can selectively
 * allow connections to and between constructs via--for example-- the `instance.connections`
 * object. Think of it as "allowing connections to your instance", rather than
 * "adding ingress rules a security group". See the [Allowing
 * Connections](https://docs.aws.amazon.com/cdk/api/latest/docs/aws-ec2-readme.html#allowing-connections)
 * section in the library documentation for examples.
 *
 * Direct manipulation of the Security Group through `addIngressRule` and
 * `addEgressRule` is possible, but mutation through the `.connections` object
 * is recommended. If you peer two constructs with security groups this way,
 * appropriate rules will be created in both.
 *
 * If you have an existing security group you want to use in your CDK application,
 * you would import it like this:
 *
 * ```ts
 * const securityGroup = SecurityGroup.fromSecurityGroupId(this, 'SG', 'sg-12345', {
 *    mutable: false
 * });
 * ```
 *
 * @stability stable
 */
class SecurityGroup extends SecurityGroupBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.securityGroupName,
        });
        this.directIngressRules = [];
        this.directEgressRules = [];
        const groupDescription = props.description || this.node.path;
        this.allowAllOutbound = props.allowAllOutbound !== false;
        this.securityGroup = new ec2_generated_1.CfnSecurityGroup(this, 'Resource', {
            groupName: this.physicalName,
            groupDescription,
            securityGroupIngress: core_1.Lazy.any({ produce: () => this.directIngressRules }, { omitEmptyArray: true }),
            securityGroupEgress: core_1.Lazy.any({ produce: () => this.directEgressRules }, { omitEmptyArray: true }),
            vpcId: props.vpc.vpcId,
        });
        this.securityGroupId = this.securityGroup.attrGroupId;
        this.securityGroupVpcId = this.securityGroup.attrVpcId;
        this.securityGroupName = this.securityGroup.ref;
        this.addDefaultEgressRule();
    }
    /**
     * Look up a security group by id.
     *
     * @stability stable
     */
    static fromLookup(scope, id, securityGroupId) {
        if (core_1.Token.isUnresolved(securityGroupId)) {
            throw new Error('All arguments to look up a security group must be concrete (no Tokens)');
        }
        const attributes = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.SECURITY_GROUP_PROVIDER,
            props: { securityGroupId },
            dummyValue: {
                securityGroupId: 'sg-12345',
                allowAllOutbound: true,
            },
        }).value;
        return SecurityGroup.fromSecurityGroupId(scope, id, attributes.securityGroupId, {
            allowAllOutbound: attributes.allowAllOutbound,
            mutable: true,
        });
    }
    /**
     * Import an existing security group into this app.
     *
     * @stability stable
     */
    static fromSecurityGroupId(scope, id, securityGroupId, options = {}) {
        class MutableImport extends SecurityGroupBase {
            constructor() {
                var _a;
                super(...arguments);
                this.securityGroupId = securityGroupId;
                this.allowAllOutbound = (_a = options.allowAllOutbound) !== null && _a !== void 0 ? _a : true;
            }
            addEgressRule(peer, connection, description, remoteRule) {
                // Only if allowAllOutbound has been disabled
                if (options.allowAllOutbound === false) {
                    super.addEgressRule(peer, connection, description, remoteRule);
                }
            }
        }
        class ImmutableImport extends SecurityGroupBase {
            constructor() {
                var _a;
                super(...arguments);
                this.securityGroupId = securityGroupId;
                this.allowAllOutbound = (_a = options.allowAllOutbound) !== null && _a !== void 0 ? _a : true;
            }
            addEgressRule(_peer, _connection, _description, _remoteRule) {
                // do nothing
            }
            addIngressRule(_peer, _connection, _description, _remoteRule) {
                // do nothing
            }
        }
        return options.mutable !== false
            ? new MutableImport(scope, id)
            : new ImmutableImport(scope, id);
    }
    /**
     * Add an ingress rule for the current security group.
     *
     * `remoteRule` controls where the Rule object is created if the peer is also a
     * securityGroup and they are in different stack. If false (default) the
     * rule object is created under the current SecurityGroup object. If true and the
     * peer is also a SecurityGroup, the rule object is created under the remote
     * SecurityGroup object.
     *
     * @stability stable
     */
    addIngressRule(peer, connection, description, remoteRule) {
        if (!peer.canInlineRule || !connection.canInlineRule) {
            super.addIngressRule(peer, connection, description, remoteRule);
            return;
        }
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        this.addDirectIngressRule({
            ...peer.toIngressRuleConfig(),
            ...connection.toRuleJson(),
            description,
        });
    }
    /**
     * Add an egress rule for the current security group.
     *
     * `remoteRule` controls where the Rule object is created if the peer is also a
     * securityGroup and they are in different stack. If false (default) the
     * rule object is created under the current SecurityGroup object. If true and the
     * peer is also a SecurityGroup, the rule object is created under the remote
     * SecurityGroup object.
     *
     * @stability stable
     */
    addEgressRule(peer, connection, description, remoteRule) {
        if (this.allowAllOutbound) {
            // In the case of "allowAllOutbound", we don't add any more rules. There
            // is only one rule which allows all traffic and that subsumes any other
            // rule.
            core_1.Annotations.of(this).addWarning('Ignoring Egress rule since \'allowAllOutbound\' is set to true; To add customize rules, set allowAllOutbound=false on the SecurityGroup');
            return;
        }
        else {
            // Otherwise, if the bogus rule exists we can now remove it because the
            // presence of any other rule will get rid of EC2's implicit "all
            // outbound" rule anyway.
            this.removeNoTrafficRule();
        }
        if (!peer.canInlineRule || !connection.canInlineRule) {
            super.addEgressRule(peer, connection, description, remoteRule);
            return;
        }
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        const rule = {
            ...peer.toEgressRuleConfig(),
            ...connection.toRuleJson(),
            description,
        };
        if (isAllTrafficRule(rule)) {
            // We cannot allow this; if someone adds the rule in this way, it will be
            // removed again if they add other rules. We also can't automatically switch
            // to "allOutbound=true" mode, because we might have already emitted
            // EgressRule objects (which count as rules added later) and there's no way
            // to recall those. Better to prevent this for now.
            throw new Error('Cannot add an "all traffic" egress rule in this way; set allowAllOutbound=true on the SecurityGroup instead.');
        }
        this.addDirectEgressRule(rule);
    }
    /**
     * Add a direct ingress rule
     */
    addDirectIngressRule(rule) {
        if (!this.hasIngressRule(rule)) {
            this.directIngressRules.push(rule);
        }
    }
    /**
     * Return whether the given ingress rule exists on the group
     */
    hasIngressRule(rule) {
        return this.directIngressRules.findIndex(r => ingressRulesEqual(r, rule)) > -1;
    }
    /**
     * Add a direct egress rule
     */
    addDirectEgressRule(rule) {
        if (!this.hasEgressRule(rule)) {
            this.directEgressRules.push(rule);
        }
    }
    /**
     * Return whether the given egress rule exists on the group
     */
    hasEgressRule(rule) {
        return this.directEgressRules.findIndex(r => egressRulesEqual(r, rule)) > -1;
    }
    /**
     * Add the default egress rule to the securityGroup
     *
     * This depends on allowAllOutbound:
     *
     * - If allowAllOutbound is true, we *TECHNICALLY* don't need to do anything, because
     *   EC2 is going to create this default rule anyway. But, for maximum readability
     *   of the template, we will add one anyway.
     * - If allowAllOutbound is false, we add a bogus rule that matches no traffic in
     *   order to get rid of the default "all outbound" rule that EC2 creates by default.
     *   If other rules happen to get added later, we remove the bogus rule again so
     *   that it doesn't clutter up the template too much (even though that's not
     *   strictly necessary).
     */
    addDefaultEgressRule() {
        if (this.allowAllOutbound) {
            this.directEgressRules.push(ALLOW_ALL_RULE);
        }
        else {
            this.directEgressRules.push(MATCH_NO_TRAFFIC);
        }
    }
    /**
     * Remove the bogus rule if it exists
     */
    removeNoTrafficRule() {
        const i = this.directEgressRules.findIndex(r => egressRulesEqual(r, MATCH_NO_TRAFFIC));
        if (i > -1) {
            this.directEgressRules.splice(i, 1);
        }
    }
}
exports.SecurityGroup = SecurityGroup;
/**
 * Egress rule that purposely matches no traffic
 *
 * This is used in order to disable the "all traffic" default of Security Groups.
 *
 * No machine can ever actually have the 255.255.255.255 IP address, but
 * in order to lock it down even more we'll restrict to a nonexistent
 * ICMP traffic type.
 */
const MATCH_NO_TRAFFIC = {
    cidrIp: '255.255.255.255/32',
    description: 'Disallow all traffic',
    ipProtocol: 'icmp',
    fromPort: 252,
    toPort: 86,
};
/**
 * Egress rule that matches all traffic
 */
const ALLOW_ALL_RULE = {
    cidrIp: '0.0.0.0/0',
    description: 'Allow all outbound traffic by default',
    ipProtocol: '-1',
};
/**
 * Compare two ingress rules for equality the same way CloudFormation would (discarding description)
 */
function ingressRulesEqual(a, b) {
    return a.cidrIp === b.cidrIp
        && a.cidrIpv6 === b.cidrIpv6
        && a.fromPort === b.fromPort
        && a.toPort === b.toPort
        && a.ipProtocol === b.ipProtocol
        && a.sourceSecurityGroupId === b.sourceSecurityGroupId
        && a.sourceSecurityGroupName === b.sourceSecurityGroupName
        && a.sourceSecurityGroupOwnerId === b.sourceSecurityGroupOwnerId;
}
/**
 * Compare two egress rules for equality the same way CloudFormation would (discarding description)
 */
function egressRulesEqual(a, b) {
    return a.cidrIp === b.cidrIp
        && a.cidrIpv6 === b.cidrIpv6
        && a.fromPort === b.fromPort
        && a.toPort === b.toPort
        && a.ipProtocol === b.ipProtocol
        && a.destinationPrefixListId === b.destinationPrefixListId
        && a.destinationSecurityGroupId === b.destinationSecurityGroupId;
}
/**
 * Whether this rule refers to all traffic
 */
function isAllTrafficRule(rule) {
    return rule.cidrIp === '0.0.0.0/0' && rule.ipProtocol === '-1';
}
//# sourceMappingURL=data:application/json;base64,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