# -*- coding: utf-8 -*-
# vim: tabstop=4 shiftwidth=4 softtabstop=4
#
# Copyright (C) 2020, GEM Foundation
#
# OpenQuake is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenQuake is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with OpenQuake. If not, see <http://www.gnu.org/licenses/>.
"""
Module :mod:`openquake.hazardlib.gsim.lanzano_2020` implements
:class:`~openquake.hazardlib.gsim.lanzano_2020.LanzanoEtAl2020`
"""

import numpy as np
from scipy.constants import g as gravity_acc

from openquake.baselib.general import CallableDict
from openquake.hazardlib import const
from openquake.hazardlib.imt import PGA, SA
from openquake.hazardlib.gsim.base import GMPE, CoeffsTable

CONSTS = {'Mh': 5.0,
          'Rref': 1.0,
          'PseudoDepth': 6.0}


def _get_distance_term(C, mag, ctx):
    term1 = C['c1']*(mag-C['Mref']) + C['c2']
    tmp = np.sqrt(ctx.rjb ** 2 + CONSTS['PseudoDepth'] ** 2)
    term2 = np.log10(tmp / CONSTS['Rref'])
    term3 = C['c3'] * (tmp-CONSTS['Rref'])
    return term1 * term2 + term3


def _get_magnitude_term(C, mag):
    return np.where(mag <= CONSTS['Mh'],
                    C['b1'] * (mag-CONSTS['Mh']),
                    C['b2'] * (mag-CONSTS['Mh']))


_get_site_correction = CallableDict()


@_get_site_correction.add("ref")
def _get_site_correction_1(kind, ctx, C):
    """
    Compute the fourth term of the equation 1 described on paragraph :
    The functional form Fs in Eq. (1) represents the site amplification and
    it is given by FS = sj Cj , for j = 1,...,3, where sj are the
    coefficients to be determined through the regression analysis,
    while Cj are dummy variables used to denote two different
    site classes
    """
    s_others = np.zeros(len(ctx.siteclass))
    list_sites = [x.decode('utf-8') for x in ctx.siteclass]
    idx = [index for index, value in enumerate(list_sites) if value != 'A']
    s_others[idx] = 1.0
    return (C['s_other'] * s_others)


@_get_site_correction.add("ec8")
def _get_site_correction_2(kind, ctx, C):
    ssb, ssc, ssd, sse = _get_site_type_dummy_variables(kind, ctx)
    return ((C['sB'] * ssb) + (C['sC'] * ssc) +
            (C['sD'] * ssd) + (C['sE'] * sse))


@_get_site_correction.add("cluster")
def _get_site_correction_3(kind, ctx, C):
    cl2, cl3, cl4, cl5, cl6, cl7, cl8, cl9 = _get_site_type_dummy_variables(
        kind, ctx)
    return ((C['s2'] * cl2) + (C['s3'] * cl3) + (C['s4'] * cl4) +
            (C['s5'] * cl5) + (C['s6'] * cl6) + (C['s7'] * cl7) +
            (C['s8'] * cl8) + (C['s9'] * cl9))


_get_site_type_dummy_variables = CallableDict()


@_get_site_type_dummy_variables.add("ec8")
def _get_site_type_dummy_variables_1(kind, ctx):
    """
    Get site type dummy variables for five different EC8 site classes
    """
    ssb = np.zeros(len(ctx.siteclass))
    ssc = np.zeros(len(ctx.siteclass))
    ssd = np.zeros(len(ctx.siteclass))
    sse = np.zeros(len(ctx.siteclass))

    list_sites = [x.decode('utf-8') for x in ctx.siteclass]
    idx = [index for index, value in enumerate(list_sites) if value == 'B']
    ssb[idx] = 1.0
    idx = [index for index, value in enumerate(list_sites) if value == 'C']
    ssc[idx] = 1.0
    idx = [index for index, value in enumerate(list_sites) if value == 'D']
    ssd[idx] = 1.0
    idx = [index for index, value in enumerate(list_sites) if value == 'E']
    sse[idx] = 1.0
    return ssb, ssc, ssd, sse


@_get_site_type_dummy_variables.add("cluster")
def _get_site_type_dummy_variables_2(kind, ctx):
    """
    Get site type dummy variables.
    The recording sites are classified into 9 different clusters.
    """
    cl2 = np.zeros(len(ctx.siteclass))
    cl3 = np.zeros(len(ctx.siteclass))
    cl4 = np.zeros(len(ctx.siteclass))
    cl5 = np.zeros(len(ctx.siteclass))
    cl6 = np.zeros(len(ctx.siteclass))
    cl7 = np.zeros(len(ctx.siteclass))
    cl8 = np.zeros(len(ctx.siteclass))
    cl9 = np.zeros(len(ctx.siteclass))

    list_sites = [x.decode('utf-8') for x in ctx.siteclass]
    idx = [index for index, value in enumerate(list_sites) if value == '2']
    cl2[idx] = 1.0
    idx = [index for index, value in enumerate(list_sites) if value == '3']
    cl3[idx] = 1.0
    idx = [index for index, value in enumerate(list_sites) if value == '4']
    cl4[idx] = 1.0
    idx = [index for index, value in enumerate(list_sites) if value == '5']
    cl5[idx] = 1.0
    idx = [index for index, value in enumerate(list_sites) if value == '6']
    cl6[idx] = 1.0
    idx = [index for index, value in enumerate(list_sites) if value == '7']
    cl7[idx] = 1.0
    idx = [index for index, value in enumerate(list_sites) if value == '8']
    cl8[idx] = 1.0
    idx = [index for index, value in enumerate(list_sites) if value == '9']
    cl9[idx] = 1.0
    return cl2, cl3, cl4, cl5, cl6, cl7, cl8, cl9


class LanzanoEtAl2020_ref(GMPE):
    """
    Implements the GMM proposed in Lanzano et al (2020):
    "Methodology to identify the reference rock sites in regions of
    medium-to-high seismicity: an application in Central Italy" by
    Lanzano G, Felicetta C, Pacor F, Spallarossa D, and Traversa P.

    The site term for Reference Rock Sites is zero,
    whereas for other sites, the site term is provided.
    """
    kind = "ref"

    #: Supported tectonic region type is 'active shallow crust'
    DEFINED_FOR_TECTONIC_REGION_TYPE = const.TRT.ACTIVE_SHALLOW_CRUST

    #: Set of :mod:`intensity measure types <openquake.hazardlib.imt>`
    #: this GSIM can calculate. A set should contain classes from module
    #: :mod:`openquake.hazardlib.imt`.
    DEFINED_FOR_INTENSITY_MEASURE_TYPES = {PGA, SA}

    #: Supported intensity measure component is the geometric mean of two
    #: horizontal components
    DEFINED_FOR_INTENSITY_MEASURE_COMPONENT = const.IMC.GEOMETRIC_MEAN

    #: Supported standard deviation types are inter-event, intra-event
    #: and total
    DEFINED_FOR_STANDARD_DEVIATION_TYPES = {
        const.StdDev.TOTAL, const.StdDev.INTER_EVENT, const.StdDev.INTRA_EVENT}

    #: Required site parameter is not set
    REQUIRES_SITES_PARAMETERS = {'siteclass'}

    #: Required rupture parameters is magnitude, ev_lat, ev_lon
    REQUIRES_RUPTURE_PARAMETERS = {'mag'}

    #: Required distance measure is Rjb
    REQUIRES_DISTANCES = {'rjb'}

    def compute(self, ctx: np. recarray, imts, mean, sig, tau, phi):
        for m, imt in enumerate(imts):
            # Ergodic coeffs
            C = self.COEFFS[imt]
            # Get mean
            mean[m] = (C['a'] + _get_magnitude_term(C, ctx.mag) +
                       _get_distance_term(C, ctx.mag, ctx) +
                       _get_site_correction(self.kind, ctx, C))
            # To natural logarithm and fraction of g
            mean[m] = np.log(10.0 ** mean[m] / (gravity_acc * 100))
            # Get stds

            sig[m] = np.log(10.0 ** np.sqrt(
                C['tau'] ** 2 + C['phi_S2S'] ** 2 + C['phi_0'] ** 2))
            tau[m] = np.log(10.0 ** C['tau'])
            phi[m] = np.log(10.0 ** np.sqrt(
                C['phi_S2S'] ** 2 + C['phi_0'] ** 2))

    COEFFS = CoeffsTable(sa_damping=5., table="""\
IMT                 a                   b1                  b2                  c1                  c2                  c3                      s_other             Mref                tau                 phi_S2S             phi_0               sigma
pga                 2.88013426131621    0.569253138264182   0.214397115956832   0.166899352569838   -1.41318304541776   -0.00598907637537312    0.339427441478476   3.81278167434967    0.155737446367176   0.261441590479063   0.214894236857538   0.372539112668422
0.0399042298483639  2.96950031744096    0.449412610288790   0.140808497431903   0.214380350170792   -1.43231636898807   -0.00665274291717771    0.374110802998023   3.91588329541421    0.180685154866461   0.265270823737769   0.214870079160844   0.386244593533967
0.0422654268808115  2.96701583486237    0.445920211119872   0.141800079005613   0.214078600148658   -1.42719956860868   -0.00678001214718295    0.375920466025263   3.91849370907382    0.181360681993709   0.276283985201019   0.214429922978836   0.393960314397897
0.0447828034034931  2.96598735658502    0.442679651115467   0.127770640908337   0.217575718411917   -1.41397233240366   -0.00711711116729072    0.370352671728875   3.90559740317707    0.181058398702317   0.278768115394770   0.214583509749297   0.395651220847006
0.047438330170778   2.95428111593663    0.436321339758675   0.117401665387745   0.221048043142706   -1.40493554134354   -0.00725473727420865    0.377406385803140   3.90345771578653    0.180819220583177   0.290809983699458   0.214655887643877   0.404157379311287
0.0502260170768458  2.95571351639957    0.437565210530119   0.119664687993080   0.219736304387148   -1.39528347591264   -0.00744954325202958    0.380681129705774   3.89049290740867    0.182428015050688   0.293128019752825   0.215383631577814   0.406932580891717
0.0532197977647685  2.96232170527409    0.436941405915149   0.119035183061854   0.218870976574165   -1.39109563836478   -0.00764995607265580    0.379699821198647   3.85568068743395    0.184089556198512   0.291814276935471   0.216005130780751   0.407066030820979
0.056369785794814   2.97382613284666    0.444871389410847   0.120379247375472   0.214803037338119   -1.38757472687494   -0.00773536330394386    0.384519555927780   3.82965731674099    0.184460551205423   0.293824311402061   0.216473628215677   0.408924507256246
0.0597014925373134  2.97423344907176    0.450746786819342   0.128323408783896   0.210138957856001   -1.38525479433739   -0.00782018252518949    0.387251787117516   3.77876433547759    0.183531183440022   0.299122850271667   0.217916326600324   0.413092847007207
0.0632511068943707  2.98722379064012    0.465866765134967   0.138172152794754   0.201122161481644   -1.37786888083567   -0.00794584379054016    0.391040388122758   3.74094838379900    0.183628944798924   0.299250442423620   0.218834211894349   0.413713462379604
0.0669792364367046  2.98965122688604    0.476267877769067   0.148387079595120   0.195109128317236   -1.36181787148354   -0.00815986980709281    0.392566561094644   3.71010728849464    0.183135414231686   0.299358273464202   0.219424184190015   0.413885163354200
0.07097232079489    2.99406714176123    0.492764546798953   0.156505842984994   0.185287903950039   -1.34672424225956   -0.00828337039765202    0.393987927837048   3.67218427235496    0.182143117771820   0.302136402472195   0.219693515100451   0.415605295475974
0.075187969924812   2.99510982014303    0.505551938787962   0.165388273259372   0.179090685475501   -1.33509839223843   -0.00834958195796843    0.388611252647993   3.62925939506460    0.180686608961789   0.308759957759658   0.220729600421694   0.420359273331964
0.0796178343949045  2.99062554164733    0.518297063079211   0.167796521471313   0.173755871096163   -1.32299623477137   -0.00840019230997627    0.387497537256311   3.57369676430942    0.179397878475918   0.308861534851303   0.222267332941238   0.420692065298568
0.0843170320404722  2.98717858606940    0.532217307425021   0.179017881417699   0.166643489823472   -1.30957326859521   -0.00851069120536993    0.385228963248079   3.49723289013590    0.178099650585862   0.308594625618820   0.223130250062572   0.420401280912192
0.0893655049151028  2.99386822180326    0.546299832530299   0.191949737465628   0.158798283651350   -1.30074927741242   -0.00847961516003442    0.384579651726741   3.45178858667030    0.176333223170197   0.307347969871203   0.224018175988720   0.419213935062819
0.0946073793755913  3.00686059195449    0.561559678764369   0.203227737692687   0.151294827189644   -1.29613312518347   -0.00836450876661932    0.385620101017497   3.42285916721909    0.174947438397337   0.306175862655325   0.224936246743905   0.418266159489095
0.100200400801603   3.01768264473694    0.574991391888360   0.212763940061066   0.145082972273000   -1.29088313654399   -0.00822829379421259    0.385331557635117   3.40976737395031    0.173376789735285   0.301957633463985   0.225862000481105   0.415032007063805
0.106157112526539   3.03238088377701    0.590122391972946   0.225363518682195   0.137949574959235   -1.28571694487147   -0.00813407020240352    0.383668588867056   3.39124207412723    0.171466687068837   0.297133021962885   0.226877667648822   0.411293488391110
0.112485939257593   3.04863364899145    0.609149920954030   0.239527312385211   0.129380184108262   -1.28017052705485   -0.00801699556317075    0.381692642590264   3.36832679835694    0.168803922615572   0.293968205336596   0.227737739196876   0.408382844759335
0.119189511323004   3.05459081472761    0.623966139131395   0.245882881929482   0.123224218330793   -1.27401480730974   -0.00788333689678662    0.380084026286217   3.34045021321693    0.166573549970318   0.291026795213173   0.228187488455797   0.405601865096216
0.126262626262626   3.06439379055793    0.640131474582024   0.255464818644867   0.116472772078115   -1.27031334507669   -0.00771431426255746    0.378959822228332   3.31438604895075    0.164658410638499   0.288157049901183   0.228576229054310   0.402981352038079
0.133689839572193   3.07682145638902    0.651823251684664   0.265338831963279   0.112668824166680   -1.27264375519493   -0.00749247840608884    0.374909941254531   3.30960538014624    0.162705498945908   0.285252981425333   0.228566249330842   0.400106077350053
0.141643059490085   3.09165407132118    0.662326867682481   0.272647735119004   0.110246518837694   -1.27783312604418   -0.00725996834585422    0.369991467671697   3.31695134515147    0.161106631687377   0.282043996912899   0.228510376176143   0.397141228013936
0.149925037481259   3.12262058349222    0.679767499653559   0.284085983346150   0.104449840824454   -1.28849153855018   -0.00698451090789442    0.364384572614720   3.32730719305991    0.158831248112219   0.277296055323239   0.228880410993254   0.393073415803148
0.158982511923688   3.14885713725955    0.695720885988155   0.299091084619285   0.0982239575530748  -1.29742524659949   -0.00666898551091210    0.358487379504894   3.36468112228562    0.156036463215858   0.272944621594082   0.229028174787414   0.388973069963082
0.168350168350168   3.16799918416449    0.712139848304798   0.315352126046833   0.0911868589684849  -1.30057461079258   -0.00638553169442288    0.352024221649069   3.40764965029756    0.153463039624170   0.268984601796931   0.228793234772298   0.385032420468868
0.17825311942959    3.17643101076960    0.724296706821514   0.327511273585619   0.0870219794246487  -1.30377257481523   -0.00610800964659948    0.347827051973820   3.44854035836812    0.151425898055137   0.264394240351555   0.228583700551068   0.380899757272830
0.188679245283019   3.18336514460574    0.734818971662827   0.340261685270005   0.0838290659740484  -1.31028916854977   -0.00584506927084642    0.342874476484811   3.46965075722036    0.149223188966194   0.259632513065228   0.228497431550143   0.376679808578610
0.2                 3.19305456504480    0.739880666722022   0.344133541107693   0.0849723768557619  -1.32336282162587   -0.00552517795623317    0.336235372774562   3.52085298608413    0.147008788049248   0.255564330209715   0.227948497752366   0.372673084977718
0.211864406779661   3.20961837356910    0.748900759602657   0.355173076419168   0.0839784279913528  -1.33571862180792   -0.00522506302304435    0.327015738999233   3.57845764048298    0.145032739852761   0.252372600529084   0.227400741528940   0.369374501522402
0.224215246636771   3.22504867417842    0.760904688562204   0.369049804987820   0.0815458086655374  -1.34514805960421   -0.00492480443526148    0.318911727465050   3.65867332686577    0.142656765545449   0.248678942221869   0.226553568182298   0.365402091286583
0.237529691211401   3.23341367785240    0.769840154202946   0.382615958851896   0.0802602433518650  -1.35439278822072   -0.00457528620924709    0.311998247382918   3.75998590365660    0.140338357943396   0.244540171185430   0.225529191958790   0.361051473420664
0.251889168765743   3.25342369876741    0.783821889434033   0.396903874842023   0.0761344526730204  -1.37074337788675   -0.00418887426184403    0.305514360955855   3.85559206523876    0.137364836052785   0.240717914913337   0.224581910699507   0.356722927995208
0.266666666666667   3.27137645754185    0.796722125302770   0.409742471232562   0.0727660028580962  -1.38727608269733   -0.00384219844909158    0.297433044326606   3.93832021697493    0.134720325980347   0.237366353126711   0.223448431161200   0.352734394718358
0.282485875706215   3.28656846730028    0.805262193236139   0.416193616384299   0.0728257301976143  -1.40568090006138   -0.00348563316607277    0.289479410073700   4.02965431622306    0.133545626347184   0.235460538917125   0.221539644084675   0.349794101729530
0.29940119760479    3.28819683667019    0.810071227507866   0.423933356229202   0.0744105707230608  -1.41950149328261   -0.00311351595051403    0.281784607738253   4.12135517338210    0.132111268731269   0.234685890876018   0.219488893027383   0.347428019688168
0.317460317460317   3.28975454921290    0.817943050679782   0.435058595787690   0.0742472119560204  -1.43098291471894   -0.00274650317765909    0.272157232507921   4.21752867453363    0.131057002518500   0.235479535327033   0.217581075152039   0.346364076848750
0.335570469798658   3.28416969844908    0.821856159202589   0.442108607337370   0.0761313076384498  -1.44044541236609   -0.00242472245737293    0.263076522689601   4.30337532615948    0.129939900807196   0.235641003570201   0.216088176506826   0.345116154955691
0.355871886120996   3.28265108584414    0.826955323082914   0.455075873896069   0.0772864546846015  -1.45379626344089   -0.00209917944196348    0.253586920977278   4.37150105070936    0.128896506122099   0.236069441630284   0.214463591349900   0.344002503736260
0.377358490566038   3.28078477845710    0.829162916465316   0.463033825697455   0.0807096099240982  -1.47349346053823   -0.00169173546607541    0.245659412286841   4.42261927759405    0.128179219090633   0.236960245921420   0.212108773970417   0.342885115379857
0.398406374501992   3.28298523318796    0.831579575175803   0.465715534815069   0.0843512156702941  -1.49233369512269   -0.00125375656187198    0.236120404874670   4.52471928638493    0.127848199852711   0.238556483586350   0.209676662016121   0.342372108473932
0.421940928270042   3.28040108207849    0.835534057975528   0.466242071062153   0.0868676460287490  -1.50049381512147   -0.000914747859753860   0.226850694393758   4.67809776009534    0.127453451128601   0.240345559002239   0.207706715705872   0.342275400351836
0.448430493273543   3.27164101499570    0.841805551506318   0.472685283965138   0.0882484167699649  -1.50801423195180   -0.000649053756197937   0.217604301323686   4.76435931741129    0.127454772725846   0.241268863425636   0.205806365059684   0.341777183920131
0.4739336492891     3.25427332385959    0.845842090020494   0.478118955568619   0.0904951781703437  -1.51347519495109   -0.000384447076954428   0.209600285599014   4.83573266152396    0.127654779565605   0.241713103704024   0.203984466372725   0.341072763159917
0.50251256281407    3.22945850035441    0.848019454378850   0.486191860871108   0.0929451435471202  -1.51809635324631   -0.000181073149053531   0.203194029984933   4.87194107479204    0.127918866085496   0.241077404604772   0.202301293953040   0.339716594894401
0.531914893617021   3.19664498990885    0.848446665965265   0.489956295486256   0.0966566959328404  -1.51601133674253   -1.11034123040917e-05   0.197956442282899   4.92317892436622    0.128285192606971   0.239378893762443   0.200402410848550   0.337521068520248
0.564971751412429   3.15857902884973    0.853932181199138   0.496373730494690   0.0977419282827402  -1.50354633203235   0                       0.190013416615062   4.97435923206417    0.128682045101678   0.237724220197530   0.198632757340566   0.335450213725957
0.598802395209581   3.12023294332872    0.858590026835093   0.504288608897990   0.0998479910458280  -1.48771674544264   0                       0.180529545294340   5.04139600424271    0.129118468532308   0.236305321229164   0.197156489314105   0.333740116012460
0.632911392405063   3.07567187950527    0.860553758800281   0.514711887268942   0.103002948808826   -1.46892141417287   0                       0.172246319375269   5.11688826761694    0.129780886689869   0.235087714180640   0.195301892691632   0.332042378616978
0.671140939597316   3.03352185807801    0.865497323479999   0.526073607624367   0.104298951368332   -1.45277661571946   0                       0.164503843965017   5.18682007898963    0.130782092545896   0.234561413443722   0.193735473550264   0.331144147040447
0.709219858156028   2.99045099723792    0.868832983595948   0.532123745174996   0.106887787616691   -1.43653210395383   0                       0.156234214723763   5.24801550913774    0.131540282535024   0.234357881676696   0.192980850637456   0.330859594610974
0.75187969924812    2.94527039090750    0.873711862979336   0.536093860973510   0.108842931077943   -1.41920467696280   0                       0.148535455977646   5.31420697534731    0.132385219479885   0.233822241982703   0.192430183259319   0.330496690772798
0.793650793650794   2.88927259226585    0.874550425328731   0.533834744120818   0.113603574946381   -1.39487541189935   0                       0.142630374833733   5.38040018280612    0.133830843222321   0.232819437671445   0.191575896015073   0.329874080656882
0.840336134453782   2.83183355049569    0.874044507509935   0.530515912834306   0.118585668724929   -1.37627175919420   0                       0.140102286059115   5.40779525588456    0.134909612577040   0.231666845180319   0.190791392986836   0.329046328591197
0.892857142857143   2.77400330531240    0.872169106171724   0.526971476283775   0.124605168424757   -1.35767621142309   0                       0.139783026465033   5.43297116197112    0.136263355632829   0.229580611376181   0.189923828661332   0.327637024619561
0.943396226415094   2.70699764565709    0.862880675796033   0.524678651104824   0.134390271465349   -1.33520855850413   0                       0.139126871844014   5.44263481178222    0.137699659377362   0.227123429883225   0.189469033616106   0.326258736732034
1                   2.64253961392495    0.850392462015420   0.526672501485505   0.144807676103154   -1.31893895529236   0                       0.136054117268224   5.42555199253122    0.139671499828279   0.224886770430956   0.188743301749503   0.325124931886883
1.06382978723404    2.58198094686660    0.840500630302220   0.529145028013100   0.153807371568472   -1.30337606745826   0                       0.133116438985350   5.42640427462940    0.140591440837698   0.223022620008032   0.188567965812833   0.324134108052295
1.12359550561798    2.52983382781120    0.835264647233755   0.529330590108875   0.159844301413384   -1.29509203309204   0                       0.129481033918707   5.41946802931943    0.141158100646388   0.221666598802283   0.188766574043390   0.323565310066989
1.19047619047619    2.47654613189075    0.829374993517708   0.529712846154301   0.165495697570091   -1.28901860307041   0                       0.127328866254361   5.41016784801165    0.141637804086200   0.220376104312637   0.188699678807744   0.322853625781902
1.26582278481013    2.41700805651900    0.819267009666242   0.531464650299461   0.172870475876989   -1.28176084135794   0                       0.126453617275670   5.39481294214211    0.142219308181721   0.218650527284014   0.188732858004736   0.321954152624097
1.33333333333333    2.35308194295324    0.806802580512782   0.530571377024434   0.181209876125612   -1.26473959858758   0                       0.126703712110570   5.42290207743670    0.143122378940171   0.216301106787949   0.188593701711122   0.320683283749386
1.40845070422535    2.29058371312852    0.794152776115596   0.528932857469600   0.189685153755549   -1.25013148669068   0                       0.126381225970586   5.43636835779885    0.143266415792102   0.213531628453062   0.188848231688200   0.319037108900348
1.49253731343284    2.23215955852249    0.782654754898940   0.520401701140748   0.198744852382002   -1.23852699168497   0                       0.126980707183152   5.43592656578532    0.143116079005257   0.211648703124094   0.189311743187387   0.317987926991886
1.58730158730159    2.16816735848730    0.768194024944963   0.511042814200092   0.209061970595742   -1.22736286752316   0                       0.132054341244563   5.42775765002479    0.142115010877398   0.209670255708877   0.190121314836408   0.316708709701570
1.69491525423729    2.10504271397491    0.752914237995734   0.503320666019917   0.218996492741626   -1.22433079340902   0                       0.137473910078798   5.38928937691258    0.140891758075500   0.208476451614753   0.191169896945598   0.316004506090775
1.78571428571429    2.04844539515705    0.739054373788888   0.502062381457535   0.227272240476616   -1.22534558528107   0                       0.142083882293085   5.36281450651231    0.140170549633741   0.206009699696809   0.192410175424453   0.314816541751960
1.88679245283019    1.99368854223898    0.726219641123204   0.498985591165613   0.235617595094693   -1.22951127258319   0                       0.145245477681060   5.32448916905747    0.138910021832205   0.204629716970958   0.193587975914055   0.314079002245516
2                   1.94393949985108    0.713681634778468   0.492595644795684   0.243560419093234   -1.24194587668475   0                       0.149386127461626   5.26896508895249    0.136974047294487   0.202213143668361   0.194681443923737   0.312334611775104
    """)


class LanzanoEtAl2020_EC8(LanzanoEtAl2020_ref):
    kind = "ec8"

    COEFFS = CoeffsTable(sa_damping=5., table="""\
IMT                 a                   b1                  b2                  c1                  c2                  c3                      sB                  sC                  sD                  sE                  Mref                tau                 phi_S2S             phi_0               sigma
pga                 3.09622481409268    0.569215009619483   0.214024666583574   0.167003447922139   -1.41369462894651   -0.00597908881876995    0.0783650714707006  0.220928097624637   0.244235244347513   0.305962491781761   3.81278167434967    0.155842280528824   0.276943549784854   0.214881081744956   0.383612337470712
0.0399042298483639  3.19963423965516    0.449695151702073   0.140914140252521   0.214184072490509   -1.43178818773985   -0.00666299538914285    0.0988736079186288  0.194986447026218   0.195086233748200   0.296791629314916   3.91588329541421    0.180712786617934   0.287736417803533   0.214863708231210   0.402014639649472
0.0422654268808115  3.20286314587029    0.446181021475098   0.141604709013285   0.214015389432652   -1.42686000229189   -0.00678633482522061    0.0933386832119378  0.199914253313631   0.194778159260513   0.310036242983507   3.91849370907382    0.181394944726811   0.297904476400645   0.214420598364486   0.409423248039202
0.0447828034034931  3.20195902757759    0.442889523091371   0.127239736955223   0.217643166744411   -1.41365669088863   -0.00712257081402402    0.0869706381308448  0.201794558358320   0.195877804022348   0.314716092171694   3.90559740317707    0.181141677635475   0.299282453982991   0.214572430011704   0.410394471648684
0.047438330170778   3.19829562024923    0.436370605445957   0.116938768294341   0.221159842673568   -1.40471405452361   -0.00725824435717939    0.0843755018147887  0.205319252993514   0.199215154741629   0.321544392665617   3.90345771578653    0.180895900542560   0.311038833065752   0.214645791324090   0.418976488888283
0.0502260170768458  3.20389528150748    0.437559435823492   0.119237404814594   0.219851054998869   -1.39514482015589   -0.00745152618558132    0.0830219037172368  0.206893198740547   0.198108682461941   0.316906293485742   3.89049290740867    0.182469597115031   0.313526505478255   0.215374576501468   0.421877033875885
0.0532197977647685  3.21164622261669    0.436930645245299   0.118619420981526   0.218976129775893   -1.39110179813360   -0.00764938826142540    0.0809363888016552  0.210060893273962   0.203476014623524   0.311984213089102   3.85568068743395    0.184141158242122   0.311804809583287   0.215996256945511   0.421645097745120
0.056369785794814   3.22723739216709    0.444863209697147   0.119984866509938   0.214908306904227   -1.38765933407754   -0.00773323174710182    0.0803982346693111  0.215785557632515   0.218599094476961   0.310939045573278   3.82965731674099    0.184521882887253   0.313891084359244   0.216465029120839   0.423595381156080
0.0597014925373134  3.23109064820279    0.450726631277826   0.127892486157593   0.210263477691661   -1.38538116285720   -0.00781719997240878    0.0776540760504841  0.223447975556145   0.235262344230633   0.318227735889258   3.77876433547759    0.183595095053594   0.318642452080618   0.217907343577233   0.427462023553641
0.0632511068943707  3.24572675738833    0.465834635097601   0.137720500401068   0.201254778581099   -1.37805300504986   -0.00794204943732778    0.0790200879296248  0.232784580016723   0.245130098981947   0.326904723406055   3.74094838379900    0.183694238121714   0.318622554171126   0.218825211562795   0.427958383912678
0.0669792364367046  3.24870607976854    0.476215440405085   0.148038873819638   0.195233625622500   -1.36207134815277   -0.00815474480275948    0.0798342105519315  0.238342769017570   0.246860555673714   0.335256001540413   3.71010728849464    0.183196943752912   0.318476028559546   0.219415667944378   0.427938472571637
0.07097232079489    3.25440882998770    0.492743910472091   0.156113577491470   0.185395052194928   -1.34701484096709   -0.00827746912027736    0.0799930824172659  0.237726392940763   0.247671073196404   0.344062157857888   3.67218427235496    0.182209782329997   0.321210269971103   0.219684130218873   0.429694728128792
0.075187969924812   3.25673174808184    0.505524638889157   0.165018085580684   0.179202347158246   -1.33547967969720   -0.00834179635130920    0.0720415397004562  0.239111003738797   0.243239601586464   0.359458060544564   3.62925939506460    0.180752653415244   0.326445800002847   0.220719203693727   0.433538174715149
0.0796178343949045  3.25321897204021    0.518239235178815   0.167369936425834   0.173893523479864   -1.32345674351826   -0.00839094768129500    0.0694514848968736  0.239876778129922   0.244036126818100   0.365936351866060   3.57369676430942    0.179468946272345   0.326238033923348   0.222255981306717   0.433633576514653
0.0843170320404722  3.24874876444362    0.532148491918561   0.178626970911536   0.166777127095399   -1.31009253040426   -0.00850049864355339    0.0679386816308804  0.240127646113891   0.249610988138037   0.373985489144765   3.49723289013590    0.178172843295205   0.325443263846220   0.223118924237446   0.432944493467296
0.0893655049151028  3.25208206857443    0.546226788241969   0.191557672223314   0.158935574256478   -1.30128422910802   -0.00846909858758203    0.0713247167209465  0.239572179529037   0.262052054653001   0.376976668409059   3.45178858667030    0.176409574187433   0.324097865951924   0.224006515106518   0.431669646129710
0.0946073793755913  3.26242703925608    0.561484050847901   0.202824415998568   0.151436084064874   -1.29667910871982   -0.00835374179955186    0.0757013170227871  0.237632354219958   0.277433094670840   0.385738302543117   3.42285916721909    0.175029516580018   0.322980834279065   0.224924015586862   0.430746751320952
0.100200400801603   3.27108727871657    0.574903397712465   0.212352177193363   0.145231813268720   -1.29144027144834   -0.00821704572764337    0.0784710120893966  0.232004924273247   0.282766042570635   0.393705060629440   3.40976737395031    0.173462559329035   0.319047689130915   0.225849033583833   0.427654619289337
0.106157112526539   3.28332957086968    0.590030039380447   0.224947048613537   0.138100536129578   -1.28629064031996   -0.00812240745704994    0.0803689361404923  0.226626146613774   0.280815507056983   0.394977573116009   3.39124207412723    0.171554874001092   0.314527471755206   0.226864095060110   0.424058867269269
0.112485939257593   3.29846261472460    0.609059456957550   0.239117318382888   0.129528170314386   -1.28076017179699   -0.00800501968372083    0.0798065861278704  0.224650733199228   0.279289731268861   0.391504347874585   3.36832679835694    0.168899543329551   0.311467950444182   0.227723289192870   0.421185512965071
0.119189511323004   3.30215015577018    0.623877367160436   0.245464665823835   0.123370297549159   -1.27462758138976   -0.00787111464158963    0.0806065879736338  0.226170363756075   0.285389477932813   0.391382017479658   3.34045021321693    0.166675198215900   0.308368057576187   0.228172621186032   0.418251390544639
0.126262626262626   3.30790770962964    0.640054832382534   0.255047349186275   0.116610840825840   -1.27094420330801   -0.00770183174575741    0.0840481520190820  0.228315886486009   0.298042310873476   0.397822898778841   3.31438604895075    0.164767933190352   0.305187142371479   0.228561169610766   0.415364745651977
0.133689839572193   3.31640957835696    0.651741580089852   0.264928308754329   0.112805963844453   -1.27328946081043   -0.00747984109330962    0.0847012971480669  0.224334353658626   0.300505614224387   0.404127825642599   3.30960538014624    0.162818328821269   0.301937789807370   0.228550846296729   0.412203501267106
0.141643059490085   3.32807056723513    0.662240675347832   0.272247514283620   0.110381670192632   -1.27848108468785   -0.00724737269498501    0.0838116573934200  0.218258811479494   0.296424316580088   0.404917503960741   3.31695134515147    0.161223353302960   0.298514636559810   0.228494705368810   0.409040081498458
0.149925037481259   3.35449288972164    0.679684681348386   0.283695798706554   0.104579708202804   -1.28914186601711   -0.00697199103209487    0.0840173327797600  0.214513723841689   0.293844272551574   0.390139388179460   3.32730719305991    0.158946837991341   0.293718202325114   0.228864405999480   0.404862193861117
0.158982511923688   3.37472598789677    0.695655290679400   0.298719314265759   0.0983377449256202  -1.29805978865248   -0.00665670351201972    0.0858838053354033  0.212917468474514   0.290666974209616   0.366929166826285   3.36468112228562    0.156150261335907   0.289328482250076   0.229012063753344   0.400674930711880
0.168350168350168   3.38627835693906    0.712095128821718   0.315009379850371   0.0912795055337445  -1.30117732324527   -0.00637382616467229    0.0891474642435151  0.213514910338250   0.281962471172506   0.347168176122972   3.40764965029756    0.153572565865003   0.285179951550472   0.228777335292354   0.396549122929506
0.17825311942959    3.38970795982048    0.724276802576824   0.327191102491142   0.0870945671489454  -1.30435190446106   -0.00609661999662537    0.0916032971695423  0.211902824714388   0.277868014550463   0.330245471850372   3.44854035836812    0.151534499739534   0.280670886002998   0.228567651284033   0.392405430740348
0.188679245283019   3.39412899134541    0.734805279216581   0.339957717859006   0.0838963050432978  -1.31086173173080   -0.00583381765959161    0.0903128879802022  0.204950764441877   0.276843756461665   0.304703496545681   3.46965075722036    0.149334424469412   0.276109062194276   0.228480892507186   0.388253142676647
0.2                 3.39934625643954    0.739877263606662   0.343867363003084   0.0850274624176442  -1.32392248750909   -0.00551434849585528    0.0893081360584368  0.199085669738270   0.282126554127032   0.283824859878492   3.52085298608413    0.147117403710775   0.271721932280313   0.227932165658151   0.383965377473492
0.211864406779661   3.40801163053112    0.748917616917848   0.354948773057719   0.0840148874735013  -1.33623608449606   -0.00521530192238278    0.0900147561813341  0.195267344972634   0.277328175492482   0.262674311429384   3.57845764048298    0.145139202578178   0.267929007091082   0.227384758149707   0.380204115186360
0.224215246636771   3.41436150589724    0.760958229455359   0.368876685358862   0.0815521731678663  -1.34560903302408   -0.00491637088829089    0.0936301134590080  0.193778158085303   0.267310016902343   0.243509231187416   3.65867332686577    0.142758142879515   0.263743165196154   0.226538411557322   0.375845708312611
0.237529691211401   3.41257661111910    0.769922292308309   0.382487465666110   0.0802412627662878  -1.35479625242721   -0.00456810567338997    0.0999407777939304  0.192204615552805   0.260771369027808   0.229530909198299   3.75998590365660    0.140430647218454   0.259096410762655   0.225515323012680   0.371091468053091
0.251889168765743   3.42315200732003    0.783924931663786   0.396817940956767   0.0760949852366324  -1.37111606029871   -0.00418277480453128    0.105328558702396   0.194227002414696   0.257171629130660   0.216146613041688   3.85559206523876    0.137446537143925   0.254612059791837   0.224569358664726   0.366265270555897
0.266666666666667   3.43265569195030    0.796833794877122   0.409683528696794   0.0727141657257167  -1.38764203111499   -0.00383689962478242    0.107376786587862   0.196446988817663   0.255790982071480   0.199884852135306   3.93832021697493    0.134794389395408   0.250384509849153   0.223437044152382   0.361643530129373
0.282485875706215   3.43918687517265    0.805371050579381   0.416167982891067   0.0727658845477654  -1.40604287865834   -0.00348114224143090    0.109794369144517   0.197285010026344   0.258572374935562   0.187618803868144   4.02965431622306    0.133609214620132   0.247476570651516   0.221529612903814   0.358010397399513
0.29940119760479    3.43251871862511    0.810191932460880   0.423932052329737   0.0743382351188832  -1.41983783075678   -0.00311009639046373    0.112165138897087   0.199316084457890   0.253099748188341   0.175150238057967   4.12135517338210    0.132167708989503   0.245784369672225   0.219479963932581   0.355034807085945
0.317460317460317   3.42460772341363    0.818077117427406   0.435073236913366   0.0741615823277106  -1.43129319524247   -0.00274412301096435    0.113754582320710   0.202129869319461   0.242461569942870   0.168532657993420   4.21752867453363    0.131106639021884   0.245435090723561   0.217573038267011   0.353221405827614
0.335570469798658   3.41132701110000    0.821989672518417   0.442137166376901   0.0760396380586900  -1.44072905299899   -0.00242327550417692    0.113930146458068   0.202562083521928   0.229316237140681   0.158582000165554   4.30337532615948    0.129980700475068   0.244709936707088   0.216080999107739   0.351381464785175
0.355871886120996   3.40286909533266    0.827084965827964   0.455122340429552   0.0771925174992777  -1.45405670458465   -0.00209856184324134    0.112714795737835   0.201740377980378   0.219745635726443   0.142316469237887   4.37150105070936    0.128930070960566   0.244246880206049   0.214457387936934   0.349673380355178
0.377358490566038   3.39421911766429    0.829284172870576   0.463091230363724   0.0806192331295485  -1.47374797240310   -0.00169139747139159    0.112533331509557   0.201936285778053   0.224136793680392   0.129609008172232   4.42261927759405    0.128207986151423   0.244184826327372   0.212103530211266   0.347924452503146
0.398406374501992   3.38846282817065    0.831699085862653   0.465773215249402   0.0842634448552648  -1.49258966512096   -0.00125355774752787    0.111790811103647   0.202384693287173   0.230401750192610   0.122569994127818   4.52471928638493    0.127875480022580   0.244661023157760   0.209672128417671   0.346660577624286
0.421940928270042   3.37814682262724    0.835655781777694   0.466300213949373   0.0867783572864325  -1.50075434047457   -0.000914772966118503   0.111128532832286   0.203255194073888   0.233630825012836   0.115777529091337   4.67809776009534    0.127479131508322   0.245414411396064   0.207702901541458   0.345860748856723
0.448430493273543   3.36169440545200    0.841927101555865   0.472744001916549   0.0881596246771244  -1.50827451276048   -0.000649183334859945   0.110515939316651   0.203374953494526   0.235510853564878   0.110790692963840   4.76435931741129    0.127477687582633   0.245406559626123   0.205803179568278   0.344717114543734
0.4739336492891     3.33748704609449    0.845960939460849   0.478178762536367   0.0904075362119720  -1.51373750133881   -0.000384620202164106   0.110097512787664   0.204842677090889   0.236937518604768   0.106375852852947   4.83573266152396    0.127674763151133   0.245037135608114   0.203981821937821   0.343442319252884
0.50251256281407    3.30688365675788    0.848141667545658   0.486248243449805   0.0928563522806452  -1.51836390637140   -0.000181312442441958   0.109792931538991   0.210529563588497   0.235440712817265   0.104474557835202   4.87194107479204    0.127937780669839   0.243659593668336   0.202299114908985   0.341559665654920
0.531914893617021   3.26962758750691    0.848571880765696   0.490009635696778   0.0965665571837709  -1.51628284982201   -1.14554674178244e-05   0.109345508589837   0.214664808899653   0.231392524100303   0.100065539283231   4.92317892436622    0.128302510023781   0.241432822537201   0.200400515923378   0.338986295680280
0.564971751412429   3.22575125000796    0.854059548714028   0.496421307528724   0.0976528431507788  -1.50384595740557   0                       0.107750492078372   0.214309597579942   0.234462454782952   0.0949671760704507  4.97435923206417    0.128697017913695   0.239115273905439   0.198631146675934   0.336442222475936
0.598802395209581   3.18032195355069    0.858717917092154   0.504328339288512   0.0997611241498396  -1.48801558242897   0                       0.105965415464686   0.215122359721965   0.234447051038040   0.0902351459174824  5.04139600424271    0.129132597515025   0.236922939173419   0.197155193930428   0.334182401303979
0.632911392405063   3.12969599342702    0.860680857477604   0.514739990106072   0.102920700591011   -1.46921495245466   0                       0.104007091860906   0.217138823283802   0.238837183799889   0.0831868059152130  5.11688826761694    0.129795125882540   0.234946302047456   0.195300795720151   0.331947195134351
0.671140939597316   3.08167003549235    0.865623142889097   0.526085054309111   0.104222755365539   -1.45305958847834   0                       0.102000767824658   0.221597580237019   0.247709631737664   0.0776137782854617  5.18682007898963    0.130796319226745   0.233541303104406   0.193734647018428   0.330427497091758
0.709219858156028   3.03286395013636    0.868955778615260   0.532115525835085   0.106819261394035   -1.43680172801105   0                       0.0993888468710527  0.225050139024320   0.255052664451380   0.0682606270576663  5.24801550913774    0.131555574196004   0.232484640860578   0.192980288552615   0.329541149339446
0.75187969924812    2.98224373836750    0.873833684481640   0.536072224114084   0.108778777713116   -1.41946631968248   0                       0.0975068706915807  0.225985121064962   0.255023469262961   0.0625075645528434  5.31420697534731    0.132399272400623   0.231366251462034   0.192429945728343   0.328769210329709
0.793650793650794   2.92149112529368    0.874672310623641   0.533808189332436   0.113540377014167   -1.39513887953536   0                       0.0968533762697133  0.227295330311784   0.252860271790018   0.0600427498825333  5.38040018280612    0.133845219337568   0.229930093914102   0.191575881786130   0.327847082202973
0.840336134453782   2.86208171168817    0.874165591886444   0.530485032118853   0.118523752539371   -1.37653540134863   0                       0.0963968427924872  0.229362842651477   0.248436498776210   0.0610960943148969  5.40779525588456    0.134922746079090   0.228570719682557   0.190791622577324   0.326879434274187
0.892857142857143   2.80414237408826    0.872294532644521   0.526937101956937   0.124542312930588   -1.35794728609928   0                       0.0963199969892136  0.229721437003002   0.241512558795230   0.0633335626699320  5.43297116197112    0.136274246454832   0.226526480786109   0.189924227946962   0.325509030758627
0.943396226415094   2.73647525772624    0.863009118919517   0.524639761402111   0.134326344053352   -1.33548359993338   0                       0.0961849393221239  0.232151078325998   0.238040305937048   0.0687307012316626  5.44263481178222    0.137706342550425   0.223944562181621   0.189469553919254   0.324056963467373
1                   2.67027255410007    0.850522625333204   0.526622035937866   0.144745191423902   -1.31920593033648   0                       0.0946385403789653  0.234051758291895   0.235577129192792   0.0722640352806900  5.42555199253122    0.139675462055607   0.221442258595876   0.188743904207177   0.322754039429188
1.06382978723404    2.60814909798572    0.840637572293547   0.529091583598674   0.153741015166071   -1.30365270143653   0                       0.0934927702899721  0.234264900285285   0.227244637555784   0.0717124935786620  5.42640427462940    0.140591799492390   0.219474365281209   0.188568784713226   0.321703648826557
1.12359550561798    2.55343694632877    0.835403699661745   0.529272181384477   0.159777617332747   -1.29536708772951   0                       0.0930852030209751  0.232023615367258   0.220449806657645   0.0701491971411274  5.41946802931943    0.141154010022802   0.218058984345668   0.188767732466215   0.321103459993533
1.19047619047619    2.49773300593265    0.829512031528763   0.529650585732474   0.165430940782849   -1.28928535841108   0                       0.0937680189595533  0.232378098387697   0.218033005536975   0.0726652117337561  5.41016784801165    0.141631138495558   0.216615869875291   0.188701076560722   0.320296598122041
1.26582278481013    2.43781884724583    0.819401174829966   0.531395999529396   0.172809111214510   -1.28201980611010   0                       0.0932912386561557  0.231062455203663   0.217618965778722   0.0756804589087512  5.39481294214211    0.142210468665833   0.214878578696235   0.188734369522711   0.319401445237502
1.33333333333333    2.37457931221065    0.806940621835840   0.530505061888970   0.181145840226252   -1.26500653476655   0                       0.0928808645242125  0.228861582339021   0.215598237698816   0.0791030833456832  5.42290207743670    0.143109668987489   0.212653720754524   0.188595203276370   0.318229685930315
1.40845070422535    2.31184857770289    0.794294282790635   0.528865386875931   0.189618732194913   -1.25040792401231   0                       0.0930325099500343  0.226392857135906   0.214452601538314   0.0831052304128341  5.43636835779885    0.143250802556560   0.209962937751439   0.188849708262603   0.316653501438376
1.49253731343284    2.25317652042637    0.782792662495419   0.520338860508533   0.198677166180589   -1.23880873647719   0                       0.0944033457882101  0.223485173320326   0.212927005095662   0.0863248636789122  5.43592656578532    0.143097262098112   0.208233644765261   0.189313317897844   0.315717610477608
1.58730158730159    2.19242325676074    0.768330596718917   0.511001678062195   0.208987311840043   -1.22767558620483   0                       0.0964409303342747  0.221176566649186   0.210228187299556   0.0879736592252349  5.42775765002479    0.142098164916949   0.206665404353414   0.190122895716330   0.314720818035425
1.69491525423729    2.13252141026874    0.753059553799640   0.503288089620650   0.218914420440457   -1.22467008362080   0                       0.0985050567301254  0.222580794273341   0.207346599214287   0.0911993875930848  5.38928937691258    0.140875426285663   0.205758352112638   0.191171508804092   0.314211602226443
1.78571428571429    2.07942475394364    0.739205470943634   0.502036093681869   0.227184089427837   -1.22570352221520   0                       0.0993474839501075  0.222183035893090   0.207355783004241   0.0936847717006886  5.36281450651231    0.140153477287391   0.203606483136903   0.192411692012974   0.313242488169183
1.88679245283019    2.02779850462564    0.726376190890401   0.498967433504031   0.235523531390321   -1.22988364953583   0                       0.0992543085407831  0.219718789977752   0.204667966710654   0.0944318147653520  5.32448916905747    0.138889791141205   0.202611986244569   0.193589302051092   0.312759986126495
2                   1.98274976771010    0.713847205659568   0.492576019222848   0.243460410235115   -1.24233675653874   0                       0.0982523590836060  0.218213843281353   0.198442435540576   0.0941642829303047  5.26896508895249    0.136955854666769   0.200689363666195   0.194682324083019   0.311342791993990
    """)


class LanzanoEtAl2020_Cluster(LanzanoEtAl2020_ref):
    kind = "cluster"

    COEFFS = CoeffsTable(sa_damping=5., table="""\
IMT     a       b1      b2      c1      c2      c3          s2      s3      s4      s5      s6      s7      s8      s9      Mref    tau     phi_S2S phi_0   sigma
pga     2.843   0.553   0.207   0.175   -1.461  -5.262E-03  0.741   0.633   0.716   0.466   0.300   0.509   0.959   0.406   3.813   0.154   0.120   0.210   0.287
0.040   2.935   0.436   0.138   0.222   -1.477  -6.022E-03  0.680   0.628   0.731   0.405   0.326   0.557   1.008   0.355   3.916   0.178   0.151   0.211   0.315
0.042   2.935   0.434   0.137   0.221   -1.472  -6.159E-03  0.686   0.638   0.750   0.413   0.331   0.575   1.024   0.362   3.918   0.179   0.152   0.211   0.316
0.045   2.935   0.430   0.123   0.224   -1.460  -6.450E-03  0.685   0.631   0.753   0.409   0.325   0.582   1.024   0.359   3.906   0.179   0.156   0.211   0.317
0.047   2.928   0.423   0.113   0.228   -1.454  -6.538E-03  0.686   0.641   0.769   0.416   0.329   0.596   1.040   0.369   3.903   0.179   0.161   0.211   0.320
0.050   2.929   0.423   0.117   0.227   -1.444  -6.721E-03  0.693   0.644   0.778   0.421   0.329   0.602   1.033   0.370   3.890   0.180   0.163   0.211   0.322
0.053   2.931   0.421   0.117   0.227   -1.439  -6.922E-03  0.697   0.646   0.788   0.425   0.331   0.609   1.037   0.374   3.856   0.182   0.163   0.212   0.323
0.056   2.946   0.430   0.118   0.223   -1.435  -7.030E-03  0.705   0.646   0.797   0.429   0.330   0.613   1.051   0.377   3.830   0.182   0.165   0.212   0.325
0.060   2.949   0.437   0.127   0.217   -1.432  -7.107E-03  0.705   0.646   0.805   0.434   0.328   0.619   1.070   0.383   3.779   0.182   0.168   0.213   0.327
0.063   2.963   0.451   0.135   0.209   -1.427  -7.208E-03  0.708   0.646   0.810   0.437   0.329   0.623   1.091   0.388   3.741   0.183   0.169   0.214   0.328
0.067   2.967   0.462   0.143   0.202   -1.412  -7.391E-03  0.715   0.639   0.808   0.441   0.332   0.626   1.101   0.393   3.710   0.182   0.167   0.214   0.327
0.071   2.969   0.476   0.155   0.193   -1.399  -7.481E-03  0.722   0.639   0.812   0.447   0.337   0.629   1.116   0.402   3.672   0.181   0.167   0.214   0.326
0.075   2.976   0.492   0.162   0.186   -1.390  -7.534E-03  0.733   0.639   0.816   0.451   0.340   0.631   1.119   0.409   3.629   0.180   0.166   0.215   0.326
0.080   2.975   0.505   0.165   0.180   -1.381  -7.553E-03  0.735   0.638   0.818   0.455   0.342   0.632   1.125   0.416   3.574   0.178   0.165   0.217   0.325
0.084   2.970   0.518   0.175   0.174   -1.371  -7.645E-03  0.737   0.639   0.819   0.458   0.344   0.629   1.133   0.424   3.497   0.177   0.162   0.217   0.324
0.089   2.977   0.531   0.187   0.167   -1.365  -7.600E-03  0.735   0.641   0.819   0.462   0.344   0.622   1.128   0.432   3.452   0.176   0.159   0.218   0.322
0.095   2.990   0.543   0.197   0.161   -1.365  -7.465E-03  0.735   0.645   0.821   0.466   0.344   0.614   1.116   0.442   3.423   0.174   0.157   0.219   0.321
0.100   3.001   0.556   0.205   0.155   -1.362  -7.323E-03  0.740   0.645   0.815   0.469   0.345   0.603   1.093   0.448   3.410   0.173   0.153   0.220   0.319
0.106   3.013   0.570   0.218   0.149   -1.355  -7.248E-03  0.740   0.644   0.804   0.470   0.343   0.589   1.071   0.450   3.391   0.171   0.149   0.221   0.317
0.112   3.028   0.588   0.231   0.140   -1.349  -7.134E-03  0.742   0.650   0.795   0.473   0.340   0.578   1.052   0.453   3.368   0.169   0.146   0.222   0.315
0.119   3.033   0.603   0.237   0.134   -1.342  -7.025E-03  0.744   0.656   0.789   0.479   0.338   0.567   1.042   0.455   3.340   0.167   0.144   0.223   0.313
0.126   3.042   0.620   0.247   0.127   -1.337  -6.868E-03  0.743   0.663   0.783   0.489   0.337   0.557   1.032   0.459   3.314   0.165   0.142   0.223   0.312
0.134   3.050   0.631   0.256   0.123   -1.337  -6.663E-03  0.741   0.667   0.776   0.498   0.335   0.542   1.025   0.461   3.310   0.163   0.140   0.223   0.310
0.142   3.058   0.644   0.265   0.120   -1.336  -6.471E-03  0.747   0.671   0.766   0.507   0.333   0.525   1.008   0.464   3.317   0.161   0.138   0.224   0.308
0.150   3.081   0.663   0.276   0.113   -1.341  -6.222E-03  0.766   0.673   0.753   0.512   0.329   0.507   0.975   0.466   3.327   0.159   0.136   0.224   0.307
0.159   3.099   0.678   0.292   0.108   -1.345  -5.930E-03  0.792   0.670   0.738   0.513   0.326   0.489   0.933   0.467   3.365   0.156   0.135   0.224   0.305
0.168   3.113   0.694   0.308   0.101   -1.345  -5.656E-03  0.813   0.666   0.719   0.511   0.322   0.471   0.899   0.467   3.408   0.154   0.132   0.224   0.302
0.178   3.120   0.705   0.320   0.097   -1.347  -5.393E-03  0.826   0.666   0.697   0.512   0.319   0.453   0.864   0.467   3.449   0.152   0.129   0.223   0.300
0.189   3.128   0.715   0.332   0.094   -1.352  -5.156E-03  0.827   0.665   0.678   0.511   0.314   0.432   0.825   0.466   3.470   0.149   0.127   0.223   0.297
0.200   3.137   0.722   0.336   0.095   -1.362  -4.867E-03  0.818   0.662   0.658   0.514   0.306   0.413   0.776   0.465   3.521   0.147   0.125   0.223   0.295
0.212   3.154   0.733   0.346   0.092   -1.372  -4.582E-03  0.805   0.661   0.638   0.513   0.298   0.394   0.736   0.464   3.578   0.145   0.124   0.222   0.293
0.224   3.170   0.745   0.359   0.090   -1.380  -4.281E-03  0.795   0.662   0.615   0.513   0.290   0.372   0.697   0.459   3.659   0.143   0.123   0.221   0.291
0.238   3.178   0.753   0.371   0.089   -1.387  -3.943E-03  0.790   0.660   0.590   0.515   0.281   0.347   0.660   0.455   3.760   0.141   0.123   0.220   0.289
0.252   3.197   0.768   0.386   0.084   -1.400  -3.595E-03  0.789   0.656   0.569   0.522   0.275   0.324   0.625   0.452   3.856   0.138   0.121   0.219   0.286
0.267   3.213   0.782   0.399   0.080   -1.413  -3.277E-03  0.784   0.654   0.549   0.529   0.266   0.302   0.582   0.448   3.938   0.135   0.119   0.218   0.283
0.282   3.225   0.792   0.404   0.080   -1.428  -2.949E-03  0.776   0.657   0.530   0.537   0.261   0.280   0.560   0.445   4.030   0.134   0.116   0.216   0.280
0.299   3.223   0.797   0.412   0.082   -1.438  -2.602E-03  0.769   0.662   0.512   0.539   0.253   0.260   0.545   0.440   4.121   0.133   0.114   0.214   0.277
0.317   3.225   0.807   0.424   0.081   -1.447  -2.253E-03  0.766   0.668   0.490   0.543   0.243   0.238   0.527   0.432   4.218   0.132   0.114   0.212   0.275
0.336   3.224   0.812   0.430   0.082   -1.456  -1.938E-03  0.770   0.667   0.466   0.547   0.231   0.215   0.505   0.423   4.303   0.131   0.113   0.211   0.272
0.356   3.224   0.818   0.443   0.083   -1.469  -1.628E-03  0.776   0.660   0.442   0.551   0.218   0.193   0.490   0.416   4.372   0.130   0.111   0.209   0.270
0.377   3.226   0.822   0.451   0.085   -1.488  -1.237E-03  0.787   0.653   0.421   0.557   0.203   0.172   0.472   0.412   4.423   0.130   0.111   0.207   0.268
0.398   3.232   0.826   0.455   0.088   -1.507  -8.046E-04  0.802   0.641   0.401   0.563   0.189   0.150   0.443   0.405   4.525   0.130   0.111   0.204   0.266
0.422   3.230   0.832   0.455   0.089   -1.515  -4.634E-04  0.814   0.627   0.381   0.569   0.179   0.131   0.420   0.401   4.678   0.130   0.111   0.202   0.265
0.448   3.223   0.842   0.462   0.089   -1.523  -2.121E-04  0.816   0.612   0.362   0.572   0.171   0.115   0.401   0.398   4.764   0.130   0.112   0.200   0.264
0.474   3.201   0.848   0.469   0.090   -1.525  0           0.815   0.597   0.345   0.576   0.165   0.101   0.383   0.395   4.836   0.130   0.114   0.199   0.263
0.503   3.159   0.850   0.479   0.092   -1.512  0           0.806   0.583   0.329   0.582   0.159   0.088   0.360   0.389   4.872   0.130   0.114   0.197   0.262
0.532   3.112   0.850   0.483   0.096   -1.496  0           0.798   0.568   0.314   0.586   0.154   0.077   0.340   0.383   4.923   0.131   0.113   0.194   0.260
0.565   3.074   0.858   0.490   0.096   -1.484  0           0.789   0.553   0.300   0.590   0.150   0.066   0.328   0.378   4.974   0.132   0.112   0.193   0.259
0.599   3.038   0.865   0.498   0.097   -1.469  0           0.777   0.538   0.286   0.589   0.145   0.052   0.313   0.368   5.041   0.132   0.110   0.191   0.257
0.633   2.995   0.868   0.507   0.100   -1.451  0           0.768   0.524   0.274   0.591   0.139   0.039   0.290   0.357   5.117   0.132   0.108   0.189   0.255
0.671   2.953   0.874   0.518   0.101   -1.435  0           0.759   0.509   0.261   0.597   0.135   0.028   0.268   0.348   5.187   0.133   0.108   0.188   0.254
0.709   2.910   0.880   0.523   0.103   -1.420  0           0.745   0.498   0.247   0.604   0.130   0.019   0.252   0.340   5.248   0.134   0.108   0.187   0.254
0.752   2.866   0.887   0.526   0.104   -1.404  0           0.725   0.487   0.236   0.609   0.124   0.011   0.239   0.333   5.314   0.135   0.107   0.186   0.254
0.794   2.811   0.888   0.525   0.108   -1.380  0           0.701   0.475   0.226   0.610   0.118   0.002   0.228   0.327   5.380   0.136   0.107   0.185   0.254
0.840   2.758   0.889   0.523   0.113   -1.363  0           0.680   0.466   0.216   0.613   0.112   -0.005  0.220   0.321   5.408   0.138   0.108   0.184   0.254
0.893   2.705   0.888   0.520   0.118   -1.345  0           0.663   0.457   0.207   0.614   0.106   -0.011  0.209   0.314   5.433   0.139   0.108   0.183   0.254
0.943   2.640   0.880   0.517   0.128   -1.322  0           0.645   0.447   0.202   0.611   0.103   -0.013  0.200   0.311   5.443   0.140   0.108   0.183   0.255
1.000   2.579   0.867   0.520   0.138   -1.306  0           0.630   0.435   0.197   0.604   0.098   -0.016  0.192   0.305   5.426   0.142   0.108   0.182   0.255
1.064   2.519   0.857   0.522   0.147   -1.290  0           0.616   0.426   0.193   0.597   0.094   -0.018  0.183   0.299   5.426   0.143   0.108   0.182   0.256
1.124   2.467   0.853   0.523   0.153   -1.283  0           0.602   0.419   0.191   0.589   0.093   -0.019  0.180   0.295   5.419   0.144   0.109   0.182   0.256
1.190   2.415   0.848   0.523   0.158   -1.278  0           0.591   0.411   0.188   0.582   0.090   -0.019  0.180   0.291   5.410   0.144   0.109   0.182   0.257
1.266   2.360   0.840   0.525   0.164   -1.273  0           0.580   0.405   0.186   0.573   0.088   -0.018  0.180   0.287   5.395   0.145   0.109   0.182   0.257
1.333   2.297   0.828   0.523   0.172   -1.256  0           0.569   0.399   0.184   0.565   0.087   -0.016  0.180   0.284   5.423   0.146   0.109   0.182   0.258
1.408   2.237   0.814   0.520   0.181   -1.241  0           0.559   0.395   0.183   0.555   0.086   -0.015  0.184   0.279   5.436   0.146   0.109   0.182   0.258
1.493   2.180   0.803   0.511   0.190   -1.230  0           0.552   0.392   0.184   0.545   0.086   -0.013  0.188   0.275   5.436   0.146   0.109   0.183   0.258
1.587   2.119   0.788   0.502   0.201   -1.219  0           0.545   0.393   0.191   0.536   0.089   -0.008  0.196   0.275   5.428   0.145   0.108   0.184   0.257
1.695   2.057   0.772   0.496   0.212   -1.215  0           0.540   0.393   0.200   0.531   0.092   -0.002  0.210   0.275   5.389   0.143   0.106   0.185   0.257
1.786   2.002   0.758   0.495   0.220   -1.217  0           0.540   0.394   0.210   0.528   0.096   0.005   0.227   0.276   5.363   0.143   0.105   0.186   0.257
1.887   1.947   0.745   0.492   0.229   -1.221  0           0.546   0.397   0.219   0.524   0.099   0.011   0.243   0.277   5.324   0.141   0.104   0.187   0.257
2.000   1.899   0.732   0.484   0.237   -1.233  0           0.555   0.400   0.228   0.517   0.100   0.018   0.258   0.277   5.269   0.139   0.102   0.188   0.255
    """)
