# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['trycast']
install_requires = \
['coverage']

setup_kwargs = {
    'name': 'trycast',
    'version': '0.7.0',
    'description': '',
    'long_description': '# trycast\n\nThe trycast module defines 2 functions, `trycast()` and `isassignable()`:\n\n\n### trycast()\n\ntrycast() parses JSON-like values whose shape is defined by\n[typed dictionaries](https://www.python.org/dev/peps/pep-0589/#abstract)\n(TypedDicts) and other standard Python type hints.\n\nHere is an example of parsing a `Point2D` object defined as a `TypedDict`:\n\n```python\nfrom bottle import HTTPResponse, request, route\nfrom trycast import trycast\nfrom typing import TypedDict\n\nclass Point2D(TypedDict):\n    x: float\n    y: float\n    name: str\n\n@route(\'/draw_point\')\ndef draw_point_endpoint() -> HTTPResponse:\n    request_json = request.json  # type: object\n    if (point := trycast(Point2D, request_json)) is not None:\n        draw_point(point)  # type is narrowed to Point2D\n        return HTTPResponse(status=200)\n    else:\n        return HTTPResponse(status=400)  # Bad Request\n\ndef draw_point(point: Point2D) -> None:\n    ...\n```\n\nIn this example the `trycast` function is asked to parse a `request_json`\ninto a `Point2D` object, returning the original object (with its type narrowed\nappropriately) if parsing was successful.\n\nMore complex types can be parsed as well, such as the `Shape` in the following\nexample, which is a tagged union that can be either a `Circle` or `Rect` value:\n\n```python\nfrom bottle import HTTPResponse, request, route\nfrom trycast import trycast\nfrom typing import Literal, TypedDict, Union\n\nclass Point2D(TypedDict):\n    x: float\n    y: float\n\nclass Circle(TypedDict):\n    type: Literal[\'circle\']\n    center: Point2D  # a nested TypedDict!\n    radius: float\n\nclass Rect(TypedDict):\n    type: Literal[\'rect\']\n    x: float\n    y: float\n    width: float\n    height: float\n\nShape = Union[Circle, Rect]  # a Tagged Union!\n\n@route(\'/draw_shape\')\ndef draw_shape_endpoint() -> HTTPResponse:\n    request_json = request.json  # type: object\n    if (shape := trycast(Shape, request_json)) is not None:\n        draw_shape(shape)  # type is narrowed to Shape\n        return HTTPResponse(status=200)  # OK\n    else:\n        return HTTPResponse(status=400)  # Bad Request\n```\n\n> **Important:** Current limitations in the mypy typechecker require that you\n> add an extra `cast(Optional[Shape], ...)` around the call to `trycast`\n> in the example so that it is accepted by the typechecker without complaining:\n> \n> ```python\n> shape = cast(Optional[Shape], trycast(Shape, request_json))\n> if shape is not None:\n>     ...\n> ```\n> \n> These limitations are in the process of being resolved by\n> [introducing TypeForm support to mypy](https://github.com/python/mypy/issues/9773).\n\n\n### isassignable()\n\n`isassignable(value, T)` checks whether `value` is assignable to a variable\nof type `T` (using PEP 484 static typechecking rules), but at *runtime*.\n\nIt is similar to Python\'s builtin `isinstance()` method but\nadditionally supports checking against TypedDict types, Union types,\nLiteral types, and many others.\n\nHere is an example of checking assignability to a `Shape` object defined as a\nUnion of TypedDicts:\n\n```python\nclass Circle(TypedDict):\n    type: Literal[\'circle\']\n    ...\n\nclass Rect(TypedDict):\n    type: Literal[\'rect\']\n    ...\n\nShape = Union[Circle, Rect]  # a Tagged Union!\n\n@route(\'/draw_shape\')\ndef draw_shape_endpoint() -> HTTPResponse:\n    request_json = request.json  # type: object\n    if isassignable(request_json, Shape):\n        draw_shape(request_json)  # type is narrowed to Shape\n        return HTTPResponse(status=200)  # OK\n    else:\n        return HTTPResponse(status=400)  # Bad Request\n```\n\n> **Important:** Current limitations in the mypy typechecker prevent the\n> automatic narrowing of the type of `request_json` in the above example to\n> `Shape`, so you must add an additional `cast()` to narrow the type manually:\n> \n> ```python\n> if isassignable(request_json, Shape):\n>     shape = cast(Shape, request_json)  # type is manually narrowed to Shape\n>     draw_shape(shape)\n> ```\n> \n> These limitations are in the process of being resolved by\n> [introducing TypeForm support to mypy](https://github.com/python/mypy/issues/9773).\n\n\n## Motivation & Alternatives\n\nWhy use typed dictionaries to represent data structures instead of classes,\nnamed tuples, or other formats?\n\nTyped dictionaries are the natural form that JSON data comes in over the wire.\nThey can be trivially serialized and deserialized without any additional logic.\nFor applications that use a lot of JSON data - such as web applications - \nusing typed dictionaries is very convenient for representing data structures.\n\nOther alternatives for representing data structures in Python include\n[dataclasses], [named tuples], [attrs], and plain classes.\n\n[dataclasses]: https://www.python.org/dev/peps/pep-0557/#abstract\n[named tuples]: https://docs.python.org/3/library/typing.html#typing.NamedTuple\n[attrs]: https://www.attrs.org/en/stable/\n\n\n## Recommendations while using trycast\n\n- So that `trycast()` can recognize TypedDicts with mixed required and\n  not-required keys correctly:\n    * Use Python 3.9+ if possible.\n    * Prefer using `typing.TypedDict`, unless you must use Python 3.8.\n      In Python 3.8 prefer `typing_extensions.TypedDict` instead.\n    * Avoid using `mypy_extensions.TypedDict` in general.\n\n\n## More Information\n\nA presentation about trycast was given at the 2021 PyCon Typing Summit:\n\n[![2021 PyCon Typing Summit Presentation](https://raw.githubusercontent.com/davidfstr/trycast/main/README/TypingSummit2021_Presentation_FirstSlide.png)](https://youtu.be/ld9rwCvGdhc?t=1782)\n\n\n## License\n\n[MIT](LICENSE.md)\n\n\n## Changelog\n\n### Future\n\n* See the [Roadmap](https://github.com/davidfstr/trycast/wiki/Roadmap).\n\n### v0.7.0\n\n* Finalize the initial API:\n    * Alter `trycast()` to use `strict=True` by default rather than\n      `strict=False`. **(Breaking change)**\n    * Define trycast\'s `__all__` to export only the\n      `trycast` and `isassignable` functions.\n* Add support for additional type checkers, in addition to [Mypy]:\n    * Add support for the [Pyright] type checker and\n      [Pylance] language server extension (for Visual Studio Code).\n    * Add support for the [Pyre] type checker.\n    * Add support for the [Pytype] type checker.\n* Extend `trycast()` to recognize special `Any` and `NoReturn` values.\n* Fix `trycast()` to provide better diagnostic error when given a tuple\n  of types as its `tp` argument. Was broken in v0.6.0.\n\n[Mypy]: http://mypy-lang.org/\n[Pyright]: https://github.com/microsoft/pyright#readme\n[Pylance]: https://github.com/microsoft/pylance-release#readme\n[Pyre]: https://pyre-check.org/\n[Pytype]: https://google.github.io/pytype/\n\n### v0.6.1\n\n* Fix `trycast(..., eval=False)` to not use `typing.get_type_hints()`,\n  which internally calls `eval()`.\n* Fix `trycast()` and `isassignable()` to avoid swallowing KeyboardInterrupt\n  and other non-Exception BaseExceptions.\n\n### v0.6.0\n\n* Extend `trycast()` to recognize a stringified type argument.\n* Extend `trycast()` to report a better error message when given\n  a type argument with an unresolved forward reference (`ForwardRef`).\n* Fix `strict` argument to `trycast` to be passed to inner calls of `trycast`\n  correctly.\n    * This also fixes `isassignable()`\'s use of strict matching to be correct.\n* Alter `trycast()` to interpret a type argument of `None` or `"None"` as an\n  alias for `type(None)`, as consistent with\n  [PEP 484](https://peps.python.org/pep-0484/#using-none).\n* Alter `TypeNotSupportedError` to extend `TypeError` rather than `ValueError`.\n  **(Breaking change)**\n    * This is consistent with `trycast`\'s and `isinstance`\'s behavior of using\n      a `TypeError` rather than a `ValueError` when there is a problem with its\n      `tp` argument.\n* Drop support for Python 3.6. **(Breaking change)**\n    * Python 3.6 is end-of-life.\n\n### v0.5.0\n\n* `isassignable()` is introduced to the API:\n    * `isassignable()` leverages `trycast()` to enable type-checking\n      of values against type objects (i.e. type forms) provided at\n      runtime, using the same PEP 484 typechecking rules used by\n      typecheckers such as mypy.\n* Extend `trycast()` to recognize `Required[]` and `NotRequired[]` from\n  [PEP 655], as imported from `typing_extensions`.\n* Extend `trycast()` to support a `strict` parameter that controls whether it\n  accepts `mypy_extensions.TypedDict` or Python 3.8 `typing.TypedDict`\n  instances (which lack certain runtime type information necessary for\n  accurate runtime typechecking).\n    * For now `strict=False` by default for backward compatibility\n      with earlier versions of `trycast()`, but this default is expected\n      to be altered to `strict=True` when/before trycast v1.0.0 is released.\n* Rename primary development branch from `master` to `main`.\n\n[PEP 655]: https://www.python.org/dev/peps/pep-0655/\n\n### v0.4.0\n\n* Upgrade development status from Alpha to Beta:\n    * trycast is thoroughly tested.\n    * trycast has high code coverage (92% on Python 3.9).\n    * trycast has been in production use for over a year\n      at [at least one company] without issues\n* Add support for Python 3.10.\n* Setup continuous integration with GitHub Actions, against Python 3.6 - 3.10.\n* Migrate to the Black code style.\n* Introduce Black and isort code formatters.\n* Introduce flake8 linter.\n* Introduce coverage.py code coverage reports.\n\n[at least one company]: https://dafoster.net/projects/techsmart-platform/\n\n### v0.3.0\n\n* TypedDict improvements & fixes:\n    * Fix `trycast()` to recognize custom Mapping subclasses as TypedDicts.\n* Extend `trycast()` to recognize more JSON-like values:\n    * Extend `trycast()` to recognize `Mapping` and `MutableMapping` values.\n    * Extend `trycast()` to recognize `tuple[T, ...]` and `Tuple[T, ...]` values.\n    * Extend `trycast()` to recognize `Sequence` and `MutableSequence` values.\n* Extend `trycast()` to recognize `tuple[T1, T2, etc]` and `Tuple[T1, T2, etc]` values.\n* Documentation improvements:\n    * Improve introduction.\n    * Outline motivation to use trycast and note alternatives.\n\n### v0.2.0\n\n* TypedDict improvements & fixes:\n    * Fix `trycast()` to recognize TypedDicts from `mypy_extensions`.\n    * Extend `trycast()` to recognize TypedDicts that contain forward-references\n      to other types.\n        - Unfortunately there appears to be no easy way to support arbitrary kinds\n          of types that contain forward-references.\n        - In particular {Union, Optional} types and collection types (List, Dict)\n          with forward-references remain unsupported by `trycast()`.\n    * Recognize TypedDicts that have mixed required and not-required keys correctly.\n        - Exception: Does not work for mypy_extensions.TypedDict or\n          Python 3.8\'s typing.TypedDict due to insufficient runtime\n          type annotation information.\n    * Fix recognition of a total=False TypedDict so that extra keys are disallowed.\n* Alter `typing_extensions` to be an optional dependency of `trycast`.\n\n### v0.1.0\n\n* Add support for Python 3.6, 3.7, and 3.9, in addition to 3.8.\n\n### v0.0.2\n\n* Fix README to appear on PyPI.\n* Add other package metadata, such as the supported Python versions.\n\n### v0.0.1a\n\n* Initial release.\n* Supports typechecking all types found in JSON.\n',
    'author': 'David Foster',
    'author_email': 'david@dafoster.net',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/davidfstr/trycast',
    'py_modules': modules,
    'install_requires': install_requires,
    'python_requires': '>=3.7.4,<4.0.0',
}


setup(**setup_kwargs)
