# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['esdbclient', 'esdbclient.protos.Grpc']

package_data = \
{'': ['*']}

install_requires = \
['grpcio>=1.44.0,<1.45.0', 'protobuf>=3.20.0,<3.21.0', 'typing_extensions']

setup_kwargs = {
    'name': 'esdbclient',
    'version': '0.4.8',
    'description': 'Python gRPC Client for EventStoreDB',
    'long_description': '# Python gRPC Client for EventStoreDB\n\nThis package provides a Python gRPC client for\n[EventStoreDB](https://www.eventstore.com/). It has been\ndeveloped and tested to work with EventStoreDB LTS version 21.10,\nand with Python versions 3.7, 3.8, 3.9, and 3.10.\n\nMethods have typing annotations, the static typing is checked\nwith mypy, and the test coverage is 100%.\n\nNot all the features of the EventStoreDB API are presented\nby this client in its current form, however many of the most\nuseful aspects are presented in an easy-to-use interface (see below).\n\n## Installation\n\nUse pip to install this package from\n[the Python Package Index](https://pypi.org/project/esdbclient/).\n\n    $ pip install esdbclient\n\nIt is recommended to install Python packages into a Python virtual environment.\n\n## Getting started\n\n### Start EventStoreDB\n\nUse Docker to run EventStoreDB from the official container image on DockerHub.\n\n    $ docker run -d --name my-eventstoredb -it -p 2113:2113 -p 1113:1113 eventstore/eventstore:21.10.2-buster-slim --insecure\n\nPlease note, this will start the server without SSL/TLS enabled, allowing\nonly "insecure" connections. This version of this Python client does not\nsupport SSL/TLS connections. A future version of this library will support\n"secure" connections.\n\n### Construct client\n\nThe class `EsdbClient` can be constructed with a `uri` that indicates the\nhostname and port number of the EventStoreDB server.\n\n```python\nfrom esdbclient import EsdbClient\n\nclient = EsdbClient(uri=\'localhost:2113\')\n```\n\n### Append events\n\nThe method `append_events()` can be used to append events to\na stream. If the append operation is successful, this method\nwill return the database "commit position" as it was when the\noperation was completed. Otherwise, an exception will be raised.\n\nThe commit position value can be used to wait for downstream\nprocessing to have proceeded the appended events, so that for\nexample a user interface that depends on eventually consistent\nmaterialised views can wait after making a command before making\na query.\n\nA "commit position" is a monotonically increasing integer representing\nthe position of the recorded event in a "total order" of all recorded\nevents in the database. The sequence of commit positions is not gapless.\nIt represents the position of the event record on disk, and there are\nusually large differences between successive commits.\n\nThree arguments are required, `stream_name`, `expected_position`\nand `events`.\n\nThe `stream_name` argument is a string that uniquely identifies\nthe stream in the database.\n\nThe `expected_position` argument is an optional integer that specifies\nthe expected position of the end of the stream in the database: either\na positive integer representing the expected current position of the stream,\nor `None` if the stream is expected not to exist. If there is a mismatch\nwith the actual position of the end of the stream, an exception\n`ExpectedPositionError` will be raised by the client. This accomplishes\noptimistic concurrency control when appending new events. If you need to\nget the current position of the end of a steam, use the `get_stream_position()`\nmethod (see below). If you wish to disable optimistic concurrency, set the\n`expected_position` to a negative integer.\n\nThe `events` argument is a sequence of new event objects to be appended to the\nnamed stream. The class `NewEvent` can be used to construct new event objects.\n\nIn the example below, a stream is created by appending a new event with\n`expected_position=None`.\n\n```python\nfrom uuid import uuid4\n\nfrom esdbclient import NewEvent\n\n# Construct new event object.\nevent1 = NewEvent(\n    type=\'OrderCreated\',\n    data=b\'{}\',\n    metadata=b\'{}\'\n)\n\n# Define stream name.\nstream_name1 = str(uuid4())\n\n# Append list of events to new stream.\ncommit_position1 = client.append_events(\n    stream_name=stream_name1,\n    expected_position=None,\n    events=[event1],\n)\n```\n\nIn the example below, two subsequent events are appended to an existing\nstream. The sequences of stream positions are zero-based, and so when a\nstream only has one recorded event, the expected position of the end of\nthe stream is `0`.\n\n```python\nevent2 = NewEvent(\n    type=\'OrderUpdated\',\n    data=b\'{}\',\n    metadata=b\'{}\',\n)\nevent3 = NewEvent(\n    type=\'OrderDeleted\',\n    data=b\'{}\',\n    metadata=b\'{}\',\n)\n\ncommit_position2 = client.append_events(\n    stream_name=stream_name1,\n    expected_position=0,\n    events=[event2, event3],\n)\n```\n\nPlease note, whilst the append operation is atomic, so that either all\nor none of a given list of events will be recorded, by design it is only\npossible with EventStoreDB to atomically record events in one stream.\n\n### Get current stream position\n\nThe method `get_stream_position()` can be used to get the\nposition of the end of a stream (the position of the last\nrecorded event in the stream).\n\n```python\nstream_position = client.get_stream_position(\n    stream_name=stream_name1\n)\n\nassert stream_position == 2\n```\n\nThe sequence of stream positions is gapless. It is zero-based, so that\nthe position of the end of the stream when one event has been appended\nis `0`. The position is `1` after two events have been appended, `2`\nafter three events have been appended, and so on.\n\nIf a stream does not exist, the returned stream position is `None`,\nwhich corresponds to the required expected position when appending\nevents to a stream that does not exist (see above).\n\n```python\nstream_position = client.get_stream_position(\n    stream_name=\'stream-unknown\'\n)\n\nassert stream_position == None\n```\n\nThis method takes an optional argument `timeout` which is a float that sets\na deadline for the completion of the gRPC operation.\n\n\n### Read stream events\n\nThe method `read_stream_events()` can be used to read the recorded\nevents in a stream. An iterable object of recorded events is returned.\n\nOne argument is required, `stream_name`, which is the name of the\nstream to be read. By default, the recorded events in the stream\nare returned in the order they were recorded.\n\nThe example below shows how to read the recorded events of a stream\nforwards from the start of the stream to the end of the stream.\n\n```python\nresponse = client.read_stream_events(\n    stream_name=stream_name1\n)\n```\n\nThe iterable object is actually a Python generator, and we need to\niterate over it to actually get the recorded events from gRPC. So\nlet\'s convert it into a list, so that we can call `len()`, and so\nthat we can index it to check we have the individual events recorded\nabove.\n\n```python\nevents = list(response)\n```\n\nNow that we have an actual list of events, we can check we have the\nthree events that we recorded in the stream above.\n\n```python\nassert len(events) == 3\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 0\nassert events[0].type == event1.type\nassert events[0].data == event1.data\n\nassert events[1].stream_name == stream_name1\nassert events[1].stream_position == 1\nassert events[1].type == event2.type\nassert events[1].data == event2.data\n\nassert events[2].stream_name == stream_name1\nassert events[2].stream_position == 2\nassert events[2].type == event3.type\nassert events[2].data == event3.data\n```\n\nThe method `read_stream_events()` also supports four optional arguments,\n`position`, `backwards`, `limit`, and `timeout`.\n\nThe argument `position` is an optional integer that can be used to indicate\nthe position in the stream from which to start reading. This argument is `None`\nby default, which means the stream will be read either from the start of the\nstream (the default behaviour), or from the end of the stream if `backwards` is\n`True`. When reading a stream from a specific position in the stream, the\nrecorded event at that position WILL be included, both when reading forwards\nfrom that position, and when reading backwards from that position.\n\nThe argument `backwards` is a boolean, by default `False`, which means the\nstream will be read forwards by default, so that events are returned in the\norder they were appended, If `backwards` is `True`, the stream will be read\nbackwards, so that events are returned in reverse order.\n\nThe argument `limit` is an integer which limits the number of events that will\nbe returned.\n\nThe argument `timeout` is a float which sets a deadline for the completion of\nthe gRPC operation.\n\nThe example below shows how to read recorded events in a stream forwards from\na specific stream position to the end of the stream.\n\n```python\nevents = list(\n    client.read_stream_events(\n        stream_name=stream_name1,\n        position=1,\n    )\n)\n\nassert len(events) == 2\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 1\nassert events[0].type == event2.type\nassert events[0].data == event2.data\n\nassert events[1].stream_name == stream_name1\nassert events[1].stream_position == 2\nassert events[1].type == event3.type\nassert events[1].data == event3.data\n```\n\nThe example below shows how to read the recorded events in a stream backwards from\nthe end of the stream to the start of the stream.\n\n```python\nevents = list(\n    client.read_stream_events(\n        stream_name=stream_name1,\n        backwards=True,\n    )\n)\n\nassert len(events) == 3\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 2\nassert events[0].type == event3.type\nassert events[0].data == event3.data\n\nassert events[1].stream_name == stream_name1\nassert events[1].stream_position == 1\nassert events[1].type == event2.type\nassert events[1].data == event2.data\n```\n\nThe example below shows how to read a limited number (two) of the recorded events\nin stream forwards from the start of the stream.\n\n```python\nevents = list(\n    client.read_stream_events(\n        stream_name=stream_name1,\n        limit=2,\n    )\n)\n\nassert len(events) == 2\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 0\nassert events[0].type == event1.type\nassert events[0].data == event1.data\n\nassert events[1].stream_name == stream_name1\nassert events[1].stream_position == 1\nassert events[1].type == event2.type\nassert events[1].data == event2.data\n```\n\nThe example below shows how to read a limited number of the recorded events\nin a stream backwards from a given stream position.\n\n```python\nevents = list(\n    client.read_stream_events(\n        stream_name=stream_name1,\n        position=2,\n        backwards=True,\n        limit=1,\n    )\n)\n\nassert len(events) == 1\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 2\nassert events[0].type == event3.type\nassert events[0].data == event3.data\n```\n\n### Read all recorded events\n\nThe method `read_all_events()` can be used to read all recorded events\nin the database in the order they were committed. An iterable object of\nrecorded events is returned.\n\nThe example below shows how to read all events in the database in the\norder they were recorded.\n\n```python\nevents = list(client.read_all_events())\n\nassert len(events) >= 3\n```\n\nThe method `read_stream_events()` supports six optional arguments,\n`position`, `backwards`, `filter_exclude`, `filter_include`, `limit`,\nand `timeout`.\n\nThe argument `position` is an optional integer that can be used to specify\nthe commit position from which to start reading. This argument is `None` by\ndefault, meaning that all the events will be read either from the start, or\nfrom the end if `backwards` is `True` (see below). Please note, if specified,\nthe specified position must be an actually existing commit position, because\nany other number will result in a server error (at least in EventStoreDB v21.10).\nPlease also note, when reading forwards the event at the given position\nWILL be included. However when reading backwards, the event at the given position\nwill NOT be included.\n\nThe argument `backwards` is a boolean which is by default `False` meaning all the\nevents will be read forwards by default, so that events are returned in the\norder they were committed, If `backwards` is `True`, all the events will be read\nbackwards, so that events are returned in reverse order.\n\nThe argument `filter_exclude` is a sequence of regular expressions that\nmatch the type strings of recorded events that should not be included.\nThis argument is ignored if `filter_include` is set.\n\nThe argument `filter_include` is a sequence of regular expressions\nthat match the type strings of recorded events that should be included. By\ndefault, this argument is an empty tuple. If this argument is set to a\nnon-empty sequence, the `filter_include` argument is ignored.\n\nPlease note, the filtering happens on the EventStoreDB server, and the\n`limit` argument is applied after filtering. See below for more information\nabout filter regular expressions.\n\nThe argument `limit` is an integer which limits the number of events that will\nbe returned.\n\nThe argument `timeout` is a float which sets a deadline for the completion of\nthe gRPC operation.\n\nThe example below shows how to read all recorded events from a particular commit position.\n\n```python\nevents = list(\n    client.read_all_events(\n        position=commit_position1\n    )\n)\n\nassert len(events) == 3\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 0\nassert events[0].type == event1.type\nassert events[0].data == event1.data\n\nassert events[1].stream_name == stream_name1\nassert events[1].stream_position == 1\nassert events[1].type == event2.type\nassert events[1].data == event2.data\n\nassert events[2].stream_name == stream_name1\nassert events[2].stream_position == 2\nassert events[2].type == event3.type\nassert events[2].data == event3.data\n```\n\nThe example below shows how to read all recorded events in reverse order.\n\n```python\nevents = list(\n    client.read_all_events(\n        backwards=True\n    )\n)\n\nassert len(events) >= 3\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 2\nassert events[0].type == event3.type\nassert events[0].data == event3.data\n\nassert events[1].stream_name == stream_name1\nassert events[1].stream_position == 1\nassert events[1].type == event2.type\nassert events[1].data == event2.data\n\nassert events[2].stream_name == stream_name1\nassert events[2].stream_position == 0\nassert events[2].type == event1.type\nassert events[2].data == event1.data\n```\n\nThe example below shows how to read a limited number (one) of the recorded events\nin the database forwards from a specific commit position.\n\n```python\nevents = list(\n    client.read_all_events(\n        position=commit_position1,\n        limit=1,\n    )\n)\n\nassert len(events) == 1\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 0\nassert events[0].type == event1.type\nassert events[0].data == event1.data\n```\n\nThe example below shows how to read a limited number (one) of the recorded events\nin the database backwards from the end. This gives the last recorded event.\n\n```python\nevents = list(\n    client.read_all_events(\n        backwards=True,\n        limit=1,\n    )\n)\n\nassert len(events) == 1\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 2\nassert events[0].type == event3.type\nassert events[0].data == event3.data\n```\n\n### Get current commit position\n\nThe method `get_commit_position()` can be used to get the current\ncommit position of the database.\n\n```python\ncommit_position = client.get_commit_position()\n```\n\nThis method is provided as a convenience when testing, and otherwise isn\'t\nvery useful. In particular, when reading all events (see above) or subscribing\nto all events with a catch-up subscription (see below), the commit position\nwould normally be read from the downstream database, so that you are reading\nfrom the last position that was successfully processed.\n\nThis method takes an optional argument `timeout` which is a float that sets\na deadline for the completion of the gRPC operation.\n\n### Catch-up subscriptions\n\nThe method `subscribe_all_events()` can be used to create a\n"catch-up subscription" to EventStoreDB. The optional argument\n`position` can be used to specify a commit position from which\nto receive recorded events. Please note, returned events are those\nafter the given commit position.\n\nThis method returns a subscription object, which is an iterable object,\nfrom which recorded events can be obtained by iteration.\n\nThe example below shows how to subscribe to receive all recorded\nevents from a specific commit position. Three already-existing\nevents are received, and then three new events are recorded, which\nare then received via the subscription.\n\n```python\n\n# Get the commit position (usually from database of materialised views).\ncommit_position = client.get_commit_position()\n\n# Append three events.\nstream_name1 = str(uuid4())\nevent1 = NewEvent(\n    type=\'OrderCreated\',\n    data=b\'{}\',\n    metadata=b\'{}\',\n)\nevent2 = NewEvent(\n    type=\'OrderUpdated\',\n    data=b\'{}\',\n    metadata=b\'{}\',\n)\nevent3 = NewEvent(\n    type=\'OrderDeleted\',\n    data=b\'{}\',\n    metadata=b\'{}\',\n)\nclient.append_events(\n    stream_name=stream_name1,\n    expected_position=None,\n    events=[event1, event2, event3],\n)\n\n# Subscribe from the commit position.\nsubscription = client.subscribe_all_events(\n    position=commit_position\n)\n\n# Catch up by receiving the three events from the subscription.\nevents = []\nfor event in subscription:\n    # Check the stream name is \'stream_name1\'.\n    assert event.stream_name == stream_name1\n    events.append(event)\n    if len(events) == 3:\n        break\n\n# Append three more events.\nstream_name = str(uuid4())\nevent4 = NewEvent(\n    type=\'OrderCreated\',\n    data=b\'{}\',\n    metadata=b\'{}\',\n)\nevent5 = NewEvent(\n    type=\'OrderUpdated\',\n    data=b\'{}\',\n    metadata=b\'{}\',\n)\nevent6 = NewEvent(\n    type=\'OrderDeleted\',\n    data=b\'{}\',\n    metadata=b\'{}\',\n)\nclient.append_events(\n    stream_name=stream_name,\n    expected_position=None,\n    events=[event4, event5, event6],\n)\n\n# Receive the three new events from the same subscription.\nevents = []\nfor event in subscription:\n    # Check the stream name is \'stream_name2\'.\n    assert event.stream_name == stream_name\n    events.append(event)\n    if len(events) == 3:\n        break\n```\n\nCatch-up subscriptions are not registered in EventStoreDB (they are not\n"persistent subscriptions). It is simply a streaming gRPC call which is\nkept open by the server, with newly recorded events sent to the client\nas the client iterates over the subscription. This kind of subscription\nis closed as soon as the subscription object goes out of memory.\n\n```python\n# End the subscription.\ndel subscription\n```\n\nTo accomplish "exactly once" processing of the events, the commit position\nshould be recorded atomically and uniquely along with the result of processing\nreceived events, for example in the same database as materialised views when\nimplementing eventually-consistent CQRS, or in the same database as a downstream\nanalytics or reporting or archiving application. This avoids "dual writing" in\nthe processing of events.\n\nThe subscription object might be used directly when processing events. It might\nalso be used within a thread dedicated to receiving events, with recorded events\nput on a queue for processing in a different thread. This package doesn\'t provide\nsuch thread or queue objects, you would need to do that yourself. Just make sure\nto reconstruct the subscription (and the queue) using your last recorded commit\nposition when resuming the subscription after an error, to be sure all events\nare processed once.\n\nMany catch-up subscriptions can be created, and all will receive all the events\nthey are subscribed to receive. Received events do not need to be (and indeed\ncannot be) acknowledged back to the EventStoreDB server. Acknowledging events\nis an aspect of "persistent subscriptions", which is a feature of EventStoreDB\nthat is not (currently) supported by this client. Whilst there are some advantages\nof persistent subscribers, by recording in the upstream server the position in\nthe commit sequence of events that have been processed, there is a danger of\n"dual writing" in the consumption of events. The danger is that if the event\nis successfully processed but then the acknowledgment fails, the event may be\nprocessed more than once. On the other hand, if the acknowledgment is successful\nbut then the processing fails, the event may not be been processed. The only\nprotection against this danger is to avoid "dual writing" by atomically recording\nthe commit position of an event that has been processed along with the results of\nprocess the event, that is with both things being recorded in the same transaction.\n\nThis method also support three other optional arguments, `filter_exclude`,\n`filter_include`, and `timeout`.\n\nThe argument `filter_exclude` is a sequence of regular expressions that\nmatch the type strings of recorded events that should not be included.\nThis argument is ignored if `filter_include` is set.\n\nThe argument `filter_include` is a sequence of regular expressions\nthat match the type strings of recorded events that should be included. By\ndefault, this argument is an empty tuple. If this argument is set to a\nnon-empty sequence, the `filter_include` argument is ignored.\n\nPlease note, in this version of this Python client, the filtering happens\nwithin the client (rather than on the server as when reading all events) because\npassing these filter options in the read request for subscriptions seems to cause\nan error in EventStoreDB v21.10. See below for more information about filter\nregular expressions.\n\nThe argument `timeout` is a float which sets a deadline for the completion of\nthe gRPC operation. This probably isn\'t very useful, but is included for\ncompleteness and consistency with the other methods.\n\n### The NewEvent class\n\nThe `NewEvent` class can be used to define new events.\n\nThe attribute `type` is a unicode string, used to specify the type of the event\nto be recorded.\n\nThe attribute `data` is a byte string, used to specify the data of the event\nto be recorded. Please note, in this version of this Python client,\nwriting JSON event data to EventStoreDB isn\'t supported, but it might be in\na future version.\n\nThe attribute `metadata` is a byte string, used to specify metadata for the event\nto be recorded.\n\n```python\nnew_event = NewEvent(\n    type=\'OrderCreated\',\n    data=b\'{}\',\n    metadata=b\'{}\',\n)\n```\n\n### The RecordedEvent class\n\nThe `RecordedEvent` class is used when reading recorded events.\n\nThe attribute `type` is a unicode string, used to indicate the type of the event\nthat was recorded.\n\nThe attribute `data` is a byte string, used to indicate the data of the event\nthat was recorded.\n\nThe attribute `metadata` is a byte string, used to indicate metadata for the event\nthat was recorded.\n\nThe attribute `stream_name` is a unicode string, used to indicate the type of\nthe name of the stream in which the event was recorded.\n\nThe attribute `stream_position` is an integer, used to indicate\nthe position in the stream at which the event was recorded.\n\nThe attribute `commit_position` is an integer, used to indicate\nthe position in total order of all recorded events at which the\nevent was recorded.\n\n```python\nfrom esdbclient.events import RecordedEvent\n\nrecorded_event = RecordedEvent(\n    type=\'OrderCreated\',\n    data=b\'{}\',\n    metadata=b\'{}\',\n    stream_name=\'stream1\',\n    stream_position=0,\n    commit_position=512,\n)\n```\n\n### Filter regular expressions\n\nThe `filter_exclude` and `filter_include` arguments in `read_all_events()` and\n`subscribe_all_events()` are applied to the `type` attribute of recorded events.\n\nThe default value of the `filter_exclude` arguments is designed to exclude\nEventStoreDB "system events", which otherwise would be included. System\nevents, by convention in EventStoreDB, all have `type` strings that\nstart with the `$` sign.\n\nPlease note, characters that have a special meaning in regular expressions\nwill need to be escaped (with double-backslash) when matching these characters\nin type strings.\n\nFor example, to match EventStoreDB system events, use the sequence `[\'\\\\$.*\']`.\nPlease note, the constant `ESDB_EVENTS_REGEX` is set to `\'\\\\$.*\'`. You\ncan import this value (`from esdbclient import ESDB_EVENTS_REGEX`) and use\nit when building longer sequences of regular expressions. For example,\nto exclude system events and snapshots, you might use the sequence\n`[ESDB_EVENTS_REGEX, \'.*Snapshot\']` as the value of the `filter_exclude`\nargument.\n\n\n### Stop EventStoreDB\n\nUse Docker to stop and remove the EventStoreDB container.\n\n    $ docker stop my-eventstoredb\n\t$ docker rm my-eventstoredb\n\n\n## Developers\n\nClone the project repository, set up a virtual environment, and install\ndependencies.\n\nUse your IDE (e.g. PyCharm) to open the project repository. Create a\nPoetry virtual environment, and then update packages.\n\n    $ make update-packages\n\nAlternatively, use the ``make install`` command to create a dedicated\nPython virtual environment for this project.\n\n    $ make install\n\nThe ``make install`` command uses the build tool Poetry to create a dedicated\nPython virtual environment for this project, and installs popular development\ndependencies such as Black, isort and pytest.\n\nAdd tests in `./tests`. Add code in `./esdbclient`.\n\nStart EventStoreDB.\n\n    $ make start-eventstoredb\n\nRun tests.\n\n    $ make test\n\nStop EventStoreDB.\n\n    $ make stop-eventstoredb\n\nCheck the formatting of the code.\n\n    $ make lint\n\nReformat the code.\n\n    $ make fmt\n\nAdd dependencies in `pyproject.toml` and then update installed packages.\n\n    $ make update-packages\n',
    'author': 'John Bywater',
    'author_email': 'john.bywater@appropriatesoftware.net',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/pyeventsourcing/esdbclient',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
