"""
Copyright (c) 2022 Windhover Labs, L.L.C. All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
 notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
 notice, this list of conditions and the following disclaimer in
 the documentation and/or other materials provided with the
 distribution.
3. Neither the name Windhover Labs nor the names of its
 contributors may be used to endorse or promote products derived
 from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

"""
"""
This module can generate an XTCE file. It gathers the data from a sqlite database, which is generated by juicer.
You can learn more about juicer here [1]. At the time of writing we try our best to
follow the XTCE Standard Version 1.2. You can read more about the XTCE standard here [2]. It should be noted that
while we do try to stay 100%-compliant to XTCE, we use our XTCE files with YAMCS[3]. Thus we lean towards YAMCS-flavor
of XTCE, which is mostly complaint with the standard.

[1]:https://github.com/WindhoverLabs/juicer
[2]:https://www.omg.org/spec/XTCE/1.2/PDF
[3]:https://yamcs.org/
"""
from pathlib import Path
import random
from typing import Union

try:
    from xtce_generator.src.xtce import xtce
except ModuleNotFoundError:
    import xtce.xtce as xtce
import argparse
import sqlite3
import logging
import yaml
from enum import Enum
import sys
import os

# FIXME:Not sure if I should move these to another file. Especially given the fact there will be more constants in the future.
_ROOT_SPACESYSTEM_KEY = "root_spacesystem"
_CPU_ID_KEY = "cpu_id"


# # TODO:Implementation to enable nesting of spacesystems beyond depth of root+1
# class NameSpace(xtce.SpaceSystemType):
#     def __init__(self, shortDescription=None, LongDescription=None, AliasSet=None, AncillaryDataSet=None, name=None,
#                  operationalStatus=None, base=None, Header=None, TelemetryMetaData=None, CommandMetaData=None,
#                  ServiceSet=None, SpaceSystem=None, gds_collector_=None, **kwargs_):
#         super(NameSpace, self).__init__(shortDescription, LongDescription, AliasSet, AncillaryDataSet, name,
#                                         operationalStatus, base, Header, TelemetryMetaData,
#                                         CommandMetaData,
#                                         ServiceSet, SpaceSystem, gds_collector_, **kwargs_)
#         self.__namespace_dict = dict({name:{name: self}})
#
#     def __getitem__(self, key: str):
#         pass


class RefType(int, Enum):
    """
    Used by XTCEManager to resolve different types based on typeref values.
    """
    BaseType = 0
    AGGREGATE = 1
    ENUM = 2


class XTCEManager:
    UNKNOWN_TYPE = 'UNKNOWN'  # A type for anything that, for some reason, we don't understand from the database
    BASE_TYPE_NAMESPACE = 'BaseType'
    NAMESPACE_SEPARATOR = '/'
    ARRAY_BASE_NAME = "Array"

    def __init__(self, root_space_system: str, file_path: str, sqlite_path: str, config: dict, cpu_id: str = None):
        """
        Instantiates a XTCEManager instance. An XTCEManager is a class that manages an xtce class internally
        and provides utilities such as serialization tools(through the write_to_file method) and functionality
        to add namespaces, BaseType, etc to our root spacesystem. It reads all data(except for base types)
        from the sqlite database at sqlite_path. Beware that the database is assumed to have been generated by
        our juicer tool. You can check out juicer here[1]
        [1]:https://github.com/WindhoverLabs/juicer
        :param root_space_system: The name of the root spacesystem in the output XTCE file.
        :param file_name: The name of the output file that has all of the XTCE.
        :param destination_dir: In which directory to write the outputX XTCE file to. The default is the current
        working directory.
        :param sqlite_path: The file path to the sqlite database that XTCE will be generated from.
        :param config: An optional configuration dict to configure things like parent containers. Very useful for
        decoupling protocols such as CCSDS and MAVLink in a ground system.
        :param cpu_id: The id of the flight computer(CPD, PPD, etc).
        """

        self.root = xtce.SpaceSystemType(name=root_space_system)

        # This is a horrid hack, I know. Will try to come up with something more elegant.
        # The problem is sometimes CPU_ID is present, sometimes it is not.
        self.nested_root = False
        if cpu_id:
            # FIXME:This will work, but it's really wack. We should be using nested namespaces(?)
            self.nested_root = True
            self.root.add_SpaceSystem(xtce.SpaceSystemType(name=cpu_id))
            self.root = self.root.get_SpaceSystem()[len(self.root.get_SpaceSystem()) - 1]

        self.telemetry_metadata = xtce.TelemetryMetaDataType()
        self.command_metadata = xtce.CommandMetaDataType()
        self.parameter_type_set = xtce.ParameterTypeSetType()
        self.argument_type_set = xtce.ArgumentTypeSetType()
        self.output_file = Path(file_path)
        self.telemetry_metadata.set_ParameterTypeSet(self.parameter_type_set)
        self.command_metadata.set_ArgumentTypeSet(self.argument_type_set)
        self.root.set_TelemetryMetaData(self.telemetry_metadata)
        self.root.set_CommandMetaData(self.command_metadata)

        if not(XTCEManager.NAMESPACE_SEPARATOR in root_space_system):
            self.__namespace_dict = dict({XTCEManager.NAMESPACE_SEPARATOR + root_space_system: self.root})
        else:
            self.__namespace_dict = dict({root_space_system: self.root})

        # FIXME: Would like to avoid initializing database connection in constructor.
        db_handle = sqlite3.connect(sqlite_path)

        self.db_cursor = db_handle.cursor()

        # FIXME: We need a validator for our yaml config
        self.custom_config = config

    def __get_telemetry_base_container_length(self) -> int:
        """
        Retrieves the length of our telemetry base container from the user-provided yaml configuration file, if it is
        provided.
        :return: The length of the base container in bits. If it cannot be retrieved, None is returned.
        """
        out_length = None

        # FIXME: If we had a validator for our config file, we wouldn't have to be checking for every key
        if self.custom_config:
            if 'global' in self.custom_config.keys():
                if 'TelemetryMetaData' in self.custom_config['global'].keys():
                    if 'BaseContainer' in self.custom_config['global']['TelemetryMetaData']:
                        if 'size' in self.custom_config['global']['TelemetryMetaData']['BaseContainer']:
                            out_length = self.custom_config['global']['TelemetryMetaData']['BaseContainer']['size']

        return out_length

    def __get_command_base_container_length(self):
        """
        Retrieves the length of our command base container from the user-provided yaml configuration file, if it is
        provided.
        :return: The length of the base container in bits. If it cannot be retrieved, None is returned.
        """
        out_length = None

        if self.custom_config:
            if 'global' in self.custom_config.keys():
                if 'CommandMetaData' in self.custom_config['global'].keys():
                    if 'BaseContainer' in self.custom_config['global']['CommandMetaData']:
                        if 'size' in self.custom_config['global']['CommandMetaData']['BaseContainer']:
                            out_length = self.custom_config['global']['CommandMetaData']['BaseContainer']['size']
        return out_length

    def __get_endianness_postfix(self, little_endian: bool) -> str:
        endianness = '_LE' if little_endian else '_BE'
        return endianness

    def __get_int_argtype(self, bit_size: int, little_endian: bool) -> xtce.IntegerArgumentType:
        """
        Factory function to construct a IntegerArgumentType.
        :param bit_size: The size of the type in bits.
        :param little_endian:
        :return: A IntegerArgumentType with an encoding of signed int, size of bit_size and endianness dependent on the
        little_endian flag.
        """
        endianness = self.__get_endianness_postfix(little_endian)

        base_type_name = 'int' + str(bit_size) + endianness

        arg_type = xtce.IntegerArgumentType(name=base_type_name, signed=True)

        bit_order = xtce.BitOrderType.LEAST_SIGNIFICANT_BIT_FIRST if little_endian else \
            xtce.BitOrderType.MOST_SIGNIFICANT_BIT_FIRST

        byte_order = xtce.ByteOrderCommonType.LEAST_SIGNIFICANT_BYTE_FIRST if little_endian else \
            xtce.ByteOrderCommonType.MOST_SIGNIFICANT_BYTE_FIRST

        if bit_size > 8:
            base_type_data_encoding = xtce.IntegerDataEncodingType(sizeInBits=bit_size,
                                                                   bitOrder=bit_order,
                                                                   byteOrder=byte_order,
                                                                   encoding=xtce.IntegerEncodingType.TWOS_COMPLEMENT
                                                                   )
        else:
            base_type_data_encoding = xtce.IntegerDataEncodingType(sizeInBits=bit_size,
                                                                   bitOrder=bit_order,
                                                                   encoding=xtce.IntegerEncodingType.TWOS_COMPLEMENT
                                                                   )

        arg_type.set_IntegerDataEncoding(base_type_data_encoding)

        return arg_type

    def __get_uint_argtype(self, bit_size: int, little_endian: bool) -> xtce.IntegerArgumentType:
        """
        Factory function to construct a IntegerArgumentType.
        :param bit_size:
        :param little_endian:
        :return:
        """
        endianness = self.__get_endianness_postfix(little_endian)

        base_type_name = 'uint' + str(bit_size) + endianness

        arg_type = xtce.IntegerArgumentType(name=base_type_name, signed=False)
        bit_order = xtce.BitOrderType.LEAST_SIGNIFICANT_BIT_FIRST if little_endian else \
            xtce.BitOrderType.MOST_SIGNIFICANT_BIT_FIRST

        byte_order = xtce.ByteOrderCommonType.LEAST_SIGNIFICANT_BYTE_FIRST if little_endian else \
            xtce.ByteOrderCommonType.MOST_SIGNIFICANT_BYTE_FIRST

        if bit_size > 8:
            base_type_data_encoding = xtce.IntegerDataEncodingType(sizeInBits=bit_size,
                                                                   bitOrder=bit_order,
                                                                   byteOrder=byte_order,
                                                                   encoding=xtce.IntegerEncodingType.UNSIGNED
                                                                   )
        else:
            base_type_data_encoding = xtce.IntegerDataEncodingType(sizeInBits=bit_size,
                                                                   bitOrder=bit_order,
                                                                   encoding=xtce.IntegerEncodingType.UNSIGNED
                                                                   )

        arg_type.set_IntegerDataEncoding(base_type_data_encoding)

        return arg_type

    def __get_string_argtype(self, bit_size: int, little_endian: bool) -> xtce.StringArgumentType:
        """
        Factory function to construct a StringArgumentType.
        :param bit_size: The size of the type in bits.
        :param little_endian:
        :return: A StringArgumentType with an encoding of TerminatorChar as '\0' and a max size of(defined by SizeInBits)
        of bit_size and endianness dependent on the little_endian flag.
        """
        endianness = self.__get_endianness_postfix(little_endian)

        base_type_name = 'string' + str(bit_size) + endianness

        arg_type = xtce.StringArgumentType(name=base_type_name, signed=True)

        bit_order = xtce.BitOrderType.LEAST_SIGNIFICANT_BIT_FIRST if little_endian else \
            xtce.BitOrderType.MOST_SIGNIFICANT_BIT_FIRST

        byte_order = xtce.ByteOrderCommonType.LEAST_SIGNIFICANT_BYTE_FIRST if little_endian else \
            xtce.ByteOrderCommonType.MOST_SIGNIFICANT_BYTE_FIRST

        size_in_bits = xtce.SizeInBitsType()
        size_in_bits.set_TerminationChar("00")
        size_in_bits.set_Fixed(xtce.FixedType(bit_size))

        if bit_size >= 8:
            base_type_data_encoding = xtce.StringDataEncodingType(bitOrder=bit_order,
                                                                  byteOrder=byte_order,
                                                                  encoding=xtce.StringEncodingType.UTF_8
                                                                  )
        else:
            logging.warning('Creating a string that is less than 8 bits. I hope you know what you are doing.')
            base_type_data_encoding = xtce.StringDataEncodingType(sizeInBits=size_in_bits,
                                                                  bitOrder=bit_order,
                                                                  encoding=xtce.StringEncodingType.UTF_8
                                                                  )

        base_type_data_encoding.set_SizeInBits(size_in_bits)
        arg_type.set_StringDataEncoding(base_type_data_encoding)

        return arg_type

    def __get_string_paramtype(self, bit_size: int, little_endian: bool) -> xtce.StringParameterType:
        """
        Factory function to construct a StringParameterType.
        :param bit_size: The size of the type in bits.
        :param little_endian:
        :return: A StringParameterType with an encoding of TerminatorChar as '\0' and a max size of(defined by SizeInBits)
        of bit_size and endianness dependent on the little_endian flag.
        """
        endianness = self.__get_endianness_postfix(little_endian)

        base_type_name = 'string' + str(bit_size) + endianness

        param_type = xtce.StringParameterType(name=base_type_name)

        bit_order = xtce.BitOrderType.LEAST_SIGNIFICANT_BIT_FIRST if little_endian else \
            xtce.BitOrderType.MOST_SIGNIFICANT_BIT_FIRST

        byte_order = xtce.ByteOrderCommonType.LEAST_SIGNIFICANT_BYTE_FIRST if little_endian else \
            xtce.ByteOrderCommonType.MOST_SIGNIFICANT_BYTE_FIRST

        size_in_bits = xtce.SizeInBitsType()
        size_in_bits.set_TerminationChar("00")
        size_in_bits.set_Fixed(xtce.FixedType(bit_size))

        if bit_size >= 8:
            base_type_data_encoding = xtce.StringDataEncodingType(bitOrder=bit_order,
                                                                  byteOrder=byte_order,
                                                                  encoding=xtce.StringEncodingType.UTF_8
                                                                  )
        else:
            logging.warning('Creating a string that is less than 8 bits. I hope you know what you are doing.')
            base_type_data_encoding = xtce.StringDataEncodingType(bitOrder=bit_order,
                                                                  encoding=xtce.StringEncodingType.UTF_8
                                                                  )
        base_type_data_encoding.set_SizeInBits(size_in_bits)
        param_type.set_StringDataEncoding(base_type_data_encoding)

        return param_type

    def __get_padding_argtype(self, bit_size: int, little_endian: bool) -> xtce.IntegerArgumentType:
        """
        Factory function to construct a IntegerArgumentType.
        :param bit_size:
        :param little_endian:
        :return:
        """
        endianness = self.__get_endianness_postfix(little_endian)

        base_type_name = '_padding' + str(bit_size) + endianness

        arg_type = xtce.IntegerArgumentType(name=base_type_name, signed=False)
        bit_order = xtce.BitOrderType.LEAST_SIGNIFICANT_BIT_FIRST if little_endian else \
            xtce.BitOrderType.MOST_SIGNIFICANT_BIT_FIRST

        byte_order = xtce.ByteOrderCommonType.LEAST_SIGNIFICANT_BYTE_FIRST if little_endian else \
            xtce.ByteOrderCommonType.MOST_SIGNIFICANT_BYTE_FIRST

        if bit_size > 8:
            base_type_data_encoding = xtce.IntegerDataEncodingType(sizeInBits=bit_size,
                                                                   bitOrder=bit_order,
                                                                   byteOrder=byte_order,
                                                                   encoding=xtce.IntegerEncodingType.UNSIGNED
                                                                   )
        else:
            base_type_data_encoding = xtce.IntegerDataEncodingType(sizeInBits=bit_size,
                                                                   bitOrder=bit_order,
                                                                   encoding=xtce.IntegerEncodingType.UNSIGNED
                                                                   )

        arg_type.set_IntegerDataEncoding(base_type_data_encoding)

        return arg_type

    def __get_float_argtype(self, bit_size: int, little_endian: bool) -> xtce.FloatArgumentType:
        """
        Factory function to construct a IntegerParameterType.
        :param bit_size:
        :param little_endian:
        :return:
        """
        endianness = self.__get_endianness_postfix(little_endian)

        base_type_name = 'float' + str(bit_size) + endianness

        bit_order = xtce.BitOrderType.LEAST_SIGNIFICANT_BIT_FIRST if little_endian else \
            xtce.BitOrderType.MOST_SIGNIFICANT_BIT_FIRST

        byte_order = xtce.ByteOrderCommonType.LEAST_SIGNIFICANT_BYTE_FIRST if little_endian else \
            xtce.ByteOrderCommonType.MOST_SIGNIFICANT_BYTE_FIRST

        if bit_size > 8:
            encoding = xtce.FloatDataEncodingType(sizeInBits=str(bit_size),
                                                  bitOrder=bit_order,
                                                  byteOrder=byte_order)
        else:
            encoding = xtce.FloatDataEncodingType(sizeInBits=str(bit_size),
                                                  bitOrder=bit_order)

        param_type = xtce.FloatArgumentType(name=base_type_name, sizeInBits=str(bit_size))
        param_type.set_FloatDataEncoding(encoding)

        return param_type

    def __get_int_paramtype(self, bit_size: int, little_endian: bool) -> xtce.IntegerDataType:
        """
        Factory function to construct a IntegerParameterType.
        :param bit_size:
        :param little_endian:
        :return:
        """
        endianness = self.__get_endianness_postfix(little_endian)

        base_type_name = 'int' + str(bit_size) + endianness

        param_type = xtce.IntegerParameterType(name=base_type_name, signed=True)

        bit_size = bit_size
        bit_order = xtce.BitOrderType.LEAST_SIGNIFICANT_BIT_FIRST if little_endian else \
            xtce.BitOrderType.MOST_SIGNIFICANT_BIT_FIRST

        byte_order = xtce.ByteOrderCommonType.LEAST_SIGNIFICANT_BYTE_FIRST if little_endian else \
            xtce.ByteOrderCommonType.MOST_SIGNIFICANT_BYTE_FIRST

        if bit_size > 8:
            base_type_data_encoding = xtce.IntegerDataEncodingType(sizeInBits=bit_size,
                                                                   bitOrder=bit_order,
                                                                   byteOrder=byte_order,
                                                                   encoding=xtce.IntegerEncodingType.TWOS_COMPLEMENT
                                                                   )
        else:
            base_type_data_encoding = xtce.IntegerDataEncodingType(sizeInBits=bit_size,
                                                                   bitOrder=bit_order,
                                                                   encoding=xtce.IntegerEncodingType.TWOS_COMPLEMENT
                                                                   )

        param_type.set_IntegerDataEncoding(base_type_data_encoding)

        return param_type

    def __get_uint_paramtype(self, bit_size: int, little_endian: bool) -> xtce.IntegerDataType:
        """
        Factory function to construct a IntegerParameterType.
        :param bit_size:
        :param little_endian:
        :return:
        """
        endianness = self.__get_endianness_postfix(little_endian)

        base_type_name = 'uint' + str(bit_size) + endianness

        param_type = xtce.IntegerParameterType(name=base_type_name, signed=False, sizeInBits=bit_size)

        bit_order = xtce.BitOrderType.LEAST_SIGNIFICANT_BIT_FIRST if little_endian else \
            xtce.BitOrderType.MOST_SIGNIFICANT_BIT_FIRST

        byte_order = xtce.ByteOrderCommonType.LEAST_SIGNIFICANT_BYTE_FIRST if little_endian else \
            xtce.ByteOrderCommonType.MOST_SIGNIFICANT_BYTE_FIRST

        if bit_size > 8:
            base_type_data_encoding = xtce.IntegerDataEncodingType(sizeInBits=bit_size,
                                                                   bitOrder=bit_order,
                                                                   byteOrder=byte_order,
                                                                   encoding=xtce.IntegerEncodingType.UNSIGNED
                                                                   )
        else:
            base_type_data_encoding = xtce.IntegerDataEncodingType(sizeInBits=bit_size,
                                                                   bitOrder=bit_order,
                                                                   encoding=xtce.IntegerEncodingType.UNSIGNED
                                                                   )

        param_type.set_IntegerDataEncoding(base_type_data_encoding)

        return param_type

    def __get_padding_paramtype(self, bit_size: int, little_endian: bool) -> xtce.IntegerDataType:
        """
        Factory function to construct a IntegerParameterType.
        :param bit_size:
        :param little_endian:
        :return:
        """
        endianness = self.__get_endianness_postfix(little_endian)

        base_type_name = '_padding' + str(bit_size) + endianness

        param_type = xtce.IntegerParameterType(name=base_type_name, signed=False, sizeInBits=bit_size)

        bit_order = xtce.BitOrderType.LEAST_SIGNIFICANT_BIT_FIRST if little_endian else \
            xtce.BitOrderType.MOST_SIGNIFICANT_BIT_FIRST

        byte_order = xtce.ByteOrderCommonType.LEAST_SIGNIFICANT_BYTE_FIRST if little_endian else \
            xtce.ByteOrderCommonType.MOST_SIGNIFICANT_BYTE_FIRST

        if bit_size > 8:
            base_type_data_encoding = xtce.IntegerDataEncodingType(sizeInBits=bit_size,
                                                                   bitOrder=bit_order,
                                                                   byteOrder=byte_order,
                                                                   encoding=xtce.IntegerEncodingType.UNSIGNED
                                                                   )
        else:
            base_type_data_encoding = xtce.IntegerDataEncodingType(sizeInBits=bit_size,
                                                                   bitOrder=bit_order,
                                                                   encoding=xtce.IntegerEncodingType.UNSIGNED
                                                                   )

        param_type.set_IntegerDataEncoding(base_type_data_encoding)

        return param_type

    def __get_float_paramtype(self, bit_size: int, little_endian: bool) -> xtce.FloatDataType:
        """
        Factory function to construct a IntegerParameterType.
        :param bit_size:
        :param little_endian:
        :return:
        """
        endianness = self.__get_endianness_postfix(little_endian)

        base_type_name = 'float' + str(bit_size) + endianness

        bit_order = xtce.BitOrderType.LEAST_SIGNIFICANT_BIT_FIRST if little_endian else \
            xtce.BitOrderType.MOST_SIGNIFICANT_BIT_FIRST

        byte_order = xtce.ByteOrderCommonType.LEAST_SIGNIFICANT_BYTE_FIRST if little_endian else \
            xtce.ByteOrderCommonType.MOST_SIGNIFICANT_BYTE_FIRST

        if bit_size > 8:
            encoding = xtce.FloatDataEncodingType(sizeInBits=str(bit_size),
                                                  bitOrder=bit_order,
                                                  byteOrder=byte_order)
        else:
            encoding = xtce.FloatDataEncodingType(sizeInBits=str(bit_size),
                                                  bitOrder=bit_order)

        param_type = xtce.FloatParameterType(name=base_type_name, sizeInBits=str(bit_size))
        param_type.set_FloatDataEncoding(encoding)

        return param_type

    def __add_telemetry_base_types(self):
        """
        Adds all supported base types for our ground system to the TelemetryMetaData element of
        the namespace 'BaseType', which is created to hold all of the base types. Base types are the types tha are not
        user-defined such as int16, int32, etc. Check our docs for more details on base types. Please note that these
        base types are stored as *ParameterTypes on the xtce. This also adds a variable-sized string type.
        :return:
        """
        base_set = xtce.ParameterTypeSetType()
        base_space_system = self[self.BASE_TYPE_NAMESPACE]

        # Add base types such as int, float, bool, etc
        for bit in range(1, 65):
            if bit > 1:
                base_set.add_IntegerParameterType(self.__get_int_paramtype(bit, True))
                base_set.add_IntegerParameterType(self.__get_int_paramtype(bit, False))

            base_set.add_IntegerParameterType(self.__get_padding_paramtype(bit, True))
            base_set.add_IntegerParameterType(self.__get_padding_paramtype(bit, False))
            base_set.add_IntegerParameterType(self.__get_uint_paramtype(bit, True))
            base_set.add_IntegerParameterType(self.__get_uint_paramtype(bit, False))

        # NOTE: For right now, only singed 32-bit floating types are supported
        # Add floating types
        base_set.add_FloatParameterType(self.__get_float_paramtype(32, True))
        base_set.add_FloatParameterType(self.__get_float_paramtype(32, False))
        base_set.add_FloatParameterType(self.__get_float_paramtype(64, True))
        base_set.add_FloatParameterType(self.__get_float_paramtype(64, False))
        base_set.add_IntegerParameterType(
            xtce.IntegerParameterType(name=XTCEManager.UNKNOWN_TYPE, signed=False, sizeInBits='32'))
        base_set.add_BooleanParameterType(
            xtce.BooleanParameterType(name='boolean8_LE', IntegerDataEncoding=xtce.IntegerDataEncodingType()))
        base_set.add_BooleanParameterType(
            xtce.BooleanParameterType(name='boolean8_BE', IntegerDataEncoding=xtce.IntegerDataEncodingType(sizeInBits=8,
                                                                                                           bitOrder=xtce.BitOrderType.MOST_SIGNIFICANT_BIT_FIRST,
                                                                                                           encoding=xtce.IntegerEncodingType.TWOS_COMPLEMENT)))

        # Add string type
        size_in_bits = xtce.SizeInBitsType()
        size_in_bits.set_TerminationChar("00")
        str_encoding = xtce.StringDataEncodingType()

        str_encoding.set_SizeInBits(size_in_bits)
        str_type = xtce.StringParameterType(name="string")

        str_type.set_StringDataEncoding(str_encoding)

        base_set.add_StringParameterType(str_type)

        base_space_system.get_TelemetryMetaData().set_ParameterTypeSet(base_set)

    def __add_commands_base_types(self):
        """
        Adds all supported base types for our ground system to the CommandMetaData element of
        the namespace 'BaseType', which is created to hold all of the base types. Base types are the types tha are not
        user-defined such as int16, int32, etc. Check our docs for more details on base types. Please note that these
        base types are stored as *ArgumentTypes on the xtce. This also adds a variable-sized string type.
        :return:
        """
        base_set = xtce.ArgumentTypeSetType()
        base_space_system = self[self.BASE_TYPE_NAMESPACE]

        # Add int types
        for bit in range(1, 65):
            if bit > 1:
                base_set.add_IntegerArgumentType(self.__get_int_argtype(bit, True))
                base_set.add_IntegerArgumentType(self.__get_int_argtype(bit, False))

            base_set.add_IntegerArgumentType(self.__get_padding_argtype(bit, True))
            base_set.add_IntegerArgumentType(self.__get_padding_argtype(bit, False))

            base_set.add_IntegerArgumentType(self.__get_uint_argtype(bit, True))
            base_set.add_IntegerArgumentType(self.__get_uint_argtype(bit, False))

        # NOTE: For right now, only singed little-endian 32-bit floating types are supported
        base_set.add_FloatArgumentType(self.__get_float_argtype(32, True))
        base_set.add_FloatArgumentType(self.__get_float_argtype(32, False))
        base_set.add_IntegerArgumentType(xtce.IntegerParameterType(name='UNKNOWN', signed=False, sizeInBits='32'))

        base_set.add_BooleanArgumentType(
            xtce.BooleanArgumentType(name='boolean8_LE', IntegerDataEncoding=xtce.IntegerDataEncodingType()))
        base_set.add_BooleanArgumentType(
            xtce.BooleanArgumentType(name='boolean8_BE', IntegerDataEncoding=xtce.IntegerDataEncodingType(sizeInBits=8,
                                                                                                          bitOrder=xtce.BitOrderType.MOST_SIGNIFICANT_BIT_FIRST,
                                                                                                          encoding=xtce.IntegerEncodingType.TWOS_COMPLEMENT)))

        str_type = xtce.StringArgumentType(name="string", StringDataEncoding=xtce.StringDataEncodingType(
            SizeInBits=xtce.SizeInBitsType()))

        base_set.add_StringArgumentType(str_type)

        base_space_system.get_CommandMetaData().set_ArgumentTypeSet(base_set)

    def add_base_types(self):
        """
        Create a namespace BaseType and add all base types to it. Please refer to the docs for how we define a base type
        in our ground system.
        :return:
        """
        self.add_namespace(self.BASE_TYPE_NAMESPACE)
        self.__add_telemetry_base_types()
        self.__add_commands_base_types()

    def __get_all_arg_basetypes(self) -> set([xtce.NameDescriptionType]):
        """
        Get all base types names in BASE_TYPE_NAMESPACE. Do note that the base type names returned do not have the
        namespace include in them. In other words, even though the absolute reference path to 'int16_LE' is
        'BASE_TYPE_NAMESPACE/int16_LE', this function will return 'int16_LE' for that type. As the name suggests,
        this function only returns type names that are *ArgumentTypes.
        """
        all_types = self[
                        self.BASE_TYPE_NAMESPACE].get_CommandMetaData().get_ArgumentTypeSet().get_IntegerArgumentType() + \
                    self[self.BASE_TYPE_NAMESPACE].get_CommandMetaData().get_ArgumentTypeSet().get_FloatArgumentType() + \
                    self[
                        self.BASE_TYPE_NAMESPACE].get_CommandMetaData().get_ArgumentTypeSet().get_BooleanArgumentType() + \
                    self[self.BASE_TYPE_NAMESPACE].get_CommandMetaData().get_ArgumentTypeSet().get_StringArgumentType()

        all_type_names = []

        for type in all_types:
            all_type_names.append(type.get_name())

        return set(all_type_names)

    def __get_all_param_basetypes(self):
        """
        Get all base types names in BASE_TYPE_NAMESPACE. Do note that the base type names returned do not have the
        namespace include in them. In other words, even though the absolute reference path to 'int16_LE' is
        'BASE_TYPE_NAMESPACE/int16_LE', this function will return 'int16_LE' for that type. As the name suggests,
        this function only returns type names that are *ParameterTypes.
        """
        all_types = self[
                        self.BASE_TYPE_NAMESPACE].get_TelemetryMetaData().get_ParameterTypeSet().get_IntegerParameterType() + \
                    self[
                        self.BASE_TYPE_NAMESPACE].get_TelemetryMetaData().get_ParameterTypeSet().get_FloatParameterType() + \
                    self[
                        self.BASE_TYPE_NAMESPACE].get_TelemetryMetaData().get_ParameterTypeSet().get_BooleanParameterType() + \
                    self[
                        self.BASE_TYPE_NAMESPACE].get_TelemetryMetaData().get_ParameterTypeSet().get_StringParameterType() + \
                    self[
                        self.BASE_TYPE_NAMESPACE].get_TelemetryMetaData().get_ParameterTypeSet().get_ArrayParameterType()

        all_type_names = []

        for type in all_types:
            all_type_names.append(type.get_name())

        return set(all_type_names)

    def __get_all_basetypes(self) -> set([xtce.NameDescriptionType]):
        """
        Get all base types names in BASE_TYPE_NAMESPACE. Do note that the base type names returned do not have the
        namespace include in them. In other words, even though the absolute reference path to 'int16_LE' is
        'BASE_TYPE_NAMESPACE/int16_LE', this function will return 'int16_LE' for that type. This does mean that this function
        makes no distinction between Argument Types and Parameter Types. If you are interested in *ArgumentTypes or
        *ParameterTypes specifically then look at the __get_all_arg_basetypes and __get_all_arg_basetypes functions.

        :return: A set of type names.
        """
        all_types = self[
                        self.BASE_TYPE_NAMESPACE].get_TelemetryMetaData().get_ParameterTypeSet().get_IntegerParameterType() + \
                    self[
                        self.BASE_TYPE_NAMESPACE].get_CommandMetaData().get_ArgumentTypeSet().get_IntegerArgumentType() + \
                    self[
                        self.BASE_TYPE_NAMESPACE].get_TelemetryMetaData().get_ParameterTypeSet().get_FloatParameterType() + \
                    self[self.BASE_TYPE_NAMESPACE].get_CommandMetaData().get_ArgumentTypeSet().get_FloatArgumentType() + \
                    self[
                        self.BASE_TYPE_NAMESPACE].get_TelemetryMetaData().get_ParameterTypeSet().get_BooleanParameterType() + \
                    self[
                        self.BASE_TYPE_NAMESPACE].get_CommandMetaData().get_ArgumentTypeSet().get_BooleanArgumentType() + \
                    self[
                        self.BASE_TYPE_NAMESPACE].get_TelemetryMetaData().get_ParameterTypeSet().get_StringParameterType() + \
                    self[
                        self.BASE_TYPE_NAMESPACE].get_CommandMetaData().get_ArgumentTypeSet().get_StringArgumentType() + \
                    self[
                        self.BASE_TYPE_NAMESPACE].get_TelemetryMetaData().get_ParameterTypeSet().get_ArrayParameterType() + \
                    self[self.BASE_TYPE_NAMESPACE].get_CommandMetaData().get_ArgumentTypeSet().get_ArrayArgumentType()

        all_type_names = []

        for type in all_types:
            all_type_names.append(type.get_name())

        return set(all_type_names)

    def __basetype_exists(self, type_name) -> bool:
        """
        Checks if the base type with type_name exists in the BaseType namespace. type_name is assumed to just be the type
        name without the namespace. For example "string320_LE", NOT "BaseType/string320_LE".
        """
        return type_name in self.__get_all_basetypes()

    def __arg_basetype_exists(self, type_name) -> bool:
        """
        Checks if the base type with type_name exists in the BaseType namespace. type_name is assumed to just be the type
        name without the namespace. For example "string320_LE", NOT "BaseType/string320_LE". As the function name suggests,
        only *ArgumentTypes are checked.
        """
        return type_name in self.__get_all_arg_basetypes()

    def __param_basetype_exists(self, type_name: str) -> bool:
        """
        Checks if the base type with type_name exists in the BaseType namespace. type_name is assumed to just be the type
        name without the namespace. For example "string320_LE", NOT "BaseType/string320_LE". As the function name suggests,
        only *ParameterTypes are checked.
        """
        return type_name in self.__get_all_param_basetypes()

    def __get_basetype_name(self, basename: str, bit_size: int, little_endian: bool):
        """
        A factory function that constructs the base type name.
        :param basename: The basename of this type. Please note that we mean by this is a name like 'int'. NOT 'int32',
        'int32_t', etc. The caller should pass just 'int' for basetype and pass in the bit size as a separate argument
        to bit_size. Same goes for float, strings, etc.
        :param bit_size: How many bits does this base type contain.
        :param little_endian: A bool describing whether the type is encoded using Little Endian or big Endian.
        :return: The full basetype name. Note that this function DOES check the BaseType namespace of our root SpaceSystem.
        We return the fully-constructed basetype name even if we don't find it in out BaseType namespace. However,
        we do warn the user that this base type name was not found.

        NOTE: strings are special. They are special because, as opposed intrinsic types such as integer, they very
        in size in non-predictable way.  Therefore the type 'string' is passed to this function, then the string type will
        be created if it does not exist. Not sure if this is the best way to handle strings, but this is what we will do
        for now.
        """
        # FIXME: This name assembling should be in a function to avoid code duplication and silly mistakes

        typename = basename + str(bit_size) + ('_LE' if little_endian else '_BE')
        out_type_ref = None
        all_basetypes = self.__get_all_basetypes()
        logging.debug(f'all base types-->{all_basetypes}')
        logging.debug(f'basename:{basename}')

        if typename in all_basetypes:
            out_type_ref = XTCEManager.BASE_TYPE_NAMESPACE + XTCEManager.NAMESPACE_SEPARATOR + typename
        else:
            logging.warning(
                f'{typename} was looked up as a base type, but it was not found in the BaseType namespace. ')

        return out_type_ref

    def is_little_endian(self, elf_id: str):
        return self.db_cursor.execute('SELECT little_endian FROM elfs where id=?',
                                      (elf_id,)).fetchone()[0] == 1

    # FIXME: Finish implementation. This is for stand-alone arrays
    def __handle_array(self, symbol_record: tuple, multiplicity):
        for index in range(multiplicity):
            self.__get_aggregate_paramtype()

    def __get_array(self, field_id) -> []:
        """
        Fetches all of the dimension_lists records that match field_id.
        :return: A list with all of the records fetched from dimension_lists. If no records are found,
        an empty list is returned.
        """
        dim_list_records = self.db_cursor.execute('SELECT * FROM dimension_lists WHERE field_id=?',
                                                  (field_id,)).fetchall()
        return dim_list_records

    def __is_array(self, field_id) -> bool:
        """
        Return True if the field with field_id is an array. Otherwise, return False.
        """
        dim_list_records = self.db_cursor.execute('SELECT * FROM dimension_lists WHERE field_id=?',
                                                  (field_id,)).fetchall()
        return len(dim_list_records) > 0

    def __get_dimension_list_param_type(self, field_id) -> xtce.DimensionListType:
        """
        Return the size of the array mapped to field_id in the database.
        This function assumes the following schema for the dimension_lists table:
        (id, field_id, dim_order, upper_bound)
        If the field is not an array, this function returns 0.
        """
        # In XTCE DimensionListType is the parameter version of this type
        dim_list_param_type = xtce.DimensionListType()
        dim_list_records = self.db_cursor.execute('SELECT * FROM dimension_lists WHERE field_id=?',
                                                  (field_id,)).fetchall()
        # Enforce order by dim_order
        dim_list_records.sort(key=lambda dim_list_record: dim_list_record[3])

        if len(dim_list_records) > 0:
            for dim in dim_list_records:
                new_dimension = xtce.DimensionType()
                upper_bound = dim[3]
                new_dimension.set_StartingIndex(xtce.FixedType(0))
                new_dimension.set_EndingIndex(xtce.FixedType(upper_bound))
                dim_list_param_type.add_Dimension(new_dimension)

        return dim_list_param_type

    def __get_array_param_type(self, field_id: [], type_ref: str) -> xtce.ArrayParameterType:
        """
        Construct a ArrayParameterType with the dimensions of the field_id in the dimension_lists table. This function
        assumes thar type_ref exists. If field_id is not an array, then a ArrayParameterType with an empty DimensionListType
        is returned. It is assumed that type_ref is a fully qualified name such as "/simlink/apps/to/TO_MessageFlowDiagTlm_t".
        """
        out_array_type = xtce.ArrayParameterType()
        dimensions = self.__get_dimension_list_param_type(field_id)
        out_array_type.set_DimensionList(dimensions)
        out_array_name = self.ARRAY_BASE_NAME

        # Clean up type_ref name to avoid circular references in BaseType namespace.
        if type_ref.find('/') != 1:
            type_ref_name = type_ref.replace(XTCEManager.NAMESPACE_SEPARATOR, "_")
            #     Strip the root from name
            if type_ref.find(XTCEManager.BASE_TYPE_NAMESPACE) == 0:
                type_ref = type_ref[type_ref.find(XTCEManager.BASE_TYPE_NAMESPACE + XTCEManager.NAMESPACE_SEPARATOR):]
            else:
                type_ref = type_ref[type_ref.find(self.root.get_name() + XTCEManager.NAMESPACE_SEPARATOR):]
        else:
            type_ref_name = type_ref

        for dim in dimensions.get_Dimension():
            out_array_name += '_' + str(dim.get_EndingIndex().get_FixedValue()) + 'Dim'
            out_array_name += '_' + type_ref_name
        out_array_type.set_name(out_array_name)
        out_array_type.set_arrayTypeRef(type_ref)

        return out_array_type

    def __is_base_type(self, type_name: str) -> tuple:
        """
        Checks if type_name is a base type as it appears in the database.
        :return: A tuple of the form (bool, str), where the bool is whether this is a basetype or not and what the
        base type maps to in our BaseType namespace. Please note that this function does not pre-append the BaseType
        namespace to the type, that is the responsibility of the caller. Please note that padding types are also
        considered base types. Padding types have the form of _padding[Number Of Bits] such as _padding8.

        NOTE:While strings are considered a base type, it should be noted that, as opposed to all of the other base types,
        they are created as needed. This is because we can't really predict their sizes, like we do
        ints, not even a range as a string could be of any size. Thus they are created on the fly.
        """
        out_base_type = (False, '')

        if type_name == 'int64' \
                or type_name == 'int32' \
                or type_name == 'int16' \
                or type_name == 'int8' \
                or type_name == 'int':
            out_base_type = (True, 'int')
        elif type_name == 'uint8' \
                or type_name == 'uint16' \
                or type_name == 'uint32' \
                or type_name == 'unsigned int' \
                or type_name == 'unsigned' \
                or type_name == 'uint64':
            out_base_type = (True, 'uint')
        # FIXME: char types need to be handled properly
        elif type_name == 'char':
            out_base_type = (True, 'int')
        elif type_name == 'boolean':
            out_base_type = (True, 'boolean')
        elif type_name == 'float' or type_name == 'double':
            out_base_type = (True, 'float')
        elif type_name[:8] == '_padding':
            out_base_type = (True, '_padding')
        elif type_name == 'string':
            out_base_type = (True, 'string')

        return out_base_type

    def __is_type_in_config(self, spacesystem: str, type_name: str):
        """
        Check if the type with name of type_name which is part of spacesystem in the config file provided by the user,
        that is if a configuration file was provided by the user. For now, our configuration only supports mapping
        to BaseTypes.
        :param spacesystem: 
        :param type_name: The reference to the type. The namespace of the type must be explicitly stated, meaning
        that types such as 'int16_LE' MUST be passed as 'BaseType/int16_LE'
        :return: A tuple of the form (bool, typeref). If the type_name is not found in the configuration dict, then the
        tuple is returned as (False, XTCEManager.UNKNOWN_TYPE)
        """
        does_type_exist = (False, self.UNKNOWN_TYPE)

        if self.custom_config and self.namespace_exists(spacesystem):
            if spacesystem in self.custom_config['spacesystems'] and \
                    type_name in self.custom_config['spacesystems'][spacesystem]['remap'].keys():
                does_type_exist = (True, self.custom_config['spacesystems'][spacesystem]['remap'][type_name])

        return does_type_exist

    def __is__symbol_enum(self, symbol_id: int):
        """
        Checks the database to see if the symbol id is actually an enum or not.
        :param symbol_id:
        :return: True if the symbol is an enum. Otherwise, False is returned.
        """
        return len(self.db_cursor.execute('SELECT * FROM enumerations where symbol=?',
                                          (symbol_id,)).fetchall()) > 0

    def __is__symbol_string(self, symbol_id: int):
        """
        Checks the database to see if the symbol id is a actually a string or not.
        :param symbol_id:
        :return: True if the symbol is a string. Otherwise, False is returned.
        """
        # NOTE: This could be cached when the XTCEManager instance is created.
        is_string = False
        symbol_name, = self.db_cursor.execute('SELECT name FROM symbols where id=?',
                                              (symbol_id,)).fetchone()
        if symbol_name == 'string':
            is_string = True

        return is_string

    def __get_enum_from_symbol_id(self, symbol_id: int):
        """
        Attempts to fetch an enumeration name that is mapped to symbol_id from the database.
        :param symbol_id:
        :return: The name of the enumeration if it is found. Otherwise, None is returned.
        """
        enums = self.db_cursor.execute('SELECT * FROM symbols where id=?',
                                       (symbol_id,)).fetchall()

        out_enum_name = None

        if len(enums) > 0:
            out_enum_name = enums[0][2]

        return out_enum_name

    def __get_enum_param_type(self, symbol_id: int) -> xtce.EnumeratedParameterType:
        """
        Factory function that creates a EnumeratedParameterType that is described by the symbol with symbol_id
        in the database. This function assumes that the caller has ensured that there is an enumeration
        associated to the symbol_id in the database.
        :param symbol_id: id of symbol that points to this enumeration type in the database.
        :return:
        """
        symbol_elf, symbol_name, byte_size = self.db_cursor.execute(
            'SELECT elf, name,byte_size FROM symbols where id=?',
            (symbol_id,)).fetchone()
        out_enum = xtce.EnumeratedParameterType(name=symbol_name)
        enum_list = xtce.EnumerationListType()

        bit_size = byte_size * 8

        little_endian = self.is_little_endian(symbol_elf)

        bit_order = xtce.BitOrderType.LEAST_SIGNIFICANT_BIT_FIRST if little_endian else \
            xtce.BitOrderType.MOST_SIGNIFICANT_BIT_FIRST

        byte_order = xtce.ByteOrderCommonType.LEAST_SIGNIFICANT_BYTE_FIRST if little_endian else \
            xtce.ByteOrderCommonType.MOST_SIGNIFICANT_BYTE_FIRST

        for value, name in self.db_cursor.execute('SELECT value, name FROM enumerations where symbol=?',
                                                  (symbol_id,)).fetchall():
            enum_list.add_Enumeration(xtce.ValueEnumerationType(label=name, value=value))

        out_enum.set_EnumerationList(enum_list)

        if bit_size > 8:
            out_enum.set_IntegerDataEncoding(xtce.IntegerDataEncodingType(encoding=xtce.IntegerEncodingType.UNSIGNED,
                                                                          sizeInBits=bit_size,
                                                                          bitOrder=bit_order,
                                                                          byteOrder=byte_order))
        else:
            out_enum.set_IntegerDataEncoding(xtce.IntegerDataEncodingType(encoding=xtce.IntegerEncodingType.UNSIGNED,
                                                                          sizeInBits=bit_size,
                                                                          bitOrder=bit_order, ))

        return out_enum

    def __get_enum_arg_type(self, symbol_id: int) -> xtce.EnumeratedArgumentType:
        """
        Factory function that creates a EnumeratedArgumentType that is described by the symbol with symbol_id
        in the database. This function assumes that the caller has ensured that there is an enumeration
        associated to the symbol_id in the database.
        :param symbol_id: id of symbol that points to this enumeration type in the database.
        :return:
        """
        symbol_elf, symbol_name, byte_size = self.db_cursor.execute(
            'SELECT elf, name,byte_size FROM symbols where id=?',
            (symbol_id,)).fetchone()
        out_enum = xtce.EnumeratedArgumentType(name=symbol_name)
        enum_list = xtce.EnumerationListType()

        bit_size = byte_size * 8

        little_endian = self.is_little_endian(symbol_elf)

        bit_order = xtce.BitOrderType.LEAST_SIGNIFICANT_BIT_FIRST if little_endian else \
            xtce.BitOrderType.MOST_SIGNIFICANT_BIT_FIRST

        byte_order = xtce.ByteOrderCommonType.LEAST_SIGNIFICANT_BYTE_FIRST if little_endian else \
            xtce.ByteOrderCommonType.MOST_SIGNIFICANT_BYTE_FIRST

        for value, name in self.db_cursor.execute('SELECT value, name FROM enumerations where symbol=?',
                                                  (symbol_id,)).fetchall():
            enum_list.add_Enumeration(xtce.ValueEnumerationType(label=name, value=value))

        out_enum.set_EnumerationList(enum_list)

        if bit_size > 8:
            out_enum.set_IntegerDataEncoding(xtce.IntegerDataEncodingType(encoding=xtce.IntegerEncodingType.UNSIGNED,
                                                                          sizeInBits=bit_size,
                                                                          bitOrder=bit_order,
                                                                          byteOrder=byte_order))
        else:
            out_enum.set_IntegerDataEncoding(xtce.IntegerDataEncodingType(encoding=xtce.IntegerEncodingType.UNSIGNED,
                                                                          sizeInBits=bit_size,
                                                                          bitOrder=bit_order, ))

        return out_enum

    def __aggregate_param_exists(self, param_name: str, namespace: str) -> bool:
        """
        Checks if the aggregate type with type_name exists in the telemetry child of the namespace space system.
        :return:
        """
        does_aggregate_exist = False

        if self.namespace_exists(namespace):
            if self[namespace].get_TelemetryMetaData().get_ParameterSet():
                types = [name.get_name() for name in
                         self[namespace].get_TelemetryMetaData().get_ParameterSet().get_Parameter() if
                         name.get_name() == param_name]

                # FIXME: This length should ALWAYS be 1 or 0. Will revise.
                if len(types) > 0:
                    does_aggregate_exist = True

        return does_aggregate_exist

    def __aggregate_paramtype_exists(self, type_name: str, namespace: str):
        """
        Checks if the aggregate type with type_name exists in the telemetry child of the namespace space system.
        :return:
        """
        does_aggregate_exist = False

        if self.namespace_exists(namespace):
            if self[namespace].get_TelemetryMetaData().get_ParameterTypeSet():
                types = [aggregate_name.get_name() for aggregate_name in
                         self[namespace].get_TelemetryMetaData().get_ParameterTypeSet().get_AggregateParameterType() if
                         aggregate_name.get_name() == type_name]

                if len(types) > 0:
                    does_aggregate_exist = True

        return does_aggregate_exist

    def find_aggregate_param_type(self, type_name: str, namespace: str) -> Union[xtce.AggregateParameterType, None]:
        """
        Returns a parameter type with the name of type_name. Note that this type_name is the same
        name of a symbol that appears in the database. The namespace refers to the spacesystem inside the XTCE. The
        namespace is the same as the modules in the database. Please note that the type returned is local to the namespace.
        :param type_name:
        :param namespace:
        :return: The  param type object. If the parameter with name of type_name does not exist None is returned.
        """
        out_param_type_ref = None
        if self.namespace_exists(namespace):
            if self[namespace].get_TelemetryMetaData().get_ParameterTypeSet():
                types = [aggregate_name for aggregate_name in
                         self[namespace].get_TelemetryMetaData().get_ParameterTypeSet().get_AggregateParameterType() if
                         aggregate_name.get_name() == type_name]

                # FIXME: There should only be one type in the list.
                if len(types) > 0:
                    out_param_type_ref = types[0]

        return out_param_type_ref

    def find_aggregate_arg_type(self, type_name: str, namespace: str) -> xtce.AggregateArgumentType:
        """
        Returns an argument type with the name of type_name. Note that this type_name is the same
        name of a symbol that appears in the database. The namespace refers to the spacesystem inside the XTCE. The
        namespace is the same as the modules in the database. Please note that the type returned is local to the namespace.
        :param type_name:
        :param namespace:
        :return: The name of the type which may be used as a typeRef inside of a argument type parameter. If the argument
        with name of type_name does not exist None is returned.
        """
        out_arg_type_ref = None
        if self.namespace_exists(namespace):
            if self[namespace].get_CommandMetaData().get_ArgumentTypeSet():
                types = [aggregate_name for aggregate_name in
                         self[namespace].get_CommandMetaData().get_ArgumentTypeSet().get_AggregateArgumentType() if
                         aggregate_name.get_name() == type_name]

                # FIXME: There should only be one type in the list.
                if len(types) > 0:
                    out_arg_type_ref = types[0]

        return out_arg_type_ref

    def __enumeration_paramtype_exists(self, symbol_id: int, namespace: str):
        """
        Checks if the enumerated type with symbol_id exists in the telemetry child of the namespace space system.
        :return:
        """
        does_enum_exist = False

        type_name = self.db_cursor.execute('SELECT * FROM enumerations WHERE symbol=?',
                                           (symbol_id,)).fetchall()

        if len(type_name) > 0:
            enum_symbol = self.db_cursor.execute('SELECT * FROM symbols WHERE id=?',
                                                 (symbol_id,)).fetchall()[0][2]
            if self.namespace_exists(namespace):
                if self[namespace].get_TelemetryMetaData().get_ParameterTypeSet():
                    types = [enum_name.get_name() for enum_name in
                             self[
                                 namespace].get_TelemetryMetaData().get_ParameterTypeSet().get_EnumeratedParameterType()
                             if
                             enum_name.get_name() == enum_symbol]

                    if len(types) > 0:
                        does_enum_exist = True

        return does_enum_exist

    def __enumeration_argtype_exists(self, symbol_id: int, namespace: str):
        """
        Checks if the enumerated type with symbol_id exists in the command metadata child of the namespace space system.
        :return:
        """
        does_enum_exist = False

        type_name = self.db_cursor.execute('SELECT * FROM enumerations WHERE symbol=?',
                                           (symbol_id,)).fetchall()

        if len(type_name) > 0:
            enum_symbol = self.db_cursor.execute('SELECT * FROM symbols WHERE id=?',
                                                 (symbol_id,)).fetchall()[0][2]
            if self.namespace_exists(namespace):
                if self[namespace].get_CommandMetaData().get_ArgumentTypeSet():
                    types = [enum_name.get_name() for enum_name in
                             self[
                                 namespace].get_CommandMetaData().get_ArgumentTypeSet().get_EnumeratedArgumentType()
                             if
                             enum_name.get_name() == enum_symbol]

                    if len(types) > 0:
                        does_enum_exist = True

        return does_enum_exist

    def __get_aggregate_paramtype(self, symbol_record: tuple, module_name: str,
                                  header_present: bool = True, header_size: int = 0) -> Union[
        xtce.AggregateParameterType, None]:
        """
        A factory function to create an aggregateParamType type pointed to by symbol_id.
        :param symbol_record: A tuple containing the symbol record of the database in the form of
        (id, elf, name, byte_size)
        :return: If the symbol is processed successfully, an AggregateParameterType representing that symbol(struct) is returned.
        If the symbol already exists, then the AggregateParameterType representing this symbol is returned.
        Beware that if this function finds a field of the symbol record
        whose type does not exist(such as a field that has a struct type not defined in our xtce), then this function takes
        the liberty of adding it to the telemetry object in the xtce object.
        """

        # FIXME: This entire function needs to be decoupled; it's far too big
        out_param = xtce.AggregateParameterType(name=symbol_record[2])

        if out_param.get_name() == "PX4_PositionSetpointTripletMsg_t":
            print("break")

        # If the symbol exists already in our xtce, we don't need to explore it any further
        if self.__aggregate_paramtype_exists(symbol_record[2], module_name):
            return self.find_aggregate_param_type(symbol_record[2], module_name)

        logging.debug(f'symbol record-->{symbol_record}')

        symbol_id = str(symbol_record[0])

        if header_present:
            fields = list(filter(lambda record: record[3] >= header_size / 8,
                                 self.db_cursor.execute('SELECT * FROM fields where symbol=?',
                                                        (symbol_id,)).fetchall()))
        else:
            fields = self.db_cursor.execute('SELECT * FROM fields where symbol=?',
                                            (symbol_id,)).fetchall()

        # Enforce ordering of fields by offset.
        fields.sort(key=lambda record: record[3])

        logging.debug(f'root fields-->{fields}')

        type_ref_name = None

        member_list = xtce.MemberListType()
        out_param.set_MemberList(member_list)
        symbol_id = str(symbol_record[0])

        for field_id, field_symbol, field_name, field_byte_offset, field_type, field_little_endian, bit_size, bit_offset in fields:
            # If this field is standalone array, ignore it for now
            if field_type == field_symbol:
                continue

            elif self.__is_array(field_id):
                logging.debug(f'comparing {field_type} and {field_symbol}')

                symbol_type = self.db_cursor.execute('SELECT * FROM symbols where id=?',
                                                     (field_type,)).fetchone()
                # The symbol_type is expected, as per our schema, to have the form of (id, elf ,name, byte_size)
                if symbol_type:
                    logging.debug(f'symbol_type$$$$-->{symbol_type}')
                    base_type_val = self.__is_base_type(symbol_type[2])

                    if self.__is__symbol_enum(field_type) is True:
                        if self.__enumeration_paramtype_exists(field_type, module_name) is True:
                            type_ref_name = self.__get_enum_from_symbol_id(field_type)

                        else:
                            new_enum = self.__get_enum_param_type(field_type)
                            self[
                                module_name].get_TelemetryMetaData().get_ParameterTypeSet().add_EnumeratedParameterType(
                                new_enum)
                            type_ref_name = new_enum.get_name()

                    elif base_type_val[0]:
                        # This is a basetype, so we can just get a type from our BaseType namespace
                        # TODO: Make a distinction between unsigned and int types
                        type_ref_name = self.__get_basetype_name(base_type_val[1], symbol_type[3] * 8,
                                                                 self.is_little_endian(symbol_type[1]))
                    else:
                        logging.debug(f'field type-->{field_type}')
                        child_symbol = self.db_cursor.execute('SELECT * FROM symbols where id=?',
                                                              (field_type,)).fetchone()

                        logging.debug(f'field_symbol id:{field_symbol}')
                        logging.debug(f'child symbol-->{child_symbol}')
                        child = self.__get_aggregate_paramtype(child_symbol, module_name, False)
                        if self.__aggregate_paramtype_exists(child_symbol[2], module_name) is False:
                            self[module_name].get_TelemetryMetaData().get_ParameterTypeSet().add_AggregateParameterType(
                                child)
                        type_ref_name = child.get_name()

                    if self.__is__symbol_string(field_type) is True:
                        field_multiplicity = self.__get_array(field_id)[0][3] + 1
                        # FIXME: Don't love the way I'm handling this, looks kind of ugly and wack. Will revise.
                        endianness_postfix = self.__get_endianness_postfix(field_little_endian)
                        string_type_name = 'string' + str(field_multiplicity * 8) + endianness_postfix
                        if self.__param_basetype_exists(string_type_name) is False:
                            # Create a string type if it does not exist.
                            new_string_type = self.__get_string_paramtype(field_multiplicity * 8, field_little_endian)
                            self[
                                XTCEManager.BASE_TYPE_NAMESPACE].get_TelemetryMetaData().get_ParameterTypeSet().add_StringParameterType(
                                new_string_type)
                            type_ref_name = XTCEManager.BASE_TYPE_NAMESPACE + XTCEManager.NAMESPACE_SEPARATOR + new_string_type.get_name()
                        else:
                            type_ref_name = XTCEManager.BASE_TYPE_NAMESPACE + XTCEManager.NAMESPACE_SEPARATOR + string_type_name
                        # If the field is a string, then it is a special kind of array that ends in a null terminator.

                        member = xtce.MemberType()
                        member.set_name(f'{field_name}')
                        member.set_typeRef(type_ref_name)
                        member_list.add_Member(member)

                    else:
                        if base_type_val[0] is True:
                            type_ref_name = type_ref_name
                        else:
                            type_ref_name = module_name.rstrip(
                                XTCEManager.NAMESPACE_SEPARATOR) + XTCEManager.NAMESPACE_SEPARATOR + type_ref_name

                        new_array = self.__get_array_param_type(field_id, type_ref_name)

                        if self.__param_basetype_exists(new_array.get_name()) is False:
                            self[
                                XTCEManager.BASE_TYPE_NAMESPACE].get_TelemetryMetaData().get_ParameterTypeSet().add_ArrayParameterType(
                                new_array)

                        member = xtce.MemberType()
                        member.set_name(f'{field_name}')
                        member.set_typeRef(
                            XTCEManager.BASE_TYPE_NAMESPACE + XTCEManager.NAMESPACE_SEPARATOR + new_array.get_name())
                        member_list.add_Member(member)

                else:
                    type_ref_name = 'BaseType/UNKNOWN'
                    logging.warning('BaseType/UNKNOWN is being used as array type')

            else:
                logging.debug('else block')
                member = xtce.MemberType()
                member.set_name(field_name)
                symbol_type = self.db_cursor.execute('SELECT * FROM symbols where id=?',
                                                     (field_type,)).fetchone()
                # The symbol_type is expected, as per our schema, to have the form of (id, elf ,name, byte_size)
                if symbol_type:
                    logging.debug(f'symbol_type$$$$-->{symbol_type}')
                    base_type_val = self.__is_base_type(symbol_type[2])

                    if self.__is__symbol_enum(field_type) is True:
                        if self.__enumeration_paramtype_exists(field_type, module_name) is True:
                            type_ref_name = self.__get_enum_from_symbol_id(field_type)

                        else:
                            new_enum = self.__get_enum_param_type(field_type)
                            self[
                                module_name].get_TelemetryMetaData().get_ParameterTypeSet().add_EnumeratedParameterType(
                                new_enum)
                            type_ref_name = new_enum.get_name()

                    elif base_type_val[0]:
                        #     TODO: Make a distinction between unsigned and int types
                        type_ref_name = self.__get_basetype_name(base_type_val[1], symbol_type[3] * 8,
                                                                 self.is_little_endian(symbol_type[1]))
                    else:
                        logging.debug(f'field type-->{field_type}')
                        child_symbol = self.db_cursor.execute('SELECT * FROM symbols where id=?',
                                                              (field_type,)).fetchone()

                        logging.debug(f'field_symbol id:{field_symbol}')
                        logging.debug(f'child symbol-->{child_symbol}')
                        logging.debug(f'field id-->{field_id})')
                        child = self.__get_aggregate_paramtype(child_symbol, module_name, False)
                        if self.__aggregate_paramtype_exists(child_symbol[2], module_name) is False:
                            self[module_name].get_TelemetryMetaData().get_ParameterTypeSet().add_AggregateParameterType(
                                child)
                        type_ref_name = child.get_name()
                else:
                    type_ref_name = 'BaseType/UNKNOWN'

                member.set_typeRef(type_ref_name)
                member_list.add_Member(member)

        logging.debug(f'out_param--> {out_param.get_name()}')
        return out_param

    def __aggregate_argtype_exists(self, type_name: str, namespace: str):
        """
        Checks if the aggregate type with type_name exists in the command child of the space system namespace.
        :return:
        """
        does_aggregate_exist = False

        if self.namespace_exists(namespace):
            if self[namespace].get_CommandMetaData().get_ArgumentTypeSet():
                types = [aggregate_name.get_name() for aggregate_name in
                         self[namespace].get_CommandMetaData().get_ArgumentTypeSet().get_AggregateArgumentType() if
                         aggregate_name.get_name() == type_name]

                if len(types) > 0:
                    does_aggregate_exist = True

        return does_aggregate_exist

    def __get_aggregate_argtype(self, symbol_record: tuple, module_name: str,
                                header_present: bool = True, header_size: int = 0) -> Union[
        xtce.AggregateArgumentType, None]:
        """
        A factory function to create an aggregateArgumentType type pointed to by symbol_id.
        :param symbol_record: A tuple containing the symbol record of the database in the form of
        (id, elf, name, byte_size)
        :return: If the symbol is processed successfully, an aggregateArgumentType representing that symbol(struct) is returned.
        If the symbol already exists, then a reference to the aggregateArgumentType that represents that symbol  is returned.
        Beware that if this function finds a field of the symbol record
        whose type does not exist(such as a field that has a struct type not defined in our xtce), then function takes
        the liberty of adding it to the telemetry object in the xtce object.
        """

        out_param = xtce.AggregateArgumentType(name=symbol_record[2])

        # If the symbol exists already in our xtce, we don't need to explore it any further
        if self.__aggregate_argtype_exists(symbol_record[2], module_name):
            return self.find_aggregate_arg_type(symbol_record[2], module_name)

        logging.debug(f'symbol record-->{symbol_record}')

        symbol_id = str(symbol_record[0])

        if header_present:
            fields = list(filter(lambda record: record[3] >= header_size / 8,
                                 self.db_cursor.execute('SELECT * FROM fields where symbol=?',
                                                        (symbol_id,)).fetchall()))
        else:
            fields = self.db_cursor.execute('SELECT * FROM fields where symbol=?',
                                            (symbol_id,)).fetchall()

        # Enforce ordering of fields by offset.
        fields.sort(key=lambda record: record[3])

        logging.debug(f'root fields-->{fields}')

        type_ref_name = None

        member_list = xtce.MemberListType()
        out_param.set_MemberList(member_list)
        for field_id, field_symbol, field_name, field_byte_offset, field_type, field_little_endian, bit_size, bit_offset in fields:
            field_multiplicity = 0
            if self.__is_array(field_id):
                # Add 1 to upper bound since it is a zero-indexed and inclusive bound
                field_multiplicity = self.__get_array(field_id)[0][3] + 1
            # TODO: When YAMCS adds support for array arguments, update this to use XTCE ArgumentArrayType

            if field_type == field_symbol:
                # This means this field is a standalone array; we skip those for now
                logging.warning(f'Stand-alone arrays are not parsed at the moment.')
                continue
            elif field_multiplicity > 0:

                logging.debug(f'comparing{field_type} and {field_symbol}')

                symbol_type = self.db_cursor.execute('SELECT * FROM symbols where id=?',
                                                     (field_type,)).fetchone()
                # The symbol_type is expected, as per our schema, to have the form of (id, elf ,name, byte_size)
                if symbol_type:
                    logging.debug(f'symbol_type$$$$-->{symbol_type}')
                    base_type_val = self.__is_base_type(symbol_type[2])

                    if self.__is__symbol_enum(field_type) is True:
                        if self.__enumeration_argtype_exists(field_type, module_name) is True:
                            type_ref_name = self.__get_enum_from_symbol_id(field_type)

                        else:
                            new_enum = self.__get_enum_arg_type(field_type)
                            self[
                                module_name].get_CommandMetaData().get_ArgumentTypeSet().add_EnumeratedArgumentType(
                                new_enum)
                            type_ref_name = new_enum.get_name()

                    elif base_type_val[0]:
                        #     TODO: Make a distinction between unsigned and int types
                        type_ref_name = self.__get_basetype_name(base_type_val[1], symbol_type[3] * 8,
                                                                 self.is_little_endian(symbol_type[1]))
                    else:
                        logging.debug(f'field type-->{field_type}')
                        child_symbol = self.db_cursor.execute('SELECT * FROM symbols where id=?',
                                                              (field_type,)).fetchone()

                        logging.debug(f'field_symbol id:{field_symbol}')
                        logging.debug(f'child symbol-->{child_symbol}')
                        child = self.__get_aggregate_argtype(child_symbol, module_name, False)
                        if self.__aggregate_argtype_exists(child_symbol[2], module_name) is False:
                            self[module_name].get_CommandMetaData().get_ArgumentTypeSet().add_AggregateArgumentType(
                                child)
                        type_ref_name = child.get_name()

                    if self.__is__symbol_string(field_type) is True:
                        # FIXME: Don't love the way I'm handling this, looks kind of ugly and wack. Will revise.
                        endianness_postfix = self.__get_endianness_postfix(field_little_endian)
                        string_type_name = 'string' + str(field_multiplicity * 8) + endianness_postfix
                        if self.__arg_basetype_exists(string_type_name) is False:
                            # Create a string type if it does not exist.
                            new_string_type = self.__get_string_argtype(field_multiplicity * 8, field_little_endian)
                            self[
                                XTCEManager.BASE_TYPE_NAMESPACE].get_CommandMetaData().get_ArgumentTypeSet().add_StringArgumentType(
                                new_string_type)
                            type_ref_name = XTCEManager.BASE_TYPE_NAMESPACE + XTCEManager.NAMESPACE_SEPARATOR + new_string_type.get_name()
                        else:
                            type_ref_name = XTCEManager.BASE_TYPE_NAMESPACE + XTCEManager.NAMESPACE_SEPARATOR + string_type_name

                        # If the field is a string, then it is a special kind of array that ends in a null terminator.
                        member = xtce.MemberType()
                        member.set_name(f'{field_name}')
                        member.set_typeRef(type_ref_name)
                        member_list.add_Member(member)

                    else:
                        # It's not a string, so it must be a regular array
                        dim_list = self.__get_array(field_id)
                        if len(dim_list) > 1:
                            logging.warning(f'Array {field_name} has more than one dimension.'
                                            f'This array will be flattened. '
                                            f'Look at ticket:https://github.com/WindhoverLabs/xtce_generator/issues/35')
                            field_multiplicity = 1
                            for dim in dim_list:
                                field_multiplicity *= dim[0][3] + 1

                        for index in range(field_multiplicity):
                            child_symbol = self.db_cursor.execute('SELECT * FROM symbols where id=?',
                                                                  (field_type,)).fetchone()

                            # FIXME: This entire function needs to be decoupled (?)
                            logging.debug(f'field_symbol id on array:{field_symbol}')
                            logging.debug(f'child symbol-->{child_symbol}')

                            member = xtce.MemberType()
                            member.set_name(f'{field_name}_{index}_')
                            member.set_typeRef(type_ref_name)
                            member_list.add_Member(member)

                else:
                    type_ref_name = 'BaseType/UNKNOWN'
                    logging.warning('BaseType/UNKNOWN is being used as array type')

            else:
                logging.debug('else block')
                member = xtce.MemberType()
                member.set_name(field_name)
                symbol_type = self.db_cursor.execute('SELECT * FROM symbols where id=?',
                                                     (field_type,)).fetchone()
                # The symbol_type is expected, as per our schema, to have the form of (id, elf ,name, byte_size)
                if symbol_type:
                    logging.debug(f'symbol_type$$$$-->{symbol_type}')
                    base_type_val = self.__is_base_type(symbol_type[2])

                    if self.__is__symbol_enum(field_type) is True:
                        if self.__enumeration_argtype_exists(field_type, module_name) is True:
                            type_ref_name = self.__get_enum_from_symbol_id(field_type)

                        else:
                            new_enum = self.__get_enum_arg_type(field_type)
                            self[
                                module_name].get_CommandMetaData().get_ArgumentTypeSet().add_EnumeratedArgumentType(
                                new_enum)
                            type_ref_name = new_enum.get_name()

                    elif base_type_val[0]:
                        #     TODO: Make a distinction between unsigned and int types
                        type_ref_name = self.__get_basetype_name(base_type_val[1], symbol_type[3] * 8,
                                                                 self.is_little_endian(symbol_type[1]))
                    else:
                        logging.debug(f'field type-->{field_type}')
                        child_symbol = self.db_cursor.execute('SELECT * FROM symbols where id=?',
                                                              (field_type,)).fetchone()

                        logging.debug(f'field_symbol id:{field_symbol}')
                        logging.debug(f'child symbol-->{child_symbol}')
                        logging.debug(f'field id-->{field_id})')
                        child = self.__get_aggregate_argtype(child_symbol, module_name, False)
                        if self.__aggregate_argtype_exists(child_symbol[2], module_name) is False:
                            self[module_name].get_CommandMetaData().get_ArgumentTypeSet().add_AggregateArgumentType(
                                child)
                        type_ref_name = child.get_name()

                else:
                    type_ref_name = 'BaseType/UNKNOWN'

                member.set_typeRef(type_ref_name)
                member_list.add_Member(member)

        logging.debug(f'out_param--> {out_param.get_name()}')
        return out_param

    def add_namespace(self, namespace_name: str):
        """
        Add a namespace to the root SpaceSystem. Please note that namespace is a synonym for SpaceSystem;
        they are the same thing in xtce-speak.
        :param namespace_name: The name of the new namespace.
        :return:
        """
        new_namespace = xtce.SpaceSystemType(name=namespace_name)
        self.root.add_SpaceSystem(new_namespace)
        new_namespace.set_TelemetryMetaData(xtce.TelemetryMetaDataType())
        new_namespace.set_CommandMetaData(xtce.CommandMetaDataType())

        self.__namespace_dict[namespace_name] = new_namespace

    def __get_telemetry_header_length(self, symbol_id):
        """
        Calculate the the size of the telemetry header inside of the struct with id of symbol_id in the database.
        :return: The size of the telemetry header in bits.
        """
        offsets = self.db_cursor.execute('select byte_offset from fields where symbol=?', (symbol_id,)).fetchall()

        offsets.sort()
        return offsets[1]

    def __get_apid(self, message_id: int, offset: int = 11):
        bits = bin(message_id)
        apid = int(bits[len(bits) - offset: len(bits)], 2)
        return apid

    def add_telemetry_containers(self, module_name: str, module_id: int, parent_container: str = None):
        """
        Iterate through all of the rows of telemetry and build our containers for each message in the database.
        :return:
        """
        module_space_system = self[module_name]
        container_set = xtce.ContainerSetType()
        base_paramtype_set = xtce.ParameterTypeSetType()
        module_space_system.get_TelemetryMetaData().set_ParameterTypeSet(base_paramtype_set)
        base_param_set = xtce.ParameterSetType()
        module_space_system.get_TelemetryMetaData().set_ParameterSet(base_param_set)
        module_space_system.get_TelemetryMetaData().set_ContainerSet(container_set)

        for tlm in self.db_cursor.execute('select * from telemetry where module=?',
                                          (module_id,)).fetchall():
            tlm_name = tlm[1]
            tlm_message_id = tlm[2]
            tlm_macro = tlm[3]
            tlm_symbol_id = tlm[4]
            tlm_module = tlm[5]
            min_rate = tlm[6]

            default_rate = None
            if min_rate is not None:
                default_rate = xtce.RateInStreamType(minimumValue=min_rate)

            seq_container = xtce.SequenceContainerType(name=str(tlm_name), DefaultRateInStream=default_rate)
            container_entry_list = xtce.EntryListType()
            seq_container.set_EntryList(container_entry_list)

            logging.debug(f'message id:{tlm_message_id}')

            for symbol in self.db_cursor.execute('select * from symbols where id=?',
                                                 (tlm_symbol_id,)).fetchall():
                logging.debug(f'symbol{symbol} for tlm:{tlm_name}')

                aggregate_type = self.__get_aggregate_paramtype(symbol, module_name,
                                                                header_size=self.__get_telemetry_base_container_length())

                if aggregate_type and len(aggregate_type.get_MemberList().get_Member()) > 0:
                    if self.__aggregate_paramtype_exists(symbol[2], module_name) is False:
                        base_paramtype_set.add_AggregateParameterType(aggregate_type)
                    telemetry_param = xtce.ParameterType(name=tlm_name,
                                                         parameterTypeRef=aggregate_type.get_name())

                    container_param_ref = xtce.ParameterRefEntryType(parameterRef=telemetry_param.get_name())

                    # Ensure that we do not duplicate a parameter name.
                    if self.__aggregate_param_exists(telemetry_param.get_name(), module_name) is False:
                        logging.warning(
                            f'The parameter {telemetry_param.get_name()} is being used multiple times. This might be a '
                            f'sign of something wrong in configuration or flight software.')
                        base_param_set.add_Parameter(telemetry_param)

                    container_entry_list.add_ParameterRefEntry(container_param_ref)
                if parent_container:
                    seq_container.set_BaseContainer(
                        xtce.BaseContainerType(containerRef=parent_container + '/cfs-tlm-hdr'))
                    comparison = xtce.ComparisonType()
                    comparison.set_parameterRef(parent_container + '/ccsds-apid')
                    comparison.set_value(self.__get_apid(tlm_message_id))
                    base_container_restriction = xtce.RestrictionCriteriaType()
                    comparison_list = xtce.ComparisonListType()
                    comparison_list.add_Comparison(comparison)
                    base_container_restriction.set_ComparisonList(comparison_list)
                    seq_container.get_BaseContainer().set_RestrictionCriteria(base_container_restriction)

                    container_set.add_SequenceContainer(seq_container)

    def __get_command_argument_assignment_list(self, apid: tuple, command_code: tuple, command_length: tuple):
        # FIXME: There does not seem to be point to passing self here. Will re-visit.
        """
        Factory function that creates a ArgumentAssignmentListType containing three ArgumentAssignment objects;
        one for apid, one for command_code and another one for command_length.
        :param apid: A tuple of the form (arg_name: str, arg_value: int)
        :param command_code:A tuple of the form (arg_name: str, arg_value: int)
        :param command_length:A tuple of the form (arg_name: str, arg_value: int)
        :return:
        """
        out_argument_list = xtce.ArgumentAssignmentListType()

        apid_arg = xtce.ArgumentAssignmentType(argumentName=apid[0], argumentValue=apid[1])
        command_code_arg = xtce.ArgumentAssignmentType(argumentName=command_code[0], argumentValue=command_code[1])
        command_length_arg = xtce.ArgumentAssignmentType(argumentName=command_length[0],
                                                         argumentValue=command_length[1])

        out_argument_list.add_ArgumentAssignment(apid_arg)
        out_argument_list.add_ArgumentAssignment(command_code_arg)
        out_argument_list.add_ArgumentAssignment(command_length_arg)

        return out_argument_list

    def __get_command_length(self, symbol_id: int):
        """
        Calculate the size of a command(in bytes) that uses the struct that has symbol_id as its id on the database.
        This function assumes the fields table schema is (id,symbol, name, byte_offset, multiplicity, little_endian,
        bit_size, bit_offset).
        This function ALWAYS subtract 7 from the total size.
        :param symbol_id: The id of the struct used to calculate the size(length) of a command.
        :return:
        """

        out_length = 0
        fields = self.db_cursor.execute('SELECT * FROM fields where symbol=?',
                                        (symbol_id,)).fetchall()
        for field_id, field_symbol, field_name, field_byte_offset, field_type, field_little_endian, bit_offset, bit_size in \
                fields:
            field_multiplicity = self.__get_array(field_id)

            if len(field_multiplicity) > 0:
                field_multiplicity = field_multiplicity[0][3]
            else:
                field_multiplicity = 0

            size_of_symbol = self.db_cursor.execute('SELECT byte_size from symbols where id=?',
                                                    (field_type,)).fetchone()[0]
            if field_multiplicity > 0:
                size_of_symbol *= field_multiplicity
            out_length += size_of_symbol

        if out_length < self.custom_config['global']['CommandMetaData']['BaseContainer']['size'] / 8:
            out_length += int(self.custom_config['global']['CommandMetaData']['BaseContainer']['size'] / 8 - out_length)

        return out_length - 7

    def __get_argtype_from_typeref(self, type_ref: str, namesapce: str):
        out_arg_type = self.UNKNOWN_TYPE
        if self.BASE_TYPE_NAMESPACE in type_ref:
            out_arg_type = RefType.BaseType
        elif type_ref in [aggregate_type.get_name() for aggregate_type in
                          self[namesapce].get_CommandMetaData().get_ArgumentTypeSet().get_AggregateArgumentType()
                          if aggregate_type.get_name() == type_ref]:
            out_arg_type = RefType.AGGREGATE
        elif type_ref in [enum_type.get_name() for enum_type in
                          self[namesapce].get_CommandMetaData().get_ArgumentTypeSet().get_EnumeratedArgumentType()
                          if enum_type.get_name() == type_ref]:
            out_arg_type = RefType.ENUM

        return out_arg_type

    def __extract_members_from_aggregate_argtype(self, aggregate: xtce.AggregateArgumentType, namespace: str):
        """
        Returns a list of the members inside of a xtce.AggregateArgumentType, essentially flattening the
        entire Aggregate. This is especially useful when populating an ArgumentList in CommandMetaData
        objects.
        :return:
        """
        out_members = xtce.MemberListType()
        for member in aggregate.get_MemberList().get_Member():
            type_ref = member.get_typeRef()
            new_member = xtce.MemberType()
            arg_type = self.__get_argtype_from_typeref(type_ref, namespace)
            if arg_type == RefType.BaseType:
                new_member.set_name(member.get_name())
                new_member.set_typeRef(type_ref)

                out_members.add_Member(new_member)
            elif arg_type == RefType.ENUM:
                new_member.set_name(member.get_name())
                new_member.set_typeRef(type_ref)

                out_members.add_Member(new_member)
            elif arg_type == RefType.AGGREGATE:
                # FIXME: Make this more readable
                aggregate_members = self.__extract_members_from_aggregate_argtype(
                    [aggregate_type for aggregate_type in self[namespace].
                        get_CommandMetaData().get_ArgumentTypeSet().get_AggregateArgumentType()
                     if aggregate_type.get_name() == type_ref][0], namespace)
                for aggregate_member in aggregate_members.get_Member():
                    new_member = xtce.MemberType()

                    new_member.set_name(member.get_name() + '.' + aggregate_member.get_name())
                    new_member.set_typeRef(aggregate_member.get_typeRef())

                    out_members.add_Member(new_member)

        return out_members

    def add_command_containers(self, module_name: str, module_id: int, parent_command: str = None):
        """
        Iterate through all of the rows of the commands table and build our containers for each command in the database.
        :return:
        """
        meta_command_set = xtce.MetaCommandSetType()
        base_argtype_set = xtce.ArgumentTypeSetType()
        self[module_name].get_CommandMetaData().set_ArgumentTypeSet(base_argtype_set)
        self[module_name].get_CommandMetaData().set_MetaCommandSet(meta_command_set)

        for command in self.db_cursor.execute('select * from commands where module=?',
                                              (module_id,)).fetchall():
            command_name = command[1]
            command_code = command[2]
            command_message_id = command[3]
            command_macro = command[4]
            command_symbol_id = command[5]
            command_module = command[6]

            meta_command = xtce.MetaCommandType(name=command_name)
            command_container = xtce.CommandContainerType(
                name=command_name + '-container')
            container_entry_list = xtce.CommandContainerEntryListType()
            command_container.set_EntryList(container_entry_list)

            base_arg_set = xtce.ArgumentListType()
            logging.debug(f'message id:{command_message_id}')

            meta_command.set_ArgumentList(base_arg_set)

            for symbol in self.db_cursor.execute('select * from symbols where id=?',
                                                 (command_symbol_id,)).fetchall():
                logging.debug(f'symbol{symbol} for tlm:{command_name}')

                aggregate_type = self.__get_aggregate_argtype(symbol, module_name,
                                                              header_size=self.__get_command_base_container_length())

                if aggregate_type and len(aggregate_type.get_MemberList().get_Member()) > 0:
                    if self.__aggregate_argtype_exists(symbol[2], module_name) is False:
                        base_argtype_set.add_AggregateArgumentType(aggregate_type)

                    # If the list is not flattened, then YAMCS does not allow us to send the command from the Web Interface
                    for member in self.__extract_members_from_aggregate_argtype(aggregate_type,
                                                                                module_name).get_Member():
                        command_arg = xtce.ArgumentType(name=member.get_name(),
                                                        argumentTypeRef=member.get_typeRef())
                        container_entry_list.add_ArgumentRefEntry(
                            xtce.ArgumentArgumentRefEntryType(argumentRef=command_arg.get_name()))
                        base_arg_set.add_Argument(command_arg)
                    # container_entry_list.setA('container_arg_ref')
                    meta_command.set_ArgumentList(base_arg_set)

                # NOTE: It is assumed that the arguments to be set on the base command are apid, command_code and
                # command_length
            base_command = None
            if parent_command:
                base_command = xtce.BaseMetaCommandType(metaCommandRef=parent_command)
                argument_assignment_list = self.__get_command_argument_assignment_list(
                    ('ccsds-apid', self.__get_apid(command_message_id)),
                    ('cfs-cmd-code', command_code),
                    ('ccsds-length', self.__get_command_length(command_symbol_id)))

                base_command.set_ArgumentAssignmentList(argument_assignment_list)

            if base_command:
                meta_command.set_BaseMetaCommand(base_command)
                meta_command.set_CommandContainer(command_container)
                meta_command_set.add_MetaCommand(meta_command)

    def __inspect_parent_modules(self, child_module_name: str, out_parent_modules: list):
        """
        inspect all parent modules of child_module_name and add them to out_parent_modules.
        Very useful for constructing qualified namespaces such as "obc/simlink/apps/simlink"
        """
        out_parent_modules.append(child_module_name)
        parent_key = \
        self.db_cursor.execute('select parent_module from modules where name=?', (child_module_name,)).fetchone()[0]
        if parent_key is not None:
            new_parent = self.db_cursor.execute('select name from modules where id=?', (parent_key,)).fetchone()
            self.__inspect_parent_modules(new_parent[0], out_parent_modules)

    def __get_qualified_namespace(self, module_list):
        qualified_name = XTCEManager.NAMESPACE_SEPARATOR + self.root.get_name() + XTCEManager.NAMESPACE_SEPARATOR
        for name in module_list:
            qualified_name += name
            qualified_name += XTCEManager.NAMESPACE_SEPARATOR

        return qualified_name

    def add_aggregate_types(self):
        """
        Iterate through all of the symbols in the database and add them to the TelemetryMetaDataType and
        CommandMetaDataType children of our root SpaceSystem.
        :return:
        """
        for module_id in set(self.db_cursor.execute('select module from telemetry').fetchall()):
            module = self.db_cursor.execute('select id, name from modules where id=?', (module_id[0],)).fetchone()
            logging.info(f'Adding telemetry containers to namespace "{module[1]}".')

            modules = []
            self.__inspect_parent_modules(module[1], modules)
            modules.reverse()
            qualified_module_name = self.__get_qualified_namespace(modules)
            self.add_telemetry_containers(qualified_module_name, module[0],
                                          self.custom_config['global']['TelemetryMetaData']['BaseContainer'][
                                              'container_ref'])

        for module_id in set(self.db_cursor.execute('select module from commands').fetchall()):
            module = self.db_cursor.execute('select id, name from modules where id=?', (module_id[0],)).fetchone()
            logging.info(f'Adding command containers to namespace "{module[1]}"')
            modules = []
            self.__inspect_parent_modules(module[1], modules)
            modules.reverse()
            qualified_module_name = self.__get_qualified_namespace(modules)
            self.add_command_containers(qualified_module_name, module[0],
                                        self.custom_config['global']['CommandMetaData']['BaseContainer'][
                                            'container_ref'])

    def __get_namespace(self, namespace_name: str) -> xtce.SpaceSystemType:
        """
        Returns a namespace SpaceSystemType object that has the name of namespace_name.
        :param namespace_name:
        :return:
        """
        namespace_name = namespace_name.rstrip(XTCEManager.NAMESPACE_SEPARATOR)
        return self.__namespace_dict[namespace_name]

    def namespace_exists(self, namespace_name: str) -> xtce.SpaceSystemType:
        """
        Returns whether or not the namespace exists. Please if you are a user(and not an API author)
        always use this function instead of accessing __namespace_dict directly,
        :param namespace_name:
        :return:
        """
        namespace_name = namespace_name.rstrip(XTCEManager.NAMESPACE_SEPARATOR)
        return namespace_name in self.__namespace_dict

    def __query_spacesystem_from_qualified_name(self, qualified_name: str):
        """
        """
        qualified_name = qualified_name.strip(XTCEManager.NAMESPACE_SEPARATOR)
        current_name = ""
        current_space_system = self.root
        for space_system_name in qualified_name.split(XTCEManager.NAMESPACE_SEPARATOR):
            current_name += XTCEManager.NAMESPACE_SEPARATOR + space_system_name
            if not (current_name in self.__namespace_dict):
                new_namespace = xtce.SpaceSystemType(name=space_system_name)
                current_space_system.add_SpaceSystem(new_namespace)
                new_namespace.set_TelemetryMetaData(xtce.TelemetryMetaDataType())
                new_namespace.set_CommandMetaData(xtce.CommandMetaDataType())
                self.__namespace_dict[current_name] = new_namespace
                current_space_system = new_namespace
            else:
                current_space_system = self.__get_namespace(current_name)
        current_name += XTCEManager.NAMESPACE_SEPARATOR

    def __getitem__(self, key: str) -> xtce.SpaceSystemType:
        """
        Returns a reference to the namespace with the name of key. If the namespace does not exist, then a new namespace
        with the name of key is created. New namespaces are guaranteed to have an empty CommandMetaDataType object and
        an empty TelemetryMetaDataType object.
        :param key: The name of the namespace.
        :return:
        """
        if not (XTCEManager.NAMESPACE_SEPARATOR in key):
            if key not in self.__namespace_dict:
                key = XTCEManager.NAMESPACE_SEPARATOR + self.root.get_name() + XTCEManager.NAMESPACE_SEPARATOR + key
                # self.__namespace_dict[key] = xtce.SpaceSystemType(name=key)
                # self.__get_namespace(key).set_CommandMetaData(xtce.CommandMetaDataType())
                # self.__get_namespace(key).set_TelemetryMetaData(xtce.TelemetryMetaDataType())
                self.add_namespace(
                    XTCEManager.NAMESPACE_SEPARATOR + self.root.get_name() + XTCEManager.NAMESPACE_SEPARATOR + key)
        else:
            self.__query_spacesystem_from_qualified_name(key)

        return self.__get_namespace(key)

    def write_to_file(self, namespace: str):
        """
        Writes the current xtce spacesystem to a file.
        :return:
        """
        xtce_file = open(self.output_file, 'w+')

        # Like I said before; this is a horrid, horrid hack and will try to come up with a much more elegant solution.
        if self.nested_root:
            out_root = xtce.SpaceSystemType(name=self.custom_config[_ROOT_SPACESYSTEM_KEY])
            out_root.add_SpaceSystem(self[namespace])

        else:
            out_root = self[XTCEManager.NAMESPACE_SEPARATOR + namespace]

        out_root.export(xtce_file, 0, namespacedef_='xmlns:xml="http://www.w3.org/XML/1998/namespace" '
                                                    'xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" '
                                                    'xmlns:xtce="http://www.omg.org/spec/XTCE/20180204" '
                                                    'xsi:schemaLocation="http://www.omg.org/spec/XTCE/20180204 SpaceSystem.xsd "',
                        namespaceprefix_='xtce:')

    def resolve_root_sapcesystem(config_data: dict):
        root_space_system = None
        if _ROOT_SPACESYSTEM_KEY in config_data:
            if _CPU_ID_KEY in config_data:
                root_space_system = f"{config_data[_ROOT_SPACESYSTEM_KEY]}" \
                                    f"{XTCEManager.NAMESPACE_SEPARATOR}" \
                                    f"{config_data[_CPU_ID_KEY]}"
        else:
            logging.error(f'No {_ROOT_SPACESYSTEM_KEY} key found in configuration ')

        return root_space_system


def parse_cli() -> argparse.Namespace:
    """
    Parses cli arguments.
    :return: The namespace that has all of the arguments that have been parsed.
    """

    parser = argparse.ArgumentParser(description='Takes in path to sqlite database.')

    parser.add_argument('--sqlite_path', type=str,
                        help='The file path to the sqlite database', required=True)
    parser.add_argument('--log_level', type=str, default='0', choices=['0', '1', '2', '3',
                                                                       '4'],
                        help='[(0=SILENT), (1=ERRORS), (2=WARNINGS), (3=INFO), (4=DEBUG)]')

    parser.add_argument('--config_yaml', type=str, required=True,
                        help='An config file to apply extra settings to the xtce generation such as'
                             'mapping an Aggregate Type to a base type, or to another aggregate type altogether. This'
                             'is where the root spacesystem is defined as well.')
    parser.add_argument('--output_path', type=str, default=None, required=True,
                        help='The file path to write the output xtce file to. The default is the current directory.')

    return parser.parse_args()


def read_yaml(yaml_file: str) -> dict:
    yaml_data = yaml.load(open(yaml_file, 'r'),
                          Loader=yaml.FullLoader)
    return yaml_data


logging_map = {'1': logging.ERROR,
               '2': logging.WARNING,
               '3': logging.INFO,
               '4': logging.DEBUG,
               }


def set_log_level(log_level: str):
    if log_level == '0':
        for key, level in logging_map.items():
            logging.disable(level)
    else:
        logging.getLogger().setLevel(logging_map[log_level])

    logging.getLogger().name = 'xtce_generator'


def generate_xtce(database_path: str, config_yaml: dict, output_path: str, root_spacesystem: str = '/airliner',
                  log_level: str = '0', cpu_id: str = None):
    """
    Generate an xtce file from the database at database_path. This database is assumed to have
    been generated by juicer. To read about more about juicer, look at [1]. It should at least comply with the schema
    used by juicer.
    [1]:https://github.com/WindhoverLabs/juicer
    :param output_path:
    :param database_path: The database file path.
    :param config_yaml: An optional configuration YAML file that can be used to configure the xtce file. This is
    particularly useful for mapping parent container in the xtce file.
    :param root_spacesystem: The name of the root spacesystem that will be used in the output xtce file.
    :param cpu_id: The id of the flight computer(CPD, PPD, etc).
    :param log_level:
    :return:
    """
    set_log_level(log_level)

    logging.info('Building xtce object...')

    xtce_obj = XTCEManager(root_spacesystem, output_path, database_path, config_yaml, cpu_id)

    logging.info('Adding base_types to xtce...')
    xtce_obj.add_base_types()

    logging.info('Adding aggregate types to xtce...')
    xtce_obj.add_aggregate_types()

    logging.info('Writing xtce object to file...')
    xtce_obj.write_to_file(namespace=root_spacesystem)

    logging.info(f'XTCE file has been written to "{xtce_obj.output_file}"')


def main():
    logging.info('Parsing CLI arguments...')

    args = parse_cli()

    if args.config_yaml:
        config_data = read_yaml(args.config_yaml)
    else:
        config_data = None

    if _ROOT_SPACESYSTEM_KEY in config_data:
        if _CPU_ID_KEY in config_data:
            cpu_id = config_data[_CPU_ID_KEY]
            generate_xtce(args.sqlite_path, config_data, args.output_path, config_data[_ROOT_SPACESYSTEM_KEY],
                          args.log_level, cpu_id=cpu_id)
        else:
            generate_xtce(args.sqlite_path, config_data, args.output_path, config_data[_ROOT_SPACESYSTEM_KEY],
                          args.log_level)
    else:
        logging.error(f'No root_spacesystem key found in configuration "{args.config_yaml}"')


if __name__ == '__main__':
    main()
