"use strict";
const autoscaling = require("@aws-cdk/aws-autoscaling");
const ec2 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const user_data_1 = require("../lib/user-data");
function newFixtures(spot = false) {
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'my-stack', { env: { region: 'us-west-33' } });
    const vpc = new ec2.Vpc(stack, 'vpc');
    const asg = new autoscaling.AutoScalingGroup(stack, 'ASG', {
        instanceType: new ec2.InstanceType('m4.xlarge'),
        machineImage: new ec2.AmazonLinuxImage(),
        spotPrice: spot ? '0.01' : undefined,
        vpc
    });
    return { stack, vpc, asg };
}
module.exports = {
    'default user data'(test) {
        // GIVEN
        const { asg, stack } = newFixtures();
        // WHEN
        const userData = stack.resolve(user_data_1.renderUserData('my-cluster-name', asg));
        // THEN
        test.deepEqual(userData, [
            'set -o xtrace',
            '/etc/eks/bootstrap.sh my-cluster-name --kubelet-extra-args "--node-labels lifecycle=OnDemand" --use-max-pods true',
            '/opt/aws/bin/cfn-signal --exit-code $? --stack my-stack --resource ASG46ED3070 --region us-west-33'
        ]);
        test.done();
    },
    '--use-max-pods=true'(test) {
        // GIVEN
        const { asg, stack } = newFixtures();
        // WHEN
        const userData = stack.resolve(user_data_1.renderUserData('my-cluster-name', asg, {
            useMaxPods: true
        }));
        // THEN
        test.deepEqual(userData[1], '/etc/eks/bootstrap.sh my-cluster-name --kubelet-extra-args "--node-labels lifecycle=OnDemand" --use-max-pods true');
        test.done();
    },
    '--use-max-pods=false'(test) {
        // GIVEN
        const { asg, stack } = newFixtures();
        // WHEN
        const userData = stack.resolve(user_data_1.renderUserData('my-cluster-name', asg, {
            useMaxPods: false
        }));
        // THEN
        test.deepEqual(userData[1], '/etc/eks/bootstrap.sh my-cluster-name --kubelet-extra-args "--node-labels lifecycle=OnDemand" --use-max-pods false');
        test.done();
    },
    '--aws-api-retry-attempts'(test) {
        // GIVEN
        const { asg, stack } = newFixtures();
        // WHEN
        const userData = stack.resolve(user_data_1.renderUserData('my-cluster-name', asg, {
            awsApiRetryAttempts: 123
        }));
        // THEN
        test.deepEqual(userData[1], '/etc/eks/bootstrap.sh my-cluster-name --kubelet-extra-args "--node-labels lifecycle=OnDemand" --use-max-pods true --aws-api-retry-attempts 123');
        test.done();
    },
    '--docker-config-json'(test) {
        // GIVEN
        const { asg } = newFixtures();
        // WHEN
        const userData = user_data_1.renderUserData('my-cluster-name', asg, {
            dockerConfigJson: '{"docker":123}'
        });
        // THEN
        test.deepEqual(userData[1], `/etc/eks/bootstrap.sh my-cluster-name --kubelet-extra-args "--node-labels lifecycle=OnDemand" --use-max-pods true --docker-config-json '{"docker":123}'`);
        test.done();
    },
    '--enable-docker-bridge=true'(test) {
        // GIVEN
        const { asg, stack } = newFixtures();
        // WHEN
        const userData = stack.resolve(user_data_1.renderUserData('my-cluster-name', asg, {
            enableDockerBridge: true
        }));
        // THEN
        test.deepEqual(userData[1], `/etc/eks/bootstrap.sh my-cluster-name --kubelet-extra-args "--node-labels lifecycle=OnDemand" --use-max-pods true --enable-docker-bridge`);
        test.done();
    },
    '--enable-docker-bridge=false'(test) {
        // GIVEN
        const { asg, stack } = newFixtures();
        // WHEN
        const userData = stack.resolve(user_data_1.renderUserData('my-cluster-name', asg, {
            enableDockerBridge: false
        }));
        // THEN
        test.deepEqual(userData[1], `/etc/eks/bootstrap.sh my-cluster-name --kubelet-extra-args "--node-labels lifecycle=OnDemand" --use-max-pods true`);
        test.done();
    },
    '--kubelet-extra-args'(test) {
        // GIVEN
        const { asg, stack } = newFixtures();
        // WHEN
        const userData = stack.resolve(user_data_1.renderUserData('my-cluster-name', asg, {
            kubeletExtraArgs: '--extra-args-for --kubelet'
        }));
        // THEN
        test.deepEqual(userData[1], `/etc/eks/bootstrap.sh my-cluster-name --kubelet-extra-args "--node-labels lifecycle=OnDemand  --extra-args-for --kubelet" --use-max-pods true`);
        test.done();
    },
    'arbitrary additional bootstrap arguments can be passed through "additionalArgs"'(test) {
        // GIVEN
        const { asg, stack } = newFixtures();
        // WHEN
        const userData = stack.resolve(user_data_1.renderUserData('my-cluster-name', asg, {
            additionalArgs: '--apiserver-endpoint 1111 --foo-bar'
        }));
        // THEN
        test.deepEqual(userData[1], `/etc/eks/bootstrap.sh my-cluster-name --kubelet-extra-args "--node-labels lifecycle=OnDemand" --use-max-pods true --apiserver-endpoint 1111 --foo-bar`);
        test.done();
    },
    'if asg has spot instances, the correct label and taint is used'(test) {
        // GIVEN
        const { asg, stack } = newFixtures(true);
        // WHEN
        const userData = stack.resolve(user_data_1.renderUserData('my-cluster-name', asg, {
            kubeletExtraArgs: '--node-labels X=y'
        }));
        // THEN
        test.deepEqual(userData[1], `/etc/eks/bootstrap.sh my-cluster-name --kubelet-extra-args "--node-labels lifecycle=Ec2Spot --register-with-taints=spotInstance=true:PreferNoSchedule --node-labels X=y" --use-max-pods true`);
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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