"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const assert_2 = require("assert");
const batch = require("../lib");
describe('Batch Compute Evironment', () => {
    let expectedManagedDefaultComputeProps;
    let defaultServiceRole;
    let stack;
    let vpc;
    beforeEach(() => {
        // GIVEN
        stack = new cdk.Stack();
        vpc = new ec2.Vpc(stack, 'test-vpc');
        defaultServiceRole = {
            ServiceRole: {
                'Fn::GetAtt': [
                    'testcomputeenvResourceServiceInstanceRole105069A5',
                    'Arn'
                ],
            },
        };
        expectedManagedDefaultComputeProps = (overrides) => {
            return {
                ComputeResources: {
                    AllocationStrategy: batch.AllocationStrategy.BEST_FIT,
                    InstanceRole: {
                        'Fn::GetAtt': [
                            'testcomputeenvInstanceProfileCBD87EAB',
                            'Arn'
                        ]
                    },
                    InstanceTypes: [
                        'optimal'
                    ],
                    MaxvCpus: 256,
                    MinvCpus: 0,
                    Subnets: [
                        {
                            Ref: 'testvpcPrivateSubnet1Subnet865FB50A'
                        },
                        {
                            Ref: 'testvpcPrivateSubnet2Subnet23D3396F'
                        }
                    ],
                    Type: batch.ComputeResourceType.ON_DEMAND,
                    ...overrides,
                }
            };
        };
    });
    describe('when validating props', () => {
        test('should deny setting compute resources when using type unmanaged', () => {
            // THEN
            assert_2.throws(() => {
                // WHEN
                new batch.ComputeEnvironment(stack, 'test-compute-env', {
                    managed: false,
                    computeResources: {
                        vpc
                    },
                });
            });
        });
        test('should deny if creating a managed environment with no provided compute resource props', () => {
            // THEN
            assert_2.throws(() => {
                // WHEN
                new batch.ComputeEnvironment(stack, 'test-compute-env', {
                    managed: true,
                });
            });
        });
    });
    describe('using spot resources', () => {
        test('should provide a spot fleet role if one is not given and allocationStrategy is BEST_FIT', () => {
            // WHEN
            new batch.ComputeEnvironment(stack, 'test-compute-env', {
                managed: true,
                computeResources: {
                    type: batch.ComputeResourceType.SPOT,
                    allocationStrategy: batch.AllocationStrategy.BEST_FIT,
                    vpc
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Batch::ComputeEnvironment', {
                Type: 'MANAGED',
                ...expectedManagedDefaultComputeProps({
                    Type: batch.ComputeResourceType.SPOT,
                    SpotIamFleetRole: {
                        'Fn::Join': [
                            '',
                            [
                                'arn',
                                {
                                    Ref: 'AWS::Partition'
                                },
                                'iam::',
                                {
                                    Ref: 'AWS::AccountId'
                                },
                                ':role/aws-service-role/spotfleet.amazonaws.com/AWSServiceRoleForEC2SpotFleet',
                            ],
                        ],
                    },
                }),
            }, assert_1.ResourcePart.Properties));
        });
        describe('with a bid percentage', () => {
            test('should deny my bid if set below 0', () => {
                // THEN
                assert_2.throws(() => {
                    // WHEN
                    new batch.ComputeEnvironment(stack, 'test-compute-env', {
                        managed: true,
                        computeResources: {
                            vpc,
                            type: batch.ComputeResourceType.SPOT,
                            bidPercentage: -1,
                        },
                    });
                });
            });
            test('should deny my bid if above 100', () => {
                // THEN
                assert_2.throws(() => {
                    // WHEN
                    new batch.ComputeEnvironment(stack, 'test-compute-env', {
                        managed: true,
                        computeResources: {
                            vpc,
                            type: batch.ComputeResourceType.SPOT,
                            bidPercentage: 101,
                        },
                    });
                });
            });
        });
    });
    describe('with properties specified', () => {
        test('renders the correct cloudformation properties', () => {
            // WHEN
            const props = {
                computeEnvironmentName: 'my-test-compute-env',
                computeResources: {
                    allocationStrategy: batch.AllocationStrategy.BEST_FIT,
                    vpc,
                    computeResourcesTags: new cdk.Tag('foo', 'bar'),
                    desiredvCpus: 1,
                    ec2KeyPair: 'my-key-pair',
                    image: new ecs.EcsOptimizedAmi({
                        generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
                        hardwareType: ecs.AmiHardwareType.STANDARD,
                    }),
                    instanceRole: new iam.CfnInstanceProfile(stack, 'Instance-Profile', {
                        roles: [new iam.Role(stack, 'Ecs-Instance-Role', {
                                assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
                                managedPolicies: [
                                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2ContainerServiceforEC2Role')
                                ]
                            }).roleName]
                    }).attrArn,
                    instanceTypes: [
                        ec2.InstanceType.of(ec2.InstanceClass.T2, ec2.InstanceSize.MICRO),
                    ],
                    maxvCpus: 4,
                    minvCpus: 1,
                    securityGroups: [
                        new ec2.SecurityGroup(stack, 'test-sg', {
                            vpc,
                            allowAllOutbound: true,
                        }),
                    ],
                    type: batch.ComputeResourceType.ON_DEMAND,
                    vpcSubnets: {
                        subnetType: ec2.SubnetType.PRIVATE,
                    },
                },
                enabled: false,
                managed: true,
            };
            new batch.ComputeEnvironment(stack, 'test-compute-env', props);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Batch::ComputeEnvironment', {
                ComputeEnvironmentName: 'my-test-compute-env',
                Type: 'MANAGED',
                State: 'DISABLED',
                ServiceRole: {
                    'Fn::GetAtt': [
                        'testcomputeenvResourceServiceInstanceRole105069A5',
                        'Arn'
                    ],
                },
                ComputeResources: {
                    AllocationStrategy: batch.AllocationStrategy.BEST_FIT,
                    DesiredvCpus: props.computeResources.desiredvCpus,
                    Ec2KeyPair: props.computeResources.ec2KeyPair,
                    ImageId: {
                        Ref: 'SsmParameterValueawsserviceecsoptimizedamiamazonlinux2recommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter'
                    },
                    InstanceRole: {
                        'Fn::GetAtt': [
                            props.computeResources.instanceRole ? "InstanceProfile" : '',
                            'Arn'
                        ]
                    },
                    InstanceTypes: [
                        props.computeResources.instanceTypes ? props.computeResources.instanceTypes[0].toString() : '',
                    ],
                    MaxvCpus: props.computeResources.maxvCpus,
                    MinvCpus: props.computeResources.minvCpus,
                    SecurityGroupIds: [
                        {
                            'Fn::GetAtt': [
                                'testsg872EB48A',
                                'GroupId'
                            ]
                        }
                    ],
                    Subnets: [
                        {
                            Ref: `${vpc.node.uniqueId}PrivateSubnet1Subnet865FB50A`
                        },
                        {
                            Ref: `${vpc.node.uniqueId}PrivateSubnet2Subnet23D3396F`
                        }
                    ],
                    Tags: {
                        key: 'foo',
                        props: {},
                        defaultPriority: 100,
                        value: 'bar'
                    },
                    Type: 'EC2'
                },
            }, assert_1.ResourcePart.Properties));
        });
        describe('with no allocation strategy specified', () => {
            test('should default to a best_fit strategy', () => {
                // WHEN
                new batch.ComputeEnvironment(stack, 'test-compute-env', {
                    managed: true,
                    computeResources: {
                        vpc,
                    },
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Batch::ComputeEnvironment', {
                    Type: 'MANAGED',
                    ServiceRole: {
                        'Fn::GetAtt': [
                            'testcomputeenvResourceServiceInstanceRole105069A5',
                            'Arn'
                        ],
                    },
                }, assert_1.ResourcePart.Properties));
            });
        });
        describe('with a min vcpu value', () => {
            test('should deny less than 0', () => {
                // THEN
                assert_2.throws(() => {
                    // WHEN
                    new batch.ComputeEnvironment(stack, 'test-compute-env', {
                        computeResources: {
                            vpc,
                            minvCpus: -1,
                        },
                    });
                });
            });
            test('cannot be greater than the max vcpu value', () => {
                // THEN
                assert_2.throws(() => {
                    // WHEN
                    new batch.ComputeEnvironment(stack, 'test-compute-env', {
                        computeResources: {
                            vpc,
                            minvCpus: 2,
                            maxvCpus: 1,
                        },
                    });
                });
            });
        });
        describe('with no min vcpu value provided', () => {
            test('should default to 0', () => {
                // WHEN
                new batch.ComputeEnvironment(stack, 'test-compute-env', {
                    managed: true,
                    computeResources: {
                        vpc,
                    },
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Batch::ComputeEnvironment', {
                    ...defaultServiceRole,
                    ...expectedManagedDefaultComputeProps({
                        MinvCpus: 0,
                    }),
                }, assert_1.ResourcePart.Properties));
            });
        });
        describe('with no max vcpu value provided', () => {
            test('should default to 256', () => {
                // WHEN
                new batch.ComputeEnvironment(stack, 'test-compute-env', {
                    managed: true,
                    computeResources: {
                        vpc,
                    },
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Batch::ComputeEnvironment', {
                    ...expectedManagedDefaultComputeProps({
                        MaxvCpus: 256,
                    }),
                }, assert_1.ResourcePart.Properties));
            });
        });
        describe('with no instance role specified', () => {
            test('should generate a role for me', () => {
                // WHEN
                new batch.ComputeEnvironment(stack, 'test-compute-env', {
                    managed: true,
                    computeResources: {
                        vpc,
                    },
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResource('AWS::Batch::ComputeEnvironment'));
                assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role'));
            });
        });
        describe('with no instance type defined', () => {
            test('should default to optimal matching', () => {
                // WHEN
                new batch.ComputeEnvironment(stack, 'test-compute-env', {
                    managed: true,
                    computeResources: {
                        vpc,
                    },
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Batch::ComputeEnvironment', {
                    ...expectedManagedDefaultComputeProps({
                        InstanceTypes: ['optimal'],
                    }),
                }, assert_1.ResourcePart.Properties));
            });
        });
        describe('with no type specified', () => {
            test('should default to EC2', () => {
                // WHEN
                new batch.ComputeEnvironment(stack, 'test-compute-env', {
                    managed: true,
                    computeResources: {
                        vpc,
                    },
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Batch::ComputeEnvironment', {
                    ...expectedManagedDefaultComputeProps({
                        Type: batch.ComputeResourceType.ON_DEMAND,
                    }),
                }, assert_1.ResourcePart.Properties));
            });
        });
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29tcHV0ZS1lbnZpcm9ubWVudC50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiY29tcHV0ZS1lbnZpcm9ubWVudC50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsNENBQXFGO0FBQ3JGLGdDQUE4QjtBQUM5Qix3Q0FBd0M7QUFDeEMsd0NBQXdDO0FBQ3hDLHdDQUF3QztBQUN4QyxxQ0FBcUM7QUFDckMsbUNBQThCO0FBQzlCLGdDQUFnQztBQUVoQyxRQUFRLENBQUMsMEJBQTBCLEVBQUUsR0FBRyxFQUFFO0lBQ3hDLElBQUksa0NBQXVDLENBQUM7SUFDNUMsSUFBSSxrQkFBdUIsQ0FBQztJQUU1QixJQUFJLEtBQWdCLENBQUM7SUFDckIsSUFBSSxHQUFZLENBQUM7SUFFakIsVUFBVSxDQUFDLEdBQUcsRUFBRTtRQUNkLFFBQVE7UUFDUixLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDeEIsR0FBRyxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFFckMsa0JBQWtCLEdBQUc7WUFDbkIsV0FBVyxFQUFFO2dCQUNYLFlBQVksRUFBRTtvQkFDWixtREFBbUQ7b0JBQ25ELEtBQUs7aUJBQ047YUFDRjtTQUNGLENBQUM7UUFFRixrQ0FBa0MsR0FBRyxDQUFDLFNBQWMsRUFBRSxFQUFFO1lBQ3RELE9BQU87Z0JBQ0wsZ0JBQWdCLEVBQUU7b0JBQ2hCLGtCQUFrQixFQUFFLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxRQUFRO29CQUNyRCxZQUFZLEVBQUU7d0JBQ1osWUFBWSxFQUFFOzRCQUNaLHVDQUF1Qzs0QkFDdkMsS0FBSzt5QkFDTjtxQkFDRjtvQkFDRCxhQUFhLEVBQUU7d0JBQ2IsU0FBUztxQkFDVjtvQkFDRCxRQUFRLEVBQUUsR0FBRztvQkFDYixRQUFRLEVBQUUsQ0FBQztvQkFDWCxPQUFPLEVBQUU7d0JBQ1A7NEJBQ0UsR0FBRyxFQUFFLHFDQUFxQzt5QkFDM0M7d0JBQ0Q7NEJBQ0UsR0FBRyxFQUFFLHFDQUFxQzt5QkFDM0M7cUJBQ0Y7b0JBQ0QsSUFBSSxFQUFFLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxTQUFTO29CQUN6QyxHQUFHLFNBQVM7aUJBQ2I7YUFDRixDQUFDO1FBQ0osQ0FBQyxDQUFDO0lBQ0osQ0FBQyxDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsdUJBQXVCLEVBQUUsR0FBRyxFQUFFO1FBQ3JDLElBQUksQ0FBQyxpRUFBaUUsRUFBRSxHQUFHLEVBQUU7WUFDM0UsT0FBTztZQUNQLGVBQU0sQ0FBQyxHQUFHLEVBQUU7Z0JBQ1YsT0FBTztnQkFDUCxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsa0JBQWtCLEVBQUU7b0JBQ3RELE9BQU8sRUFBRSxLQUFLO29CQUNkLGdCQUFnQixFQUFFO3dCQUNoQixHQUFHO3FCQUNKO2lCQUNGLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsdUZBQXVGLEVBQUUsR0FBRyxFQUFFO1lBQ2pHLE9BQU87WUFDUCxlQUFNLENBQUMsR0FBRyxFQUFFO2dCQUNWLE9BQU87Z0JBQ1AsSUFBSSxLQUFLLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLGtCQUFrQixFQUFFO29CQUN0RCxPQUFPLEVBQUUsSUFBSTtpQkFDZCxDQUFDLENBQUM7WUFDTCxDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsc0JBQXNCLEVBQUUsR0FBRyxFQUFFO1FBQ3BDLElBQUksQ0FBQyx5RkFBeUYsRUFBRSxHQUFHLEVBQUU7WUFDbkcsT0FBTztZQUNQLElBQUksS0FBSyxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRSxrQkFBa0IsRUFBRTtnQkFDdEQsT0FBTyxFQUFFLElBQUk7Z0JBQ2IsZ0JBQWdCLEVBQUU7b0JBQ2hCLElBQUksRUFBRSxLQUFLLENBQUMsbUJBQW1CLENBQUMsSUFBSTtvQkFDcEMsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLGtCQUFrQixDQUFDLFFBQVE7b0JBQ3JELEdBQUc7aUJBQ0o7YUFDRixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxnQ0FBZ0MsRUFBRTtnQkFDbEUsSUFBSSxFQUFFLFNBQVM7Z0JBQ2YsR0FBRyxrQ0FBa0MsQ0FBQztvQkFDcEMsSUFBSSxFQUFFLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJO29CQUNwQyxnQkFBZ0IsRUFBRTt3QkFDaEIsVUFBVSxFQUFFOzRCQUNWLEVBQUU7NEJBQ0Y7Z0NBQ0UsS0FBSztnQ0FDTDtvQ0FDRSxHQUFHLEVBQUUsZ0JBQWdCO2lDQUN0QjtnQ0FDRCxPQUFPO2dDQUNQO29DQUNFLEdBQUcsRUFBRSxnQkFBZ0I7aUNBQ3RCO2dDQUNELDhFQUE4RTs2QkFDL0U7eUJBQ0Y7cUJBQ0Y7aUJBQ0YsQ0FBQzthQUNILEVBQUUscUJBQVksQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDO1FBQy9CLENBQUMsQ0FBQyxDQUFDO1FBRUgsUUFBUSxDQUFDLHVCQUF1QixFQUFFLEdBQUcsRUFBRTtZQUNyQyxJQUFJLENBQUMsbUNBQW1DLEVBQUUsR0FBRyxFQUFFO2dCQUM3QyxPQUFPO2dCQUNQLGVBQU0sQ0FBQyxHQUFHLEVBQUU7b0JBQ1YsT0FBTztvQkFDUCxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsa0JBQWtCLEVBQUU7d0JBQ3RELE9BQU8sRUFBRSxJQUFJO3dCQUNiLGdCQUFnQixFQUFFOzRCQUNoQixHQUFHOzRCQUNILElBQUksRUFBRSxLQUFLLENBQUMsbUJBQW1CLENBQUMsSUFBSTs0QkFDcEMsYUFBYSxFQUFFLENBQUMsQ0FBQzt5QkFDbEI7cUJBQ0YsQ0FBQyxDQUFDO2dCQUNMLENBQUMsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsaUNBQWlDLEVBQUUsR0FBRyxFQUFFO2dCQUMzQyxPQUFPO2dCQUNQLGVBQU0sQ0FBQyxHQUFHLEVBQUU7b0JBQ1YsT0FBTztvQkFDUCxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsa0JBQWtCLEVBQUU7d0JBQ3RELE9BQU8sRUFBRSxJQUFJO3dCQUNiLGdCQUFnQixFQUFFOzRCQUNoQixHQUFHOzRCQUNILElBQUksRUFBRSxLQUFLLENBQUMsbUJBQW1CLENBQUMsSUFBSTs0QkFDcEMsYUFBYSxFQUFFLEdBQUc7eUJBQ25CO3FCQUNGLENBQUMsQ0FBQztnQkFDTCxDQUFDLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQywyQkFBMkIsRUFBRSxHQUFHLEVBQUU7UUFDekMsSUFBSSxDQUFDLCtDQUErQyxFQUFFLEdBQUcsRUFBRTtZQUN6RCxPQUFPO1lBQ1AsTUFBTSxLQUFLLEdBQUc7Z0JBQ1osc0JBQXNCLEVBQUUscUJBQXFCO2dCQUM3QyxnQkFBZ0IsRUFBRTtvQkFDaEIsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLGtCQUFrQixDQUFDLFFBQVE7b0JBQ3JELEdBQUc7b0JBQ0gsb0JBQW9CLEVBQUUsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUM7b0JBQy9DLFlBQVksRUFBRSxDQUFDO29CQUNmLFVBQVUsRUFBRSxhQUFhO29CQUN6QixLQUFLLEVBQUUsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO3dCQUM3QixVQUFVLEVBQUUsR0FBRyxDQUFDLHFCQUFxQixDQUFDLGNBQWM7d0JBQ3BELFlBQVksRUFBRSxHQUFHLENBQUMsZUFBZSxDQUFDLFFBQVE7cUJBQzNDLENBQUM7b0JBQ0YsWUFBWSxFQUFHLElBQUksR0FBRyxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRSxrQkFBa0IsRUFBRTt3QkFDbkUsS0FBSyxFQUFFLENBQUUsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxtQkFBbUIsRUFBRTtnQ0FDaEQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLG1CQUFtQixDQUFDO2dDQUN4RCxlQUFlLEVBQUU7b0NBQ2YsR0FBRyxDQUFDLGFBQWEsQ0FBQyx3QkFBd0IsQ0FBQyxrREFBa0QsQ0FBQztpQ0FDL0Y7NkJBQ0YsQ0FBQyxDQUFDLFFBQVEsQ0FBQztxQkFDYixDQUFDLENBQUMsT0FBTztvQkFDVixhQUFhLEVBQUU7d0JBQ2IsR0FBRyxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxFQUFFLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUM7cUJBQ2xFO29CQUNELFFBQVEsRUFBRSxDQUFDO29CQUNYLFFBQVEsRUFBRSxDQUFDO29CQUNYLGNBQWMsRUFBRTt3QkFDZCxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRTs0QkFDdEMsR0FBRzs0QkFDSCxnQkFBZ0IsRUFBRSxJQUFJO3lCQUN2QixDQUFDO3FCQUNIO29CQUNELElBQUksRUFBRSxLQUFLLENBQUMsbUJBQW1CLENBQUMsU0FBUztvQkFDekMsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxHQUFHLENBQUMsVUFBVSxDQUFDLE9BQU87cUJBQ25DO2lCQUN3QjtnQkFDM0IsT0FBTyxFQUFFLEtBQUs7Z0JBQ2QsT0FBTyxFQUFFLElBQUk7YUFDZCxDQUFDO1lBRUYsSUFBSSxLQUFLLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLGtCQUFrQixFQUFFLEtBQUssQ0FBQyxDQUFDO1lBRS9ELE9BQU87WUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGdDQUFnQyxFQUFFO2dCQUNsRSxzQkFBc0IsRUFBRSxxQkFBcUI7Z0JBQzdDLElBQUksRUFBRSxTQUFTO2dCQUNmLEtBQUssRUFBRSxVQUFVO2dCQUNqQixXQUFXLEVBQUU7b0JBQ1gsWUFBWSxFQUFFO3dCQUNaLG1EQUFtRDt3QkFDbkQsS0FBSztxQkFDTjtpQkFDRjtnQkFDRCxnQkFBZ0IsRUFBRTtvQkFDaEIsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLGtCQUFrQixDQUFDLFFBQVE7b0JBQ3JELFlBQVksRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsWUFBWTtvQkFDakQsVUFBVSxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVO29CQUM3QyxPQUFPLEVBQUU7d0JBQ1AsR0FBRyxFQUFFLG1IQUFtSDtxQkFDekg7b0JBQ0QsWUFBWSxFQUFFO3dCQUNaLFlBQVksRUFBRTs0QkFDWixLQUFLLENBQUMsZ0JBQWdCLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsRUFBRTs0QkFDNUQsS0FBSzt5QkFDTjtxQkFDRjtvQkFDRCxhQUFhLEVBQUU7d0JBQ2IsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRTtxQkFDL0Y7b0JBQ0QsUUFBUSxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxRQUFRO29CQUN6QyxRQUFRLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLFFBQVE7b0JBQ3pDLGdCQUFnQixFQUFFO3dCQUNoQjs0QkFDRSxZQUFZLEVBQUU7Z0NBQ1osZ0JBQWdCO2dDQUNoQixTQUFTOzZCQUNWO3lCQUNGO3FCQUNGO29CQUNELE9BQU8sRUFBRTt3QkFDUDs0QkFDRSxHQUFHLEVBQUUsR0FBRyxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsOEJBQThCO3lCQUN4RDt3QkFDRDs0QkFDRSxHQUFHLEVBQUUsR0FBRyxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsOEJBQThCO3lCQUN4RDtxQkFDRjtvQkFDRCxJQUFJLEVBQUU7d0JBQ0osR0FBRyxFQUFFLEtBQUs7d0JBQ1YsS0FBSyxFQUFFLEVBQUU7d0JBQ1QsZUFBZSxFQUFFLEdBQUc7d0JBQ3BCLEtBQUssRUFBRSxLQUFLO3FCQUNiO29CQUNELElBQUksRUFBRSxLQUFLO2lCQUNaO2FBQ0YsRUFBRSxxQkFBWSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUM7UUFDL0IsQ0FBQyxDQUFDLENBQUM7UUFFSCxRQUFRLENBQUMsdUNBQXVDLEVBQUUsR0FBRyxFQUFFO1lBQ3JELElBQUksQ0FBQyx1Q0FBdUMsRUFBRSxHQUFHLEVBQUU7Z0JBQ2pELE9BQU87Z0JBQ1AsSUFBSSxLQUFLLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLGtCQUFrQixFQUFFO29CQUN0RCxPQUFPLEVBQUUsSUFBSTtvQkFDYixnQkFBZ0IsRUFBRTt3QkFDaEIsR0FBRztxQkFDSjtpQkFDRixDQUFDLENBQUM7Z0JBRUgsT0FBTztnQkFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGdDQUFnQyxFQUFFO29CQUNsRSxJQUFJLEVBQUUsU0FBUztvQkFDZixXQUFXLEVBQUU7d0JBQ1gsWUFBWSxFQUFFOzRCQUNaLG1EQUFtRDs0QkFDbkQsS0FBSzt5QkFDTjtxQkFDRjtpQkFDRixFQUFFLHFCQUFZLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQztZQUMvQixDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsUUFBUSxDQUFDLHVCQUF1QixFQUFFLEdBQUcsRUFBRTtZQUNyQyxJQUFJLENBQUMseUJBQXlCLEVBQUUsR0FBRyxFQUFFO2dCQUNuQyxPQUFPO2dCQUNQLGVBQU0sQ0FBQyxHQUFHLEVBQUU7b0JBQ1YsT0FBTztvQkFDUCxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsa0JBQWtCLEVBQUU7d0JBQ3RELGdCQUFnQixFQUFFOzRCQUNoQixHQUFHOzRCQUNILFFBQVEsRUFBRSxDQUFDLENBQUM7eUJBQ2I7cUJBQ0YsQ0FBQyxDQUFDO2dCQUNMLENBQUMsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsMkNBQTJDLEVBQUUsR0FBRyxFQUFFO2dCQUNyRCxPQUFPO2dCQUNQLGVBQU0sQ0FBQyxHQUFHLEVBQUU7b0JBQ1YsT0FBTztvQkFDUCxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsa0JBQWtCLEVBQUU7d0JBQ3RELGdCQUFnQixFQUFFOzRCQUNoQixHQUFHOzRCQUNILFFBQVEsRUFBRSxDQUFDOzRCQUNYLFFBQVEsRUFBRSxDQUFDO3lCQUNaO3FCQUNGLENBQUMsQ0FBQztnQkFDTCxDQUFDLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxRQUFRLENBQUMsaUNBQWlDLEVBQUUsR0FBRyxFQUFFO1lBQy9DLElBQUksQ0FBQyxxQkFBcUIsRUFBRSxHQUFHLEVBQUU7Z0JBQy9CLE9BQU87Z0JBQ1AsSUFBSSxLQUFLLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLGtCQUFrQixFQUFFO29CQUN0RCxPQUFPLEVBQUUsSUFBSTtvQkFDYixnQkFBZ0IsRUFBRTt3QkFDaEIsR0FBRztxQkFDSjtpQkFDRixDQUFDLENBQUM7Z0JBRUgsT0FBTztnQkFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGdDQUFnQyxFQUFFO29CQUNsRSxHQUFHLGtCQUFrQjtvQkFDckIsR0FBRyxrQ0FBa0MsQ0FBQzt3QkFDcEMsUUFBUSxFQUFFLENBQUM7cUJBQ1osQ0FBQztpQkFDSCxFQUFFLHFCQUFZLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQztZQUMvQixDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsUUFBUSxDQUFDLGlDQUFpQyxFQUFFLEdBQUcsRUFBRTtZQUMvQyxJQUFJLENBQUMsdUJBQXVCLEVBQUUsR0FBRyxFQUFFO2dCQUNqQyxPQUFPO2dCQUNQLElBQUksS0FBSyxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRSxrQkFBa0IsRUFBRTtvQkFDdEQsT0FBTyxFQUFFLElBQUk7b0JBQ2IsZ0JBQWdCLEVBQUU7d0JBQ2hCLEdBQUc7cUJBQ0o7aUJBQ0YsQ0FBQyxDQUFDO2dCQUVILE9BQU87Z0JBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxnQ0FBZ0MsRUFBRTtvQkFDbEUsR0FBRyxrQ0FBa0MsQ0FBQzt3QkFDcEMsUUFBUSxFQUFFLEdBQUc7cUJBQ2QsQ0FBQztpQkFDSCxFQUFFLHFCQUFZLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQztZQUMvQixDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsUUFBUSxDQUFDLGlDQUFpQyxFQUFFLEdBQUcsRUFBRTtZQUMvQyxJQUFJLENBQUMsK0JBQStCLEVBQUUsR0FBRyxFQUFFO2dCQUN6QyxPQUFPO2dCQUNQLElBQUksS0FBSyxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRSxrQkFBa0IsRUFBRTtvQkFDdEQsT0FBTyxFQUFFLElBQUk7b0JBQ2IsZ0JBQWdCLEVBQUU7d0JBQ2hCLEdBQUc7cUJBQ0o7aUJBQ0YsQ0FBQyxDQUFDO2dCQUVILE9BQU87Z0JBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGdDQUFnQyxDQUFDLENBQUMsQ0FBQztnQkFDakUsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQztZQUNuRCxDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsUUFBUSxDQUFDLCtCQUErQixFQUFFLEdBQUcsRUFBRTtZQUM3QyxJQUFJLENBQUMsb0NBQW9DLEVBQUUsR0FBRyxFQUFFO2dCQUM5QyxPQUFPO2dCQUNQLElBQUksS0FBSyxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRSxrQkFBa0IsRUFBRTtvQkFDdEQsT0FBTyxFQUFFLElBQUk7b0JBQ2IsZ0JBQWdCLEVBQUU7d0JBQ2hCLEdBQUc7cUJBQ0o7aUJBQ0YsQ0FBQyxDQUFDO2dCQUVILE9BQU87Z0JBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxnQ0FBZ0MsRUFBRTtvQkFDbEUsR0FBRyxrQ0FBa0MsQ0FBQzt3QkFDcEMsYUFBYSxFQUFFLENBQUUsU0FBUyxDQUFFO3FCQUM3QixDQUFDO2lCQUNILEVBQUUscUJBQVksQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDO1lBQy9CLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxRQUFRLENBQUMsd0JBQXdCLEVBQUUsR0FBRyxFQUFFO1lBQ3RDLElBQUksQ0FBQyx1QkFBdUIsRUFBRSxHQUFHLEVBQUU7Z0JBQ2pDLE9BQU87Z0JBQ1AsSUFBSSxLQUFLLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLGtCQUFrQixFQUFFO29CQUN0RCxPQUFPLEVBQUUsSUFBSTtvQkFDYixnQkFBZ0IsRUFBRTt3QkFDaEIsR0FBRztxQkFDSjtpQkFDRixDQUFDLENBQUM7Z0JBRUgsT0FBTztnQkFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGdDQUFnQyxFQUFFO29CQUNsRSxHQUFHLGtDQUFrQyxDQUFDO3dCQUNwQyxJQUFJLEVBQUUsS0FBSyxDQUFDLG1CQUFtQixDQUFDLFNBQVM7cUJBQzFDLENBQUM7aUJBQ0gsRUFBRSxxQkFBWSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUM7WUFDL0IsQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQge2V4cGVjdCwgaGF2ZVJlc291cmNlLCBoYXZlUmVzb3VyY2VMaWtlLCBSZXNvdXJjZVBhcnR9IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQgJ0Bhd3MtY2RrL2Fzc2VydC9qZXN0JztcbmltcG9ydCAqIGFzIGVjMiBmcm9tICdAYXdzLWNkay9hd3MtZWMyJztcbmltcG9ydCAqIGFzIGVjcyBmcm9tICdAYXdzLWNkay9hd3MtZWNzJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7dGhyb3dzfSBmcm9tICdhc3NlcnQnO1xuaW1wb3J0ICogYXMgYmF0Y2ggZnJvbSBcIi4uL2xpYlwiO1xuXG5kZXNjcmliZSgnQmF0Y2ggQ29tcHV0ZSBFdmlyb25tZW50JywgKCkgPT4ge1xuICBsZXQgZXhwZWN0ZWRNYW5hZ2VkRGVmYXVsdENvbXB1dGVQcm9wczogYW55O1xuICBsZXQgZGVmYXVsdFNlcnZpY2VSb2xlOiBhbnk7XG5cbiAgbGV0IHN0YWNrOiBjZGsuU3RhY2s7XG4gIGxldCB2cGM6IGVjMi5WcGM7XG5cbiAgYmVmb3JlRWFjaCgoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICB2cGMgPSBuZXcgZWMyLlZwYyhzdGFjaywgJ3Rlc3QtdnBjJyk7XG5cbiAgICBkZWZhdWx0U2VydmljZVJvbGUgPSB7XG4gICAgICBTZXJ2aWNlUm9sZToge1xuICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAndGVzdGNvbXB1dGVlbnZSZXNvdXJjZVNlcnZpY2VJbnN0YW5jZVJvbGUxMDUwNjlBNScsXG4gICAgICAgICAgJ0FybidcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgfTtcblxuICAgIGV4cGVjdGVkTWFuYWdlZERlZmF1bHRDb21wdXRlUHJvcHMgPSAob3ZlcnJpZGVzOiBhbnkpID0+IHtcbiAgICAgIHJldHVybiB7XG4gICAgICAgIENvbXB1dGVSZXNvdXJjZXM6IHtcbiAgICAgICAgICBBbGxvY2F0aW9uU3RyYXRlZ3k6IGJhdGNoLkFsbG9jYXRpb25TdHJhdGVneS5CRVNUX0ZJVCxcbiAgICAgICAgICBJbnN0YW5jZVJvbGU6IHtcbiAgICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgICAndGVzdGNvbXB1dGVlbnZJbnN0YW5jZVByb2ZpbGVDQkQ4N0VBQicsXG4gICAgICAgICAgICAgICdBcm4nXG4gICAgICAgICAgICBdXG4gICAgICAgICAgfSxcbiAgICAgICAgICBJbnN0YW5jZVR5cGVzOiBbXG4gICAgICAgICAgICAnb3B0aW1hbCdcbiAgICAgICAgICBdLFxuICAgICAgICAgIE1heHZDcHVzOiAyNTYsXG4gICAgICAgICAgTWludkNwdXM6IDAsXG4gICAgICAgICAgU3VibmV0czogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBSZWY6ICd0ZXN0dnBjUHJpdmF0ZVN1Ym5ldDFTdWJuZXQ4NjVGQjUwQSdcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFJlZjogJ3Rlc3R2cGNQcml2YXRlU3VibmV0MlN1Ym5ldDIzRDMzOTZGJ1xuICAgICAgICAgICAgfVxuICAgICAgICAgIF0sXG4gICAgICAgICAgVHlwZTogYmF0Y2guQ29tcHV0ZVJlc291cmNlVHlwZS5PTl9ERU1BTkQsXG4gICAgICAgICAgLi4ub3ZlcnJpZGVzLFxuICAgICAgICB9XG4gICAgICB9O1xuICAgIH07XG4gIH0pO1xuXG4gIGRlc2NyaWJlKCd3aGVuIHZhbGlkYXRpbmcgcHJvcHMnLCAoKSA9PiB7XG4gICAgdGVzdCgnc2hvdWxkIGRlbnkgc2V0dGluZyBjb21wdXRlIHJlc291cmNlcyB3aGVuIHVzaW5nIHR5cGUgdW5tYW5hZ2VkJywgKCkgPT4ge1xuICAgICAgLy8gVEhFTlxuICAgICAgdGhyb3dzKCgpID0+IHtcbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBuZXcgYmF0Y2guQ29tcHV0ZUVudmlyb25tZW50KHN0YWNrLCAndGVzdC1jb21wdXRlLWVudicsIHtcbiAgICAgICAgICBtYW5hZ2VkOiBmYWxzZSxcbiAgICAgICAgICBjb21wdXRlUmVzb3VyY2VzOiB7XG4gICAgICAgICAgICB2cGNcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgdGVzdCgnc2hvdWxkIGRlbnkgaWYgY3JlYXRpbmcgYSBtYW5hZ2VkIGVudmlyb25tZW50IHdpdGggbm8gcHJvdmlkZWQgY29tcHV0ZSByZXNvdXJjZSBwcm9wcycsICgpID0+IHtcbiAgICAgIC8vIFRIRU5cbiAgICAgIHRocm93cygoKSA9PiB7XG4gICAgICAgIC8vIFdIRU5cbiAgICAgICAgbmV3IGJhdGNoLkNvbXB1dGVFbnZpcm9ubWVudChzdGFjaywgJ3Rlc3QtY29tcHV0ZS1lbnYnLCB7XG4gICAgICAgICAgbWFuYWdlZDogdHJ1ZSxcbiAgICAgICAgfSk7XG4gICAgICB9KTtcbiAgICB9KTtcbiAgfSk7XG5cbiAgZGVzY3JpYmUoJ3VzaW5nIHNwb3QgcmVzb3VyY2VzJywgKCkgPT4ge1xuICAgIHRlc3QoJ3Nob3VsZCBwcm92aWRlIGEgc3BvdCBmbGVldCByb2xlIGlmIG9uZSBpcyBub3QgZ2l2ZW4gYW5kIGFsbG9jYXRpb25TdHJhdGVneSBpcyBCRVNUX0ZJVCcsICgpID0+IHtcbiAgICAgIC8vIFdIRU5cbiAgICAgIG5ldyBiYXRjaC5Db21wdXRlRW52aXJvbm1lbnQoc3RhY2ssICd0ZXN0LWNvbXB1dGUtZW52Jywge1xuICAgICAgICBtYW5hZ2VkOiB0cnVlLFxuICAgICAgICBjb21wdXRlUmVzb3VyY2VzOiB7XG4gICAgICAgICAgdHlwZTogYmF0Y2guQ29tcHV0ZVJlc291cmNlVHlwZS5TUE9ULFxuICAgICAgICAgIGFsbG9jYXRpb25TdHJhdGVneTogYmF0Y2guQWxsb2NhdGlvblN0cmF0ZWd5LkJFU1RfRklULFxuICAgICAgICAgIHZwY1xuICAgICAgICB9LFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpCYXRjaDo6Q29tcHV0ZUVudmlyb25tZW50Jywge1xuICAgICAgICBUeXBlOiAnTUFOQUdFRCcsXG4gICAgICAgIC4uLmV4cGVjdGVkTWFuYWdlZERlZmF1bHRDb21wdXRlUHJvcHMoe1xuICAgICAgICAgIFR5cGU6IGJhdGNoLkNvbXB1dGVSZXNvdXJjZVR5cGUuU1BPVCxcbiAgICAgICAgICBTcG90SWFtRmxlZXRSb2xlOiB7XG4gICAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAgICcnLFxuICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgJ2FybicsXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAnaWFtOjonLFxuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgIFJlZjogJ0FXUzo6QWNjb3VudElkJ1xuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgJzpyb2xlL2F3cy1zZXJ2aWNlLXJvbGUvc3BvdGZsZWV0LmFtYXpvbmF3cy5jb20vQVdTU2VydmljZVJvbGVGb3JFQzJTcG90RmxlZXQnLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9LFxuICAgICAgICB9KSxcbiAgICAgIH0sIFJlc291cmNlUGFydC5Qcm9wZXJ0aWVzKSk7XG4gICAgfSk7XG5cbiAgICBkZXNjcmliZSgnd2l0aCBhIGJpZCBwZXJjZW50YWdlJywgKCkgPT4ge1xuICAgICAgdGVzdCgnc2hvdWxkIGRlbnkgbXkgYmlkIGlmIHNldCBiZWxvdyAwJywgKCkgPT4ge1xuICAgICAgICAvLyBUSEVOXG4gICAgICAgIHRocm93cygoKSA9PiB7XG4gICAgICAgICAgLy8gV0hFTlxuICAgICAgICAgIG5ldyBiYXRjaC5Db21wdXRlRW52aXJvbm1lbnQoc3RhY2ssICd0ZXN0LWNvbXB1dGUtZW52Jywge1xuICAgICAgICAgICAgbWFuYWdlZDogdHJ1ZSxcbiAgICAgICAgICAgIGNvbXB1dGVSZXNvdXJjZXM6IHtcbiAgICAgICAgICAgICAgdnBjLFxuICAgICAgICAgICAgICB0eXBlOiBiYXRjaC5Db21wdXRlUmVzb3VyY2VUeXBlLlNQT1QsXG4gICAgICAgICAgICAgIGJpZFBlcmNlbnRhZ2U6IC0xLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9KTtcbiAgICAgICAgfSk7XG4gICAgICB9KTtcblxuICAgICAgdGVzdCgnc2hvdWxkIGRlbnkgbXkgYmlkIGlmIGFib3ZlIDEwMCcsICgpID0+IHtcbiAgICAgICAgLy8gVEhFTlxuICAgICAgICB0aHJvd3MoKCkgPT4ge1xuICAgICAgICAgIC8vIFdIRU5cbiAgICAgICAgICBuZXcgYmF0Y2guQ29tcHV0ZUVudmlyb25tZW50KHN0YWNrLCAndGVzdC1jb21wdXRlLWVudicsIHtcbiAgICAgICAgICAgIG1hbmFnZWQ6IHRydWUsXG4gICAgICAgICAgICBjb21wdXRlUmVzb3VyY2VzOiB7XG4gICAgICAgICAgICAgIHZwYyxcbiAgICAgICAgICAgICAgdHlwZTogYmF0Y2guQ29tcHV0ZVJlc291cmNlVHlwZS5TUE9ULFxuICAgICAgICAgICAgICBiaWRQZXJjZW50YWdlOiAxMDEsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0pO1xuICAgICAgICB9KTtcbiAgICAgIH0pO1xuICAgIH0pO1xuICB9KTtcblxuICBkZXNjcmliZSgnd2l0aCBwcm9wZXJ0aWVzIHNwZWNpZmllZCcsICgpID0+IHtcbiAgICB0ZXN0KCdyZW5kZXJzIHRoZSBjb3JyZWN0IGNsb3VkZm9ybWF0aW9uIHByb3BlcnRpZXMnLCAoKSA9PiB7XG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBwcm9wcyA9IHtcbiAgICAgICAgY29tcHV0ZUVudmlyb25tZW50TmFtZTogJ215LXRlc3QtY29tcHV0ZS1lbnYnLFxuICAgICAgICBjb21wdXRlUmVzb3VyY2VzOiB7XG4gICAgICAgICAgYWxsb2NhdGlvblN0cmF0ZWd5OiBiYXRjaC5BbGxvY2F0aW9uU3RyYXRlZ3kuQkVTVF9GSVQsXG4gICAgICAgICAgdnBjLFxuICAgICAgICAgIGNvbXB1dGVSZXNvdXJjZXNUYWdzOiBuZXcgY2RrLlRhZygnZm9vJywgJ2JhcicpLFxuICAgICAgICAgIGRlc2lyZWR2Q3B1czogMSxcbiAgICAgICAgICBlYzJLZXlQYWlyOiAnbXkta2V5LXBhaXInLFxuICAgICAgICAgIGltYWdlOiBuZXcgZWNzLkVjc09wdGltaXplZEFtaSh7XG4gICAgICAgICAgICBnZW5lcmF0aW9uOiBlYzIuQW1hem9uTGludXhHZW5lcmF0aW9uLkFNQVpPTl9MSU5VWF8yLFxuICAgICAgICAgICAgaGFyZHdhcmVUeXBlOiBlY3MuQW1pSGFyZHdhcmVUeXBlLlNUQU5EQVJELFxuICAgICAgICAgIH0pLFxuICAgICAgICAgIGluc3RhbmNlUm9sZTogIG5ldyBpYW0uQ2ZuSW5zdGFuY2VQcm9maWxlKHN0YWNrLCAnSW5zdGFuY2UtUHJvZmlsZScsIHtcbiAgICAgICAgICAgIHJvbGVzOiBbIG5ldyBpYW0uUm9sZShzdGFjaywgJ0Vjcy1JbnN0YW5jZS1Sb2xlJywge1xuICAgICAgICAgICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZWMyLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgICAgICAgbWFuYWdlZFBvbGljaWVzOiBbXG4gICAgICAgICAgICAgICAgaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdzZXJ2aWNlLXJvbGUvQW1hem9uRUMyQ29udGFpbmVyU2VydmljZWZvckVDMlJvbGUnKVxuICAgICAgICAgICAgICBdXG4gICAgICAgICAgICB9KS5yb2xlTmFtZV1cbiAgICAgICAgICB9KS5hdHRyQXJuLFxuICAgICAgICAgIGluc3RhbmNlVHlwZXM6IFtcbiAgICAgICAgICAgIGVjMi5JbnN0YW5jZVR5cGUub2YoZWMyLkluc3RhbmNlQ2xhc3MuVDIsIGVjMi5JbnN0YW5jZVNpemUuTUlDUk8pLFxuICAgICAgICAgIF0sXG4gICAgICAgICAgbWF4dkNwdXM6IDQsXG4gICAgICAgICAgbWludkNwdXM6IDEsXG4gICAgICAgICAgc2VjdXJpdHlHcm91cHM6IFtcbiAgICAgICAgICAgIG5ldyBlYzIuU2VjdXJpdHlHcm91cChzdGFjaywgJ3Rlc3Qtc2cnLCB7XG4gICAgICAgICAgICAgIHZwYyxcbiAgICAgICAgICAgICAgYWxsb3dBbGxPdXRib3VuZDogdHJ1ZSxcbiAgICAgICAgICAgIH0pLFxuICAgICAgICAgIF0sXG4gICAgICAgICAgdHlwZTogYmF0Y2guQ29tcHV0ZVJlc291cmNlVHlwZS5PTl9ERU1BTkQsXG4gICAgICAgICAgdnBjU3VibmV0czoge1xuICAgICAgICAgICAgc3VibmV0VHlwZTogZWMyLlN1Ym5ldFR5cGUuUFJJVkFURSxcbiAgICAgICAgICB9LFxuICAgICAgICB9IGFzIGJhdGNoLkNvbXB1dGVSZXNvdXJjZXMsXG4gICAgICAgIGVuYWJsZWQ6IGZhbHNlLFxuICAgICAgICBtYW5hZ2VkOiB0cnVlLFxuICAgICAgfTtcblxuICAgICAgbmV3IGJhdGNoLkNvbXB1dGVFbnZpcm9ubWVudChzdGFjaywgJ3Rlc3QtY29tcHV0ZS1lbnYnLCBwcm9wcyk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpCYXRjaDo6Q29tcHV0ZUVudmlyb25tZW50Jywge1xuICAgICAgICBDb21wdXRlRW52aXJvbm1lbnROYW1lOiAnbXktdGVzdC1jb21wdXRlLWVudicsXG4gICAgICAgIFR5cGU6ICdNQU5BR0VEJyxcbiAgICAgICAgU3RhdGU6ICdESVNBQkxFRCcsXG4gICAgICAgIFNlcnZpY2VSb2xlOiB7XG4gICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAndGVzdGNvbXB1dGVlbnZSZXNvdXJjZVNlcnZpY2VJbnN0YW5jZVJvbGUxMDUwNjlBNScsXG4gICAgICAgICAgICAnQXJuJ1xuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICAgIENvbXB1dGVSZXNvdXJjZXM6IHtcbiAgICAgICAgICBBbGxvY2F0aW9uU3RyYXRlZ3k6IGJhdGNoLkFsbG9jYXRpb25TdHJhdGVneS5CRVNUX0ZJVCxcbiAgICAgICAgICBEZXNpcmVkdkNwdXM6IHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuZGVzaXJlZHZDcHVzLFxuICAgICAgICAgIEVjMktleVBhaXI6IHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuZWMyS2V5UGFpcixcbiAgICAgICAgICBJbWFnZUlkOiB7XG4gICAgICAgICAgICBSZWY6ICdTc21QYXJhbWV0ZXJWYWx1ZWF3c3NlcnZpY2VlY3NvcHRpbWl6ZWRhbWlhbWF6b25saW51eDJyZWNvbW1lbmRlZGltYWdlaWRDOTY1ODRCNkYwMEE0NjRFQUQxOTUzQUZGNEIwNTExOFBhcmFtZXRlcidcbiAgICAgICAgICB9LFxuICAgICAgICAgIEluc3RhbmNlUm9sZToge1xuICAgICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAgIHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuaW5zdGFuY2VSb2xlID8gXCJJbnN0YW5jZVByb2ZpbGVcIiA6ICcnLFxuICAgICAgICAgICAgICAnQXJuJ1xuICAgICAgICAgICAgXVxuICAgICAgICAgIH0sXG4gICAgICAgICAgSW5zdGFuY2VUeXBlczogW1xuICAgICAgICAgICAgcHJvcHMuY29tcHV0ZVJlc291cmNlcy5pbnN0YW5jZVR5cGVzID8gcHJvcHMuY29tcHV0ZVJlc291cmNlcy5pbnN0YW5jZVR5cGVzWzBdLnRvU3RyaW5nKCkgOiAnJyxcbiAgICAgICAgICBdLFxuICAgICAgICAgIE1heHZDcHVzOiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLm1heHZDcHVzLFxuICAgICAgICAgIE1pbnZDcHVzOiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLm1pbnZDcHVzLFxuICAgICAgICAgIFNlY3VyaXR5R3JvdXBJZHM6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAgICAgJ3Rlc3RzZzg3MkVCNDhBJyxcbiAgICAgICAgICAgICAgICAnR3JvdXBJZCdcbiAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgfVxuICAgICAgICAgIF0sXG4gICAgICAgICAgU3VibmV0czogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBSZWY6IGAke3ZwYy5ub2RlLnVuaXF1ZUlkfVByaXZhdGVTdWJuZXQxU3VibmV0ODY1RkI1MEFgXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBSZWY6IGAke3ZwYy5ub2RlLnVuaXF1ZUlkfVByaXZhdGVTdWJuZXQyU3VibmV0MjNEMzM5NkZgXG4gICAgICAgICAgICB9XG4gICAgICAgICAgXSxcbiAgICAgICAgICBUYWdzOiB7XG4gICAgICAgICAgICBrZXk6ICdmb28nLFxuICAgICAgICAgICAgcHJvcHM6IHt9LFxuICAgICAgICAgICAgZGVmYXVsdFByaW9yaXR5OiAxMDAsXG4gICAgICAgICAgICB2YWx1ZTogJ2JhcidcbiAgICAgICAgICB9LFxuICAgICAgICAgIFR5cGU6ICdFQzInXG4gICAgICAgIH0sXG4gICAgICB9LCBSZXNvdXJjZVBhcnQuUHJvcGVydGllcykpO1xuICAgIH0pO1xuXG4gICAgZGVzY3JpYmUoJ3dpdGggbm8gYWxsb2NhdGlvbiBzdHJhdGVneSBzcGVjaWZpZWQnLCAoKSA9PiB7XG4gICAgICB0ZXN0KCdzaG91bGQgZGVmYXVsdCB0byBhIGJlc3RfZml0IHN0cmF0ZWd5JywgKCkgPT4ge1xuICAgICAgICAvLyBXSEVOXG4gICAgICAgIG5ldyBiYXRjaC5Db21wdXRlRW52aXJvbm1lbnQoc3RhY2ssICd0ZXN0LWNvbXB1dGUtZW52Jywge1xuICAgICAgICAgIG1hbmFnZWQ6IHRydWUsXG4gICAgICAgICAgY29tcHV0ZVJlc291cmNlczoge1xuICAgICAgICAgICAgdnBjLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuXG4gICAgICAgIC8vIFRIRU5cbiAgICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkJhdGNoOjpDb21wdXRlRW52aXJvbm1lbnQnLCB7XG4gICAgICAgICAgVHlwZTogJ01BTkFHRUQnLFxuICAgICAgICAgIFNlcnZpY2VSb2xlOiB7XG4gICAgICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAgICAgJ3Rlc3Rjb21wdXRlZW52UmVzb3VyY2VTZXJ2aWNlSW5zdGFuY2VSb2xlMTA1MDY5QTUnLFxuICAgICAgICAgICAgICAnQXJuJ1xuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LCBSZXNvdXJjZVBhcnQuUHJvcGVydGllcykpO1xuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICBkZXNjcmliZSgnd2l0aCBhIG1pbiB2Y3B1IHZhbHVlJywgKCkgPT4ge1xuICAgICAgdGVzdCgnc2hvdWxkIGRlbnkgbGVzcyB0aGFuIDAnLCAoKSA9PiB7XG4gICAgICAgIC8vIFRIRU5cbiAgICAgICAgdGhyb3dzKCgpID0+IHtcbiAgICAgICAgICAvLyBXSEVOXG4gICAgICAgICAgbmV3IGJhdGNoLkNvbXB1dGVFbnZpcm9ubWVudChzdGFjaywgJ3Rlc3QtY29tcHV0ZS1lbnYnLCB7XG4gICAgICAgICAgICBjb21wdXRlUmVzb3VyY2VzOiB7XG4gICAgICAgICAgICAgIHZwYyxcbiAgICAgICAgICAgICAgbWludkNwdXM6IC0xLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9KTtcbiAgICAgICAgfSk7XG4gICAgICB9KTtcblxuICAgICAgdGVzdCgnY2Fubm90IGJlIGdyZWF0ZXIgdGhhbiB0aGUgbWF4IHZjcHUgdmFsdWUnLCAoKSA9PiB7XG4gICAgICAgIC8vIFRIRU5cbiAgICAgICAgdGhyb3dzKCgpID0+IHtcbiAgICAgICAgICAvLyBXSEVOXG4gICAgICAgICAgbmV3IGJhdGNoLkNvbXB1dGVFbnZpcm9ubWVudChzdGFjaywgJ3Rlc3QtY29tcHV0ZS1lbnYnLCB7XG4gICAgICAgICAgICBjb21wdXRlUmVzb3VyY2VzOiB7XG4gICAgICAgICAgICAgIHZwYyxcbiAgICAgICAgICAgICAgbWludkNwdXM6IDIsXG4gICAgICAgICAgICAgIG1heHZDcHVzOiAxLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9KTtcbiAgICAgICAgfSk7XG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIGRlc2NyaWJlKCd3aXRoIG5vIG1pbiB2Y3B1IHZhbHVlIHByb3ZpZGVkJywgKCkgPT4ge1xuICAgICAgdGVzdCgnc2hvdWxkIGRlZmF1bHQgdG8gMCcsICgpID0+IHtcbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBuZXcgYmF0Y2guQ29tcHV0ZUVudmlyb25tZW50KHN0YWNrLCAndGVzdC1jb21wdXRlLWVudicsIHtcbiAgICAgICAgICBtYW5hZ2VkOiB0cnVlLFxuICAgICAgICAgIGNvbXB1dGVSZXNvdXJjZXM6IHtcbiAgICAgICAgICAgIHZwYyxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcblxuICAgICAgICAvLyBUSEVOXG4gICAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpCYXRjaDo6Q29tcHV0ZUVudmlyb25tZW50Jywge1xuICAgICAgICAgIC4uLmRlZmF1bHRTZXJ2aWNlUm9sZSxcbiAgICAgICAgICAuLi5leHBlY3RlZE1hbmFnZWREZWZhdWx0Q29tcHV0ZVByb3BzKHtcbiAgICAgICAgICAgIE1pbnZDcHVzOiAwLFxuICAgICAgICAgIH0pLFxuICAgICAgICB9LCBSZXNvdXJjZVBhcnQuUHJvcGVydGllcykpO1xuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICBkZXNjcmliZSgnd2l0aCBubyBtYXggdmNwdSB2YWx1ZSBwcm92aWRlZCcsICgpID0+IHtcbiAgICAgIHRlc3QoJ3Nob3VsZCBkZWZhdWx0IHRvIDI1NicsICgpID0+IHtcbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBuZXcgYmF0Y2guQ29tcHV0ZUVudmlyb25tZW50KHN0YWNrLCAndGVzdC1jb21wdXRlLWVudicsIHtcbiAgICAgICAgICBtYW5hZ2VkOiB0cnVlLFxuICAgICAgICAgIGNvbXB1dGVSZXNvdXJjZXM6IHtcbiAgICAgICAgICAgIHZwYyxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcblxuICAgICAgICAvLyBUSEVOXG4gICAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpCYXRjaDo6Q29tcHV0ZUVudmlyb25tZW50Jywge1xuICAgICAgICAgIC4uLmV4cGVjdGVkTWFuYWdlZERlZmF1bHRDb21wdXRlUHJvcHMoe1xuICAgICAgICAgICAgTWF4dkNwdXM6IDI1NixcbiAgICAgICAgICB9KSxcbiAgICAgICAgfSwgUmVzb3VyY2VQYXJ0LlByb3BlcnRpZXMpKTtcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgZGVzY3JpYmUoJ3dpdGggbm8gaW5zdGFuY2Ugcm9sZSBzcGVjaWZpZWQnLCAoKSA9PiB7XG4gICAgICB0ZXN0KCdzaG91bGQgZ2VuZXJhdGUgYSByb2xlIGZvciBtZScsICgpID0+IHtcbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBuZXcgYmF0Y2guQ29tcHV0ZUVudmlyb25tZW50KHN0YWNrLCAndGVzdC1jb21wdXRlLWVudicsIHtcbiAgICAgICAgICBtYW5hZ2VkOiB0cnVlLFxuICAgICAgICAgIGNvbXB1dGVSZXNvdXJjZXM6IHtcbiAgICAgICAgICAgIHZwYyxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcblxuICAgICAgICAvLyBUSEVOXG4gICAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkJhdGNoOjpDb21wdXRlRW52aXJvbm1lbnQnKSk7XG4gICAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OklBTTo6Um9sZScpKTtcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgZGVzY3JpYmUoJ3dpdGggbm8gaW5zdGFuY2UgdHlwZSBkZWZpbmVkJywgKCkgPT4ge1xuICAgICAgdGVzdCgnc2hvdWxkIGRlZmF1bHQgdG8gb3B0aW1hbCBtYXRjaGluZycsICgpID0+IHtcbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBuZXcgYmF0Y2guQ29tcHV0ZUVudmlyb25tZW50KHN0YWNrLCAndGVzdC1jb21wdXRlLWVudicsIHtcbiAgICAgICAgICBtYW5hZ2VkOiB0cnVlLFxuICAgICAgICAgIGNvbXB1dGVSZXNvdXJjZXM6IHtcbiAgICAgICAgICAgIHZwYyxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcblxuICAgICAgICAvLyBUSEVOXG4gICAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpCYXRjaDo6Q29tcHV0ZUVudmlyb25tZW50Jywge1xuICAgICAgICAgIC4uLmV4cGVjdGVkTWFuYWdlZERlZmF1bHRDb21wdXRlUHJvcHMoe1xuICAgICAgICAgICAgSW5zdGFuY2VUeXBlczogWyAnb3B0aW1hbCcgXSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgfSwgUmVzb3VyY2VQYXJ0LlByb3BlcnRpZXMpKTtcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgZGVzY3JpYmUoJ3dpdGggbm8gdHlwZSBzcGVjaWZpZWQnLCAoKSA9PiB7XG4gICAgICB0ZXN0KCdzaG91bGQgZGVmYXVsdCB0byBFQzInLCAoKSA9PiB7XG4gICAgICAgIC8vIFdIRU5cbiAgICAgICAgbmV3IGJhdGNoLkNvbXB1dGVFbnZpcm9ubWVudChzdGFjaywgJ3Rlc3QtY29tcHV0ZS1lbnYnLCB7XG4gICAgICAgICAgbWFuYWdlZDogdHJ1ZSxcbiAgICAgICAgICBjb21wdXRlUmVzb3VyY2VzOiB7XG4gICAgICAgICAgICB2cGMsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgLy8gVEhFTlxuICAgICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6QmF0Y2g6OkNvbXB1dGVFbnZpcm9ubWVudCcsIHtcbiAgICAgICAgICAuLi5leHBlY3RlZE1hbmFnZWREZWZhdWx0Q29tcHV0ZVByb3BzKHtcbiAgICAgICAgICAgIFR5cGU6IGJhdGNoLkNvbXB1dGVSZXNvdXJjZVR5cGUuT05fREVNQU5ELFxuICAgICAgICAgIH0pLFxuICAgICAgICB9LCBSZXNvdXJjZVBhcnQuUHJvcGVydGllcykpO1xuICAgICAgfSk7XG4gICAgfSk7XG4gIH0pO1xufSk7XG4iXX0=