import { Duration, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
/**
 * Represents a Cache Policy.
 *
 * @stability stable
 */
export interface ICachePolicy {
    /**
     * The ID of the cache policy.
     *
     * @stability stable
     * @attribute true
     */
    readonly cachePolicyId: string;
}
/**
 * Properties for creating a Cache Policy.
 *
 * @stability stable
 */
export interface CachePolicyProps {
    /**
     * A unique name to identify the cache policy.
     *
     * The name must only include '-', '_', or alphanumeric characters.
     *
     * @default - generated from the `id`
     * @stability stable
     */
    readonly cachePolicyName?: string;
    /**
     * A comment to describe the cache policy.
     *
     * @default - no comment
     * @stability stable
     */
    readonly comment?: string;
    /**
     * The default amount of time for objects to stay in the CloudFront cache.
     *
     * Only used when the origin does not send Cache-Control or Expires headers with the object.
     *
     * @default - The greater of 1 day and ``minTtl``
     * @stability stable
     */
    readonly defaultTtl?: Duration;
    /**
     * The minimum amount of time for objects to stay in the CloudFront cache.
     *
     * @default Duration.seconds(0)
     * @stability stable
     */
    readonly minTtl?: Duration;
    /**
     * The maximum amount of time for objects to stay in the CloudFront cache.
     *
     * CloudFront uses this value only when the origin sends Cache-Control or Expires headers with the object.
     *
     * @default - The greater of 1 year and ``defaultTtl``
     * @stability stable
     */
    readonly maxTtl?: Duration;
    /**
     * Determines whether any cookies in viewer requests are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     *
     * @default CacheCookieBehavior.none()
     * @stability stable
     */
    readonly cookieBehavior?: CacheCookieBehavior;
    /**
     * Determines whether any HTTP headers are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     *
     * @default CacheHeaderBehavior.none()
     * @stability stable
     */
    readonly headerBehavior?: CacheHeaderBehavior;
    /**
     * Determines whether any query strings are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     *
     * @default CacheQueryStringBehavior.none()
     * @stability stable
     */
    readonly queryStringBehavior?: CacheQueryStringBehavior;
    /**
     * Whether to normalize and include the `Accept-Encoding` header in the cache key when the `Accept-Encoding` header is 'gzip'.
     *
     * @default false
     * @stability stable
     */
    readonly enableAcceptEncodingGzip?: boolean;
    /**
     * Whether to normalize and include the `Accept-Encoding` header in the cache key when the `Accept-Encoding` header is 'br'.
     *
     * @default false
     * @stability stable
     */
    readonly enableAcceptEncodingBrotli?: boolean;
}
/**
 * A Cache Policy configuration.
 *
 * @stability stable
 * @resource AWS::CloudFront::CachePolicy
 */
export declare class CachePolicy extends Resource implements ICachePolicy {
    /**
     * Optimize cache efficiency by minimizing the values that CloudFront includes in the cache key.
     *
     * Query strings and cookies are not included in the cache key, and only the normalized 'Accept-Encoding' header is included.
     *
     * @stability stable
     */
    static readonly CACHING_OPTIMIZED: ICachePolicy;
    /**
     * Optimize cache efficiency by minimizing the values that CloudFront includes in the cache key.
     *
     * Query strings and cookies are not included in the cache key, and only the normalized 'Accept-Encoding' header is included.
     * Disables cache compression.
     *
     * @stability stable
     */
    static readonly CACHING_OPTIMIZED_FOR_UNCOMPRESSED_OBJECTS: ICachePolicy;
    /**
     * Disables caching.
     *
     * This policy is useful for dynamic content and for requests that are not cacheable.
     *
     * @stability stable
     */
    static readonly CACHING_DISABLED: ICachePolicy;
    /**
     * Designed for use with an origin that is an AWS Elemental MediaPackage endpoint.
     *
     * @stability stable
     */
    static readonly ELEMENTAL_MEDIA_PACKAGE: ICachePolicy;
    /**
     * Imports a Cache Policy from its id.
     *
     * @stability stable
     */
    static fromCachePolicyId(scope: Construct, id: string, cachePolicyId: string): ICachePolicy;
    /** Use an existing managed cache policy. */
    private static fromManagedCachePolicy;
    /**
     * The ID of the cache policy.
     *
     * @stability stable
     */
    readonly cachePolicyId: string;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props?: CachePolicyProps);
    private renderCacheKey;
}
/**
 * Determines whether any cookies in viewer requests are included in the cache key and automatically included in requests that CloudFront sends to the origin.
 *
 * @stability stable
 */
export declare class CacheCookieBehavior {
    /**
     * Cookies in viewer requests are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static none(): CacheCookieBehavior;
    /**
     * All cookies in viewer requests are included in the cache key and are automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static all(): CacheCookieBehavior;
    /**
     * Only the provided `cookies` are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static allowList(...cookies: string[]): CacheCookieBehavior;
    /**
     * All cookies except the provided `cookies` are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static denyList(...cookies: string[]): CacheCookieBehavior;
    /**
     * The behavior of cookies: allow all, none, an allow list, or a deny list.
     *
     * @stability stable
     */
    readonly behavior: string;
    /**
     * The cookies to allow or deny, if the behavior is an allow or deny list.
     *
     * @stability stable
     */
    readonly cookies?: string[];
    private constructor();
}
/**
 * Determines whether any HTTP headers are included in the cache key and automatically included in requests that CloudFront sends to the origin.
 *
 * @stability stable
 */
export declare class CacheHeaderBehavior {
    /**
     * HTTP headers are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static none(): CacheHeaderBehavior;
    /**
     * Listed headers are included in the cache key and are automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static allowList(...headers: string[]): CacheHeaderBehavior;
    /**
     * If the no headers will be passed, or an allow list of headers.
     *
     * @stability stable
     */
    readonly behavior: string;
    /**
     * The headers for the allow/deny list, if applicable.
     *
     * @stability stable
     */
    readonly headers?: string[];
    private constructor();
}
/**
 * Determines whether any URL query strings in viewer requests are included in the cache key and automatically included in requests that CloudFront sends to the origin.
 *
 * @stability stable
 */
export declare class CacheQueryStringBehavior {
    /**
     * Query strings in viewer requests are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static none(): CacheQueryStringBehavior;
    /**
     * All query strings in viewer requests are included in the cache key and are automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static all(): CacheQueryStringBehavior;
    /**
     * Only the provided `queryStrings` are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static allowList(...queryStrings: string[]): CacheQueryStringBehavior;
    /**
     * All query strings except the provided `queryStrings` are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static denyList(...queryStrings: string[]): CacheQueryStringBehavior;
    /**
     * The behavior of query strings -- allow all, none, only an allow list, or a deny list.
     *
     * @stability stable
     */
    readonly behavior: string;
    /**
     * The query strings to allow or deny, if the behavior is an allow or deny list.
     *
     * @stability stable
     */
    readonly queryStrings?: string[];
    private constructor();
}
