"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcrSourceAction = void 0;
const codepipeline = require("@aws-cdk/aws-codepipeline");
const targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * The ECR Repository source CodePipeline Action.
 *
 * Will trigger the pipeline as soon as the target tag in the repository
 * changes, but only if there is a CloudTrail Trail in the account that
 * captures the ECR event.
 */
class EcrSourceAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.repository,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'ECR',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        this.props = props;
    }
    /** The variables emitted by this action. */
    get variables() {
        return {
            registryId: this.variableExpression('RegistryId'),
            repositoryName: this.variableExpression('RepositoryName'),
            imageDigest: this.variableExpression('ImageDigest'),
            imageTag: this.variableExpression('ImageTag'),
            imageUri: this.variableExpression('ImageURI'),
        };
    }
    bound(_scope, stage, options) {
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['ecr:DescribeImages'],
            resources: [this.props.repository.repositoryArn],
        }));
        this.props.repository.onCloudTrailImagePushed(stage.pipeline.node.uniqueId + 'SourceEventRule', {
            target: new targets.CodePipeline(stage.pipeline),
            imageTag: this.props.imageTag,
        });
        // the Action Role also needs to write to the Pipeline's bucket
        options.bucket.grantWrite(options.role);
        return {
            configuration: {
                RepositoryName: this.props.repository.repositoryName,
                ImageTag: this.props.imageTag,
            },
        };
    }
}
exports.EcrSourceAction = EcrSourceAction;
//# sourceMappingURL=data:application/json;base64,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