# Copyright 2021 Curtin University
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# Author: Aniek Roelofs, James Diprose

"""
    Observatory API

    The REST API for managing and accessing data from the Observatory Platform.   # noqa: E501

    The version of the OpenAPI document: 1.0.0
    Contact: agent@observatory.academy
    Generated by: https://openapi-generator.tech
"""

import unittest

import observatory.api.server.orm as orm
import pendulum
from observatory.api.client import ApiClient, Configuration
from observatory.api.client.api.observatory_api import ObservatoryApi  # noqa: E501
from observatory.api.client.exceptions import (
    ApiException,
    NotFoundException,
)
from observatory.api.client.model.dataset import Dataset
from observatory.api.client.model.dataset_release import DatasetRelease
from observatory.api.client.model.dataset_storage import DatasetStorage
from observatory.api.client.model.organisation import Organisation
from observatory.api.client.model.telescope import Telescope
from observatory.api.client.model.telescope_type import TelescopeType
from observatory.api.tests.testing import ObservatoryApiEnvironment


def make_telescope_types(env: ObservatoryApiEnvironment, dt):
    # Add TelescopeType objects
    telescope_types = [
        ("onix", "ONIX Telescope"),
        ("jstor", "JSTOR Telescope"),
        ("google_analytics", "Google Analytics Telescope"),
    ]

    for type_id, name in telescope_types:
        env.session.add(orm.TelescopeType(type_id=type_id, name=name, created=dt, modified=dt))
    env.session.commit()

    return telescope_types


class TestObservatoryApi(unittest.TestCase):
    """ObservatoryApi unit test stubs"""

    def setUp(self):
        self.timezone = "Pacific/Auckland"
        self.host = "localhost"
        self.port = 5001
        configuration = Configuration(host=f"http://{self.host}:{self.port}")
        api_client = ApiClient(configuration)
        self.api = ObservatoryApi(api_client=api_client)  # noqa: E501
        self.env = ObservatoryApiEnvironment(host=self.host, port=self.port)

    def test_ctor(self):
        api = ObservatoryApi()
        self.assertTrue(api.api_client is not None)

    def test_delete_organisation(self):
        """Test case for delete_organisation

        delete an Organisation  # noqa: E501
        """

        with self.env.create():
            # Post telescope
            expected_id = 1

            dt = pendulum.now(self.timezone)
            self.env.session.add(orm.Organisation(name="Curtin University", created=dt, modified=dt))
            self.env.session.commit()

            result = self.api.delete_organisation(expected_id)

            with self.assertRaises(NotFoundException) as e:
                self.api.delete_organisation(expected_id)
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: Organisation with id {expected_id}"\n', e.exception.body)

    def test_delete_telescope(self):
        """Test case for delete_telescope

        delete a Telescope  # noqa: E501
        """

        with self.env.create():
            # Post telescope
            expected_id = 1

            telescope_type_name = "ONIX Telescope"
            org_name = "Curtin University"
            dt = pendulum.now(self.timezone)
            self.env.session.add(orm.TelescopeType(type_id="onix", name=telescope_type_name, created=dt, modified=dt))
            self.env.session.add(orm.Organisation(name=org_name, created=dt, modified=dt))
            self.env.session.commit()
            self.env.session.add(
                orm.Telescope(
                    organisation={"id": expected_id}, telescope_type={"id": expected_id}, created=dt, modified=dt
                )
            )
            self.env.session.commit()

            result = self.api.delete_telescope(expected_id)

            with self.assertRaises(NotFoundException) as e:
                self.api.delete_telescope(expected_id)
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: Telescope with id {expected_id}"\n', e.exception.body)

    def test_delete_telescope_type(self):
        """Test case for delete_telescope_type

        delete a TelescopeType  # noqa: E501
        """

        with self.env.create():
            # Post telescope
            expected_id = 1

            dt = pendulum.now(self.timezone)
            self.env.session.add(orm.TelescopeType(type_id="onix", name="ONIX Telescope", created=dt, modified=dt))
            self.env.session.commit()

            result = self.api.delete_telescope_type(expected_id)

            with self.assertRaises(NotFoundException) as e:
                self.api.delete_telescope_type(expected_id)
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: TelescopeType with id {expected_id}"\n', e.exception.body)

    def test_get_organisation(self):
        """Test case for get_organisation

        get an Organisation  # noqa: E501
        """

        with self.env.create():
            expected_id = 1

            # Assert that TelescopeType with given id does not exist
            with self.assertRaises(NotFoundException) as e:
                self.api.get_telescope_type(id=expected_id)

            # Add TelescopeType
            name = "Curtin University"
            gcp_project_id = "my-project-id"
            gcp_download_bucket = "my-download-bucket"
            gcp_transform_bucket = "my-transform-bucket"
            dt = pendulum.now(self.timezone)
            dt_utc = dt.in_tz(tz="UTC")
            self.env.session.add(
                orm.Organisation(
                    name=name,
                    gcp_project_id=gcp_project_id,
                    gcp_download_bucket=gcp_download_bucket,
                    gcp_transform_bucket=gcp_transform_bucket,
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Assert that TelescopeType with given id exists
            obj = self.api.get_organisation(expected_id)
            self.assertIsInstance(obj, Organisation)
            self.assertEqual(expected_id, obj.id)
            self.assertEqual(name, obj.name)
            self.assertEqual(gcp_project_id, obj.gcp_project_id)
            self.assertEqual(gcp_download_bucket, obj.gcp_download_bucket)
            self.assertEqual(gcp_transform_bucket, obj.gcp_transform_bucket)
            self.assertEqual(dt_utc, obj.created)
            self.assertEqual(dt_utc, obj.modified)

    def test_get_organisations(self):
        """Test case for get_organisations

        Get a list of Organisations  # noqa: E501
        """

        with self.env.create():
            # Add Organisation objects
            names = ["Curtin University", "Massachusetts Institute of Technology", "Harvard University"]
            dt = pendulum.now(self.timezone)
            dt_utc = dt.in_tz(tz="UTC")
            for name in names:
                self.env.session.add(orm.Organisation(name=name, created=dt, modified=dt))
            self.env.session.commit()

            # Assert that Organisation objects returned
            objects = self.api.get_organisations(limit=10)
            self.assertEqual(len(names), len(objects))
            for i, (obj, name) in enumerate(zip(objects, names)):
                expected_id = i + 1
                self.assertIsInstance(obj, Organisation)
                self.assertEqual(expected_id, obj.id)
                self.assertEqual(name, obj.name)
                self.assertEqual(dt_utc, obj.created)
                self.assertEqual(dt_utc, obj.modified)

    def test_get_telescope(self):
        """Test case for get_telescope

        get a Telescope  # noqa: E501
        """

        with self.env.create():
            expected_id = 1

            # Assert that Telescope with given id does not exist
            with self.assertRaises(NotFoundException) as e:
                self.api.get_telescope(expected_id)
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: Telescope with id {expected_id}"\n', e.exception.body)

            # Add Telescope
            telescope_type_name = "ONIX Telescope"
            org_name = "Curtin University"
            dt = pendulum.now(self.timezone)
            dt_utc = dt.in_tz(tz="UTC")
            self.env.session.add(orm.TelescopeType(type_id="onix", name=telescope_type_name, created=dt, modified=dt))
            self.env.session.add(orm.Organisation(name=org_name, created=dt, modified=dt))
            self.env.session.commit()
            self.env.session.add(
                orm.Telescope(
                    name="Curtin ONIX Telescope",
                    extra={"view_id": 123456},
                    organisation={"id": expected_id},
                    telescope_type={"id": expected_id},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Assert that Telescope with given id exists
            obj = self.api.get_telescope(expected_id)
            self.assertIsInstance(obj, Telescope)
            self.assertEqual(expected_id, obj.id)
            self.assertEqual(expected_id, obj.organisation.id)
            self.assertEqual(org_name, obj.organisation.name)
            self.assertEqual(expected_id, obj.telescope_type.id)
            self.assertEqual(telescope_type_name, obj.telescope_type.name)
            self.assertEqual(dt_utc, obj.created)
            self.assertEqual(dt_utc, obj.modified)

    def test_get_telescope_type(self):
        """Test case for get_telescope_type

        get a TelescopeType  # noqa: E501
        """

        with self.env.create():
            expected_id = 1
            type_id = "onix"

            # Assert that TelescopeType with given id does not exist
            with self.assertRaises(NotFoundException) as e:
                self.api.get_telescope_type(id=expected_id)
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: TelescopeType with id {expected_id}"\n', e.exception.body)

            # Assert that TelescopeType with given type_id does not exist
            with self.assertRaises(NotFoundException) as e:
                self.api.get_telescope_type(type_id=type_id)
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: TelescopeType with type_id {type_id}"\n', e.exception.body)

            # Assert that 400 error raised: both missing
            expected_body = '"At least one and only one of id or type_id must be specified"\n'
            with self.assertRaises(ApiException) as e:
                self.api.get_telescope_type()
            self.assertEqual(400, e.exception.status)
            self.assertEqual(expected_body, e.exception.body)

            # Assert that 400 error raised: both present
            with self.assertRaises(ApiException) as e:
                self.api.get_telescope_type(id=expected_id, type_id=type_id)
            self.assertEqual(400, e.exception.status)
            self.assertEqual(expected_body, e.exception.body)

            # Add TelescopeType
            name = "ONIX Telescope"
            dt = pendulum.now(self.timezone)
            dt_utc = dt.in_tz(tz="UTC")
            self.env.session.add(orm.TelescopeType(type_id=type_id, name=name, created=dt, modified=dt))
            self.env.session.commit()

            # Assert that TelescopeType with given id exists
            obj = self.api.get_telescope_type(id=expected_id)
            self.assertIsInstance(obj, TelescopeType)
            self.assertEqual(expected_id, obj.id)
            self.assertEqual(name, obj.name)
            self.assertEqual(dt_utc, obj.created)
            self.assertEqual(dt_utc, obj.modified)

            # Assert that TelescopeType can be fetched with type_id
            obj = self.api.get_telescope_type(type_id=type_id)
            self.assertIsInstance(obj, TelescopeType)
            self.assertEqual(expected_id, obj.id)
            self.assertEqual(name, obj.name)
            self.assertEqual(dt_utc, obj.created)
            self.assertEqual(dt_utc, obj.modified)

    def test_get_telescope_types(self):
        """Test case for get_telescope_types

        Get a list of TelescopeType objects  # noqa: E501
        """

        with self.env.create():
            # Add TelescopeType objects
            dt = pendulum.now(self.timezone)
            dt_utc = dt.in_tz(tz="UTC")
            telescope_types = make_telescope_types(self.env, dt)

            # Assert that TelescopeType objects returned
            objects = self.api.get_telescope_types(limit=10)
            self.assertEqual(len(telescope_types), len(objects))
            for i, (obj, (type_id, name)) in enumerate(zip(objects, telescope_types)):
                expected_id = i + 1
                self.assertIsInstance(obj, TelescopeType)
                self.assertEqual(expected_id, obj.id)
                self.assertEqual(type_id, obj.type_id)
                self.assertEqual(name, obj.name)
                self.assertEqual(dt_utc, obj.created)
                self.assertEqual(dt_utc, obj.modified)

    def test_get_telescopes(self):
        """Test case for get_telescopes

        Get a list of Telescope objects  # noqa: E501
        """

        with self.env.create():
            # Add TelescopeType objects
            dt = pendulum.now(self.timezone)
            make_telescope_types(self.env, dt)

            # Add Organisations
            names = ["Curtin University", "Massachusetts Institute of Technology"]
            for name in names:
                self.env.session.add(orm.Organisation(name=name, created=dt, modified=dt))
            self.env.session.commit()

            # Add Telescopes
            dt = pendulum.now(self.timezone)
            self.env.session.add(
                orm.Telescope(
                    organisation={"id": 1},
                    telescope_type={"id": 1},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.add(
                orm.Telescope(organisation={"id": 1}, telescope_type={"id": 2}, created=dt, modified=dt)
            )
            self.env.session.add(
                orm.Telescope(organisation={"id": 2}, telescope_type={"id": 1}, created=dt, modified=dt)
            )
            self.env.session.commit()

            # Assert that all Telescope objects returned
            objects = self.api.get_telescopes(limit=10)
            self.assertEqual(3, len(objects))

            # Assert that Organisation 1 Telescope objects returned
            objects = self.api.get_telescopes(organisation_id=1, limit=10)
            self.assertEqual(2, len(objects))

            # Assert that Organisation 2 Telescope objects returned
            objects = self.api.get_telescopes(organisation_id=2, limit=10)
            self.assertEqual(1, len(objects))

            # Assert that TelescopeType 1 Telescope objects returned
            objects = self.api.get_telescopes(telescope_type_id=1, limit=10)
            self.assertEqual(2, len(objects))

            # Assert that TelescopeType 2 Telescope objects returned
            objects = self.api.get_telescopes(telescope_type_id=2, limit=10)
            self.assertEqual(1, len(objects))

    def test_post_organisation(self):
        """Test case for post_organisation

        create an Organisation  # noqa: E501
        """

        with self.env.create():
            # Post telescope
            expected_id = 1
            obj = Organisation(name="Curtin University")
            result = self.api.post_organisation(obj)
            self.assertIsInstance(result, Organisation)
            self.assertEqual(expected_id, result.id)

    def test_post_telescope(self):
        """Test case for post_telescope

        create a Telescope  # noqa: E501
        """

        with self.env.create():
            # Add TelescopeType and Organisation
            dt = pendulum.now(self.timezone)
            self.env.session.add(orm.TelescopeType(type_id="onix", name="ONIX Telescope", created=dt, modified=dt))
            self.env.session.add(orm.Organisation(name="Curtin University", created=dt, modified=dt))
            self.env.session.commit()

            # Post telescope
            expected_id = 1
            obj = Telescope(
                name="Curtin University ONIX Telescope",
                extra={"view_id": 123456},
                organisation=Organisation(id=expected_id),
                telescope_type=TelescopeType(id=expected_id),
            )
            result = self.api.post_telescope(obj)
            self.assertIsInstance(result, Telescope)
            self.assertEqual(expected_id, result.id)

    def test_post_telescope_type(self):
        """Test case for post_telescope_type

        create a TelescopeType  # noqa: E501
        """

        with self.env.create():
            expected_id = 1
            obj = TelescopeType(type_id="onix", name="ONIX Telescope")
            result = self.api.post_telescope_type(obj)

            self.assertIsInstance(result, TelescopeType)
            self.assertEqual(expected_id, result.id)

    def test_put_organisation(self):
        """Test case for put_organisation

        create or update an Organisation  # noqa: E501
        """

        with self.env.create():
            # Put create
            expected_id = 1
            name = "Curtin University"
            obj = Organisation(name=name)
            result = self.api.put_organisation(obj)
            self.assertIsInstance(result, Organisation)
            self.assertEqual(expected_id, result.id)
            self.assertEqual(name, result.name)

            # Put update
            new_name = "Massachusetts Institute of Technology"
            obj = Organisation(id=expected_id, name=new_name)
            result = self.api.put_organisation(obj)
            self.assertIsInstance(result, Organisation)
            self.assertEqual(expected_id, result.id)
            self.assertEqual(new_name, result.name)

            # Put not found
            expected_id = 2
            with self.assertRaises(NotFoundException) as e:
                self.api.put_organisation(Organisation(id=expected_id, name=new_name))
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: Organisation with id {expected_id}"\n', e.exception.body)

    def test_put_telescope(self):
        """Test case for put_telescope

        create or update a Telescope  # noqa: E501
        """

        with self.env.create():
            expected_id = 1
            dt = pendulum.now(self.timezone)
            self.env.session.add(orm.TelescopeType(type_id="onix", name="ONIX Telescope", created=dt, modified=dt))
            self.env.session.add(orm.Organisation(name="Curtin University", created=dt, modified=dt))
            self.env.session.add(
                orm.Organisation(name="Massachusetts Institute of Technology", created=dt, modified=dt)
            )
            self.env.session.commit()

            # Put create
            obj = Telescope(organisation=Organisation(id=expected_id), telescope_type=TelescopeType(id=expected_id))
            result = self.api.put_telescope(obj)
            self.assertIsInstance(result, Telescope)
            self.assertEqual(expected_id, result.id)

            # Put update
            name = "Curtin ONIX Telescope"
            extra = {"view_id": 123456}
            obj = Telescope(
                id=expected_id,
                name=name,
                extra=extra,
                organisation=Organisation(id=2),
                telescope_type=TelescopeType(id=expected_id),
            )
            result = self.api.put_telescope(obj)
            self.assertIsInstance(result, Telescope)
            self.assertEqual(expected_id, result.id)
            self.assertEqual(name, result.name)
            self.assertDictEqual(extra, result.extra)
            self.assertEqual("Massachusetts Institute of Technology", result.organisation.name)

            # Put not found
            expected_id = 2
            with self.assertRaises(NotFoundException) as e:
                self.api.put_telescope(
                    Telescope(
                        id=expected_id,
                        organisation=Organisation(id=expected_id),
                        telescope_type=TelescopeType(id=expected_id),
                    )
                )
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: Telescope with id {expected_id}"\n', e.exception.body)

    def test_put_telescope_type(self):
        """Test case for put_telescope_type

        create or update a TelescopeType  # noqa: E501
        """

        with self.env.create():
            # Put create
            expected_id = 1
            name = "ONIX Telescope"
            obj = TelescopeType(type_id="onix", name=name)
            result = self.api.put_telescope_type(obj)
            self.assertIsInstance(result, TelescopeType)
            self.assertEqual(expected_id, result.id)
            self.assertEqual(name, result.name)

            # Put update
            new_name = "Google Analytics Telescope"
            obj = TelescopeType(type_id="google_analytics", id=expected_id, name=new_name)
            result = self.api.put_telescope_type(obj)
            self.assertIsInstance(result, TelescopeType)
            self.assertEqual(expected_id, result.id)
            self.assertEqual(new_name, result.name)

            # Put not found
            expected_id = 2
            with self.assertRaises(NotFoundException) as e:
                self.api.put_telescope_type(TelescopeType(id=expected_id, name=new_name))
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: TelescopeType with id {expected_id}"\n', e.exception.body)

    def test_get_dataset(self):
        """Test case for get_dataset"""

        with self.env.create():
            expected_id = 1

            # Assert that Dataset with given id does not exist
            with self.assertRaises(NotFoundException) as e:
                self.api.get_dataset(expected_id)
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: Dataset with id {expected_id}"\n', e.exception.body)

            # Add Dataset
            telescope_type_name = "ONIX Telescope"
            org_name = "Curtin University"
            dt = pendulum.now(self.timezone)
            dt_utc = dt.in_tz(tz="UTC")
            self.env.session.add(orm.TelescopeType(type_id="onix", name=telescope_type_name, created=dt, modified=dt))
            self.env.session.add(orm.Organisation(name=org_name, created=dt, modified=dt))
            self.env.session.commit()
            self.env.session.add(
                orm.Telescope(
                    name="Curtin ONIX Telescope",
                    extra={"view_id": 123456},
                    organisation={"id": expected_id},
                    telescope_type={"id": expected_id},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            self.env.session.add(
                orm.Dataset(
                    name="dataset",
                    extra={},
                    connection={"id": expected_id},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Assert that Dataset with given id exists
            obj = self.api.get_dataset(expected_id)
            self.assertIsInstance(obj, Dataset)
            self.assertEqual(expected_id, obj.id)
            self.assertEqual("dataset", obj.name)
            self.assertEqual(expected_id, obj.connection.id)
            self.assertEqual("Curtin ONIX Telescope", obj.connection.name)
            self.assertEqual(dt_utc, obj.created)
            self.assertEqual(dt_utc, obj.modified)

    def test_post_dataset(self):
        """Test case for post_dataset"""

        with self.env.create():
            expected_id = 1
            telescope_type_name = "ONIX Telescope"
            org_name = "Curtin University"
            dt = pendulum.now(self.timezone)
            dt_utc = dt.in_tz(tz="UTC")
            self.env.session.add(orm.TelescopeType(type_id="onix", name=telescope_type_name, created=dt, modified=dt))
            self.env.session.add(orm.Organisation(name=org_name, created=dt, modified=dt))
            self.env.session.commit()
            self.env.session.add(
                orm.Telescope(
                    name="Curtin ONIX Telescope",
                    extra={"view_id": 123456},
                    organisation={"id": expected_id},
                    telescope_type={"id": expected_id},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Post Dataset
            obj = Dataset(
                name="My dataset",
                connection=Telescope(id=expected_id),
            )
            result = self.api.post_dataset(obj)
            self.assertIsInstance(result, Dataset)
            self.assertEqual(expected_id, result.id)

    def test_put_dataset(self):
        """Test case for put_dataset"""

        with self.env.create():
            expected_id = 1
            telescope_type_name = "ONIX Telescope"
            org_name = "Curtin University"
            dt = pendulum.now(self.timezone)
            dt_utc = dt.in_tz(tz="UTC")
            self.env.session.add(orm.TelescopeType(type_id="onix", name=telescope_type_name, created=dt, modified=dt))
            self.env.session.add(orm.Organisation(name=org_name, created=dt, modified=dt))
            self.env.session.commit()
            self.env.session.add(
                orm.Telescope(
                    name="Curtin ONIX Telescope",
                    extra={"view_id": 123456},
                    organisation={"id": expected_id},
                    telescope_type={"id": expected_id},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Put create
            obj = Dataset(name="My dataset", connection=Telescope(id=expected_id))
            result = self.api.put_dataset(obj)
            self.assertIsInstance(result, Dataset)
            self.assertEqual(expected_id, result.id)

            # Put update
            name = "Dataset"
            extra = {}
            obj = Dataset(id=expected_id, name=name, extra=extra, connection=Telescope(id=expected_id))
            result = self.api.put_dataset(obj)
            self.assertIsInstance(result, Dataset)
            self.assertEqual(expected_id, result.id)
            self.assertEqual(name, result.name)
            self.assertDictEqual(extra, result.extra)
            self.assertEqual("Curtin ONIX Telescope", result.connection.name)

            # Put not found
            expected_id = 2
            with self.assertRaises(NotFoundException) as e:
                self.api.put_dataset(
                    Dataset(
                        id=expected_id,
                        connection=Telescope(id=expected_id),
                    )
                )
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: Dataset with id {expected_id}"\n', e.exception.body)

    def test_get_datasets(self):
        """Test case for get_datasets"""

        with self.env.create():
            expected_id = 1
            telescope_type_name = "ONIX Telescope"
            org_name = "Curtin University"
            dt = pendulum.now(self.timezone)
            dt_utc = dt.in_tz(tz="UTC")
            self.env.session.add(orm.TelescopeType(type_id="onix", name=telescope_type_name, created=dt, modified=dt))
            self.env.session.add(orm.Organisation(name=org_name, created=dt, modified=dt))
            self.env.session.commit()
            self.env.session.add(
                orm.Telescope(
                    name="Curtin ONIX Telescope",
                    extra={"view_id": 123456},
                    organisation={"id": expected_id},
                    telescope_type={"id": expected_id},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.add(
                orm.Telescope(
                    name="Curtin ONIX Telescope",
                    extra={"view_id": 123456},
                    organisation={"id": expected_id},
                    telescope_type={"id": expected_id},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Add Datasets
            dt = pendulum.now(self.timezone)
            self.env.session.add(
                orm.Dataset(
                    name="dataset",
                    extra={},
                    connection={"id": 1},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.add(
                orm.Dataset(
                    name="dataset",
                    extra={},
                    connection={"id": 2},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Assert that all Telescope objects returned
            objects = self.api.get_datasets(limit=10)
            self.assertEqual(2, len(objects))

            # Search datasets by telescope_id
            objects = self.api.get_datasets(telescope_id=1, limit=10)
            self.assertEqual(1, len(objects))

            objects = self.api.get_datasets(telescope_id=2, limit=10)
            self.assertEqual(1, len(objects))

    def test_get_dataset_storage(self):
        """Test case for get_dataset_storage"""

        with self.env.create():
            expected_id = 1

            with self.assertRaises(NotFoundException) as e:
                self.api.get_dataset_storage(expected_id)
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: DatasetStorage with id {expected_id}"\n', e.exception.body)

            expected_id = 1
            telescope_type_name = "ONIX Telescope"
            org_name = "Curtin University"
            dt = pendulum.now(self.timezone)
            dt_utc = dt.in_tz(tz="UTC")
            self.env.session.add(orm.TelescopeType(type_id="onix", name=telescope_type_name, created=dt, modified=dt))
            self.env.session.add(orm.Organisation(name=org_name, created=dt, modified=dt))
            self.env.session.commit()
            self.env.session.add(
                orm.Telescope(
                    name="Curtin ONIX Telescope",
                    extra={"view_id": 123456},
                    organisation={"id": expected_id},
                    telescope_type={"id": expected_id},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Add Datasets
            self.env.session.add(
                orm.Dataset(
                    name="dataset",
                    extra={},
                    connection={"id": 1},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            self.env.session.add(
                orm.DatasetStorage(
                    service="bigquery",
                    address="project.dataset.table",
                    extra={},
                    dataset={"id": expected_id},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Assert that DatasetStorage with given id exists
            obj = self.api.get_dataset_storage(expected_id)
            self.assertIsInstance(obj, DatasetStorage)
            self.assertEqual(expected_id, obj.id)
            self.assertEqual(expected_id, obj.dataset.id)
            self.assertEqual("dataset", obj.dataset.name)
            self.assertEqual(dt_utc, obj.created)
            self.assertEqual(dt_utc, obj.modified)

    def test_post_dataset_storage(self):
        """Test case for post_dataset_storage"""

        with self.env.create():
            expected_id = 1

            with self.assertRaises(NotFoundException) as e:
                self.api.get_dataset_storage(expected_id)
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: DatasetStorage with id {expected_id}"\n', e.exception.body)

            expected_id = 1
            telescope_type_name = "ONIX Telescope"
            org_name = "Curtin University"
            dt = pendulum.now(self.timezone)
            dt_utc = dt.in_tz(tz="UTC")
            self.env.session.add(orm.TelescopeType(type_id="onix", name=telescope_type_name, created=dt, modified=dt))
            self.env.session.add(orm.Organisation(name=org_name, created=dt, modified=dt))
            self.env.session.commit()
            self.env.session.add(
                orm.Telescope(
                    name="Curtin ONIX Telescope",
                    extra={"view_id": 123456},
                    organisation={"id": expected_id},
                    telescope_type={"id": expected_id},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Add Datasets
            self.env.session.add(
                orm.Dataset(
                    name="dataset",
                    extra={},
                    connection={"id": 1},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Post DatasetStorage
            expected_id = 1
            obj = DatasetStorage(
                extra={},
                service="bigquery",
                address="project.dataset.table",
                dataset=Dataset(id=expected_id),
            )
            result = self.api.post_dataset_storage(obj)
            self.assertIsInstance(result, DatasetStorage)
            self.assertEqual(expected_id, result.id)

    def test_put_dataset_storage(self):
        """Test case for put_dataset_storage"""

        with self.env.create():
            expected_id = 1

            with self.assertRaises(NotFoundException) as e:
                self.api.get_dataset_storage(expected_id)
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: DatasetStorage with id {expected_id}"\n', e.exception.body)

            expected_id = 1
            telescope_type_name = "ONIX Telescope"
            org_name = "Curtin University"
            dt = pendulum.now(self.timezone)
            dt_utc = dt.in_tz(tz="UTC")
            self.env.session.add(orm.TelescopeType(type_id="onix", name=telescope_type_name, created=dt, modified=dt))
            self.env.session.add(orm.Organisation(name=org_name, created=dt, modified=dt))
            self.env.session.commit()
            self.env.session.add(
                orm.Telescope(
                    name="Curtin ONIX Telescope",
                    extra={"view_id": 123456},
                    organisation={"id": expected_id},
                    telescope_type={"id": expected_id},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Add Datasets
            self.env.session.add(
                orm.Dataset(
                    name="dataset",
                    extra={},
                    connection={"id": 1},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Put create
            obj = DatasetStorage(
                dataset=Dataset(id=expected_id),
                service="bigquery",
                address="project.dataset.table",
                extra={},
            )
            result = self.api.put_dataset_storage(obj)
            self.assertIsInstance(result, DatasetStorage)
            self.assertEqual(expected_id, result.id)

            # Put update
            obj = DatasetStorage(
                id=expected_id,
                dataset=Dataset(id=expected_id),
                service="bigquery",
                address="project2.dataset.table",
                extra={},
            )
            result = self.api.put_dataset_storage(obj)
            self.assertIsInstance(result, DatasetStorage)
            self.assertEqual(expected_id, result.id)
            self.assertEqual("project2.dataset.table", result.address)
            self.assertEqual("bigquery", result.service)

            # Put not found
            expected_id = 2
            with self.assertRaises(NotFoundException) as e:
                self.api.put_dataset_storage(
                    DatasetStorage(
                        id=expected_id,
                        dataset=Dataset(id=expected_id),
                        service="bigquery",
                        address="project2.dataset.table",
                        extra={},
                    )
                )
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: DatasetStorage with id {expected_id}"\n', e.exception.body)

    def test_get_dataset_storages(self):
        """Test case for get_dataset_storages"""

        with self.env.create():
            expected_id = 1

            with self.assertRaises(NotFoundException) as e:
                self.api.get_dataset_storage(expected_id)
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: DatasetStorage with id {expected_id}"\n', e.exception.body)

            expected_id = 1
            telescope_type_name = "ONIX Telescope"
            org_name = "Curtin University"
            dt = pendulum.now(self.timezone)
            dt_utc = dt.in_tz(tz="UTC")
            self.env.session.add(orm.TelescopeType(type_id="onix", name=telescope_type_name, created=dt, modified=dt))
            self.env.session.add(orm.Organisation(name=org_name, created=dt, modified=dt))
            self.env.session.commit()
            self.env.session.add(
                orm.Telescope(
                    name="Curtin ONIX Telescope",
                    extra={"view_id": 123456},
                    organisation={"id": expected_id},
                    telescope_type={"id": expected_id},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Add Datasets
            self.env.session.add(
                orm.Dataset(
                    name="dataset1",
                    extra={},
                    connection={"id": 1},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.add(
                orm.Dataset(
                    name="dataset2",
                    extra={},
                    connection={"id": 1},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Post DatasetStorage
            expected_id = 1
            obj = DatasetStorage(
                extra={},
                service="google1",
                address="project.dataset.table",
                dataset=Dataset(id=1),
            )
            self.api.post_dataset_storage(obj)
            obj = DatasetStorage(
                extra={},
                service="google2",
                address="project.dataset.table",
                dataset=Dataset(id=2),
            )
            self.api.post_dataset_storage(obj)

            objects = self.api.get_dataset_storages(limit=10)
            self.assertEqual(2, len(objects))

            self.assertEqual(objects[0].id, 1)
            self.assertEqual(objects[1].id, 2)
            self.assertEqual(objects[0].service, "google1")
            self.assertEqual(objects[1].service, "google2")

            objects = self.api.get_dataset_storages(dataset_id=1, limit=10)
            self.assertEqual(1, len(objects))
            self.assertEqual(objects[0].dataset.name, "dataset1")

    def test_get_dataset_release(self):
        """Test case for get_dataset_release"""

        with self.env.create():
            expected_id = 1
            with self.assertRaises(NotFoundException) as e:
                self.api.get_dataset_release(id=expected_id)
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: DatasetRelease with id {expected_id}"\n', e.exception.body)

            expected_id = 1
            telescope_type_name = "ONIX Telescope"
            org_name = "Curtin University"
            dt = pendulum.now(self.timezone)
            dt_utc = dt.in_tz(tz="UTC")
            self.env.session.add(orm.TelescopeType(type_id="onix", name=telescope_type_name, created=dt, modified=dt))
            self.env.session.add(orm.Organisation(name=org_name, created=dt, modified=dt))
            self.env.session.commit()
            self.env.session.add(
                orm.Telescope(
                    name="Curtin ONIX Telescope",
                    extra={"view_id": 123456},
                    organisation={"id": expected_id},
                    telescope_type={"id": expected_id},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Add Datasets
            self.env.session.add(
                orm.Dataset(
                    name="dataset",
                    extra={},
                    connection={"id": 1},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            self.env.session.add(
                orm.DatasetRelease(
                    schema_version="schema",
                    schema_version_alt="altschema",
                    start_date=dt,
                    end_date=dt,
                    ingestion_start=dt,
                    ingestion_end=dt,
                    dataset={"id": expected_id},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Assert that DatasetRelease with given id exists
            obj = self.api.get_dataset_release(id=expected_id)
            self.assertIsInstance(obj, DatasetRelease)
            self.assertEqual(expected_id, obj.id)
            self.assertEqual(expected_id, obj.dataset.id)
            self.assertEqual("dataset", obj.dataset.name)
            self.assertEqual(dt_utc, obj.created)
            self.assertEqual(dt_utc, obj.modified)

            # Search by dataset_id
            obj = self.api.get_dataset_release(expected_id)
            self.assertIsInstance(obj, DatasetRelease)
            self.assertEqual(obj.schema_version, "schema")

            # DatasetRelease not found
            dataset_id = 2
            self.assertRaises(NotFoundException, self.api.get_dataset_release, dataset_id)

    def test_post_dataset_release(self):
        """Test case for post_dataset_release"""

        with self.env.create():
            expected_id = 1

            with self.assertRaises(NotFoundException) as e:
                self.api.get_dataset_storage(id=expected_id)
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: DatasetStorage with id {expected_id}"\n', e.exception.body)

            expected_id = 1
            telescope_type_name = "ONIX Telescope"
            org_name = "Curtin University"
            dt = pendulum.now(self.timezone)
            dt_utc = dt.in_tz(tz="UTC")
            self.env.session.add(orm.TelescopeType(type_id="onix", name=telescope_type_name, created=dt, modified=dt))
            self.env.session.add(orm.Organisation(name=org_name, created=dt, modified=dt))
            self.env.session.commit()
            self.env.session.add(
                orm.Telescope(
                    name="Curtin ONIX Telescope",
                    extra={"view_id": 123456},
                    organisation={"id": expected_id},
                    telescope_type={"id": expected_id},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Add Datasets
            self.env.session.add(
                orm.Dataset(
                    name="dataset",
                    extra={},
                    connection={"id": 1},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Post DatasetRelease
            expected_id = 1
            obj = DatasetRelease(
                schema_version="schema",
                schema_version_alt="schemaalt",
                start_date=dt,
                end_date=dt,
                ingestion_start=dt,
                ingestion_end=dt,
                dataset=Dataset(id=expected_id),
            )
            result = self.api.post_dataset_release(obj)
            self.assertIsInstance(result, DatasetRelease)
            self.assertEqual(expected_id, result.id)

    def test_put_dataset_release(self):
        """Test case for put_dataset_release"""

        with self.env.create():
            expected_id = 1

            with self.assertRaises(NotFoundException) as e:
                self.api.get_dataset_release(id=expected_id)
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: DatasetRelease with id {expected_id}"\n', e.exception.body)

            expected_id = 1
            telescope_type_name = "ONIX Telescope"
            org_name = "Curtin University"
            dt = pendulum.now(self.timezone)
            dt_utc = dt.in_tz(tz="UTC")
            self.env.session.add(orm.TelescopeType(type_id="onix", name=telescope_type_name, created=dt, modified=dt))
            self.env.session.add(orm.Organisation(name=org_name, created=dt, modified=dt))
            self.env.session.commit()
            self.env.session.add(
                orm.Telescope(
                    name="Curtin ONIX Telescope",
                    extra={"view_id": 123456},
                    organisation={"id": expected_id},
                    telescope_type={"id": expected_id},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Add Datasets
            self.env.session.add(
                orm.Dataset(
                    name="dataset",
                    extra={},
                    connection={"id": 1},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Put create
            obj = DatasetRelease(
                dataset=Dataset(id=expected_id),
                schema_version="schema",
                schema_version_alt="schemaalt",
                start_date=dt,
                end_date=dt,
                ingestion_start=dt,
                ingestion_end=dt,
            )
            result = self.api.put_dataset_release(obj)
            self.assertIsInstance(result, DatasetRelease)
            self.assertEqual(expected_id, result.id)

            # Put update
            obj = DatasetRelease(
                id=expected_id,
                schema_version="schema2",
                schema_version_alt="schemaalt2",
                start_date=dt,
                end_date=dt,
                ingestion_start=dt,
                ingestion_end=dt,
            )
            result = self.api.put_dataset_release(obj)
            self.assertIsInstance(result, DatasetRelease)
            self.assertEqual(expected_id, result.id)
            self.assertEqual("schema2", result.schema_version)
            self.assertEqual("schemaalt2", result.schema_version_alt)

            # Put not found
            expected_id = 2
            with self.assertRaises(NotFoundException) as e:
                self.api.put_dataset_release(
                    DatasetRelease(
                        id=expected_id,
                        dataset=Dataset(id=expected_id),
                        schema_version="schema2",
                        schema_version_alt="schemaalt2",
                        start_date=dt,
                        end_date=dt,
                        ingestion_start=dt,
                        ingestion_end=dt,
                    )
                )
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: DatasetRelease with id {expected_id}"\n', e.exception.body)

    def test_get_dataset_releases(self):
        """Test case for get_dataset_releases"""

        with self.env.create():
            expected_id = 1

            with self.assertRaises(NotFoundException) as e:
                self.api.get_dataset_release(id=expected_id)
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: DatasetRelease with id {expected_id}"\n', e.exception.body)

            expected_id = 1
            telescope_type_name = "ONIX Telescope"
            org_name = "Curtin University"
            dt = pendulum.now(self.timezone)
            dt_utc = dt.in_tz(tz="UTC")
            self.env.session.add(orm.TelescopeType(type_id="onix", name=telescope_type_name, created=dt, modified=dt))
            self.env.session.add(orm.Organisation(name=org_name, created=dt, modified=dt))
            self.env.session.commit()
            self.env.session.add(
                orm.Telescope(
                    name="Curtin ONIX Telescope",
                    extra={"view_id": 123456},
                    organisation={"id": expected_id},
                    telescope_type={"id": expected_id},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Add Datasets
            self.env.session.add(
                orm.Dataset(
                    name="dataset1",
                    extra={},
                    connection={"id": 1},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.add(
                orm.Dataset(
                    name="dataset2",
                    extra={},
                    connection={"id": 1},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Post DatasetRelease
            expected_id = 1
            obj = DatasetRelease(
                dataset=Dataset(id=1),
                schema_version="schema1",
                schema_version_alt="schemaalt1",
                start_date=dt,
                end_date=dt,
                ingestion_start=dt,
                ingestion_end=dt,
            )
            self.api.post_dataset_release(obj)
            obj = DatasetRelease(
                dataset=Dataset(id=2),
                schema_version="schema2",
                schema_version_alt="schemaalt2",
                start_date=dt,
                end_date=dt,
                ingestion_start=dt,
                ingestion_end=dt,
            )
            self.api.post_dataset_release(obj)

            # Assert that all DatasetRelease objects returned
            objects = self.api.get_dataset_releases(limit=10)
            self.assertEqual(2, len(objects))
            self.assertIsInstance(objects[0], DatasetRelease)
            self.assertIsInstance(objects[1], DatasetRelease)

            self.assertEqual(objects[0].id, 1)
            self.assertEqual(objects[1].id, 2)
            self.assertEqual(objects[0].schema_version, "schema1")
            self.assertEqual(objects[1].schema_version, "schema2")

            objects = self.api.get_dataset_releases(dataset_id=1, limit=10)
            self.assertEqual(1, len(objects))
            self.assertIsInstance(objects[0], DatasetRelease)
            self.assertEqual(objects[0].dataset.name, "dataset1")

    def test_delete_dataset(self):
        """Test case for delete_dataset"""

        with self.env.create():
            expected_id = 1
            telescope_type_name = "ONIX Telescope"
            org_name = "Curtin University"
            dt = pendulum.now(self.timezone)
            dt_utc = dt.in_tz(tz="UTC")
            self.env.session.add(orm.TelescopeType(type_id="onix", name=telescope_type_name, created=dt, modified=dt))
            self.env.session.add(orm.Organisation(name=org_name, created=dt, modified=dt))
            self.env.session.commit()
            self.env.session.add(
                orm.Telescope(
                    name="Curtin ONIX Telescope",
                    extra={"view_id": 123456},
                    organisation={"id": expected_id},
                    telescope_type={"id": expected_id},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Post Dataset
            obj = Dataset(
                name="My dataset",
                connection=Telescope(id=expected_id),
            )
            self.api.post_dataset(obj)
            self.api.delete_dataset(expected_id)

            with self.assertRaises(NotFoundException) as e:
                self.api.delete_dataset(expected_id)
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: Dataset with id {expected_id}"\n', e.exception.body)

    def test_delete_dataset_storage(self):
        """Test case for delete_dataset_storage"""

        with self.env.create():
            expected_id = 1

            with self.assertRaises(NotFoundException) as e:
                self.api.get_dataset_storage(expected_id)
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: DatasetStorage with id {expected_id}"\n', e.exception.body)

            expected_id = 1
            telescope_type_name = "ONIX Telescope"
            org_name = "Curtin University"
            dt = pendulum.now(self.timezone)
            dt_utc = dt.in_tz(tz="UTC")
            self.env.session.add(orm.TelescopeType(type_id="onix", name=telescope_type_name, created=dt, modified=dt))
            self.env.session.add(orm.Organisation(name=org_name, created=dt, modified=dt))
            self.env.session.commit()
            self.env.session.add(
                orm.Telescope(
                    name="Curtin ONIX Telescope",
                    extra={"view_id": 123456},
                    organisation={"id": expected_id},
                    telescope_type={"id": expected_id},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Add Datasets
            self.env.session.add(
                orm.Dataset(
                    name="dataset",
                    extra={},
                    connection={"id": 1},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Post DatasetStorage
            expected_id = 1
            obj = DatasetStorage(
                extra={},
                service="bigquery",
                address="project.dataset.table",
                dataset=Dataset(id=expected_id),
            )
            self.api.post_dataset_storage(obj)
            self.api.delete_dataset_storage(expected_id)

            with self.assertRaises(NotFoundException) as e:
                self.api.delete_dataset_storage(expected_id)
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: DatasetStorage with id {expected_id}"\n', e.exception.body)

    def test_delete_dataset_release(self):
        """Test case for delete_dataset_release"""

        with self.env.create():
            expected_id = 1

            with self.assertRaises(NotFoundException) as e:
                self.api.get_dataset_storage(id=expected_id)
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: DatasetStorage with id {expected_id}"\n', e.exception.body)

            expected_id = 1
            telescope_type_name = "ONIX Telescope"
            org_name = "Curtin University"
            dt = pendulum.now(self.timezone)
            dt_utc = dt.in_tz(tz="UTC")
            self.env.session.add(orm.TelescopeType(type_id="onix", name=telescope_type_name, created=dt, modified=dt))
            self.env.session.add(orm.Organisation(name=org_name, created=dt, modified=dt))
            self.env.session.commit()
            self.env.session.add(
                orm.Telescope(
                    name="Curtin ONIX Telescope",
                    extra={"view_id": 123456},
                    organisation={"id": expected_id},
                    telescope_type={"id": expected_id},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Add Datasets
            self.env.session.add(
                orm.Dataset(
                    name="dataset",
                    extra={},
                    connection={"id": 1},
                    created=dt,
                    modified=dt,
                )
            )
            self.env.session.commit()

            # Post DatasetRelease
            expected_id = 1
            obj = DatasetRelease(
                schema_version="schema",
                schema_version_alt="schemaalt",
                start_date=dt,
                end_date=dt,
                ingestion_start=dt,
                ingestion_end=dt,
                dataset=Dataset(id=expected_id),
            )
            self.api.post_dataset_release(obj)
            self.api.delete_dataset_release(expected_id)

            with self.assertRaises(NotFoundException) as e:
                self.api.delete_dataset_release(expected_id)
            self.assertEqual(404, e.exception.status)
            self.assertEqual(f'"Not found: DatasetRelease with id {expected_id}"\n', e.exception.body)


if __name__ == "__main__":
    unittest.main()
