"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHubRunners = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const constructs_1 = require("constructs");
const codebuild_1 = require("./providers/codebuild");
const fargate_1 = require("./providers/fargate");
const lambda_1 = require("./providers/lambda");
const secrets_1 = require("./secrets");
const utils_1 = require("./utils");
const webhook_1 = require("./webhook");
/**
 * Create all the required infrastructure to provide self-hosted GitHub runners. It creates a webhook, secrets, and a step function to orchestrate all runs. Secrets are not automatically filled. See README.md for instructions on how to setup GitHub integration.
 *
 * By default, this will create a runner provider of each available type with the defaults. This is good enough for the initial setup stage when you just want to get GitHub integration working.
 *
 * ```typescript
 * new GitHubRunners(this, 'runners');
 * ```
 *
 * Usually you'd want to configure the runner providers so the runners can run in a certain VPC or have certain permissions.
 *
 * ```typescript
 * const vpc = ec2.Vpc.fromLookup(this, 'vpc', { vpcId: 'vpc-1234567' });
 * const runnerSg = new ec2.SecurityGroup(this, 'runner security group', { vpc: vpc });
 * const dbSg = ec2.SecurityGroup.fromSecurityGroupId(this, 'database security group', 'sg-1234567');
 * const bucket = new s3.Bucket(this, 'runner bucket');
 *
 * // create a custom CodeBuild provider
 * const myProvider = new CodeBuildRunner(
 *   this, 'codebuild runner',
 *   {
 *      label: 'my-codebuild',
 *      vpc: vpc,
 *      securityGroup: runnerSg,
 *   },
 * );
 * // grant some permissions to the provider
 * bucket.grantReadWrite(myProvider);
 * dbSg.connections.allowFrom(runnerSg, ec2.Port.tcp(3306), 'allow runners to connect to MySQL database');
 *
 * // create the runner infrastructure
 * new GitHubRunners(
 *   this,
 *   'runners',
 *   {
 *     providers: [myProvider],
 *   }
 * );
 * ```
 */
class GitHubRunners extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.extraLambdaEnv = {};
        this.secrets = new secrets_1.Secrets(this, 'Secrets');
        this.extraLambdaProps = {
            vpc: this.props?.vpc,
            vpcSubnets: this.props?.vpcSubnets,
            allowPublicSubnet: this.props?.allowPublicSubnet,
            securityGroups: this.props?.securityGroup ? [this.props.securityGroup] : undefined,
            layers: this.props?.extraCertificates ? [new aws_cdk_lib_1.aws_lambda.LayerVersion(scope, 'Certificate Layer', {
                    description: 'Layer containing GitHub Enterprise Server certificate for cdk-github-runners',
                    code: aws_cdk_lib_1.aws_lambda.Code.fromAsset(this.props.extraCertificates),
                })] : undefined,
        };
        if (this.props?.extraCertificates) {
            this.extraLambdaEnv.NODE_EXTRA_CA_CERTS = '/opt/certs.pem';
        }
        if (this.props?.providers) {
            this.providers = this.props.providers;
        }
        else {
            this.providers = [
                new codebuild_1.CodeBuildRunner(this, 'CodeBuild'),
                new lambda_1.LambdaRunner(this, 'Lambda'),
                new fargate_1.FargateRunner(this, 'Fargate'),
            ];
        }
        this.checkIntersectingLabels();
        this.orchestrator = this.stateMachine(props);
        this.webhook = new webhook_1.GithubWebhookHandler(this, 'Webhook Handler', {
            orchestrator: this.orchestrator,
            secrets: this.secrets,
        });
        this.setupUrl = this.setupFunction();
        this.statusFunction();
    }
    stateMachine(props) {
        const tokenRetrieverTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Get Runner Token', {
            lambdaFunction: this.tokenRetriever(),
            payloadResponseOnly: true,
            resultPath: '$.runner',
        });
        let deleteRunnerFunction = this.deleteRunner();
        const deleteRunnerTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Delete Runner', {
            lambdaFunction: deleteRunnerFunction,
            payloadResponseOnly: true,
            resultPath: '$.delete',
            payload: aws_cdk_lib_1.aws_stepfunctions.TaskInput.fromObject({
                runnerName: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                owner: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repo: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
                runId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runId'),
                installationId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.installationId'),
                idleOnly: false,
            }),
        });
        deleteRunnerTask.addRetry({
            errors: [
                'RunnerBusy',
            ],
            interval: cdk.Duration.minutes(1),
            backoffRate: 1,
            maxAttempts: 60,
        });
        const waitForIdleRunner = new aws_cdk_lib_1.aws_stepfunctions.Wait(this, 'Wait', {
            time: aws_cdk_lib_1.aws_stepfunctions.WaitTime.duration(props?.idleTimeout ?? cdk.Duration.minutes(10)),
        });
        const deleteIdleRunnerTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Delete Idle Runner', {
            lambdaFunction: deleteRunnerFunction,
            payloadResponseOnly: true,
            resultPath: '$.delete',
            payload: aws_cdk_lib_1.aws_stepfunctions.TaskInput.fromObject({
                runnerName: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                owner: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repo: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
                runId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runId'),
                installationId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.installationId'),
                idleOnly: true,
            }),
        });
        const providerChooser = new aws_cdk_lib_1.aws_stepfunctions.Choice(this, 'Choose provider');
        for (const provider of this.providers) {
            const providerTask = provider.getStepFunctionTask({
                runnerTokenPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runner.token'),
                runnerNamePath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                githubDomainPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runner.domain'),
                ownerPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repoPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
            });
            providerChooser.when(aws_cdk_lib_1.aws_stepfunctions.Condition.and(...provider.labels.map(label => aws_cdk_lib_1.aws_stepfunctions.Condition.isPresent(`$.labels.${label}`))), providerTask);
        }
        providerChooser.otherwise(new aws_cdk_lib_1.aws_stepfunctions.Succeed(this, 'Unknown label'));
        const work = tokenRetrieverTask.next(new aws_cdk_lib_1.aws_stepfunctions.Parallel(this, 'Error Catcher', { resultPath: '$.result' })
            .branch(providerChooser)
            .branch(waitForIdleRunner.next(deleteIdleRunnerTask))
            .addCatch(deleteRunnerTask
            .next(new aws_cdk_lib_1.aws_stepfunctions.Fail(this, 'Runner Failed')), {
            resultPath: '$.error',
        }));
        const check = new aws_cdk_lib_1.aws_stepfunctions.Choice(this, 'Is self hosted?')
            .when(aws_cdk_lib_1.aws_stepfunctions.Condition.isNotPresent('$.labels.self-hosted'), new aws_cdk_lib_1.aws_stepfunctions.Succeed(this, 'No'))
            .otherwise(work);
        let logOptions;
        if (this.props?.logOptions) {
            this.stateMachineLogGroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Logs', {
                logGroupName: props?.logOptions?.logGroupName,
                retention: props?.logOptions?.logRetention ?? aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            });
            logOptions = {
                destination: this.stateMachineLogGroup,
                includeExecutionData: props?.logOptions?.includeExecutionData ?? true,
                level: props?.logOptions?.level ?? aws_cdk_lib_1.aws_stepfunctions.LogLevel.ALL,
            };
        }
        const stateMachine = new aws_cdk_lib_1.aws_stepfunctions.StateMachine(this, 'Runner Orchestrator', {
            definition: check,
            logs: logOptions,
        });
        for (const provider of this.providers) {
            provider.grantStateMachine(stateMachine);
        }
        return stateMachine;
    }
    tokenRetriever() {
        const func = new utils_1.BundledNodejsFunction(this, 'token-retriever', {
            description: 'Get token from GitHub Actions used to start new self-hosted runner',
            environment: {
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.seconds(30),
            ...this.extraLambdaProps,
        });
        this.secrets.github.grantRead(func);
        this.secrets.githubPrivateKey.grantRead(func);
        return func;
    }
    deleteRunner() {
        const func = new utils_1.BundledNodejsFunction(this, 'delete-runner', {
            description: 'Delete GitHub Actions runner on error',
            environment: {
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.seconds(30),
            ...this.extraLambdaProps,
        });
        this.secrets.github.grantRead(func);
        this.secrets.githubPrivateKey.grantRead(func);
        return func;
    }
    statusFunction() {
        const statusFunction = new utils_1.BundledNodejsFunction(this, 'status', {
            description: 'Provide user with status about self-hosted GitHub Actions runners',
            environment: {
                WEBHOOK_SECRET_ARN: this.secrets.webhook.secretArn,
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                SETUP_SECRET_ARN: this.secrets.setup.secretArn,
                WEBHOOK_URL: this.webhook.url,
                WEBHOOK_HANDLER_ARN: this.webhook.handler.latestVersion.functionArn,
                STEP_FUNCTION_ARN: this.orchestrator.stateMachineArn,
                STEP_FUNCTION_LOG_GROUP: this.stateMachineLogGroup?.logGroupName ?? '',
                SETUP_FUNCTION_URL: this.setupUrl,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.minutes(3),
            ...this.extraLambdaProps,
        });
        const providers = this.providers.map(provider => provider.status(statusFunction));
        // expose providers as stack metadata as it's too big for Lambda environment variables
        // specifically integration testing got an error because lambda update request was >5kb
        const stack = cdk.Stack.of(this);
        const f = statusFunction.node.defaultChild;
        f.addPropertyOverride('Environment.Variables.LOGICAL_ID', f.logicalId);
        f.addPropertyOverride('Environment.Variables.STACK_NAME', stack.stackName);
        f.addMetadata('providers', providers);
        statusFunction.addToRolePolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['cloudformation:DescribeStackResource'],
            resources: [stack.stackId],
        }));
        this.secrets.webhook.grantRead(statusFunction);
        this.secrets.github.grantRead(statusFunction);
        this.secrets.githubPrivateKey.grantRead(statusFunction);
        this.secrets.setup.grantRead(statusFunction);
        this.orchestrator.grantRead(statusFunction);
        new cdk.CfnOutput(this, 'status command', {
            value: `aws --region ${stack.region} lambda invoke --function-name ${statusFunction.functionName} status.json`,
        });
    }
    setupFunction() {
        const setupFunction = new utils_1.BundledNodejsFunction(this, 'setup', {
            description: 'Setup GitHub Actions integration with self-hosted runners',
            environment: {
                SETUP_SECRET_ARN: this.secrets.setup.secretArn,
                WEBHOOK_SECRET_ARN: this.secrets.webhook.secretArn,
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                WEBHOOK_URL: this.webhook.url,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.minutes(3),
            ...this.extraLambdaProps,
        });
        // this.secrets.webhook.grantRead(setupFunction);
        this.secrets.webhook.grantWrite(setupFunction);
        this.secrets.github.grantRead(setupFunction);
        this.secrets.github.grantWrite(setupFunction);
        // this.secrets.githubPrivateKey.grantRead(setupFunction);
        this.secrets.githubPrivateKey.grantWrite(setupFunction);
        this.secrets.setup.grantRead(setupFunction);
        this.secrets.setup.grantWrite(setupFunction);
        return setupFunction.addFunctionUrl({ authType: aws_lambda_1.FunctionUrlAuthType.NONE }).url;
    }
    checkIntersectingLabels() {
        // this "algorithm" is very inefficient, but good enough for the tiny datasets we expect
        for (const p1 of this.providers) {
            for (const p2 of this.providers) {
                if (p1 == p2) {
                    continue;
                }
                if (p1.labels.every(l => p2.labels.includes(l))) {
                    if (p2.labels.every(l => p1.labels.includes(l))) {
                        throw new Error(`Both ${p1.node.path} and ${p2.node.path} use the same labels [${p1.labels.join(', ')}]`);
                    }
                    aws_cdk_lib_1.Annotations.of(p1).addWarning(`Labels [${p1.labels.join(', ')}] intersect with another provider (${p2.node.path} -- [${p2.labels.join(', ')}]). If a workflow specifies the labels [${p1.labels.join(', ')}], it is not guaranteed which provider will be used. It is recommended you do not use intersecting labels`);
                }
            }
        }
    }
}
exports.GitHubRunners = GitHubRunners;
_a = JSII_RTTI_SYMBOL_1;
GitHubRunners[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.GitHubRunners", version: "0.7.6" };
//# sourceMappingURL=data:application/json;base64,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