"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ec2Runner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const common_1 = require("./common");
const ami_1 = require("./image-builders/ami");
// this script is specifically made so `poweroff` is absolutely always called
// each `{}` is a variable coming from `params` below
const linuxUserDataTemplate = `#!/bin/bash -x
TASK_TOKEN="{}"
heartbeat () {
  while true; do
    aws stepfunctions send-task-heartbeat --task-token "$TASK_TOKEN"
    sleep 60
  done
}
setup_logs () {
  cat <<EOF > /tmp/log.conf || exit 1
  {
    "logs": {
      "log_stream_name": "unknown",
      "logs_collected": {
        "files": {
          "collect_list": [
            {
              "file_path": "/var/log/runner.log",
              "log_group_name": "{}",
              "log_stream_name": "{}",
              "timezone": "UTC"
            }
          ]
        }
      }
    }
  }
EOF
  /opt/aws/amazon-cloudwatch-agent/bin/amazon-cloudwatch-agent-ctl -a fetch-config -m ec2 -s -c file:/tmp/log.conf || exit 2
}
action () {
  sudo -Hu runner /home/runner/config.sh --unattended --url "https://{}/{}/{}" --token "{}" --ephemeral --work _work --labels "{}" {} --name "{}" || exit 1
  sudo --preserve-env=AWS_REGION -Hu runner /home/runner/run.sh || exit 2
}
heartbeat &
if setup_logs && action | tee /var/log/runner.log 2>&1; then
  aws stepfunctions send-task-success --task-token "$TASK_TOKEN" --task-output '{"ok": true}'
else
  aws stepfunctions send-task-failure --task-token "$TASK_TOKEN"
fi
poweroff
`.replace(/{/g, '\\{').replace(/}/g, '\\}').replace(/\\{\\}/g, '{}');
// this script is specifically made so `poweroff` is absolutely always called
// each `{}` is a variable coming from `params` below and their order should match the linux script
const windowsUserDataTemplate = `<powershell>
$TASK_TOKEN = "{}"
Start-Job -ScriptBlock {
  while (1) {
    aws stepfunctions send-task-heartbeat --task-token "$using:TASK_TOKEN"
    sleep 60
  }
}
function setup_logs () {
  echo '{
    "logs": {
      "log_stream_name": "unknown",
      "logs_collected": {
        "files": {
         "collect_list": [
            {
              "file_path": "/actions/runner.log",
              "log_group_name": "{}",
              "log_stream_name": "{}",
              "timezone": "UTC"
            }
          ]
        }
      }
    }
  }' | Out-File -Encoding ASCII $Env:TEMP/log.conf
  & "C:/Program Files/Amazon/AmazonCloudWatchAgent/amazon-cloudwatch-agent-ctl.ps1" -a fetch-config -m ec2 -s -c file:$Env:TEMP/log.conf
}
function action () {
  cd /actions
  ./config.cmd --unattended --url "https://{}/{}/{}" --token "{}" --ephemeral --work _work --labels "{}" {} --name "{}" 2>&1 | Out-File -Encoding ASCII -Append /actions/runner.log
  if ($LASTEXITCODE -ne 0) { return 1 }
  ./run.cmd 2>&1 | Out-File -Encoding ASCII -Append /actions/runner.log
  if ($LASTEXITCODE -ne 0) { return 2 }
  return 0
}
setup_logs
$r = action
if ($r -eq 0) {
  aws stepfunctions send-task-success --task-token "$TASK_TOKEN" --task-output '{ }'
} else {
  aws stepfunctions send-task-failure --task-token "$TASK_TOKEN"
}
Stop-Computer -ComputerName localhost -Force
</powershell>
`.replace(/{/g, '\\{').replace(/}/g, '\\}').replace(/\\{\\}/g, '{}');
/**
 * GitHub Actions runner provider using EC2 to execute jobs.
 *
 * This construct is not meant to be used by itself. It should be passed in the providers property for GitHubRunners.
 */
class Ec2Runner extends common_1.BaseProvider {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.labels = props?.labels ?? ['ec2'];
        this.vpc = props?.vpc ?? aws_cdk_lib_1.aws_ec2.Vpc.fromLookup(this, 'Default VPC', { isDefault: true });
        this.securityGroups = props?.securityGroup ? [props.securityGroup] : (props?.securityGroups ?? [new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'SG', { vpc: this.vpc })]);
        this.subnets = props?.subnet ? [props.subnet] : this.vpc.selectSubnets(props?.subnetSelection).subnets;
        this.instanceType = props?.instanceType ?? aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.M5, aws_cdk_lib_1.aws_ec2.InstanceSize.LARGE);
        this.storageSize = props?.storageSize ?? cdk.Size.gibibytes(30); // 30 is the minimum for Windows
        this.spot = props?.spot ?? false;
        this.spotMaxPrice = props?.spotMaxPrice;
        const amiBuilder = props?.amiBuilder ?? new ami_1.AmiBuilder(this, 'Image Builder', {
            vpc: props?.vpc,
            subnetSelection: props?.subnetSelection,
            securityGroups: this.securityGroups,
        });
        this.ami = amiBuilder.bind();
        if (!this.ami.architecture.instanceTypeMatch(this.instanceType)) {
            throw new Error(`AMI architecture (${this.ami.architecture.name}) doesn't match runner instance type (${this.instanceType} / ${this.instanceType.architecture})`);
        }
        this.grantPrincipal = this.role = new aws_cdk_lib_1.aws_iam.Role(this, 'Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
            ],
        });
        this.grantPrincipal.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['states:SendTaskFailure', 'states:SendTaskSuccess', 'states:SendTaskHeartbeat'],
            resources: ['*'],
        }));
        this.logGroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Logs', {
            retention: props?.logRetention ?? aws_logs_1.RetentionDays.ONE_MONTH,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        this.logGroup.grantWrite(this);
    }
    /**
     * Generate step function task(s) to start a new runner.
     *
     * Called by GithubRunners and shouldn't be called manually.
     *
     * @param parameters workflow job details
     */
    getStepFunctionTask(parameters) {
        // we need to build user data in two steps because passing the template as the first parameter to stepfunctions.JsonPath.format fails on syntax
        const params = [
            aws_cdk_lib_1.aws_stepfunctions.JsonPath.taskToken,
            this.logGroup.logGroupName,
            parameters.runnerNamePath,
            parameters.githubDomainPath,
            parameters.ownerPath,
            parameters.repoPath,
            parameters.runnerTokenPath,
            this.labels.join(','),
            this.ami.runnerVersion.is(common_1.RunnerVersion.latest()) ? '' : '--disableupdate',
            parameters.runnerNamePath,
        ];
        const passUserData = new aws_cdk_lib_1.aws_stepfunctions.Pass(this, `${this.labels.join(', ')} data`, {
            parameters: {
                userdataTemplate: this.ami.os.is(common_1.Os.WINDOWS) ? windowsUserDataTemplate : linuxUserDataTemplate,
            },
            resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.ec2'),
        });
        // we use ec2:RunInstances because we must
        // we can't use fleets because they don't let us override user data, security groups or even disk size
        // we can't use requestSpotInstances because it doesn't support launch templates, and it's deprecated
        // ec2:RunInstances also seemed like the only one to immediately return an error when spot capacity is not available
        // we build a complicated chain of states here because ec2:RunInstances can only try one subnet at a time
        // if someone can figure out a good way to use Map for this, please open a PR
        // build a state for each subnet we want to try
        const instanceProfile = new aws_cdk_lib_1.aws_iam.CfnInstanceProfile(this, 'Instance Profile', {
            roles: [this.role.roleName],
        });
        const subnetRunners = this.subnets.map((subnet, index) => {
            return new aws_cdk_lib_1.aws_stepfunctions_tasks.CallAwsService(this, `${this.labels.join(', ')} subnet${index + 1}`, {
                comment: subnet.subnetId,
                integrationPattern: aws_stepfunctions_1.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
                service: 'ec2',
                action: 'runInstances',
                heartbeat: aws_cdk_lib_1.Duration.minutes(10),
                parameters: {
                    LaunchTemplate: {
                        LaunchTemplateId: this.ami.launchTemplate.launchTemplateId,
                    },
                    MinCount: 1,
                    MaxCount: 1,
                    InstanceType: this.instanceType.toString(),
                    UserData: aws_cdk_lib_1.aws_stepfunctions.JsonPath.base64Encode(aws_cdk_lib_1.aws_stepfunctions.JsonPath.format(aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.ec2.userdataTemplate'), ...params)),
                    InstanceInitiatedShutdownBehavior: aws_cdk_lib_1.aws_ec2.InstanceInitiatedShutdownBehavior.TERMINATE,
                    IamInstanceProfile: {
                        Arn: instanceProfile.attrArn,
                    },
                    MetadataOptions: {
                        HttpTokens: 'required',
                    },
                    SecurityGroupIds: this.securityGroups.map(sg => sg.securityGroupId),
                    SubnetId: subnet.subnetId,
                    BlockDeviceMappings: [{
                            DeviceName: '/dev/sda1',
                            Ebs: {
                                DeleteOnTermination: true,
                                VolumeSize: this.storageSize.toGibibytes(),
                            },
                        }],
                    InstanceMarketOptions: this.spot ? {
                        MarketType: 'spot',
                        SpotOptions: {
                            MaxPrice: this.spotMaxPrice,
                            SpotInstanceType: 'one-time',
                        },
                    } : undefined,
                },
                iamResources: ['*'],
            });
        });
        // use Parallel, so we can easily retry this whole block on failure (only 1 branch)
        const subnetIterator = new aws_cdk_lib_1.aws_stepfunctions.Parallel(this, `${this.labels.join(', ')} subnet iterator`);
        // start with the first subnet
        subnetIterator.branch(subnetRunners[0]);
        // chain up the rest of the subnets
        for (let i = 1; i < subnetRunners.length; i++) {
            subnetRunners[i - 1].addCatch(subnetRunners[i], {
                errors: ['Ec2.Ec2Exception', 'States.Timeout'],
                resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.lastSubnetError'),
            });
        }
        // jump to the end state of the Parallel block when execution a runner succeeds
        const subnetIterationDone = new aws_cdk_lib_1.aws_stepfunctions.Succeed(this, `${this.labels.join(', ')} success`);
        for (const runner of subnetRunners) {
            runner.next(subnetIterationDone);
        }
        // retry the whole Parallel block if (only the last state) failed with an Ec2Exception or timed out
        this.addRetry(subnetIterator, ['Ec2.Ec2Exception', 'States.Timeout']);
        // return Parallel block
        return passUserData.next(subnetIterator);
    }
    grantStateMachine(stateMachineRole) {
        stateMachineRole.grantPrincipal.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [this.role.roleArn],
            conditions: {
                StringEquals: {
                    'iam:PassedToService': 'ec2.amazonaws.com',
                },
            },
        }));
        stateMachineRole.grantPrincipal.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['ec2:createTags'],
            resources: [aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: 'ec2',
                    resource: '*',
                })],
        }));
        stateMachineRole.grantPrincipal.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['iam:CreateServiceLinkedRole'],
            resources: ['*'],
            conditions: {
                StringEquals: {
                    'iam:AWSServiceName': 'spot.amazonaws.com',
                },
            },
        }));
    }
    status(statusFunctionRole) {
        statusFunctionRole.grantPrincipal.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['ec2:DescribeLaunchTemplateVersions'],
            resources: ['*'],
        }));
        return {
            type: this.constructor.name,
            labels: this.labels,
            securityGroups: this.securityGroups.map(sg => sg.securityGroupId),
            roleArn: this.role.roleArn,
            ami: {
                launchTemplate: this.ami.launchTemplate.launchTemplateId || 'unknown',
                amiBuilderLogGroup: this.ami.logGroup?.logGroupName,
            },
        };
    }
    /**
     * The network connections associated with this resource.
     */
    get connections() {
        return new aws_cdk_lib_1.aws_ec2.Connections({ securityGroups: this.securityGroups });
    }
}
exports.Ec2Runner = Ec2Runner;
_a = JSII_RTTI_SYMBOL_1;
Ec2Runner[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.Ec2Runner", version: "0.7.6" };
//# sourceMappingURL=data:application/json;base64,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