"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const have_resource_1 = require("@aws-cdk/assert/lib/assertions/have-resource");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('User Pool', () => {
    test('default setup', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool');
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPool', {
            AdminCreateUserConfig: {
                AllowAdminCreateUserOnly: true,
                InviteMessageTemplate: have_resource_1.ABSENT
            },
            EmailVerificationMessage: 'Hello {username}, Your verification code is {####}',
            EmailVerificationSubject: 'Verify your new account',
            SmsVerificationMessage: 'The verification code to your new account is {####}',
            VerificationMessageTemplate: {
                DefaultEmailOption: 'CONFIRM_WITH_CODE',
                EmailMessage: 'Hello {username}, Your verification code is {####}',
                EmailSubject: 'Verify your new account',
                SmsMessage: 'The verification code to your new account is {####}',
            },
            SmsConfiguration: {
                SnsCallerArn: {
                    'Fn::GetAtt': ['PoolsmsRoleC3352CE6', 'Arn'],
                },
                ExternalId: 'Pool'
            },
        });
        expect(stack).toHaveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: 'sts:AssumeRole',
                        Condition: {
                            StringEquals: {
                                'sts:ExternalId': 'Pool'
                            }
                        },
                        Effect: 'Allow',
                        Principal: {
                            Service: 'cognito-idp.amazonaws.com'
                        }
                    }
                ],
                Version: '2012-10-17'
            },
            Policies: [
                {
                    PolicyDocument: {
                        Statement: [
                            {
                                Action: 'sns:Publish',
                                Effect: 'Allow',
                                Resource: '*'
                            }
                        ],
                        Version: '2012-10-17'
                    },
                    PolicyName: 'sns-publish'
                }
            ]
        });
    });
    test('self sign up option is correctly configured', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            selfSignUpEnabled: true
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPool', {
            AdminCreateUserConfig: {
                AllowAdminCreateUserOnly: false
            }
        });
    });
    test('email verification via link is configured correctly', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            userVerification: {
                emailStyle: lib_1.VerificationEmailStyle.LINK
            }
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            EmailVerificationMessage: 'Hello {username}, Your verification code is {####}',
            EmailVerificationSubject: 'Verify your new account',
            VerificationMessageTemplate: {
                DefaultEmailOption: 'CONFIRM_WITH_LINK',
                EmailMessageByLink: 'Hello {username}, Your verification code is {####}',
                EmailSubjectByLink: 'Verify your new account',
            }
        });
    }),
        test('user invitation messages are configured correctly', () => {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.UserPool(stack, 'Pool', {
                userInvitation: {
                    emailBody: 'invitation email body',
                    emailSubject: 'invitation email subject',
                    smsMessage: 'invitation sms'
                }
            });
            // THEN
            expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
                AdminCreateUserConfig: {
                    InviteMessageTemplate: {
                        EmailMessage: 'invitation email body',
                        EmailSubject: 'invitation email subject',
                        SMSMessage: 'invitation sms'
                    }
                }
            });
        });
    test('smsRole property is recognized', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const role = aws_iam_1.Role.fromRoleArn(stack, 'smsRole', 'arn:aws:iam::664773442901:role/sms-role');
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            smsRole: role,
            smsRoleExternalId: 'test-external-id'
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            SmsConfiguration: {
                ExternalId: 'test-external-id',
                SnsCallerArn: role.roleArn
            }
        });
    });
    test('import using id', () => {
        // GIVEN
        const stack = new core_1.Stack(undefined, undefined, {
            env: { region: 'some-region-1', account: '0123456789012' }
        });
        const userPoolId = 'test-user-pool';
        // WHEN
        const pool = lib_1.UserPool.fromUserPoolId(stack, 'userpool', userPoolId);
        expect(pool.userPoolId).toEqual(userPoolId);
        expect(pool.userPoolArn).toMatch(/cognito-idp:some-region-1:0123456789012:userpool\/test-user-pool/);
    });
    test('import using arn', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const userPoolArn = 'arn:aws:cognito-idp:us-east-1:0123456789012:userpool/test-user-pool';
        // WHEN
        const pool = lib_1.UserPool.fromUserPoolArn(stack, 'userpool', userPoolArn);
        expect(pool.userPoolId).toEqual('test-user-pool');
        expect(pool.userPoolArn).toEqual(userPoolArn);
    });
    test('support tags', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const pool = new lib_1.UserPool(stack, 'Pool', {
            userPoolName: 'myPool',
        });
        core_1.Tag.add(pool, 'PoolTag', 'PoolParty');
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'myPool',
            UserPoolTags: {
                PoolTag: 'PoolParty',
            }
        });
    });
    test('lambda triggers are defined', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        // WHEN
        const pool = new lib_1.UserPool(stack, 'Pool', {
            lambdaTriggers: {
                preSignUp: fn
            }
        });
        pool.addCustomMessageTrigger(fn);
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            LambdaConfig: {
                PreSignUp: stack.resolve(fn.functionArn),
                CustomMessage: stack.resolve(fn.functionArn)
            }
        });
    });
    test('on* API correctly appends triggers', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const createAuthChallengeLambdaFn = new lambda.Function(stack, 'createAuthChallengeLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const customMessageLambdaFn = new lambda.Function(stack, 'customMessageLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const defineAuthChallengeLambdaFn = new lambda.Function(stack, 'defineAuthChallengeLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const postAuthenticationLambdaFn = new lambda.Function(stack, 'postAuthenticationLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const postConfirmationLambdaFn = new lambda.Function(stack, 'postConfirmationLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const preAuthenticationLambdaFn = new lambda.Function(stack, 'preAuthenticationLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const preSignUpLambdaFn = new lambda.Function(stack, 'preSignUpLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const preTokenGenerationLambdaFn = new lambda.Function(stack, 'preTokenGenerationLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const userMigrationLambdaFn = new lambda.Function(stack, 'userMigrationLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const verifyAuthChallengeResponseLambdaFn = new lambda.Function(stack, 'verifyAuthChallengeResponseLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        // WHEN
        const pool = new lib_1.UserPool(stack, 'Pool', {});
        pool.addCreateAuthChallengeTrigger(createAuthChallengeLambdaFn);
        pool.addCustomMessageTrigger(customMessageLambdaFn);
        pool.addDefineAuthChallengeTrigger(defineAuthChallengeLambdaFn);
        pool.addPostAuthenticationTrigger(postAuthenticationLambdaFn);
        pool.addPostConfirmationTrigger(postConfirmationLambdaFn);
        pool.addPreAuthenticationTrigger(preAuthenticationLambdaFn);
        pool.addPreSignUpTrigger(preSignUpLambdaFn);
        pool.addPreTokenGenerationTrigger(preTokenGenerationLambdaFn);
        pool.addUserMigrationTrigger(userMigrationLambdaFn);
        pool.addVerifyAuthChallengeResponseTrigger(verifyAuthChallengeResponseLambdaFn);
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            LambdaConfig: {
                CreateAuthChallenge: stack.resolve(createAuthChallengeLambdaFn.functionArn),
                CustomMessage: stack.resolve(customMessageLambdaFn.functionArn),
                DefineAuthChallenge: stack.resolve(defineAuthChallengeLambdaFn.functionArn),
                PostAuthentication: stack.resolve(postAuthenticationLambdaFn.functionArn),
                PostConfirmation: stack.resolve(postConfirmationLambdaFn.functionArn),
                PreAuthentication: stack.resolve(preAuthenticationLambdaFn.functionArn),
                PreSignUp: stack.resolve(preSignUpLambdaFn.functionArn),
                PreTokenGeneration: stack.resolve(preTokenGenerationLambdaFn.functionArn),
                UserMigration: stack.resolve(userMigrationLambdaFn.functionArn),
                VerifyAuthChallengeResponse: stack.resolve(verifyAuthChallengeResponseLambdaFn.functionArn)
            }
        });
    });
    test('lambdas are given cognito service grant', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            lambdaTriggers: {
                preSignUp: fn
            }
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Lambda::Permission', {
            FunctionName: stack.resolve(fn.functionArn),
            Principal: 'cognito-idp.amazonaws.com'
        });
    });
    test('no username aliases specified', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool');
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameAttributes: have_resource_1.ABSENT,
            AliasAttributes: have_resource_1.ABSENT,
        });
    });
    test('fails when preferredUsername is used without username', () => {
        const stack = new core_1.Stack();
        expect(() => new lib_1.UserPool(stack, 'Pool', {
            signInAliases: { preferredUsername: true }
        })).toThrow(/username/);
    });
    test('username and email are specified as the username aliases', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            signInAliases: { username: true, email: true }
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameAttributes: have_resource_1.ABSENT,
            AliasAttributes: ['email'],
        });
    });
    test('email and phone number are specified as the username aliases', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            signInAliases: { email: true, phone: true }
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameAttributes: ['email', 'phone_number'],
            AliasAttributes: have_resource_1.ABSENT,
        });
    });
    test('email and phone number are autoverified, by default, if they are specified as signIn', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool1', {
            userPoolName: 'Pool1',
            signInAliases: { email: true }
        });
        new lib_1.UserPool(stack, 'Pool2', {
            userPoolName: 'Pool2',
            signInAliases: { email: true, phone: true }
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool1',
            AutoVerifiedAttributes: ['email'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool2',
            AutoVerifiedAttributes: ['email', 'phone_number'],
        });
    });
    test('explicit autoverify are correctly picked up', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            signInAliases: { username: true },
            autoVerify: { email: true, phone: true },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            AutoVerifiedAttributes: ['email', 'phone_number'],
        });
    });
    test('required attributes', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            requiredAttributes: {
                fullname: true,
                timezone: true,
            }
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            Schema: [
                {
                    Name: 'name',
                    Required: true
                },
                {
                    Name: 'zoneinfo',
                    Required: true
                },
            ]
        });
    });
    test('schema is absent when required attributes are specified but as false', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool1', {
            userPoolName: 'Pool1',
        });
        new lib_1.UserPool(stack, 'Pool2', {
            userPoolName: 'Pool2',
            requiredAttributes: {
                familyName: false,
            }
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool1',
            Schema: have_resource_1.ABSENT
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool2',
            Schema: have_resource_1.ABSENT
        });
    });
    test('custom attributes with default constraints', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            customAttributes: {
                'custom-string-attr': new lib_1.StringAttribute(),
                'custom-number-attr': new lib_1.NumberAttribute(),
            }
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            Schema: [
                {
                    Name: 'custom-string-attr',
                    AttributeDataType: 'String',
                    StringAttributeConstraints: have_resource_1.ABSENT,
                    NumberAttributeConstraints: have_resource_1.ABSENT,
                },
                {
                    Name: 'custom-number-attr',
                    AttributeDataType: 'Number',
                    StringAttributeConstraints: have_resource_1.ABSENT,
                    NumberAttributeConstraints: have_resource_1.ABSENT,
                }
            ]
        });
    });
    test('custom attributes with constraints', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            customAttributes: {
                'custom-string-attr': new lib_1.StringAttribute({ minLen: 5, maxLen: 50 }),
                'custom-number-attr': new lib_1.NumberAttribute({ min: 500, max: 2000 }),
            }
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            Schema: [
                {
                    AttributeDataType: 'String',
                    Name: 'custom-string-attr',
                    StringAttributeConstraints: {
                        MaxLength: '50',
                        MinLength: '5',
                    }
                },
                {
                    AttributeDataType: 'Number',
                    Name: 'custom-number-attr',
                    NumberAttributeConstraints: {
                        MaxValue: '2000',
                        MinValue: '500',
                    }
                }
            ]
        });
    });
    test('mfaTypes is ignored when mfaEnforcement is undefined or set to OFF', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool1', {
            userPoolName: 'Pool1',
            mfaSecondFactor: {
                sms: true,
                otp: true,
            }
        });
        new lib_1.UserPool(stack, 'Pool2', {
            userPoolName: 'Pool2',
            mfa: lib_1.Mfa.OFF,
            mfaSecondFactor: {
                sms: true,
                otp: true,
            }
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool1',
            MfaConfiguration: have_resource_1.ABSENT,
            EnabledMfas: have_resource_1.ABSENT,
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool2',
            MfaConfiguration: 'OFF',
            EnabledMfas: have_resource_1.ABSENT,
        });
    });
    test('sms mfa type is the default when mfaEnforcement is set to REQUIRED or OPTIONAL', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool1', {
            userPoolName: 'Pool1',
            mfa: lib_1.Mfa.OPTIONAL,
        });
        new lib_1.UserPool(stack, 'Pool2', {
            userPoolName: 'Pool2',
            mfa: lib_1.Mfa.REQUIRED,
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool1',
            MfaConfiguration: 'OPTIONAL',
            EnabledMfas: ['SMS_MFA'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool2',
            MfaConfiguration: 'ON',
            EnabledMfas: ['SMS_MFA'],
        });
    });
    test('mfa type is correctly picked up when specified', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            mfa: lib_1.Mfa.REQUIRED,
            mfaSecondFactor: {
                sms: true,
                otp: true,
            }
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            EnabledMfas: ['SMS_MFA', 'SOFTWARE_TOKEN_MFA'],
        });
    });
    test('password policy is correctly set', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            passwordPolicy: {
                tempPasswordValidity: core_1.Duration.days(2),
                minLength: 15,
                requireDigits: true,
                requireLowercase: true,
                requireUppercase: true,
                requireSymbols: true,
            }
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            Policies: {
                PasswordPolicy: {
                    TemporaryPasswordValidityDays: 2,
                    MinimumLength: 15,
                    RequireLowercase: true,
                    RequireUppercase: true,
                    RequireNumbers: true,
                    RequireSymbols: true,
                },
            },
        });
    });
    test('throws when tempPassword validity is not in round days', () => {
        const stack = new core_1.Stack();
        expect(() => new lib_1.UserPool(stack, 'Pool', {
            passwordPolicy: {
                tempPasswordValidity: core_1.Duration.hours(30),
            }
        })).toThrow();
    });
    test('temp password throws an error when above the max', () => {
        const stack = new core_1.Stack();
        expect(() => new lib_1.UserPool(stack, 'Pool', {
            passwordPolicy: {
                tempPasswordValidity: core_1.Duration.days(400),
            }
        })).toThrow(/tempPasswordValidity cannot be greater than/);
    });
    test('throws when minLength is out of range', () => {
        const stack = new core_1.Stack();
        expect(() => new lib_1.UserPool(stack, 'Pool1', {
            passwordPolicy: {
                minLength: 5,
            },
        })).toThrow(/minLength for password must be between/);
        expect(() => new lib_1.UserPool(stack, 'Pool2', {
            passwordPolicy: {
                minLength: 100,
            },
        })).toThrow(/minLength for password must be between/);
    });
    test('email transmission settings are recognized correctly', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            emailSettings: {
                from: 'from@myawesomeapp.com',
                replyTo: 'replyTo@myawesomeapp.com'
            }
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            EmailConfiguration: {
                From: 'from@myawesomeapp.com',
                ReplyToEmailAddress: 'replyTo@myawesomeapp.com'
            }
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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