"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const cognito_generated_1 = require("./cognito.generated");
/**
 * The email verification style
 */
var VerificationEmailStyle;
(function (VerificationEmailStyle) {
    /** Verify email via code */
    VerificationEmailStyle["CODE"] = "CONFIRM_WITH_CODE";
    /** Verify email via link */
    VerificationEmailStyle["LINK"] = "CONFIRM_WITH_LINK";
})(VerificationEmailStyle = exports.VerificationEmailStyle || (exports.VerificationEmailStyle = {}));
/**
 * The different ways in which a user pool's MFA enforcement can be configured.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa.html
 */
var Mfa;
(function (Mfa) {
    /** Users are not required to use MFA for sign in, and cannot configure one. */
    Mfa["OFF"] = "OFF";
    /** Users are not required to use MFA for sign in, but can configure one if they so choose to. */
    Mfa["OPTIONAL"] = "OPTIONAL";
    /** Users are required to configure an MFA, and have to use it to sign in. */
    Mfa["REQUIRED"] = "ON";
})(Mfa = exports.Mfa || (exports.Mfa = {}));
/**
 * Define a Cognito User Pool
 */
class UserPool extends core_1.Resource {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
        super(scope, id);
        this.triggers = {};
        const signIn = this.signInConfiguration(props);
        if (props.lambdaTriggers) {
            for (const t of Object.keys(props.lambdaTriggers)) {
                const trigger = props.lambdaTriggers[t];
                if (trigger !== undefined) {
                    this.addLambdaPermission(trigger, t);
                    this.triggers[t] = trigger.functionArn;
                }
            }
        }
        const emailVerificationSubject = (_b = (_a = props.userVerification) === null || _a === void 0 ? void 0 : _a.emailSubject, (_b !== null && _b !== void 0 ? _b : 'Verify your new account'));
        const emailVerificationMessage = (_d = (_c = props.userVerification) === null || _c === void 0 ? void 0 : _c.emailBody, (_d !== null && _d !== void 0 ? _d : 'Hello {username}, Your verification code is {####}'));
        const smsVerificationMessage = (_f = (_e = props.userVerification) === null || _e === void 0 ? void 0 : _e.smsMessage, (_f !== null && _f !== void 0 ? _f : 'The verification code to your new account is {####}'));
        const defaultEmailOption = (_h = (_g = props.userVerification) === null || _g === void 0 ? void 0 : _g.emailStyle, (_h !== null && _h !== void 0 ? _h : VerificationEmailStyle.CODE));
        const verificationMessageTemplate = (defaultEmailOption === VerificationEmailStyle.CODE) ? {
            defaultEmailOption,
            emailMessage: emailVerificationMessage,
            emailSubject: emailVerificationSubject,
            smsMessage: smsVerificationMessage,
        } : {
            defaultEmailOption,
            emailMessageByLink: emailVerificationMessage,
            emailSubjectByLink: emailVerificationSubject,
            smsMessage: smsVerificationMessage
        };
        const inviteMessageTemplate = {
            emailMessage: (_j = props.userInvitation) === null || _j === void 0 ? void 0 : _j.emailBody,
            emailSubject: (_k = props.userInvitation) === null || _k === void 0 ? void 0 : _k.emailSubject,
            smsMessage: (_l = props.userInvitation) === null || _l === void 0 ? void 0 : _l.smsMessage,
        };
        const selfSignUpEnabled = props.selfSignUpEnabled !== undefined ? props.selfSignUpEnabled : false;
        const adminCreateUserConfig = {
            allowAdminCreateUserOnly: !selfSignUpEnabled,
            inviteMessageTemplate: props.userInvitation !== undefined ? inviteMessageTemplate : undefined,
        };
        const passwordPolicy = this.configurePasswordPolicy(props);
        const userPool = new cognito_generated_1.CfnUserPool(this, 'Resource', {
            userPoolName: props.userPoolName,
            usernameAttributes: signIn.usernameAttrs,
            aliasAttributes: signIn.aliasAttrs,
            autoVerifiedAttributes: signIn.autoVerifyAttrs,
            lambdaConfig: core_1.Lazy.anyValue({ produce: () => this.triggers }),
            smsConfiguration: this.smsConfiguration(props),
            adminCreateUserConfig,
            emailVerificationMessage,
            emailVerificationSubject,
            smsVerificationMessage,
            verificationMessageTemplate,
            schema: this.schemaConfiguration(props),
            mfaConfiguration: props.mfa,
            enabledMfas: this.mfaConfiguration(props),
            policies: passwordPolicy !== undefined ? { passwordPolicy } : undefined,
            emailConfiguration: undefinedIfNoKeys({
                from: (_m = props.emailSettings) === null || _m === void 0 ? void 0 : _m.from,
                replyToEmailAddress: (_o = props.emailSettings) === null || _o === void 0 ? void 0 : _o.replyTo,
            }),
        });
        this.userPoolId = userPool.ref;
        this.userPoolArn = userPool.attrArn;
        this.userPoolProviderName = userPool.attrProviderName;
        this.userPoolProviderUrl = userPool.attrProviderUrl;
    }
    /**
     * Import an existing user pool based on its id.
     */
    static fromUserPoolId(scope, id, userPoolId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.userPoolId = userPoolId;
                this.userPoolArn = core_1.Stack.of(this).formatArn({
                    service: 'cognito-idp',
                    resource: 'userpool',
                    resourceName: userPoolId,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing user pool based on its ARN.
     */
    static fromUserPoolArn(scope, id, userPoolArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.userPoolArn = userPoolArn;
                this.userPoolId = core_1.Stack.of(this).parseArn(userPoolArn).resourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Attach 'Create Auth Challenge' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-create-auth-challenge.html
     * @param fn the lambda function to attach
     */
    addCreateAuthChallengeTrigger(fn) {
        this.addLambdaPermission(fn, 'CreateAuthChallenge');
        this.triggers = { ...this.triggers, createAuthChallenge: fn.functionArn };
    }
    /**
     * Attach 'Custom Message' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-message.html
     * @param fn the lambda function to attach
     */
    addCustomMessageTrigger(fn) {
        this.addLambdaPermission(fn, 'CustomMessage');
        this.triggers = { ...this.triggers, customMessage: fn.functionArn };
    }
    /**
     * Attach 'Define Auth Challenge' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-define-auth-challenge.html
     * @param fn the lambda function to attach
     */
    addDefineAuthChallengeTrigger(fn) {
        this.addLambdaPermission(fn, 'DefineAuthChallenge');
        this.triggers = { ...this.triggers, defineAuthChallenge: fn.functionArn };
    }
    /**
     * Attach 'Post Authentication' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-authentication.html
     * @param fn the lambda function to attach
     */
    addPostAuthenticationTrigger(fn) {
        this.addLambdaPermission(fn, 'PostAuthentication');
        this.triggers = { ...this.triggers, postAuthentication: fn.functionArn };
    }
    /**
     * Attach 'Post Confirmation' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-confirmation.html
     * @param fn the lambda function to attach
     */
    addPostConfirmationTrigger(fn) {
        this.addLambdaPermission(fn, 'PostConfirmation');
        this.triggers = { ...this.triggers, postConfirmation: fn.functionArn };
    }
    /**
     * Attach 'Pre Authentication' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-authentication.html
     * @param fn the lambda function to attach
     */
    addPreAuthenticationTrigger(fn) {
        this.addLambdaPermission(fn, 'PreAuthentication');
        this.triggers = { ...this.triggers, preAuthentication: fn.functionArn };
    }
    /**
     * Attach 'Pre Sign Up' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html
     * @param fn the lambda function to attach
     */
    addPreSignUpTrigger(fn) {
        this.addLambdaPermission(fn, 'PreSignUp');
        this.triggers = { ...this.triggers, preSignUp: fn.functionArn };
    }
    /**
     * Attach 'Pre Token Generation' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-token-generation.html
     * @param fn the lambda function to attach
     */
    addPreTokenGenerationTrigger(fn) {
        this.addLambdaPermission(fn, 'PreTokenGeneration');
        this.triggers = { ...this.triggers, preTokenGeneration: fn.functionArn };
    }
    /**
     * Attach 'User Migration' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-migrate-user.html
     * @param fn the lambda function to attach
     */
    addUserMigrationTrigger(fn) {
        this.addLambdaPermission(fn, 'UserMigration');
        this.triggers = { ...this.triggers, userMigration: fn.functionArn };
    }
    /**
     * Attach 'Verify Auth Challenge Response' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-verify-auth-challenge-response.html
     * @param fn the lambda function to attach
     */
    addVerifyAuthChallengeResponseTrigger(fn) {
        this.addLambdaPermission(fn, 'VerifyAuthChallengeResponse');
        this.triggers = { ...this.triggers, verifyAuthChallengeResponse: fn.functionArn };
    }
    addLambdaPermission(fn, name) {
        const normalize = name.charAt(0).toUpperCase() + name.slice(1);
        fn.addPermission(`${normalize}Cognito`, {
            principal: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com'),
            sourceArn: this.userPoolArn
        });
    }
    signInConfiguration(props) {
        var _a;
        let aliasAttrs;
        let usernameAttrs;
        let autoVerifyAttrs;
        const signIn = (_a = props.signInAliases, (_a !== null && _a !== void 0 ? _a : { username: true }));
        if (signIn.preferredUsername && !signIn.username) {
            throw new Error('username signIn must be enabled if preferredUsername is enabled');
        }
        if (signIn.username) {
            aliasAttrs = [];
            if (signIn.email) {
                aliasAttrs.push("email" /* EMAIL */);
            }
            if (signIn.phone) {
                aliasAttrs.push("phone_number" /* PHONE_NUMBER */);
            }
            if (signIn.preferredUsername) {
                aliasAttrs.push("preferred_username" /* PREFERRED_USERNAME */);
            }
            if (aliasAttrs.length === 0) {
                aliasAttrs = undefined;
            }
        }
        else {
            usernameAttrs = [];
            if (signIn.email) {
                usernameAttrs.push("email" /* EMAIL */);
            }
            if (signIn.phone) {
                usernameAttrs.push("phone_number" /* PHONE_NUMBER */);
            }
        }
        if (props.autoVerify) {
            autoVerifyAttrs = [];
            if (props.autoVerify.email) {
                autoVerifyAttrs.push("email" /* EMAIL */);
            }
            if (props.autoVerify.phone) {
                autoVerifyAttrs.push("phone_number" /* PHONE_NUMBER */);
            }
        }
        else if (signIn.email || signIn.phone) {
            autoVerifyAttrs = [];
            if (signIn.email) {
                autoVerifyAttrs.push("email" /* EMAIL */);
            }
            if (signIn.phone) {
                autoVerifyAttrs.push("phone_number" /* PHONE_NUMBER */);
            }
        }
        return { usernameAttrs, aliasAttrs, autoVerifyAttrs };
    }
    smsConfiguration(props) {
        var _a;
        if (props.smsRole) {
            return {
                snsCallerArn: props.smsRole.roleArn,
                externalId: props.smsRoleExternalId
            };
        }
        else {
            const smsRoleExternalId = this.node.uniqueId.substr(0, 1223); // sts:ExternalId max length of 1224
            const smsRole = (_a = props.smsRole, (_a !== null && _a !== void 0 ? _a : new aws_iam_1.Role(this, 'smsRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com', {
                    conditions: {
                        StringEquals: { 'sts:ExternalId': smsRoleExternalId }
                    }
                }),
                inlinePolicies: {
                    /*
                     * The UserPool is very particular that it must contain an 'sns:Publish' action as an inline policy.
                     * Ideally, a conditional that restricts this action to 'sms' protocol needs to be attached, but the UserPool deployment fails validation.
                     * Seems like a case of being excessively strict.
                     */
                    'sns-publish': new aws_iam_1.PolicyDocument({
                        statements: [
                            new aws_iam_1.PolicyStatement({
                                actions: ['sns:Publish'],
                                resources: ['*'],
                            })
                        ]
                    })
                }
            })));
            return {
                externalId: smsRoleExternalId,
                snsCallerArn: smsRole.roleArn
            };
        }
    }
    mfaConfiguration(props) {
        if (props.mfa === undefined || props.mfa === Mfa.OFF) {
            // since default is OFF, treat undefined and OFF the same way
            return undefined;
        }
        else if (props.mfaSecondFactor === undefined &&
            (props.mfa === Mfa.OPTIONAL || props.mfa === Mfa.REQUIRED)) {
            return ['SMS_MFA'];
        }
        else {
            const enabledMfas = [];
            if (props.mfaSecondFactor.sms) {
                enabledMfas.push('SMS_MFA');
            }
            if (props.mfaSecondFactor.otp) {
                enabledMfas.push('SOFTWARE_TOKEN_MFA');
            }
            return enabledMfas;
        }
    }
    configurePasswordPolicy(props) {
        var _a, _b, _c, _d, _e, _f, _g;
        const tempPasswordValidity = (_a = props.passwordPolicy) === null || _a === void 0 ? void 0 : _a.tempPasswordValidity;
        if (tempPasswordValidity !== undefined && tempPasswordValidity.toDays() > core_1.Duration.days(365).toDays()) {
            throw new Error(`tempPasswordValidity cannot be greater than 365 days (received: ${tempPasswordValidity.toDays()})`);
        }
        const minLength = (_b = props.passwordPolicy) === null || _b === void 0 ? void 0 : _b.minLength;
        if (minLength !== undefined && (minLength < 6 || minLength > 99)) {
            throw new Error(`minLength for password must be between 6 and 99 (received: ${minLength})`);
        }
        return undefinedIfNoKeys({
            temporaryPasswordValidityDays: (_c = tempPasswordValidity) === null || _c === void 0 ? void 0 : _c.toDays({ integral: true }),
            minimumLength: minLength,
            requireLowercase: (_d = props.passwordPolicy) === null || _d === void 0 ? void 0 : _d.requireLowercase,
            requireUppercase: (_e = props.passwordPolicy) === null || _e === void 0 ? void 0 : _e.requireUppercase,
            requireNumbers: (_f = props.passwordPolicy) === null || _f === void 0 ? void 0 : _f.requireDigits,
            requireSymbols: (_g = props.passwordPolicy) === null || _g === void 0 ? void 0 : _g.requireSymbols,
        });
    }
    schemaConfiguration(props) {
        const schema = [];
        if (props.requiredAttributes) {
            const stdAttributes = [];
            if (props.requiredAttributes.address) {
                stdAttributes.push("address" /* ADDRESS */);
            }
            if (props.requiredAttributes.birthdate) {
                stdAttributes.push("birthdate" /* BIRTHDATE */);
            }
            if (props.requiredAttributes.email) {
                stdAttributes.push("email" /* EMAIL */);
            }
            if (props.requiredAttributes.familyName) {
                stdAttributes.push("family_name" /* FAMILY_NAME */);
            }
            if (props.requiredAttributes.fullname) {
                stdAttributes.push("name" /* NAME */);
            }
            if (props.requiredAttributes.gender) {
                stdAttributes.push("gender" /* GENDER */);
            }
            if (props.requiredAttributes.givenName) {
                stdAttributes.push("given_name" /* GIVEN_NAME */);
            }
            if (props.requiredAttributes.lastUpdateTime) {
                stdAttributes.push("updated_at" /* LAST_UPDATE_TIME */);
            }
            if (props.requiredAttributes.locale) {
                stdAttributes.push("locale" /* LOCALE */);
            }
            if (props.requiredAttributes.middleName) {
                stdAttributes.push("middle_name" /* MIDDLE_NAME */);
            }
            if (props.requiredAttributes.nickname) {
                stdAttributes.push("nickname" /* NICKNAME */);
            }
            if (props.requiredAttributes.phoneNumber) {
                stdAttributes.push("phone_number" /* PHONE_NUMBER */);
            }
            if (props.requiredAttributes.preferredUsername) {
                stdAttributes.push("preferred_username" /* PREFERRED_USERNAME */);
            }
            if (props.requiredAttributes.profilePage) {
                stdAttributes.push("profile" /* PROFILE_URL */);
            }
            if (props.requiredAttributes.profilePicture) {
                stdAttributes.push("picture" /* PICTURE_URL */);
            }
            if (props.requiredAttributes.timezone) {
                stdAttributes.push("zoneinfo" /* TIMEZONE */);
            }
            if (props.requiredAttributes.website) {
                stdAttributes.push("website" /* WEBSITE */);
            }
            schema.push(...stdAttributes.map((attr) => {
                return { name: attr, required: true };
            }));
        }
        if (props.customAttributes) {
            const customAttrs = Object.keys(props.customAttributes).map((attrName) => {
                var _a, _b, _c, _d, _e, _f, _g, _h;
                const attrConfig = props.customAttributes[attrName].bind();
                const numberConstraints = {
                    minValue: (_b = (_a = attrConfig.numberConstraints) === null || _a === void 0 ? void 0 : _a.min) === null || _b === void 0 ? void 0 : _b.toString(),
                    maxValue: (_d = (_c = attrConfig.numberConstraints) === null || _c === void 0 ? void 0 : _c.max) === null || _d === void 0 ? void 0 : _d.toString(),
                };
                const stringConstraints = {
                    minLength: (_f = (_e = attrConfig.stringConstraints) === null || _e === void 0 ? void 0 : _e.minLen) === null || _f === void 0 ? void 0 : _f.toString(),
                    maxLength: (_h = (_g = attrConfig.stringConstraints) === null || _g === void 0 ? void 0 : _g.maxLen) === null || _h === void 0 ? void 0 : _h.toString(),
                };
                return {
                    name: attrName,
                    attributeDataType: attrConfig.dataType,
                    numberAttributeConstraints: (attrConfig.numberConstraints) ? numberConstraints : undefined,
                    stringAttributeConstraints: (attrConfig.stringConstraints) ? stringConstraints : undefined,
                };
            });
            schema.push(...customAttrs);
        }
        if (schema.length === 0) {
            return undefined;
        }
        return schema;
    }
}
exports.UserPool = UserPool;
function undefinedIfNoKeys(struct) {
    const allUndefined = Object.values(struct).reduce((acc, v) => acc && (v === undefined), true);
    return allUndefined ? undefined : struct;
}
//# sourceMappingURL=data:application/json;base64,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