"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Capture = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const _1 = require(".");
const matcher_1 = require("./matcher");
const type_1 = require("./private/type");
/**
 * Capture values while matching templates.
 *
 * Using an instance of this class within a Matcher will capture the matching value.
 * The `as*()` APIs on the instance can be used to get the captured value.
 *
 * @stability stable
 */
class Capture extends matcher_1.Matcher {
    /**
     * Initialize a new capture.
     *
     * @param pattern a nested pattern or Matcher.
     * @stability stable
     */
    constructor(pattern) {
        super();
        this.pattern = pattern;
        /** @internal */
        this._captured = [];
        this.idx = 0;
        this.name = 'Capture';
    }
    /**
     * Test whether a target matches the provided pattern.
     *
     * Every Matcher must implement this method.
     * This method will be invoked by the assertions framework. Do not call this method directly.
     *
     * @stability stable
     */
    test(actual) {
        const result = new matcher_1.MatchResult(actual);
        if (actual == null) {
            return result.recordFailure({
                matcher: this,
                path: [],
                message: `Can only capture non-nullish values. Found ${actual}`,
            });
        }
        if (this.pattern !== undefined) {
            const innerMatcher = matcher_1.Matcher.isMatcher(this.pattern) ? this.pattern : _1.Match.objectLike(this.pattern);
            const innerResult = innerMatcher.test(actual);
            if (innerResult.hasFailed()) {
                return innerResult;
            }
        }
        result.recordCapture({ capture: this, value: actual });
        return result;
    }
    /**
     * When multiple results are captured, move the iterator to the next result.
     *
     * @returns true if another capture is present, false otherwise
     * @stability stable
     */
    next() {
        if (this.idx < this._captured.length - 1) {
            this.idx++;
            return true;
        }
        return false;
    }
    /**
     * Retrieve the captured value as a string.
     *
     * An error is generated if no value is captured or if the value is not a string.
     *
     * @stability stable
     */
    asString() {
        this.validate();
        const val = this._captured[this.idx];
        if (type_1.getType(val) === 'string') {
            return val;
        }
        this.reportIncorrectType('string');
    }
    /**
     * Retrieve the captured value as a number.
     *
     * An error is generated if no value is captured or if the value is not a number.
     *
     * @stability stable
     */
    asNumber() {
        this.validate();
        const val = this._captured[this.idx];
        if (type_1.getType(val) === 'number') {
            return val;
        }
        this.reportIncorrectType('number');
    }
    /**
     * Retrieve the captured value as a boolean.
     *
     * An error is generated if no value is captured or if the value is not a boolean.
     *
     * @stability stable
     */
    asBoolean() {
        this.validate();
        const val = this._captured[this.idx];
        if (type_1.getType(val) === 'boolean') {
            return val;
        }
        this.reportIncorrectType('boolean');
    }
    /**
     * Retrieve the captured value as an array.
     *
     * An error is generated if no value is captured or if the value is not an array.
     *
     * @stability stable
     */
    asArray() {
        this.validate();
        const val = this._captured[this.idx];
        if (type_1.getType(val) === 'array') {
            return val;
        }
        this.reportIncorrectType('array');
    }
    /**
     * Retrieve the captured value as a JSON object.
     *
     * An error is generated if no value is captured or if the value is not an object.
     *
     * @stability stable
     */
    asObject() {
        this.validate();
        const val = this._captured[this.idx];
        if (type_1.getType(val) === 'object') {
            return val;
        }
        this.reportIncorrectType('object');
    }
    validate() {
        if (this._captured.length === 0) {
            throw new Error('No value captured');
        }
    }
    reportIncorrectType(expected) {
        throw new Error(`Captured value is expected to be ${expected} but found ${type_1.getType(this._captured[this.idx])}. ` +
            `Value is ${JSON.stringify(this._captured[this.idx], undefined, 2)}`);
    }
}
exports.Capture = Capture;
_a = JSII_RTTI_SYMBOL_1;
Capture[_a] = { fqn: "@aws-cdk/assertions.Capture", version: "1.138.2" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2FwdHVyZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImNhcHR1cmUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSx3QkFBMEI7QUFDMUIsdUNBQWlEO0FBQ2pELHlDQUErQzs7Ozs7Ozs7O0FBRy9DLE1BQWEsT0FBUSxTQUFRLGlCQUFPOzs7Ozs7O0lBT2xDLFlBQTZCLE9BQWE7UUFDeEMsS0FBSyxFQUFFLENBQUM7UUFEbUIsWUFBTyxHQUFQLE9BQU8sQ0FBTTtRQUwxQyxnQkFBZ0I7UUFDVCxjQUFTLEdBQVUsRUFBRSxDQUFDO1FBQ3JCLFFBQUcsR0FBRyxDQUFDLENBQUM7UUFLZCxJQUFJLENBQUMsSUFBSSxHQUFHLFNBQVMsQ0FBQztLQUN2Qjs7Ozs7Ozs7O0lBRU0sSUFBSSxDQUFDLE1BQVc7UUFDckIsTUFBTSxNQUFNLEdBQUcsSUFBSSxxQkFBVyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ3ZDLElBQUksTUFBTSxJQUFJLElBQUksRUFBRTtZQUNsQixPQUFPLE1BQU0sQ0FBQyxhQUFhLENBQUM7Z0JBQzFCLE9BQU8sRUFBRSxJQUFJO2dCQUNiLElBQUksRUFBRSxFQUFFO2dCQUNSLE9BQU8sRUFBRSw4Q0FBOEMsTUFBTSxFQUFFO2FBQ2hFLENBQUMsQ0FBQztTQUNKO1FBRUQsSUFBSSxJQUFJLENBQUMsT0FBTyxLQUFLLFNBQVMsRUFBRTtZQUM5QixNQUFNLFlBQVksR0FBRyxpQkFBTyxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLFFBQUssQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1lBQ3JHLE1BQU0sV0FBVyxHQUFHLFlBQVksQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDOUMsSUFBSSxXQUFXLENBQUMsU0FBUyxFQUFFLEVBQUU7Z0JBQzNCLE9BQU8sV0FBVyxDQUFDO2FBQ3BCO1NBQ0Y7UUFFRCxNQUFNLENBQUMsYUFBYSxDQUFDLEVBQUUsT0FBTyxFQUFFLElBQUksRUFBRSxLQUFLLEVBQUUsTUFBTSxFQUFFLENBQUMsQ0FBQztRQUN2RCxPQUFPLE1BQU0sQ0FBQztLQUNmOzs7Ozs7O0lBR00sSUFBSTtRQUNULElBQUksSUFBSSxDQUFDLEdBQUcsR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDeEMsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDO1lBQ1gsT0FBTyxJQUFJLENBQUM7U0FDYjtRQUNELE9BQU8sS0FBSyxDQUFDO0tBQ2Q7Ozs7Ozs7O0lBR00sUUFBUTtRQUNiLElBQUksQ0FBQyxRQUFRLEVBQUUsQ0FBQztRQUNoQixNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUNyQyxJQUFJLGNBQU8sQ0FBQyxHQUFHLENBQUMsS0FBSyxRQUFRLEVBQUU7WUFDN0IsT0FBTyxHQUFHLENBQUM7U0FDWjtRQUNELElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxRQUFRLENBQUMsQ0FBQztLQUNwQzs7Ozs7Ozs7SUFHTSxRQUFRO1FBQ2IsSUFBSSxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBQ2hCLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ3JDLElBQUksY0FBTyxDQUFDLEdBQUcsQ0FBQyxLQUFLLFFBQVEsRUFBRTtZQUM3QixPQUFPLEdBQUcsQ0FBQztTQUNaO1FBQ0QsSUFBSSxDQUFDLG1CQUFtQixDQUFDLFFBQVEsQ0FBQyxDQUFDO0tBQ3BDOzs7Ozs7OztJQUdNLFNBQVM7UUFDZCxJQUFJLENBQUMsUUFBUSxFQUFFLENBQUM7UUFDaEIsTUFBTSxHQUFHLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDckMsSUFBSSxjQUFPLENBQUMsR0FBRyxDQUFDLEtBQUssU0FBUyxFQUFFO1lBQzlCLE9BQU8sR0FBRyxDQUFDO1NBQ1o7UUFDRCxJQUFJLENBQUMsbUJBQW1CLENBQUMsU0FBUyxDQUFDLENBQUM7S0FDckM7Ozs7Ozs7O0lBR00sT0FBTztRQUNaLElBQUksQ0FBQyxRQUFRLEVBQUUsQ0FBQztRQUNoQixNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUNyQyxJQUFJLGNBQU8sQ0FBQyxHQUFHLENBQUMsS0FBSyxPQUFPLEVBQUU7WUFDNUIsT0FBTyxHQUFHLENBQUM7U0FDWjtRQUNELElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxPQUFPLENBQUMsQ0FBQztLQUNuQzs7Ozs7Ozs7SUFHTSxRQUFRO1FBQ2IsSUFBSSxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBQ2hCLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ3JDLElBQUksY0FBTyxDQUFDLEdBQUcsQ0FBQyxLQUFLLFFBQVEsRUFBRTtZQUM3QixPQUFPLEdBQUcsQ0FBQztTQUNaO1FBQ0QsSUFBSSxDQUFDLG1CQUFtQixDQUFDLFFBQVEsQ0FBQyxDQUFDO0tBQ3BDO0lBRU8sUUFBUTtRQUNkLElBQUksSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQy9CLE1BQU0sSUFBSSxLQUFLLENBQUMsbUJBQW1CLENBQUMsQ0FBQztTQUN0QztLQUNGO0lBRU8sbUJBQW1CLENBQUMsUUFBYztRQUN4QyxNQUFNLElBQUksS0FBSyxDQUFDLG9DQUFvQyxRQUFRLGNBQWMsY0FBTyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLElBQUk7WUFDN0csWUFBWSxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLFNBQVMsRUFBRSxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUM7S0FDekU7O0FBdEdILDBCQXVHQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IE1hdGNoIH0gZnJvbSAnLic7XG5pbXBvcnQgeyBNYXRjaGVyLCBNYXRjaFJlc3VsdCB9IGZyb20gJy4vbWF0Y2hlcic7XG5pbXBvcnQgeyBUeXBlLCBnZXRUeXBlIH0gZnJvbSAnLi9wcml2YXRlL3R5cGUnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgQ2FwdHVyZSBleHRlbmRzIE1hdGNoZXIge1xuICBwdWJsaWMgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuICAvKiogQGludGVybmFsICovXG4gIHB1YmxpYyBfY2FwdHVyZWQ6IGFueVtdID0gW107XG4gIHByaXZhdGUgaWR4ID0gMDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IHBhdHRlcm4/OiBhbnkpIHtcbiAgICBzdXBlcigpO1xuICAgIHRoaXMubmFtZSA9ICdDYXB0dXJlJztcbiAgfVxuXG4gIHB1YmxpYyB0ZXN0KGFjdHVhbDogYW55KTogTWF0Y2hSZXN1bHQge1xuICAgIGNvbnN0IHJlc3VsdCA9IG5ldyBNYXRjaFJlc3VsdChhY3R1YWwpO1xuICAgIGlmIChhY3R1YWwgPT0gbnVsbCkge1xuICAgICAgcmV0dXJuIHJlc3VsdC5yZWNvcmRGYWlsdXJlKHtcbiAgICAgICAgbWF0Y2hlcjogdGhpcyxcbiAgICAgICAgcGF0aDogW10sXG4gICAgICAgIG1lc3NhZ2U6IGBDYW4gb25seSBjYXB0dXJlIG5vbi1udWxsaXNoIHZhbHVlcy4gRm91bmQgJHthY3R1YWx9YCxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIGlmICh0aGlzLnBhdHRlcm4gIT09IHVuZGVmaW5lZCkge1xuICAgICAgY29uc3QgaW5uZXJNYXRjaGVyID0gTWF0Y2hlci5pc01hdGNoZXIodGhpcy5wYXR0ZXJuKSA/IHRoaXMucGF0dGVybiA6IE1hdGNoLm9iamVjdExpa2UodGhpcy5wYXR0ZXJuKTtcbiAgICAgIGNvbnN0IGlubmVyUmVzdWx0ID0gaW5uZXJNYXRjaGVyLnRlc3QoYWN0dWFsKTtcbiAgICAgIGlmIChpbm5lclJlc3VsdC5oYXNGYWlsZWQoKSkge1xuICAgICAgICByZXR1cm4gaW5uZXJSZXN1bHQ7XG4gICAgICB9XG4gICAgfVxuXG4gICAgcmVzdWx0LnJlY29yZENhcHR1cmUoeyBjYXB0dXJlOiB0aGlzLCB2YWx1ZTogYWN0dWFsIH0pO1xuICAgIHJldHVybiByZXN1bHQ7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBuZXh0KCk6IGJvb2xlYW4ge1xuICAgIGlmICh0aGlzLmlkeCA8IHRoaXMuX2NhcHR1cmVkLmxlbmd0aCAtIDEpIHtcbiAgICAgIHRoaXMuaWR4Kys7XG4gICAgICByZXR1cm4gdHJ1ZTtcbiAgICB9XG4gICAgcmV0dXJuIGZhbHNlO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYXNTdHJpbmcoKTogc3RyaW5nIHtcbiAgICB0aGlzLnZhbGlkYXRlKCk7XG4gICAgY29uc3QgdmFsID0gdGhpcy5fY2FwdHVyZWRbdGhpcy5pZHhdO1xuICAgIGlmIChnZXRUeXBlKHZhbCkgPT09ICdzdHJpbmcnKSB7XG4gICAgICByZXR1cm4gdmFsO1xuICAgIH1cbiAgICB0aGlzLnJlcG9ydEluY29ycmVjdFR5cGUoJ3N0cmluZycpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYXNOdW1iZXIoKTogbnVtYmVyIHtcbiAgICB0aGlzLnZhbGlkYXRlKCk7XG4gICAgY29uc3QgdmFsID0gdGhpcy5fY2FwdHVyZWRbdGhpcy5pZHhdO1xuICAgIGlmIChnZXRUeXBlKHZhbCkgPT09ICdudW1iZXInKSB7XG4gICAgICByZXR1cm4gdmFsO1xuICAgIH1cbiAgICB0aGlzLnJlcG9ydEluY29ycmVjdFR5cGUoJ251bWJlcicpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhc0Jvb2xlYW4oKTogYm9vbGVhbiB7XG4gICAgdGhpcy52YWxpZGF0ZSgpO1xuICAgIGNvbnN0IHZhbCA9IHRoaXMuX2NhcHR1cmVkW3RoaXMuaWR4XTtcbiAgICBpZiAoZ2V0VHlwZSh2YWwpID09PSAnYm9vbGVhbicpIHtcbiAgICAgIHJldHVybiB2YWw7XG4gICAgfVxuICAgIHRoaXMucmVwb3J0SW5jb3JyZWN0VHlwZSgnYm9vbGVhbicpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYXNBcnJheSgpOiBhbnlbXSB7XG4gICAgdGhpcy52YWxpZGF0ZSgpO1xuICAgIGNvbnN0IHZhbCA9IHRoaXMuX2NhcHR1cmVkW3RoaXMuaWR4XTtcbiAgICBpZiAoZ2V0VHlwZSh2YWwpID09PSAnYXJyYXknKSB7XG4gICAgICByZXR1cm4gdmFsO1xuICAgIH1cbiAgICB0aGlzLnJlcG9ydEluY29ycmVjdFR5cGUoJ2FycmF5Jyk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhc09iamVjdCgpOiB7IFtrZXk6IHN0cmluZ106IGFueSB9IHtcbiAgICB0aGlzLnZhbGlkYXRlKCk7XG4gICAgY29uc3QgdmFsID0gdGhpcy5fY2FwdHVyZWRbdGhpcy5pZHhdO1xuICAgIGlmIChnZXRUeXBlKHZhbCkgPT09ICdvYmplY3QnKSB7XG4gICAgICByZXR1cm4gdmFsO1xuICAgIH1cbiAgICB0aGlzLnJlcG9ydEluY29ycmVjdFR5cGUoJ29iamVjdCcpO1xuICB9XG5cbiAgcHJpdmF0ZSB2YWxpZGF0ZSgpOiB2b2lkIHtcbiAgICBpZiAodGhpcy5fY2FwdHVyZWQubGVuZ3RoID09PSAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ05vIHZhbHVlIGNhcHR1cmVkJyk7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSByZXBvcnRJbmNvcnJlY3RUeXBlKGV4cGVjdGVkOiBUeXBlKTogbmV2ZXIge1xuICAgIHRocm93IG5ldyBFcnJvcihgQ2FwdHVyZWQgdmFsdWUgaXMgZXhwZWN0ZWQgdG8gYmUgJHtleHBlY3RlZH0gYnV0IGZvdW5kICR7Z2V0VHlwZSh0aGlzLl9jYXB0dXJlZFt0aGlzLmlkeF0pfS4gYCArXG4gICAgICBgVmFsdWUgaXMgJHtKU09OLnN0cmluZ2lmeSh0aGlzLl9jYXB0dXJlZFt0aGlzLmlkeF0sIHVuZGVmaW5lZCwgMil9YCk7XG4gIH1cbn0iXX0=