## Autocomplete

The autocomplete is a normal text input enhanced by a panel of suggested options.

You can treat ```Autocomplete``` as a advanced select, it provides all the features of select, and more.

The only drawback compared to select is that it requires a more complex data structure of options.

Due to limitation of python typing, we split the select component into `Autocomplete` and `MultipleAutocomplete`.

see [MUI AutoComplete](https://mui.com/material-ui/react-autocomplete/) for more details.

### Basic Usage

We need to provide a list of options to the select component. The options must be a dict that contains at least a label (this is different with DataGrid which use id instead of label by default).

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        md = mui.Markdown()
        options = [
            {"label": "Option 1"},
            {"label": "Option 2"},
            {"label": "Option 3"},
        ]
        return mui.VBox([
            mui.Autocomplete("Autocomplete", options, callback=lambda x: md.write(f"Select: :green[{x}]")).prop(muiMargin="dense"),
            md,
        ]).prop(width="100%", justifyContent="space-between")

```

variant, size and ```muiMargin``` are same as select.

### MultipleAutocomplete

MultipleAutocomplete is a select component that support multiple selection.

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        md = mui.Markdown()
        options = [
            {"label": "Option 1"},
            {"label": "Option 2"},
            {"label": "Option 3"},
        ]
        return mui.VBox([
            mui.MultipleAutocomplete("Autocomplete", options, callback=lambda x: md.write(f"Select: :green[{x}]")).prop(muiMargin="dense"),
            md,
        ]).prop(width="100%", justifyContent="space-between")

```

* checkbox

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        md = mui.Markdown()
        options = [
            {"label": "Option 1"},
            {"label": "Option 2"},
            {"label": "Option 3"},
        ]
        return mui.VBox([
            mui.MultipleAutocomplete("Autocomplete", options, callback=lambda x: md.write(f"Select: :green[{x}]")).prop(muiMargin="dense", itemVariant="checkbox"),
            md,
        ]).prop(width="100%", justifyContent="space-between")
```

### Add Option

```mui.Autocomplete``` support a powerful feature that can add option on the fly.

```Python

from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        md = mui.Markdown()
        options = [
            {"label": "Option 1"},
            {"label": "Option 2"},
            {"label": "Option 3"},
        ]
        autocomplete = mui.Autocomplete("Autocomplete", options, callback=lambda x: md.write(f"Select: :green[{x}]")).prop(muiMargin="dense", **mui.Autocomplete.get_creatable_option())
        autocomplete.event_select_new_item.on(self._on_new_option)
        self.autocomplete = autocomplete
        self.md = md
        return mui.VBox([
            autocomplete,
            md,
        ]).prop(width="100%", justifyContent="space-between")

    async def _on_new_option(self, new_item):
        # autocomplete is a controlled component, so you need 
        # to update the options manually
        await self.md.write(f"New Item :green[{new_item}]")
        await self.autocomplete.update_options([*self.autocomplete.props.options, new_item], -1)


```