## TextField 

A text field is a component that allows the user to enter text. 

The TextField is one of the two UI components that need to use debounce. The other is the Slider. This makes TextField a component that support both controlled and uncontrolled mode.

### Basic Usage

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        md = mui.Markdown()
        md2 = mui.Markdown()

        return mui.VBox([
            mui.TextField("Text Field Label", callback=lambda x: md.write(f"TextField Content: :green[{x}]")),
            mui.Input("Input Label", callback=lambda x: md2.write(f"Input Content: :red[{x}]")),
            md,
            md2,
        ])

```

#### Variants

```mui.Input``` is a basic version of TextField. It has no variant.

```mui.TextField``` support three variants: ```filled```, ```outlined```, ```standard```.

* ```mui.TextField```

```Python

from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        return mui.VBox([
            mui.TextField("filled").prop(variant="filled"),
            mui.TextField("outlined").prop(variant="outlined"),
            mui.TextField("standard").prop(variant="standard"),
        ])
```

#### margin

```mui.TextField``` support three margin: ```dense```, ```normal```, ```none```.

```mui.Input``` support two margin: ```dense```, ```none```.

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        return mui.HBox([
            mui.VBox([
                mui.TextField("dense", init="init").prop(variant="outlined", muiMargin="dense"),
                mui.TextField("dense", init="init").prop(variant="outlined", muiMargin="dense"),
                mui.TextField("dense", init="init").prop(variant="outlined", muiMargin="dense"),
            ]),
            mui.VBox([
                mui.TextField("normal", init="init").prop(variant="outlined", muiMargin="normal"),
                mui.TextField("normal", init="init").prop(variant="outlined", muiMargin="normal"),
                mui.TextField("normal", init="init").prop(variant="outlined", muiMargin="normal"),
            ]),
            mui.VBox([
                mui.TextField("none", init="init").prop(variant="outlined", muiMargin="none"),
                mui.TextField("none", init="init").prop(variant="outlined", muiMargin="none"),
                mui.TextField("none", init="init").prop(variant="outlined", muiMargin="none"),
            ]),
        ])

```

we can see that the ```none``` margin will cause some problems on ```outlined``` variant.

#### multiline

both ```mui.TextField``` and ```mui.Input``` support ```multiline```.

**WARNING** when you set ```multiline``` and remain ```rows``` undefined, you must ensure ```mui.TextField``` is inside a fixed-height container, otherwise it will cause some problems.


```Python

from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        return mui.VBox([
            mui.TextField("multiline auto resize").prop(multiline=True),
            mui.Input("multiline fixed rows").prop(multiline=True, rows=5),
        ])

```

### Controlled and Uncontrolled

* controlled: state is stored in prop. when user change the prop, the input will be updated.

* uncontrolled: state is stored in frontend component (browser). when user change the prop, the prop won't be updated.

```mui.TextField``` and ```mui.Input``` are components that support both controlled and uncontrolled mode. the default is controlled mode. you can set ```value=mui.undefined``` to enable uncontrolled mode **BEFORE** start app. 

**WARNING**: don't switch between controlled and uncontrolled mode after app started! Otherwise the behavior is undefined.

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        input_for_test =  mui.TextField("Type Here", callback=self._modify_inp_value)

        ctrled_inp =  mui.TextField("controlled")
        unctrled_inp = mui.TextField("uncontrolled").prop(value=mui.undefined)
        self.ctrled_inp = ctrled_inp
        self.unctrled_inp = unctrled_inp
        return mui.VBox([
            input_for_test,
            mui.Divider("horizontal"),
            ctrled_inp,
            unctrled_inp,
        ])
    async def _modify_inp_value(self, value: str):
        await self.ctrled_inp.send_and_wait(self.ctrled_inp.update_event(value=value))
        await self.unctrled_inp.send_and_wait(self.unctrled_inp.update_event(value=value))

```

If you only check example above, you may not understand why we need uncontrolled mode. There are two reasons:

* debounced input don't support controlled mode. 

* we don't want to manage state.

### debounced input

If our network is slow, we may see lag when we type in TextField. This is because we send update event to backend every time we type in TextField. To solve this problem, we can use debounced input. this will send update event to backend only when user stop typing for a while.

When we use debounce when input is controlled, the semi-controlled mode is enabled. In this mode, the change event is sent to backend only when user stop typing for a while. But the prop value update in frontend isn't debounced.

```Python

from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        md = mui.Markdown()
        md2 = mui.Markdown()
        # standard way to configure debounce (don't support throttle here)
        inp_use_configure = mui.TextField("debounced-2", callback=lambda x: md.write(f"debounced: {x}"))
        inp_use_configure.event_change.configure(debounce=500)
        return mui.VBox([
            # 500ms debounce
            # simple way (high priority) to set debounce
            mui.TextField("debounced", callback=lambda x: md.write(f"debounced: {x}")).prop(debounce=500),
            inp_use_configure,
            mui.TextField("normal", callback=lambda x: md2.write(f"normal: {x}")),
            md,
            md2,
        ])
```